package com.thunder.livesdk;

import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;

import com.thunder.livesdk.helper.ThunderNative;
import com.thunder.livesdk.log.ThunderLog;
import com.yy.audioengine.MainHandler;

/**
 * An interface for playing audio files. One ThunderAudioFilePlayer instance should be created for each file to be played. 
 * Call {@link #enablePublish(boolean)} when the file serves as an accompaniment
 */
public class ThunderAudioFilePlayer implements Comparable<ThunderAudioFilePlayer> {

    private Object mCallbackLock = new Object();
    private IThunderAudioFilePlayerCallback mCallback = null;
    private IThunderAudioFilePlayerEventCallback mEventCallback = null;
    private long nativeCtx = 0;
    private boolean mIsDestroy = false;
    private String mFilePath = null;
    private boolean mIsMixStandard = false;


    @Deprecated
    private final int PLAY_EVENT_PLAYING = 2;


    public static final int AUDIO_PLAY_EVENT_OPEN = 1;
    public static final int AUDIO_PLAY_EVENT_PLAY = 2;
    public static final int AUDIO_PLAY_EVENT_STOP = 3;
    public static final int AUDIO_PLAY_EVENT_PAUSE =4;
    public static final int AUDIO_PLAY_EVENT_RESUME = 5;
    public static final int AUDIO_PLAY_EVENT_END = 6;
    public static final int AUDIO_PLAY_EVENT_SEEK_COMPLETE = 7;

    //errorCode
    public static final int AUDIO_PLAYER_STATUS_SUCCESS = 0;
    public static final int AUDIO_PLAYER_CREATE_FILE_DECODER_FAILED = -1;
    public static final int AUDIO_PLAYER_OPEN_FILE_DECODER_FAILED = -2;
    public static final int AUDIO_PLAYER_OPEN_FILE_FORMAT_NOT_SUPPORT = -3;
    public static final int AUDIO_PLAYER_OPEN_FILE_PATH_ERROR = -4;

    @Deprecated
    private final int AUDIO_PLAY_EVENT_VOLUME = 8;
    @Deprecated
    private final int AUDIO_PLAY_EVENT_ERROR = 9;

    private HandlerThread mPlayerQueueThread;
    private Handler mPlayerQueueHandler;

    public ThunderAudioFilePlayer() {
        nativeCtx = ThunderNative.createAudioFilePlayer(this);
        mPlayerQueueThread = new HandlerThread("PlayerQueueThread");
        mPlayerQueueThread.start();
        mPlayerQueueHandler = new Handler(mPlayerQueueThread.getLooper());
    }

    /**
     * @deprecated use {@link IThunderAudioFilePlayerEventCallback} instead
     * A callback interface of playing events
     */
    @Deprecated
    public interface IThunderAudioFilePlayerCallback {
        /**
         * @deprecated use {@link IThunderAudioFilePlayerEventCallback#onAudioFileStateChange(int, int)} instead
         * end playing
         */
        void onAudioFilePlayEnd();

        /**
         * Callback of playing error of files
         * @deprecated
         * @param errorCode = 0:  Opening file succeeded
         *                  errorCode = -1: Error in parsing file format
         *                  errorCode = -2: Error in decoding file format 
         *                  errorCode = -3: Unsupported file format
         *                  errorCode = -4: No file path exists
         */
        void onAudioFilePlayError(int errorCode);

        /**
         * @deprecated use {@link IThunderAudioFilePlayerEventCallback#onAudioFileStateChange(int, int)} instead
         * Start playing
         */
        void onAudioFilePlaying();

        /**
         * @deprecated use {@link IThunderAudioFilePlayerEventCallback#onAudioFileStateChange(int, int)} instead
         * Pause playing
         */
        void onAudioFilePause();

        /**
         * @deprecated use {@link IThunderAudioFilePlayerEventCallback#onAudioFileStateChange(int, int)} instead
         * Resume playing
         */
        void onAudioFileResume();

        /**
         * @deprecated use {@link IThunderAudioFilePlayerEventCallback#onAudioFileStateChange(int, int)} instead
         * Callback on playback being stopped by user
         */
        void onAudioFileStop();

        /**
         * @deprecated use {@link IThunderAudioFilePlayerEventCallback#onAudioFileStateChange(int, int)} instead
         * Fast forward to the specified time after completion
         * @param millisecond actual fast forwarding progress
         */
        void onAudioFileSeekComplete(int millisecond);


        /**
         * 播放音量回调
         * @deprecated use {@link IThunderAudioFilePlayerEventCallback#onAudioFileVolume(long, long, long)} instead
         * @param volume    ranges[0-100]
         * @param currentMs current playing progress（in millisecond）
         * @param totalMs   file totoal length（in millisecond）
         */
        void onAudioFileVolume(long volume, long currentMs, long totalMs);
    }

    public synchronized void setPlayerEventCallback(IThunderAudioFilePlayerEventCallback callback) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setPlayerEventCallback");
        if (mIsDestroy) {
            return;
        }
        synchronized (mCallbackLock) {
            mEventCallback = callback;
        }
    }

    /**
     * @deprecated use {@link #setPlayerEventCallback} instead
     * Set a playback callback interface
     * @param callback Callback interface
     */
    @Deprecated
    public synchronized void setPlayerNotify(IThunderAudioFilePlayerCallback callback) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setPlayerNotify");

        if (mIsDestroy) {
            return;
        }

        synchronized (mCallbackLock) {
            mCallback = callback;
        }
    }

    /**
     * Open the callback on volume of playing files
     *
     * @param interval Callback time interval (unit: millisecond) recommended to be more than 200ms, and reset to be 200ms in case of <=0;
     */
    public synchronized void enableVolumeIndication(boolean enable, int interval) {
        ThunderLog.release(ThunderLog.kLogTagCall,
                "ThunderAudioFilePlayer enableVolumeNotify enable = %b, interval=%d ",
                enable, interval);

        if (mIsDestroy) {
            ThunderLog.release(ThunderLog.kLogTagCall, "chorus: enableVolumeIndication mIsDestroy is null");
            return;
        }

        if (interval <= 0) {
            interval = 200;
        }

        ThunderNative.enableAudioFileVolumeCallback(nativeCtx, enable, interval);
    }

    /**
     * Set whether the accompaniment is a standard stream for video mixing
     * @param standard true: Standard stream false: Non-standard stream false by default
     */
    public synchronized void setMixStandard(boolean standard)
    {
        String oval = String.valueOf(standard);
        ThunderNative.makeBehaviorEvent("af_player", "setMixStandard", oval, 2);
        mIsMixStandard = standard;
        if(mIsMixStandard)
        {
            ThunderNative.enableAudioFileVolumeCallback(nativeCtx, true, 1000);
        }
    }

    /**
     * Query whether the accompaniment is a standard stream for video mixing.
     * @return boolean, true: Standard stream false: Non-standard stream
     */
    public synchronized boolean isMixStandard()
    {
        return mIsMixStandard;
    }

    /**
     * Open a file to be played, supporting mp3, aac and wav formats
     * <br>
     *
     * @param path File path
     * @return Succeeded or failed
     */
    public void open(final String path) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer open path = %s", path);
        mFilePath = path;
        if (mIsDestroy || path == null) {
            return;
        }

        if (mPlayerQueueHandler != null) {
            mPlayerQueueHandler.post(new Runnable() {
                @Override
                public void run() {
                    ThunderNative.audioFileOpen(nativeCtx, mFilePath);
                }
            });
        }
    }

    /**
     * Close play file, and {@link #open(String)} can be called again.
     */
    public void close() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer close");

        if (mIsDestroy || mFilePath == null) {
            return;
        }

        if (mPlayerQueueHandler != null) {
            mPlayerQueueHandler.post(new Runnable() {
                @Override
                public void run() {
                    ThunderNative.audioFileClose(nativeCtx);
                }
            });
        }
    }

    /**
     * Start playing
     */
    public void play() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer play");

        if (mIsDestroy || mFilePath == null) {
            return;
        }

        if (mPlayerQueueHandler != null) {
            mPlayerQueueHandler.post(new Runnable() {
                @Override
                public void run() {
                    ThunderNative.audioFilePlay(nativeCtx);
                }
            });
        }
    }

    /**
     * Stop playing, and call {@link #play()} to play again.
     */
    public void stop() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer stop");

        if (mIsDestroy || mFilePath == null) {
            return;
        }

        if (mPlayerQueueHandler != null) {
            mPlayerQueueHandler.post(new Runnable() {
                @Override
                public void run() {
                    ThunderNative.audioFileStop(nativeCtx);
                }
            });
        }
    }

    /**
     * Suspend playing, and call {@link #resume()} to continue.
     */
    public void pause() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer pause");

        if (mIsDestroy) {
            return;
        }

        if (mPlayerQueueHandler != null) {
            mPlayerQueueHandler.post(new Runnable() {
                @Override
                public void run() {
                    ThunderNative.audioFilePause(nativeCtx);
                }
            });
        }
    }

    /**
     * Continue to play
     */
    public void resume() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer resume");
        if (mIsDestroy) {
            return;
        }

        if (mPlayerQueueHandler != null) {
            mPlayerQueueHandler.post(new Runnable() {
                @Override
                public void run() {
                    ThunderNative.audioFileResume(nativeCtx);
                }
            });
        }
    }

    /**
     * Jump to the specified play time
     *
     * @param timeMS A point-in-time to be jumped (unit: millisecond), which should be no more than a value obtained by {@link #getTotalPlayTimeMS()}
     */
    public void seek(final long timeMS) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer seek timems = %d ", timeMS);

        if (mIsDestroy || mFilePath == null) {
            return;
        }

        if (mPlayerQueueHandler != null) {
            mPlayerQueueHandler.post(new Runnable() {
                @Override
                public void run() {
                    ThunderNative.audioFileSeek(nativeCtx, timeMS);
                }
            });
        }
    }

    /**
     * Get the total play time of file
     *
     * @return total play time, unit: millisecond
     */
    public long getTotalPlayTimeMS() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer getTotalPlayTimeMS ");

        if (mIsDestroy) {
            return 0;
        }
        return ThunderNative.audioFileGetTotalTime(nativeCtx);
    }

    /**
     * Get current elapsed time
     *
     * @return timeMS Elapsed time (unit: millisecond) which should be no more than a value obtained by {@link #getTotalPlayTimeMS()}
     */
    public long getCurrentPlayTimeMS() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer getCurrentPlayTimeMS ");

        if (mIsDestroy) {
            return 0;
        }
        return ThunderNative.audioFileGetCurrentPlayTime(nativeCtx);
    }

    /**
     * Set a playing volume of current file
     *
     * @param volume Playback volume [0-100]
     */
    public void setPlayVolume(int volume) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setPlayVolume volume = %d", volume);

        if (mIsDestroy) {
            return;
        }

        if (volume < 0) {
            volume = 0;
        } else if (volume > 100) {
            volume = 100;
        }
        ThunderNative.audioFileSetPlayVolume(nativeCtx, volume);
    }

    /**
     * Adjust volume of music file in mixed video being played locally. Please call this method in the channel
     *
     * @param volume Playing volume [0-100]
     * @return 0: succeeded, <0: Failed
     */
    public int setPlayerLocalVolume(int volume) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setPlayerLocalVolume volume = %d", volume);

        if (mIsDestroy) {
            return -1;
        }

        return ThunderNative.audioFileSetPlayerLocalVolume(nativeCtx, volume);
    }

    /**
     * Adjust volume of music file in mixed video being played remotely. Please call this method in the channel
     *
     * @param volume Playing volume [0-100]
     * @return 0: succeeded, <0: Failed
     */
    public int setPlayerPublishVolume(int volume) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setPlayerPublishVolume volume = %d", volume);

        if (mIsDestroy ) {
            return -1;
        }

        return ThunderNative.audioFileSetPlayerPublishVolume(nativeCtx, volume);
    }

    /**
     * Get the volume of music file played locally
     *
     * @return Succeeded call of method: return the volume value in [0-100]; <0: Method call failed
     */
    public int getPlayerLocalVolume() {
        return (int) ThunderNative.audioFileGetPlayerLocalVolume(nativeCtx);
    }

    /**
     * Get the volume of music file played remotely
     *
     * @return Succeeded call of method: return the volume value in [0-100]; <0: Method call failed
     */
    public int getPlayerPublishVolume() {
        return (int) ThunderNative.audioFileGetPlayerPublishVolume(nativeCtx);
    }

    /**
     * Get the number of audio tracks
     */
    public int getAudioTrackCount() {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer getAudioTrackCount ");

        if (mIsDestroy) {
            return 0;
        }
        return (int) ThunderNative.audioFileGetAudioTrackCount(nativeCtx);
    }

    /**
     * Select an audio track
     */
    public int selectAudioTrack(int audioTrack) {
        ThunderLog.release(ThunderLog.kLogTagCall,
                "ThunderAudioFilePlayer selectAudioTrack audioTrack = %d", audioTrack);

        if (mIsDestroy) {
            return -1;
        }
        int trackCount = getAudioTrackCount();
        if (trackCount == 0 || audioTrack < 0) {
            return -1;
        }

        if (audioTrack >= trackCount) {
            audioTrack = trackCount - 1;
        }
        long ret = ThunderNative.audioFileSelectAudioTrack(nativeCtx, audioTrack);
        return ret == 1 ? 0 : -1;
    }

    /**
     * Set a tone for audio play
     *
     * @param val -5, -4, -3, -2, -1, 0 (normal), 1, 2, 3, 4, 5
     */
    public void setSemitone(int val) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setSemitone val = %d", val);

        if (mIsDestroy) {
            return;
        }

        if (val < -5) {
            val = -5;
        } else if (val > 5) {
            val = 5;
        }

        float tone = val;
        ThunderNative.audioFileSetSemitone(nativeCtx, tone);
    }

    /**
     * Set audio playback speed
     * @param tempo ranges [0.5f, 0.75f, 1.0f, 1.25f, 1.5f, 2.0f]
     */
    public void setTempo(float tempo) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setTempo tempo=%f", tempo);

        if (tempo < 0.5) {
            tempo = 0.5f;
        } else if (tempo > 2.0f) {
            tempo = 2.0f;
        }

        ThunderNative.audioFileSetTempo(nativeCtx, tempo);
    }

    /**
     * Set the position for audio play
     * @param azimuth ranges [-90~90]
     */
    public void setPosition(int azimuth) {
        ThunderLog.release(ThunderLog.kLogTagCall,
                "ThunderAudioFilePlayer setTempo:(azimuth=%d)", azimuth);

        if(azimuth < -90) {
            azimuth = -90;
        } else if(azimuth > 90) {
            azimuth = 90;
        }
        ThunderNative.audioFileSetPosition(nativeCtx, azimuth, 0);
    }

    /**
     * Set the number of loop playback
     *
     * @return 0: succeeded call of method; <0: failed call of method
     * @cycle Positive integer: the number of loop; 0: loop canceled; -1: infinite loop
     */
    public int setLooping(int cycle) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer setLooping cycle = %d", cycle);

        if (mIsDestroy) {
            return -1;
        }

        return (int) ThunderNative.audioFileSetLooping(nativeCtx, cycle);
    }

    /**
     * Whether to set the current playing file as a live streaming accompaniment
     * <br>
     *
     * @param enable Enabled or disabled, disabled by default
     */
    public void enablePublish(boolean enable) {
        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer enablePublish: enable=%b", enable);

        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFileEnablePublish(nativeCtx, enable);
    }

    /**
     * Notify a playback event
     */
    public void onPlayEvent(final int event, final int errorCode) {
        ThunderLog.release(ThunderLog.kLogTagCall,
                "ThunderAudioFilePlayer onPlayEvent event = %d, errorCode = %d", event, errorCode);

        MainHandler.getInstance().post(new Runnable() {
            @Override
            public void run() {
                if (mIsDestroy) {
                    return;
                }
                synchronized (mCallbackLock) {
                    // 这里需要兼容老的 callback
                    if (mCallback != null) {
                        switch (event) {
                            case AUDIO_PLAY_EVENT_END:
                                mCallback.onAudioFilePlayEnd();
                                break;
                            case AUDIO_PLAY_EVENT_ERROR:
                            case AUDIO_PLAY_EVENT_OPEN :
                                mCallback.onAudioFilePlayError(errorCode);
                                break;
                            case AUDIO_PLAY_EVENT_PLAY:
                                if(errorCode != 0) {
                                    mCallback.onAudioFilePlayError(errorCode);
                                } else {
                                    mCallback.onAudioFilePlaying();
                                }
                                break;
                            case AUDIO_PLAY_EVENT_PAUSE:
                                mCallback.onAudioFilePause();
                                break;
                            case AUDIO_PLAY_EVENT_RESUME:
                                mCallback.onAudioFileResume();
                                break;
                            case AUDIO_PLAY_EVENT_STOP:
                                mCallback.onAudioFileStop();
                                break;
                            case AUDIO_PLAY_EVENT_SEEK_COMPLETE:
                                mCallback.onAudioFileSeekComplete(errorCode);
                                break;
                            default:
                                break;
                        }
                    } else if (mEventCallback != null) {
                        mEventCallback.onAudioFileStateChange(event, errorCode);
                    }
                }
            }
        });
    }

    /*

     */
    public synchronized void onAudioFileVolume(final long volume, final long currentMs, final long totalMs) {
        MainHandler.getInstance().post(new Runnable() {
            @Override
            public void run() {
                if (mIsDestroy) {
                    ThunderLog.release(ThunderLog.kLogTagCall, " chorus: onAudioFileVolume mIsDestroy is false");
                    return;
                }
                synchronized (mCallbackLock) {
                    // 需要兼容原来的旧接口
                    if (mCallback != null) {
                        ThunderLog.release(ThunderLog.kLogTagCall, " chorus: volume[%d], currentMs[%d], totalMs[%d]."
                                , volume, currentMs, totalMs);
                        mCallback.onAudioFileVolume(volume, currentMs, totalMs);
                        if( isMixStandard())
                        {
                            ThunderNative.sendAudioFilePlayerInfo((int)volume, (int)currentMs, (int)totalMs);
                        }
                    } else if (mEventCallback != null) {
                        ThunderLog.release(ThunderLog.kLogTagCall, " evt chorus: volume[%d], currentMs[%d], totalMs[%d]."
                                , volume, currentMs, totalMs);
                        mEventCallback.onAudioFileVolume(volume, currentMs, totalMs);
                        if( isMixStandard())
                        {
                            ThunderNative.sendAudioFilePlayerInfo((int)volume, (int)currentMs, (int)totalMs);
                        }
                    } else {
                        ThunderLog.release(ThunderLog.kLogTagCall, " chorus: onAudioFileVolume mIsDestroy is null");
                    }
                }
            }
        });
    }

    /*
     * Destroy this class, and do not allow other operations and package access permission
     */
    public void destroyAudioFilePlayer() {

        ThunderLog.release(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer destroyAudioFilePlayer");

        try {
            if (mPlayerQueueHandler != null) {
                mPlayerQueueHandler.removeCallbacksAndMessages(null);
                if (Build.VERSION.SDK_INT >= 18) {
                    mPlayerQueueThread.quitSafely();
                } else {
                    mPlayerQueueThread.quit();
                }
                mPlayerQueueThread.join();
                mPlayerQueueHandler = null;
                mPlayerQueueThread = null;
            }
        } catch (Exception e) {
            ThunderLog.error(ThunderLog.kLogTagCall, "ThunderAudioFilePlayer destroyAudioFilePlayer, exception: " + e.getMessage());
        }
        setPlayerNotify(null);
        setPlayerEventCallback(null);
        ThunderNative.destroyAudioFilePlayer(nativeCtx);
        mIsDestroy = true;
        nativeCtx = 0;
    }

    protected void finalize() {
        destroyAudioFilePlayer();
    }

    @Override
    public int compareTo(ThunderAudioFilePlayer audioFilePlayer) {
        if (audioFilePlayer == null) {
            throw new NullPointerException();
        }

        if (this == audioFilePlayer) {
            return 0;
        }
        return 1;
    }
}
