package com.thunder.livesdk;

import java.util.ArrayList;
import java.util.Iterator;

import static com.thunder.livesdk.ThunderRtcConstant.TranscodingWatermarkCountType.TRANSCODING_WATERMARK_IMAGE_MAX;
import static com.thunder.livesdk.ThunderRtcConstant.TranscodingWatermarkCountType.TRANSCODING_WATERMARK_TEXT_MAX;

public class LiveTranscoding {
    public LiveTranscoding() {
      mBackgroundColor = 0x000000;
    }
    // Transcoding and video mixing brackets: The sdk interface in the configuration center provides the video mixing bracket, and sdk gets output coding parameters of transcoding and video mixing from the configuration center by means of mTransCodingMode.
    private int mTransCodingMode; // refer to LiveTranscodingMode
    // Other parameters of video mixing, such as audio and video accompany address, lyrics address, etc.
    private String mAudioUrl = ""; // URL of the standard audio file; support audio files in Mp3 format
    private String mLyricUrl = ""; // URL of lyric in LRC format, and the lyric file should be encoded in UTF-8 format.
                                   // If this field is not configured, the audio will be played without lyrics;
                                   // Otherwise, the lyrics will be added to the mixed video in the default style.
    private String mMediaUrl = ""; // Video file url as standard
    private MediaStreamLayout mMediaStreamLayout; // Layout of video standard stream in video mixing.

    public String getAudioUrl() { return mAudioUrl; }
    public String getLyricUrl() { return mLyricUrl; }
    public String getMediaUrl() { return mMediaUrl; }
    public MediaStreamLayout getMediaStreamLayout() { return mMediaStreamLayout; }

    private ArrayList<TranscodingUser> mUserList = new ArrayList<>(); // All users in video mixing

    private int mBackgroundColor; // Background color, which is a HEX color value, default value: #000000 (black)
    private TranscodingImage mBackgroundImage; // Background image, only the fields "url" and "scale" are vaild
    private TranscodingTimestamp mTimestamp; // Timestamp watermark
    private CustomTranscodingOptions mTranscodingModeOptions;
    private ArrayList<TranscodingText> mTextList = new ArrayList<>(); // Text watermark, max. number: 3
    private ArrayList<TranscodingImage> mImageList = new ArrayList<>(); // Image watermark, max. number: 4

    public void setCustomTranscodingOptions(CustomTranscodingOptions options) {
        this.mTranscodingModeOptions = options;
    }

    public CustomTranscodingOptions getCustomTranscodingOptions() {
        return mTranscodingModeOptions;
    }

    public void setBackgroundColor(int color) {
        this.mBackgroundColor = color;
    }

    public int getBackgroundColor() {
        return mBackgroundColor;
    }

    public void setBackgroundImage(TranscodingImage image) {
        this.mBackgroundImage = image;
    }

    public TranscodingImage getBackgroundImage() {
        return mBackgroundImage;
    }

    public void setTimestamp(TranscodingTimestamp mTimestamp) {
        this.mTimestamp = mTimestamp;
    }

    public TranscodingTimestamp getTimestamp() {
        return mTimestamp;
    }

    public void addTexts(ArrayList<TranscodingText> texts) {
        mTextList.clear();
        if (texts.size() > TRANSCODING_WATERMARK_TEXT_MAX)
        {
            mTextList.addAll(texts.subList(0, TRANSCODING_WATERMARK_TEXT_MAX - 1));
        } else {
            mTextList.addAll(texts);
        }
    }

    public int getTextCount() {
        return mTextList.size();
    }

    public final ArrayList<TranscodingText> getTexts() {
        return mTextList;
    }

    public void addImages(ArrayList<TranscodingImage> images) {
        mImageList.clear();
        if (images.size() > TRANSCODING_WATERMARK_IMAGE_MAX) {
            mImageList.addAll(images.subList(0, TRANSCODING_WATERMARK_IMAGE_MAX - 1));
        } else {
            mImageList.addAll(images);
        }
    }

    public int getImageCount() {
        return mImageList.size();
    }

    public final ArrayList<TranscodingImage> getImages() {
        return mImageList;
    }

    /**
     * Add one user into the video mixing layout
     *
     * @param user
     * @return
     */
    public int addUser(TranscodingUser user) {
        if (mUserList != null) {
            mUserList.add(user);
        }
        return 0;
    }

    /**
     * Set video mixing layout of users in batches
     * This method is used to set all users who participate in image integration. With this method, the new user data is used to replace the original data. 
     *
     * @param users All video mixing users in the channel
     */
    public void setUsers(ArrayList<TranscodingUser> users) {
        mUserList.clear();
        if (users != null) {
            mUserList.addAll(users);
        }
    }

    /**
     * Obtain the information about the current user location
     *
     * @return Obtain the information about all users in the channel in video mixing canvas
     */
    public final ArrayList<TranscodingUser> getUsers() {
        return mUserList;
    }

    /**
     * Delete the users who participate in video mixing 
     *
     * @param uid ID of the user to be deleted
     * @return
     */
    public int removeUser(String uid) {
        Iterator<TranscodingUser> itor = mUserList.iterator();
        while (itor.hasNext()) {
            TranscodingUser tmp = itor.next();
            if (uid.equals(tmp.uid)) {
                itor.remove();
            }
        }
        return 0;
    }

    /**
     * Delete video mixing of the channel, that is, users do not use video mixing any more.
     *
     * @return
     */
    public int removeAllUser() {
        mUserList.clear();
        return 0;
    }

    /**
     * Obtain the number of people who participated in video mixing
     *
     * @return
     */
    public int getUserCount() {
        if (mUserList == null || mUserList.isEmpty()) {
            return 0;
        } else {
            return mUserList.size();
        }
    }

    /**
     * Set the output video mixing bracket
     *
     * @param transCodingMode
     */
    public void setTransCodingMode(int transCodingMode) {
        this.mTransCodingMode = transCodingMode;
    }

    /**
     * Obtain the bracket of the output video mixing
     *
     * @return
     */
    public int getTransCodingMode() {
        return mTransCodingMode;
    }

    public static class TranscodingUser {
        public TranscodingUser() {
            uid = "";
            roomId = "";
            bStandard = false;
            layoutX = 0;
            layoutY = 0;
            layoutW = 0;
            layoutH = 0;
            zOrder = 0;
            bCrop = false;
            cropX = 0;
            cropY = 0;
            cropW = 0;
            cropH = 0;
            alpha = 1.0f;
            audioRoom = 0;
        }

        public String uid;
        public String roomId;
        public boolean bStandard; // Standard stream user or not
        public int layoutX; // User's location x in video mixing canvas
        public int layoutY; // User's location y in video mixing canvas
        public int layoutW; // User's location w in video mixing canvas
        public int layoutH; // User's location h in video mixing canvas
        public int zOrder; // Layer number of the user's video frame on the live video. The value range is the integer in [0, 100], and the minimum value is 0 (default value), indicating that the image in this region is at the lowest level
        public boolean bCrop; // How to crop the source stream when the video mixing service places it in the canvas (0: mend black edge, 1: crop)
        public int cropX; // X coordinate of crop region
        public int cropY; // Y coordinate of crop region
        public int cropW; // Width of crop region
        public int cropH; // Height of crop region
        public float alpha; // Transparency of user video on the live video. The value range is [0.0, 1.0]. 0.0 indicates that the image in this region is completely transparent, while 1.0 indicates completely opaque. The default is 1.0
        public int audioRoom; // Not yet realized

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof TranscodingUser)) {
                return false;
            }

            TranscodingUser p = (TranscodingUser) obj;
            if (this.uid.equals(p.uid) && this.roomId.equals(p.roomId)) {
                return true;
            } else {
                return false;
            }
        }

        @Override
        public int hashCode() {
            String hash = uid + roomId;
            return hash.hashCode();
        }

        public String toString() {
            return "{TranscodingUser: uid " + uid + ", roomId " + roomId + ", bStandard " + bStandard +
                    ", x " + layoutX + ", y " + layoutY + ", w " + layoutW + ", h " + layoutH +
                    ", zOrder " + zOrder + ", bCrop " + bCrop + ", cropX " + cropX + ", cropY " + cropY +
                    ", cropW " + cropW + ", cropH " + cropH + ", alpha " + alpha + ", audioRoom " + audioRoom + "}";
        }
    }

    public static class MediaStreamLayout {

        public int layoutX = 0;        //Media's location x in video mixing canvas
        public int layoutY = 0;        //Media's location y in video mixing canvas
        public int layoutW = 0;        //Media's location w in video mixing canvas
        public int layoutH = 0;        //Media's location h in video mixing canvas
        public int zOrder = 0;         //Layer number of the media's video frame on the live video. The value range is the integer in [0, 100], and the minimum value is 0 (default value), indicating that the image in this region is at the lowest level
        public boolean bCrop = true;   //How to crop the source stream when the video mixing service places it in the canvas (0: mend black edge, 1: crop)
        public int cropX = 0;          //X coordinate of crop region
        public int cropY = 0;          //Y coordinate of crop region
        public int cropW = 0;          //Width of crop region
        public int cropH = 0;          //Height of crop region
        public float alpha = 0;        //Transparency of media video on the live video. The value range is [0.0, 1.0]. 0.0 indicates that the image in this region is completely transparent, while 1.0 indicates completely opaque. The default is 1.0

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof MediaStreamLayout)) {
                return false;
            }

            MediaStreamLayout p = (MediaStreamLayout) obj;
            if (this.layoutX == p.layoutX && this.layoutY == p.layoutY) {
                return true;
            } else {
                return false;
            }
        }

        @Override
        public int hashCode() {
            String hash = String.valueOf(layoutX + layoutY + layoutW + layoutH);
            return hash.hashCode();
        }
    }
 
    /**
     * Set whether external audio URL is mixed video standard stream The play progress of the standard stream is required for video mixing synchronization internally, 
     * Therefore, the client is required to play this audio using ThunderAudioFilePlayer and to call setMixStandard in ThunderAudioFilePlayer to set the standard stream. 
     * If users want to use custom-style lyrics in mixed videos, the anchor terminal may send the lyrics progress using the sendMediaExtraInfo interface and call setAudioOnlyStandardSreamUrl to set lyricUrl to null.
     * The audience terminal will be able to receive the callback on IThunderMediaExtraInfoCallback onRecvMediaExtraInfo when playing this mixed video stream. Then, the audience terminal reads the corresponding lyrics progress and renders lyrics according to the progress.
     * @param  audioUrl URL of the benchmark audio file
     * @param   lyricUrl URL of the lyrics file corresponding to the audio; blank indicates that there is no lyrics. Otherwise, the lyrics will be added to the mixed video in the default style. 
     * @return 0: Succeeded; <0: Failed
     */
     public final int setAudioOnlyStandardSreamUrl(String audioUrl, String lyricUrl)
     {
        mAudioUrl = audioUrl;
        mLyricUrl = lyricUrl;
        mMediaUrl = "";
        mMediaStreamLayout = null;

        return 0;
     }

     /**
      * Set the external video media stream as a standard stream for video mixing; the interface setAudioOnlyStandardSreamUrl cannot be used at the same time..
      * @param mediaUrl URL of the benchmark video file
      * @param layout  Layout information about the video media stream in video mixing 
      * @return 0: Succeeded; <0: Failed
      */
    public final int setMediaStandardSream(String mediaUrl,  MediaStreamLayout layout)
    {
        mMediaUrl = mediaUrl;
        mMediaStreamLayout = layout;
        mAudioUrl = "";
        mLyricUrl = "";

        return 0;
    }

}
