package com.gamepublish.sdk.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.RectF;
import android.graphics.Shader;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.NinePatchDrawable;
import android.os.Bundle;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.widget.ImageView;

import com.gamepublish.sdk.R;
import com.gamepublish.sdk.impl.LogImpl;

/**
 * Created by lulong on 2017/7/27.
 * Email:lulong@yy.com
 */

public class RoundImageView extends ImageView {
    private static final String TAG = "RoundImageView";

    public static final int TYPE_CIRCLE = 0;
    public static final int TYPE_ROUND = 1;

    public static final int CORNER_TOP_LEFT = 1;
    public static final int CORNER_TOP_RIGHT = 1 << 1;
    public static final int CORNER_BOTTOM_LEFT = 1 << 2;
    public static final int CORNER_BOTTOM_RIGHT = 1 << 3;
    public static final int CORNER_ALL = CORNER_TOP_LEFT | CORNER_TOP_RIGHT | CORNER_BOTTOM_LEFT | CORNER_BOTTOM_RIGHT;

    /**
     * 图片的类型，圆形or圆角
     */
    private int type = TYPE_ROUND;

    /**
     * 圆角大小的默认值
     */
    private static final int BODER_RADIUS_DEFAULT = 4; // dp

    /**
     * 圆角的大小
     */
    private int mBorderRadius;

    /**
     * 圆角模式，可以指定某个角不显示圆角
     */
    private int mCornerModel = CORNER_ALL;

    /**
     * 绘图的Paint
     */
    private Paint mBitmapPaint;

    private Paint mColorPaint;

    /**
     * 圆角的半径
     */
    private int mRadius;
    /**
     * 3x3 矩阵，主要用于缩小放大
     */
    private Matrix mMatrix;
    /**
     * 渲染图像，使用图像为绘制图形着色
     */
    private BitmapShader mBitmapShader;
    /**
     * view的宽度
     */
    private int mWidth;
    private RectF mRoundRect;

    public RoundImageView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init(context, attrs);

    }

    public RoundImageView(Context context) {
        super(context);
        init(context, null);
    }

    public RoundImageView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs) {
        mMatrix = new Matrix();
        mBitmapPaint = new Paint();
        mColorPaint = new Paint();
        mColorPaint.setAntiAlias(true);
        mBitmapPaint.setAntiAlias(true);

        if (attrs != null) {
            TypedArray a = context.obtainStyledAttributes(attrs,
                    R.styleable.RoundImageView);

            mBorderRadius = a.getDimensionPixelSize(
                    R.styleable.RoundImageView_borderRadius,
                    context.getResources().getDimensionPixelSize(R.dimen.dp_4)); // 默认为4dp
            type = a.getInt(R.styleable.RoundImageView_type, TYPE_ROUND); // 默认为Circle

            boolean topLeft = a.getBoolean(R.styleable.RoundImageView_cornerTopLeft, true);
            boolean topRight = a.getBoolean(R.styleable.RoundImageView_cornerTopRight, true);
            boolean bottomLeft = a.getBoolean(R.styleable.RoundImageView_cornerBottomLeft, true);
            boolean bottomRight = a.getBoolean(R.styleable.RoundImageView_cornerBottomRight, true);
            setCornerModel(topLeft, topRight, bottomLeft, bottomRight);
            a.recycle();
        }
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);

        /**
         * 如果类型是圆形，则强制改变view的宽高一致，以小值为准
         */
        if (type == TYPE_CIRCLE) {
            mWidth = Math.min(getMeasuredWidth(), getMeasuredHeight());
            mRadius = mWidth / 2;
            setMeasuredDimension(mWidth, mWidth);
        }

    }

    /**
     * 初始化BitmapShader
     */
    private void setUpShader() {
        Drawable drawable = getDrawable();
        if (drawable == null) {
            return;
        }

        Bitmap bmp = drawableToBitmap(drawable);
        if (bmp == null || bmp.isRecycled()) {
            LogImpl.INSTANCE.e(TAG, "setUpShader bmp == null: %b" + (bmp == null));
            return;
        }
        // 将bmp作为着色器，就是在指定区域内绘制bmp
        mBitmapShader = new BitmapShader(bmp, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
        final int vwidth = getWidth() - getPaddingLeft() - getPaddingRight();
        final int vheight = getHeight() - getPaddingTop() - getPaddingBottom();
        float scale = 1.0f;
        if (type == TYPE_CIRCLE) {
            // 拿到bitmap宽或高的小值
            int bSize = Math.min(bmp.getWidth(), bmp.getHeight());
            scale = mWidth * 1.0f / bSize;

        } else if (type == TYPE_ROUND) {
            // 如果图片的宽或者高与view的宽高不匹配，计算出需要缩放的比例；缩放后的图片的宽高，一定要大于我们view的宽高；所以我们这里取大值；
            scale = Math.max(getWidth() * 1.0f / bmp.getWidth(), getHeight()
                    * 1.0f / bmp.getHeight());
        }
        // shader的变换矩阵，我们这里主要用于放大或者缩小
        if (drawable instanceof NinePatchDrawable) {

        } else {
            mMatrix.setScale(scale, scale);
            float dy = (vheight - bmp.getHeight() * scale) * mPercentY;
            float dx = (vwidth - bmp.getWidth() * scale) * mPercentX;
            mMatrix.postTranslate(dx, dy);
            // 设置变换矩阵
            mBitmapShader.setLocalMatrix(mMatrix);
        }
        // 设置shader
        mBitmapPaint.setShader(mBitmapShader);
    }

    private float mPercentX, mPercentY;

    public void setMatrixTranslatePercent(float percentX, float percentY) {
        mPercentX = percentX;
        mPercentY = percentY;
        postInvalidate();
    }

    /**
     * drawable转bitmap
     *
     * @param drawable
     * @return
     */
    private Bitmap drawableToBitmap(Drawable drawable) {
        boolean isBitmapRecycled = false;
        if (drawable instanceof BitmapDrawable) {
            BitmapDrawable bd = (BitmapDrawable) drawable;
            if (bd.getBitmap() != null && !bd.getBitmap().isRecycled()) {
                return bd.getBitmap();
            } else {
                // 标记是bitmap是空的，或已经被回收
                isBitmapRecycled = true;
            }
        }
        int w = 0;
        if (drawable instanceof NinePatchDrawable) {
            w = getWidth();
        } else {
            w = drawable.getIntrinsicWidth();
        }
        int h = 0;
        if (drawable instanceof NinePatchDrawable) {
            h = getHeight();
        } else {
            h = drawable.getIntrinsicHeight();
        }

        if (w <= 0) {
            w = getMeasuredWidth();
        }
        if (h <= 0) {
            h = getMeasuredHeight();
        }
        if (w <= 0 || h <= 0) {
            return null;
        }

        Bitmap bitmap = Bitmap.createBitmap(w, h, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        drawable.setBounds(0, 0, w, h);
        if (isBitmapRecycled) {
            // 如果标记空或被回收，则直接返回bitmpa，不应该draw，否则会抛异常
            return bitmap;
        }
        drawable.draw(canvas);
        return bitmap;
    }

    private int mLoadingColor = Color.TRANSPARENT;
    private boolean mHadUseLoadingColor = false;

    public void setLoadingColor(int color) {
        mHadUseLoadingColor = true;
        mLoadingColor = color;
        invalidate();
    }

    @Override
    public void setImageDrawable(Drawable drawable) {
        mHadUseLoadingColor = false;
        super.setImageDrawable(drawable);
    }

    @Override
    public void setImageBitmap(Bitmap bm) {
        mHadUseLoadingColor = false;
        super.setImageBitmap(bm);
    }

    @Override
    public void setImageResource(int resId) {
        mHadUseLoadingColor = false;
        super.setImageResource(resId);
    }

    private Paint mPaint;

    @Override
    protected void onDraw(Canvas canvas) {
        if (mHadUseLoadingColor || getDrawable() == null) {
            mColorPaint.setColor(mLoadingColor);
            mPaint = mColorPaint;
        } else if (getDrawable() instanceof ColorDrawable) {
            int color = ((ColorDrawable) getDrawable()).getColor();
            mColorPaint.setColor(color);
            mPaint = mColorPaint;
        } else {
            setUpShader();
            mPaint = mBitmapPaint;
        }
        if (type == TYPE_ROUND && mRoundRect != null) {
            canvas.drawRoundRect(mRoundRect, mBorderRadius, mBorderRadius,
                    mPaint);
            drawerCorner(canvas, mRoundRect, mCornerModel, mBorderRadius, mPaint);
        } else {
            canvas.drawCircle(mRadius, mRadius, mRadius, mPaint);
        }
    }

    private void drawerCorner(Canvas canvas, RectF rect, int corners, int cornerRadius, Paint paint) {
        int notRoundedCorners = corners ^ CORNER_ALL;
        //哪个角不是圆角我再把你用矩形画出来
        if ((notRoundedCorners & CORNER_TOP_LEFT) != 0) {
            canvas.drawRect(0, 0, cornerRadius, cornerRadius, paint);
        }
        if ((notRoundedCorners & CORNER_TOP_RIGHT) != 0) {
            canvas.drawRect(rect.right - cornerRadius, 0, rect.right, cornerRadius, paint);
        }
        if ((notRoundedCorners & CORNER_BOTTOM_LEFT) != 0) {
            canvas.drawRect(0, rect.bottom - cornerRadius, cornerRadius, rect.bottom, paint);
        }
        if ((notRoundedCorners & CORNER_BOTTOM_RIGHT) != 0) {
            canvas.drawRect(rect.right - cornerRadius, rect.bottom - cornerRadius, rect.right, rect.bottom, paint);
        }
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        // 圆角图片的范围
        if (type == TYPE_ROUND) {
            mRoundRect = new RectF(0, 0, getWidth(), getHeight());
        }
    }

    private static final String STATE_INSTANCE = "state_instance";
    private static final String STATE_TYPE = "state_type";
    private static final String STATE_BORDER_RADIUS = "state_border_radius";

    @Override
    protected Parcelable onSaveInstanceState() {
        Bundle bundle = new Bundle();
        bundle.putParcelable(STATE_INSTANCE, super.onSaveInstanceState());
        bundle.putInt(STATE_TYPE, type);
        bundle.putInt(STATE_BORDER_RADIUS, mBorderRadius);
        return bundle;
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        if (state instanceof Bundle) {
            Bundle bundle = (Bundle) state;
            super.onRestoreInstanceState(((Bundle) state)
                    .getParcelable(STATE_INSTANCE));
            this.type = bundle.getInt(STATE_TYPE);
            this.mBorderRadius = bundle.getInt(STATE_BORDER_RADIUS);
        } else {
            super.onRestoreInstanceState(state);
        }

    }

    /**
     * 传递dp
     *
     * @param borderRadius
     */
    public void setBorderRadius(int borderRadius) {
        // int pxVal = ResolutionUtils.dip2Px(borderRadius);
        if (this.mBorderRadius != borderRadius) {
            this.mBorderRadius = borderRadius;
            invalidate();
        }
    }

    /**
     * 传递px
     *
     * @param borderRadius
     */
    public void setBorderPxRadius(int borderRadius) {
        int pxVal = borderRadius;
        if (this.mBorderRadius != pxVal) {
            this.mBorderRadius = pxVal;
            invalidate();
        }
    }

    public void setType(int type) {
        if (this.type != type) {
            this.type = type;
            if (this.type != TYPE_ROUND && this.type != TYPE_CIRCLE) {
                this.type = TYPE_CIRCLE;
            }
            requestLayout();
        }
    }

    public void setCornerModel(boolean topLeft, boolean topRight, boolean bottomLeft, boolean bottomRight) {

        int corner =
                convert(topLeft) | (convert(topRight) << 1) | (convert(bottomLeft) << 2) | (convert(bottomRight) << 3);
        if (mCornerModel != corner) {
            mCornerModel = corner;
            requestLayout();
        }
    }

    private int convert(boolean b) {
        return b ? 1 : 0;
    }
}