package com.hyphenate.util;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.provider.DocumentsContract;
import android.text.TextUtils;
import android.util.Log;
import android.webkit.MimeTypeMap;

import java.io.File;

/**
 * This is a inner class, based on {@link androidx.documentfile.provider.DocumentFile}
 */
abstract class DocumentFile {
    static final String TAG = "DocumentFile";
    private final DocumentFile mParent;
    
    DocumentFile(DocumentFile parent) {
        mParent = parent;
    }


    /**
     * Create a {@link DocumentFile} representing the filesystem tree rooted at
     * the given {@link File}. This doesn't give you any additional access to the
     * underlying files beyond what your app already has.
     * <p>
     * {@link #getUri()} will return {@code file://} Uris for files explored
     * through this tree.
     */
    public static DocumentFile fromFile(File file) {
        if(file == null) {
            return null;
        }
        return new RawDocumentFile(null, file);
    }


    /**
     * Create a {@link DocumentFile} representing the single document at the
     * given {@link Uri}. This is only useful on devices running
     * {@link android.os.Build.VERSION_CODES#KITKAT} or later, and will return
     * {@code null} when called on earlier platform versions.
     *
     * @param singleUri the {@link Intent#getData()} from a successful
     *            {@link Intent#ACTION_OPEN_DOCUMENT} or
     *            {@link Intent#ACTION_CREATE_DOCUMENT} request.
     */
    public static DocumentFile fromSingleUri(Context context, Uri singleUri) {
        if (Build.VERSION.SDK_INT >= 19) {
            return new SingleDocumentFile(null, context, singleUri);
        } else {
            return null;
        }
    }


    /**
     * Test if given Uri is backed by a
     * {@link android.provider.DocumentsProvider}.
     */
    public static boolean isDocumentUri(Context context, Uri uri) {
        if (Build.VERSION.SDK_INT >= 19) {
            return DocumentsContract.isDocumentUri(context, uri);
        } else {
            return false;
        }
    }

    /**
     * Return a Uri for the underlying document represented by this file. This
     * can be used with other platform APIs to manipulate or share the
     * underlying content. You can use {@link #isDocumentUri(Context, Uri)} to
     * test if the returned Uri is backed by a
     * {@link android.provider.DocumentsProvider}.
     *
     * @see Intent#setData(Uri)
     * @see Intent#setClipData(android.content.ClipData)
     * @see ContentResolver#openInputStream(Uri)
     * @see ContentResolver#openOutputStream(Uri)
     * @see ContentResolver#openFileDescriptor(Uri, String)
     */
    public abstract Uri getUri();

    /**
     * Return the display name of this document.
     *
     * @see android.provider.DocumentsContract.Document#COLUMN_DISPLAY_NAME
     */
    public abstract String getName();

    /**
     * Return the MIME type of this document.
     *
     * @see android.provider.DocumentsContract.Document#COLUMN_MIME_TYPE
     */
    public abstract String getType();

    /**
     * Return the parent file of this document. Only defined inside of the
     * user-selected tree; you can never escape above the top of the tree.
     * <p>
     * The underlying {@link android.provider.DocumentsProvider} only defines a
     * forward mapping from parent to child, so the reverse mapping of child to
     * parent offered here is purely a convenience method, and it may be
     * incorrect if the underlying tree structure changes.
     */
    public DocumentFile getParentFile() {
        return mParent;
    }

    /**
     * Indicates if this file represents a <em>directory</em>.
     *
     * @return {@code true} if this file is a directory, {@code false}
     *         otherwise.
     * @see android.provider.DocumentsContract.Document#MIME_TYPE_DIR
     */
    public abstract boolean isDirectory();

    /**
     * Indicates if this file represents a <em>file</em>.
     *
     * @return {@code true} if this file is a file, {@code false} otherwise.
     * @see android.provider.DocumentsContract.Document#COLUMN_MIME_TYPE
     */
    public abstract boolean isFile();

    /**
     * Indicates if this file represents a <em>virtual</em> document.
     *
     * @return {@code true} if this file is a virtual document.
     * @see android.provider.DocumentsContract.Document#FLAG_VIRTUAL_DOCUMENT
     */
    public abstract boolean isVirtual();

    /**
     * Returns the time when this file was last modified, measured in
     * milliseconds since January 1st, 1970, midnight. Returns 0 if the file
     * does not exist, or if the modified time is unknown.
     *
     * @return the time when this file was last modified.
     * @see android.provider.DocumentsContract.Document#COLUMN_LAST_MODIFIED
     */
    public abstract long lastModified();

    /**
     * Returns the length of this file in bytes. Returns 0 if the file does not
     * exist, or if the length is unknown. The result for a directory is not
     * defined.
     *
     * @return the number of bytes in this file.
     * @see android.provider.DocumentsContract.Document#COLUMN_SIZE
     */
    public abstract long length();

    /**
     * Indicates whether the current context is allowed to read from this file.
     *
     * @return {@code true} if this file can be read, {@code false} otherwise.
     */
    public abstract boolean canRead();

    /**
     * Indicates whether the current context is allowed to write to this file.
     *
     * @return {@code true} if this file can be written, {@code false}
     *         otherwise.
     * @see android.provider.DocumentsContract.Document#COLUMN_FLAGS
     * @see android.provider.DocumentsContract.Document#FLAG_SUPPORTS_DELETE
     * @see android.provider.DocumentsContract.Document#FLAG_SUPPORTS_WRITE
     * @see android.provider.DocumentsContract.Document#FLAG_DIR_SUPPORTS_CREATE
     */
    public abstract boolean canWrite();

    /**
     * Deletes this file.
     * <p>
     * Note that this method does <i>not</i> throw {@code IOException} on
     * failure. Callers must check the return value.
     *
     * @return {@code true} if this file was deleted, {@code false} otherwise.
     * @see android.provider.DocumentsContract#deleteDocument(ContentResolver,
     *      Uri)
     */
    public abstract boolean delete();

    /**
     * Returns a boolean indicating whether this file can be found.
     *
     * @return {@code true} if this file exists, {@code false} otherwise.
     */
    public abstract boolean exists();

    static class SingleDocumentFile extends DocumentFile {
        private Context mContext;
        private Uri mUri;

        SingleDocumentFile( DocumentFile parent, Context context, Uri uri) {
            super(parent);
            mContext = context;
            mUri = uri;
        }

        @Override
        public Uri getUri() {
            return mUri;
        }

        @Override
        
        public String getName() {
            return DocumentsContractApi19.getName(mContext, mUri);
        }

        @Override
        
        public String getType() {
            return DocumentsContractApi19.getType(mContext, mUri);
        }

        @Override
        public boolean isDirectory() {
            return DocumentsContractApi19.isDirectory(mContext, mUri);
        }

        @Override
        public boolean isFile() {
            return DocumentsContractApi19.isFile(mContext, mUri);
        }

        @Override
        public boolean isVirtual() {
            return DocumentsContractApi19.isVirtual(mContext, mUri);
        }

        @Override
        public long lastModified() {
            return DocumentsContractApi19.lastModified(mContext, mUri);
        }

        @Override
        public long length() {
            return DocumentsContractApi19.length(mContext, mUri);
        }

        @Override
        public boolean canRead() {
            return DocumentsContractApi19.canRead(mContext, mUri);
        }

        @Override
        public boolean canWrite() {
            return DocumentsContractApi19.canWrite(mContext, mUri);
        }

        @Override
        public boolean delete() {
            try {
                return DocumentsContract.deleteDocument(mContext.getContentResolver(), mUri);
            } catch (Exception e) {
                return false;
            }
        }

        @Override
        public boolean exists() {
            return DocumentsContractApi19.exists(mContext, mUri);
        }
    }

    static class DocumentsContractApi19 {
        private static final String TAG = "DocumentFile";

        // DocumentsContract API level 24.
        private static final int FLAG_VIRTUAL_DOCUMENT = 1 << 9;

        public static boolean isVirtual(Context context, Uri self) {
            if (!DocumentsContract.isDocumentUri(context, self)) {
                return false;
            }

            return (getFlags(context, self) & FLAG_VIRTUAL_DOCUMENT) != 0;
        }

        
        public static String getName(Context context, Uri self) {
            return queryForString(context, self, DocumentsContract.Document.COLUMN_DISPLAY_NAME, null);
        }

        
        private static String getRawType(Context context, Uri self) {
            return queryForString(context, self, DocumentsContract.Document.COLUMN_MIME_TYPE, null);
        }

        
        public static String getType(Context context, Uri self) {
            final String rawType = getRawType(context, self);
            if (DocumentsContract.Document.MIME_TYPE_DIR.equals(rawType)) {
                return null;
            } else {
                return rawType;
            }
        }

        public static long getFlags(Context context, Uri self) {
            return queryForLong(context, self, DocumentsContract.Document.COLUMN_FLAGS, 0);
        }

        public static boolean isDirectory(Context context, Uri self) {
            return DocumentsContract.Document.MIME_TYPE_DIR.equals(getRawType(context, self));
        }

        public static boolean isFile(Context context, Uri self) {
            final String type = getRawType(context, self);
            if (DocumentsContract.Document.MIME_TYPE_DIR.equals(type) || TextUtils.isEmpty(type)) {
                return false;
            } else {
                return true;
            }
        }

        public static long lastModified(Context context, Uri self) {
            return queryForLong(context, self, DocumentsContract.Document.COLUMN_LAST_MODIFIED, 0);
        }

        public static long length(Context context, Uri self) {
            return queryForLong(context, self, DocumentsContract.Document.COLUMN_SIZE, 0);
        }

        public static boolean canRead(Context context, Uri self) {
            // Ignore if grant doesn't allow read
            if (context.checkCallingOrSelfUriPermission(self, Intent.FLAG_GRANT_READ_URI_PERMISSION)
                    != PackageManager.PERMISSION_GRANTED) {
                return false;
            }

            // Ignore documents without MIME
            if (TextUtils.isEmpty(getRawType(context, self))) {
                return false;
            }

            return true;
        }

        public static boolean canWrite(Context context, Uri self) {
            // Ignore if grant doesn't allow write
            if (context.checkCallingOrSelfUriPermission(self, Intent.FLAG_GRANT_WRITE_URI_PERMISSION)
                    != PackageManager.PERMISSION_GRANTED) {
                return false;
            }

            final String type = getRawType(context, self);
            final int flags = queryForInt(context, self, DocumentsContract.Document.COLUMN_FLAGS, 0);

            // Ignore documents without MIME
            if (TextUtils.isEmpty(type)) {
                return false;
            }

            // Deletable documents considered writable
            if ((flags & DocumentsContract.Document.FLAG_SUPPORTS_DELETE) != 0) {
                return true;
            }

            if (DocumentsContract.Document.MIME_TYPE_DIR.equals(type)
                    && (flags & DocumentsContract.Document.FLAG_DIR_SUPPORTS_CREATE) != 0) {
                // Directories that allow create considered writable
                return true;
            } else if (!TextUtils.isEmpty(type)
                    && (flags & DocumentsContract.Document.FLAG_SUPPORTS_WRITE) != 0) {
                // Writable normal files considered writable
                return true;
            }

            return false;
        }

        public static boolean exists(Context context, Uri self) {
            final ContentResolver resolver = context.getContentResolver();

            Cursor c = null;
            try {
                c = resolver.query(self, new String[] {
                        DocumentsContract.Document.COLUMN_DOCUMENT_ID }, null, null, null);
                return c.getCount() > 0;
            } catch (Exception e) {
                Log.w(TAG, "Failed query: " + e);
                return false;
            } finally {
                closeQuietly(c);
            }
        }

        
        private static String queryForString(Context context, Uri self, String column,
                                              String defaultValue) {
            final ContentResolver resolver = context.getContentResolver();

            Cursor c = null;
            try {
                c = resolver.query(self, new String[] { column }, null, null, null);
                if (c.moveToFirst() && !c.isNull(0)) {
                    return c.getString(0);
                } else {
                    return defaultValue;
                }
            } catch (Exception e) {
                Log.w(TAG, "Failed query: " + e);
                return defaultValue;
            } finally {
                closeQuietly(c);
            }
        }

        private static int queryForInt(Context context, Uri self, String column,
                                       int defaultValue) {
            return (int) queryForLong(context, self, column, defaultValue);
        }

        private static long queryForLong(Context context, Uri self, String column,
                                         long defaultValue) {
            final ContentResolver resolver = context.getContentResolver();

            Cursor c = null;
            try {
                c = resolver.query(self, new String[] { column }, null, null, null);
                if (c.moveToFirst() && !c.isNull(0)) {
                    return c.getLong(0);
                } else {
                    return defaultValue;
                }
            } catch (Exception e) {
                Log.w(TAG, "Failed query: " + e);
                return defaultValue;
            } finally {
                closeQuietly(c);
            }
        }

        private static void closeQuietly( AutoCloseable closeable) {
            if (closeable != null) {
                try {
                    closeable.close();
                } catch (RuntimeException rethrown) {
                    throw rethrown;
                } catch (Exception ignored) {
                }
            }
        }

        private DocumentsContractApi19() {
        }
    }

    static class RawDocumentFile extends DocumentFile {
        private File mFile;

        RawDocumentFile(DocumentFile parent, File file) {
            super(parent);
            mFile = file;
        }

        @Override
        public Uri getUri() {
            return Uri.fromFile(mFile);
        }

        @Override
        public String getName() {
            return mFile.getName();
        }

        @Override
        public String getType() {
            if (mFile.isDirectory()) {
                return null;
            } else {
                return getTypeForName(mFile.getName());
            }
        }

        @Override
        public boolean isDirectory() {
            return mFile.isDirectory();
        }

        @Override
        public boolean isFile() {
            return mFile.isFile();
        }

        @Override
        public boolean isVirtual() {
            return false;
        }

        @Override
        public long lastModified() {
            return mFile.lastModified();
        }

        @Override
        public long length() {
            return mFile.length();
        }

        @Override
        public boolean canRead() {
            return mFile.canRead();
        }

        @Override
        public boolean canWrite() {
            return mFile.canWrite();
        }

        @Override
        public boolean delete() {
            deleteContents(mFile);
            return mFile.delete();
        }

        @Override
        public boolean exists() {
            return mFile.exists();
        }

        private static String getTypeForName(String name) {
            final int lastDot = name.lastIndexOf('.');
            if (lastDot >= 0) {
                final String extension = name.substring(lastDot + 1).toLowerCase();
                final String mime = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension);
                if (mime != null) {
                    return mime;
                }
            }

            return "application/octet-stream";
        }

        private static boolean deleteContents(File dir) {
            File[] files = dir.listFiles();
            boolean success = true;
            if (files != null) {
                for (File file : files) {
                    if (file.isDirectory()) {
                        success &= deleteContents(file);
                    }
                    if (!file.delete()) {
                        Log.w(TAG, "Failed to delete " + file);
                        success = false;
                    }
                }
            }
            return success;
        }
    }
}
