/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAGroup;
import com.hyphenate.chat.adapter.EMAGroupSetting;
import com.hyphenate.chat.adapter.EMAMucShareFile;

import java.util.ArrayList;
import java.util.List;

/**
 *  \~chinese
 *  群组。
 *
 *  \~english
 *  The group class.
 */
public class EMGroup extends EMBase<EMAGroup> {

    /**
     * \~chinese
     * 当前用户在群组的角色枚举。
     *
     * \~english
     * The enum of the group permission types.
     *
     */
    public enum EMGroupPermissionType {
        /**
         * \~chinese
         * 普通成员。
         *
         * \~english
         * The group member.
         */
        member(0),

        /**
         * \~chinese
         * 群组管理员。
         *
         * \~english
         * The group admin.
         */
        admin(1),

        /**
         * \~chinese
         * 群主。
         *
         * \~english
         * The group owner.
         */
        owner(2),

        /**
         * \~chinese
         * 未知。
         *
         * \~english
         * The unknown type.
         */
        none(-1);
        private int permissionType;
        private EMGroupPermissionType(int permissionType) {
            this.permissionType = permissionType;
        }
    }

    public EMGroup(EMAGroup group) {
        emaObject = group;
    }

    /**
     * \~chinese
     * 获取群 ID。
     * @return 群 ID。
     *
     * \~english
     * Gets the group ID.
     * @return The group ID.
     */
    public String getGroupId() {
        return emaObject.groupId();
    }

    /**
     * \~chinese
     * 获取群名称。
     * @return       群名称。
     *
     * \~english
     * Gets the group name.
     * @return      The group name.
     */
    public String getGroupName() {
        return emaObject.groupSubject();
    }

    /**
     * \~chinese
     * 获取群简介。
     * @return 群描述。
     *
     * \~english
     * Gets the group description.
     * @return  The group description.
     */
    public String getDescription() {
        return emaObject.getDescription();
    }

    /**
     * \~chinese
     * 群组是否为公开群。
     * @return   `true` 为公开群，否则为私有群。
     *
     * \~english
     * Gets whether the group is a public group. 
     * @return  If the result is `true`, the group is a public group, otherwise it is a private group.
     */
    public boolean isPublic() {

        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        switch (setting.style()) {
            case EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE:
            case EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE:
                return false;
            default:
                return true;
        }
    }

    /**
     * \~chinese
     * 获取群组是否允许成员邀请。
     * @return   - `true`：群成员可以邀请其他用户加入；
     *           - `false`：不允许群成员邀请其他用户加入。
     *
     * \~english
     * Gets whether the group member is allowed to invite other users to join the group.
     * @return - `true`: The group member can invite other users to join the group;
     *         - `false`: Do not allow the group member invite other users to join the group.
     */
    public boolean isMemberAllowToInvite()
    {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        if (setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE) {
            return true;
        }
        return false;
    }

    /**
     * \~chinese
     * 获取群组属性：成员是否能自由加入，还是需要申请或者被邀请。
     * 
     * 群组有四个类型属性，`isMemberonly`是除了 {@link EMGroupManager.EMGroupStyle#EMGroupStylePublicOpenJoin} 之外的三种属性，表示该群不是自由加入的群组。
     *
     * @return  
     * - `true`：进群需要群主邀请，群成员邀请，或者群主和管理员同意入群申请；
     * - `false`：意味着用户可以自由加入群，不需要申请和被邀请。
     *
     * \~english
     * Fetches the group property: whether users can auto join the group VS need requesting or invitation from a group member to join the group.
     * There are four types of group properties used to define the style of a group, and `isMemberOnly` contains three types including: PRIVATE_OWNER_INVITE,  PRIVATE_MEMBER_INVITE, PUBLIC_JOIN_APPROVAL. And do not include {@link EMGroupManager.EMGroupStyle#EMGroupStylePublicOpenJoin}.
     *
     * @return 
     * - `true`: Users can not join the group freely. Needs the invitation from the group owner or members, or the application been approved by the group owner or admins.
     * - `false`: Users can join freely without the group owner or member‘s invitation or the new joiner’s application been approved.
     */
    public boolean isMemberOnly()
    {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        if (setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE ||
                setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE ||
                setting.style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL) {
            return true;
        }
        return false;
    }

    /**
     * \~chinese
     * 获取群允许加入的最大成员数，在创建群时确定。
     * 需要获取群详情才能拿到正确的结果，如果没有获取则返回 0。
     * 
     * @return
     *
     * \~english
     * The max number of group members allowed in a group. The param is set when the group is created.
     * 
     * Be sure to fetch the detail specification of the group from the server first, see {@link EMGroupManager#getGroupFromServer(String)}. If not, the SDK returns 0.
     * 
     * @return  The allowed max number of group members.
     */
    public int getMaxUserCount() {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return 0;
        }
        return setting.maxUserCount();
    }

    /**
     * \~chinese
     * 获取是否已屏蔽群消息。
     * 
     * 参考：
     * 设置屏蔽群消息见 {@link EMGroupManager#blockGroupMessage(String)},
     * 取消屏蔽群消息见 {@link EMGroupManager#unblockGroupMessage(String)}。
     * 
     * @return  - `true`：用户已经屏蔽群消息；
     *          - `false`：用户没有屏蔽群消息。
     *
     * \~english
     * Gets whether the group message is blocked.
     * 
     * Reference:
     * For blocking group message, see {@link EMGroupManager#blockGroupMessage(String)},
     * unblocking see {@link EMGroupManager#unblockGroupMessage(String)}
     * @return   - `true`: The user has blocked the group messages;
     *           - `false`: The user does not blocked the group messages.
     */
    public boolean isMsgBlocked() {
        return emaObject.isMsgBlocked();
    }

    /**
     * \~chinese
     * 获取群所有者 ID。
     * @return    群所者用户 ID。
     *
     * \~english
     * Gets the user ID of the group owner.
     * @return    The user ID of the group owner.
     */
    public String getOwner() {
        return emaObject.getOwner();
    }

    /**
     * \~chinese
     * 获取群组主题，需要获取群详情。
     * @return 群组主题。
     *
     * \~english
     * The name of the group. 
     * Be sure to fetch the detail specification of the group from the server first, see {@link EMGroupManager#getGroupFromServer(String)}.
     * @return  The group name.
     */
    public String groupSubject() {
        return emaObject.groupSubject();
    }

    /**
     * \~chinese
     * 获取群成员列表。
     * 如果没有获取成员列表，返回的列表可能为空。
     * 
     * 获取成员有如下方式：
     * （1）成员人数少于 200 人时，可通过 {@link EMGroupManager#getGroupFromServer(String, boolean)},
     * 第二个参数传入 `true`，可获取最多 200 个成员。
     * （2）通过 {@link EMGroupManager#fetchGroupMembers(String, String, int)} 分页获取，也可通过其异步方法
     * {@link EMGroupManager#asyncFetchGroupMembers(String, String, int, EMValueCallBack)} 获取。
     * @return  群成员 ID 列表。
     *
     * \~english
     * Gets the member list of the group.
     * If no member is found from the server, the return might be empty.
     * 
     * Reference:
     * You can get the group members in the following ways:
     * (1) When member's number is less than 200, can use {@link EMGroupManager#getGroupFromServer(String, boolean)} to fetch,
     * the second parameter pass in true, can get up to 200 members
     * (2) Use the method of {@link EMGroupManager#fetchGroupMembers(String, String, int)} to get by page,
     * or its asynchronous method of {@link EMGroupManager#asyncFetchGroupMembers(String, String, int, EMValueCallBack)}

     * to get
     * @return  The list of the user IDs of the group members.
     */
    public List<String> getMembers() {
        List<String> members = new ArrayList<String>();
        members.addAll(emaObject.getMembers());
        return members;
    }

    /**
     * \~chinese
     * 获取成员数量。
     *
     * 包括群主、管理员、普通成员。
     * 
     * @return  成员数量。

     *
     * \~english
     * Gets the member count of the group.
     *
     * This includes the group owner, administrators, and regular members.
     * 
     * @return  The count.
     */
    public int getMemberCount() {
        return emaObject.getMemberCount();
    }

    /**
     * \~chinese
     * 返回群组名称，如果为空则返回群组 ID，此方法可用于 Adapter 排序。
     * 
     * @return  群组名称，如果为空则返回群组 ID。
     *
     * \~english
     * Gets the group name, or group ID if the group name is empty.
     * This method can be used for adapter sorting.
     * 
     * @return  The group name, or group ID if the group name is empty.
     */
    public String toString() {
        String str = getGroupName();
        return str != null ? str : getGroupId();
    }

    /**
     * \~chinese
     * 获取群组管理员列表。需要先通过 {@link EMGroupManager#getGroupFromServer(String)} 获取群组详情。
     * 
     * @return  群组管理员列表。
     *
     * \~english
     * Gets the admin list of the group.
     * 
     * Be sure to fetch the detail specification of the group from the server first, see {@link EMGroupManager#getGroupFromServer(String)}.
     * 
     * @return  The admin list of the group.
     */
    public List<String> getAdminList() {
        return emaObject.getAdminList();
    }

    /**
     * \~chinese
     * 获取群组黑名单。
     * 如果没有获取群组黑名单列表，返回的列表可能为空。
     * 
     * 参考：
     * 获取黑名单列表可调用 {@link EMGroupManager#fetchGroupBlackList(String, int, int)},
     * 调用这个方法需要群组所有者或者管理员身份。
     * 
     * 也可通过它的异步方法进行调用，如下：
     * {@link EMGroupManager#asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}
     * @return  黑名单列表。
     *
     * \~english
     * Gets the blocklist of the group.
     * 
     * If no blocklist is found from the server, the return may be empty.
     * 
     * Reference:
     * To fetch the blocklist, call {@link EMGroupManager#fetchGroupBlackList(String, int, int)}.
     * 
     * Only the group owner or admin can call this method.
     * 
     * You also can call it's asynchronous method as follows:
     * {@link EMGroupManager#asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}.
     * 
     * @return  The blocklist.
     */
    public List<String> getBlackList() {
        return emaObject.getGroupBans();
    }

    /**
     * \~chinese
     * 获取群组禁言列表。
     * 如果没有获取禁言列表，返回的列表可能为空。
     * 
     * 参考：
     * 获取禁言列表可调用 {@link EMGroupManager#fetchGroupMuteList(String, int, int)},
     * 调用这个方法需要群组所有者或者管理员身份。
     * 也可通过它的异步方法进行调用，如下
     * {@link EMGroupManager#asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}
     *
     * @return 群组禁言列表。
     *
     * \~english
     * Gets the mute list of the group.
     * 
     * If no mute list is found from the server, the return may be empty.
     * 
     * Reference:
     * You can also fetch the mute list by calling {@link EMGroupManager#fetchGroupMuteList(String, int, int)}.
     * And only the group owner or admin can call this method. 
     * 
     * You can also call it's asynchronous method, see 
     * {@link EMGroupManager#asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}.
     *
     * @return  The mute list of the group.
     */
    public List<String> getMuteList() {
        return emaObject.getGroupMuteList();
    }

    /**
     * \~chinese
     * 获取白名单列表。
     * 如果没有从服务器获取白名单列表，返回的列表可能为空。
     * 
     * 参考：
     * 获取白名单可调用 {@link EMGroupManager#fetchGroupWhiteList(String, EMValueCallBack)},
     * 调用这个方法需要群组所有者或者管理员身份。
     * @return  群组白名单列表。
     *
     * \~english
     * Gets the allowlist of the group.
     * 
     * If no allowlist is found from the server, the return may be empty.
     * 
     * Reference:
     * You can also fetch the allowlist by calling {@link EMGroupManager#fetchGroupWhiteList(String, EMValueCallBack)}.
     * And only the group owner or admin can call this method.
     * @return  The group allowlist.
     */
    public List<String> getWhiteList() { return emaObject.getWhiteList(); }

    /**
     * \~chinese
     * 返回是否全员禁言。
     * 
     * 此方法有使用局限性，建议慎重使用。
     * 
     * 在收到禁言/解禁的回调时，该状态会更新，但是只是更新内存中对象，
     * 内存中对象被回收后以及再次从数据库或者从服务端拉取后，该状态变的不可信。
     * @return  是否全员禁言。
     *
     * \~english
     * Gets whether all members are muted.
     * 
     * This method has limitations and is recommended to be used with caution.
     * 
     * The state is updated when a all-muted/all-unmuted callback is received, but only for the in-memory object.
     * After the in-memory object is collected and pulled again from the database or server, the state becomes unreliable.
     * @return  Whether all members are muted.
     */
    public boolean isAllMemberMuted() { return emaObject.isAllMemberMuted(); }

    /**
     * \~chinese
     * 获取群组订制扩展信息。
     * @return  群组定制扩展信息。
     *
     * \~english
     * Gets the customized extension of the group.
     * @return  The customized extension of the group.
     */
    public String getExtension() {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return "";
        }
        return setting.extension();
    }

    /**
     * \~chinese
     * 获取群组公告。
     * 如果没有获取群组公告，返回的值为空。
     * 
     * 获取群组公告，可调用 {@link EMGroupManager#fetchGroupAnnouncement(String)},
     * 也可调用其异步方法 {@link EMGroupManager#asyncFetchGroupAnnouncement(String, EMValueCallBack)}。
     * 
     * @return  群组公告内容。
     *
     * \~english
     * Gets the group announcement.
     * If no announcement is found from the server, the return may be empty.
     * 
     * Reference:
     * You can also call the method: {@link EMGroupManager#fetchGroupAnnouncement(String)} to get the group announcement.
     * As for its asynchronous method, see {@link EMGroupManager#asyncFetchGroupAnnouncement(String, EMValueCallBack)}.
     * @return  The content of the group announcement.
     */
    public String getAnnouncement() {
        return emaObject.getAnnouncement();
    }

    /**
     *  \~chinese
     *  群共享文件列表。
     *
     *  \~english
     *  The list of the shared file.
     */
    private List<EMMucSharedFile> shareFileList;

    /**
     * \~chinese
     * 获取群组共享文件。
     * 如果没有获取群组共享文件，返回的列表可能为空。
     * 
     * 参考：
     * 获取群组共享文件，可调用 {@link EMGroupManager#fetchGroupSharedFileList(String, int, int)},
     * 也可调用其异步方法 {@link EMGroupManager#asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}
     * @return  群组共享文件列表。
     *
     * \~english
     * Gets the group shared file list.
     * If no shared file is found from the server, the return might be empty.
     * 
     * Reference:
     * You can also call {@link EMGroupManager#fetchGroupSharedFileList(String, int, int)} to fetch the shared file list.
     * And for the asynchronous method, see {@link EMGroupManager#asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}.
     * @return  The group shared file list.
     */
    public List<EMMucSharedFile> getShareFileList()
    {
        if (shareFileList == null) {
            shareFileList = new ArrayList<EMMucSharedFile>();
        }
        else {
            shareFileList.clear();
        }
        List<EMAMucShareFile> afiles = emaObject.getShareFiles();
        for (EMAMucShareFile afile : afiles){
            shareFileList.add(new EMMucSharedFile(afile));
        }
        return shareFileList;
    }

    /**
     * \~chinese
     * 获取当前用户在群中的角色。
     * @return   返回 `EMGroupPermissionType` 中的具体类型，例如群主，管理员或者成员。
     *
     * \~english
     * Gets the current user's role in group.
     * @return   Returns the permission type of the user in a group, see `EMGroupPermissionType`.
     */
    public EMGroupPermissionType getGroupPermissionType() {
        int permissionType = emaObject.permissionType();
        if(permissionType == EMGroupPermissionType.member.permissionType) {
            return EMGroupPermissionType.member;
        }else if(permissionType == EMGroupPermissionType.admin.permissionType) {
            return EMGroupPermissionType.admin;
        }else if(permissionType == EMGroupPermissionType.owner.permissionType) {
            return EMGroupPermissionType.owner;
        }else {
            return EMGroupPermissionType.none;
        }
    }

    /**
     * \~chinese
     * 返回群组是否禁用。
     *
     * @return    - `true`：群组已经被禁用；
     *            - `false`：群组没有被禁用。
     *
     * \~english
     * Gets whether the group is disabled.
     *
     * @return  - `true`: Group has been disabled;
     *          - `false`: Group is not disabled.
     */
    public boolean isDisabled() { return emaObject.isDisabled(); }

}
