/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.preference.PreferenceManager;
import android.text.TextUtils;

import com.hyphenate.EMError;
import com.hyphenate.chat.adapter.EMAConnectionListener;
import com.hyphenate.chat.adapter.EMAREncryptUtils;
import com.hyphenate.chat.adapter.EMASessionManager;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;

import java.util.Timer;
import java.util.TimerTask;

class EMSessionManager {
	private static final String TAG = "Session";

	private Context appContext = null;

	private final static String PREF_KEY_LOGIN_USER = "easemob.chat.loginuser";
	private final static String PREF_KEY_LOGIN_PWD = "easemob.chat.loginpwd";
	private final static String PREF_KEY_LOGIN_TOKEN = "easemob.chat.login.token";
	private final static String PREF_KEY_LOGIN_WITH_TOKEN = "easemob.chat.login_with_token";
	private final static String PREF_KEY_LOGIN_PWD_GCM = "easemob.chat.loginpwd.gcm";
	private final static String PREF_KEY_LOGIN_TOKEN_GCM = "easemob.chat.login.token.gcm";
    private final static String PREF_KEY_TOKEN_TIMESTAMP = "easemob.chat.token_timestamp";
    private final static String PREF_KEY_TOKEN_AVAILABLE_PERIOD = "easemob.chat.token_available_period";


	/**
	 * the static variable for current login user
	 */
	public EMContact currentUser = null;

	private static EMSessionManager instance = new EMSessionManager();

	private String lastLoginUser = null;
	private String lastLoginPwd = null;
	private String lastLoginToken = null;
    private String tokenTimeStamp = null;
    private long tokenAvailablePeriod = 0;
    private Timer timer;


	EMClient mClient;
	EMASessionManager mSessionManager;
	private EMAREncryptUtils encryptUtils;

	static synchronized EMSessionManager getInstance(){
		if(instance.appContext == null){
			instance.appContext = EMClient.getInstance().getContext();
		}

		return instance;
	}

	void init(EMClient client, EMASessionManager sessionManager) {
		this.mClient = client;
		this.mSessionManager = sessionManager;
		encryptUtils = new EMAREncryptUtils();
	}

	/**
	 * get the username who has been logined successfully. this method will be
	 * called during sdk init.
	 *
	 * @return
	 */
	String getLastLoginUser() {
		if (lastLoginUser == null) {
			SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
			lastLoginUser = setting.getString(PREF_KEY_LOGIN_USER, "");
			currentUser = new EMContact(lastLoginUser);
		}
		return lastLoginUser;
	}

	void setLastLoginUser(String username) {
		if(username == null){
			return;
		}
		currentUser = new EMContact(username);
		lastLoginUser = username;
		SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
		Editor editor = setting.edit();
		editor.putString(PREF_KEY_LOGIN_USER, username);
		editor.commit();
	}

	String getLastLoginPwd() {
		if (lastLoginPwd == null) {
			SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
			String encryptedPwd = setting.getString(PREF_KEY_LOGIN_PWD_GCM, "");

            if(TextUtils.isEmpty(encryptedPwd)){
            	encryptedPwd = setting.getString(PREF_KEY_LOGIN_PWD, "");
            	if(TextUtils.isEmpty(encryptedPwd)) {
					lastLoginPwd = "";
					return lastLoginPwd;
            	}
				try {
					lastLoginPwd = mSessionManager.decrypt(encryptedPwd);
				} catch (Exception e) {
					e.printStackTrace();
				}
                return lastLoginPwd;
            }

			try {
				lastLoginPwd = decryptData(getSecretKey(false), encryptedPwd);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return lastLoginPwd;
	}

	void setLastLoginPwd(String pwd) {
		if(pwd == null){
			return;
		}

		lastLoginPwd = pwd;
		SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
		Editor editor = setting.edit();
		// save password in encryption mode
		try {
			String encrypedPwd = encryptData(getSecretKey(true), pwd);
			editor.putString(PREF_KEY_LOGIN_PWD_GCM, encrypedPwd);
			editor.commit();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	String getLastLoginToken() {
		if (lastLoginToken == null) {
			SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
			String encryptedToken = setting.getString(PREF_KEY_LOGIN_TOKEN_GCM, "");

			if(TextUtils.isEmpty(encryptedToken)){
				encryptedToken = setting.getString(PREF_KEY_LOGIN_TOKEN, "");
				if(TextUtils.isEmpty(encryptedToken)) {
					lastLoginToken = "";
					return lastLoginToken;
				}
				try {
					lastLoginToken = mSessionManager.decrypt(encryptedToken);
				} catch (Exception e) {
					e.printStackTrace();
				}
				return lastLoginToken;
			}

			try {
				lastLoginToken = decryptData(getSecretKey(false), encryptedToken);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return lastLoginToken;
	}

	void setLastLoginToken(String token) {
		if(token == null){
			return;
		}

		lastLoginToken = token;
		SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
		Editor editor = setting.edit();
		// save password in encryption mode
		try {
			String encrypedToken = encryptData(getSecretKey(true), token);

			editor.putString(PREF_KEY_LOGIN_TOKEN_GCM, encrypedToken);
			editor.commit();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	void clearLastLoginUser(){
		try{
			lastLoginUser = "";
			currentUser.username = "";
			currentUser.remark = "";
			SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
			Editor editor = setting.edit();
			editor.putString(PREF_KEY_LOGIN_USER, lastLoginUser);
			editor.commit();
		}catch(Exception e){
			e.printStackTrace();
		}
	}

	void clearLastLoginPwd(){
		try{
			lastLoginPwd = "";
			SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
			Editor editor = setting.edit();
			editor.putString(PREF_KEY_LOGIN_PWD, lastLoginPwd);
			editor.putString(PREF_KEY_LOGIN_PWD_GCM, lastLoginPwd);
			editor.commit();
		}catch(Exception e){
			e.printStackTrace();
		}
	}

	void clearLastLoginToken(){
		try{
			lastLoginToken = "";
			SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
			Editor editor = setting.edit();
			editor.putString(PREF_KEY_LOGIN_TOKEN, lastLoginToken);
			editor.putString(PREF_KEY_LOGIN_TOKEN_GCM, lastLoginToken);
			editor.commit();
		}catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * whether last login with token
	 * this method will be called during sdk init.
	 *
	 * @return
	 */
	boolean isLastLoginWithToken() {
		SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
		return setting.getBoolean(PREF_KEY_LOGIN_WITH_TOKEN, false);
	}

	void setLastLoginWithToken(boolean loginWithToken) {
		SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
		Editor editor = setting.edit();
		editor.putBoolean(PREF_KEY_LOGIN_WITH_TOKEN, loginWithToken);
		editor.commit();
	}

	public String getLoginUserName() {
		return currentUser.username;
	}

	// Note: Add synchronized to avoid cipher exceptions when this method is called in multiple places
	synchronized String encryptData(byte[] key, String data) {
		encryptUtils.initAESgcm(key);
		return encryptUtils.aesGcmEncrypt(data, 1);
	}

	String decryptData(byte[] key, String encryptData) {
		return encryptUtils.aesGcmDecrypt(encryptData, key, 1);
	}
    String getTokenTimeStamp() {
        if (tokenTimeStamp == null) {
            SharedPreferences sp = PreferenceManager.getDefaultSharedPreferences(appContext);
            tokenTimeStamp = sp.getString(PREF_KEY_TOKEN_TIMESTAMP, "");
        }
        return tokenTimeStamp;
    }

    void setLoginWithTokenData(boolean isLoginWithToken, String timeStamp, long period) {
        if (timeStamp == null||period<0) {
			EMLog.e(TAG,"setLoginWithTokenData, timeStamp ="+timeStamp+", period="+period);
            return;
        }
        this.tokenTimeStamp = timeStamp;
        this.tokenAvailablePeriod = period;
        SharedPreferences sp = PreferenceManager.getDefaultSharedPreferences(appContext);
        Editor editor = sp.edit();
        try {
            editor.putBoolean(PREF_KEY_LOGIN_WITH_TOKEN, isLoginWithToken);
            editor.putString(PREF_KEY_TOKEN_TIMESTAMP, timeStamp);
            editor.putLong(PREF_KEY_TOKEN_AVAILABLE_PERIOD, tokenAvailablePeriod);
            editor.commit();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    long getTokenAvailablePeriod() {
        if (tokenAvailablePeriod == 0) {
            SharedPreferences sp = PreferenceManager.getDefaultSharedPreferences(appContext);
            tokenAvailablePeriod = sp.getLong(PREF_KEY_TOKEN_AVAILABLE_PERIOD, 0);
        }
        return tokenAvailablePeriod;
    }

    synchronized void checkTokenAvailability(EMAConnectionListener connectionListener) {
        String tokenTimeStamp = getTokenTimeStamp();
        if (TextUtils.isEmpty(tokenTimeStamp)||connectionListener==null) {
            return ;
        }
        long delay = Long.valueOf(tokenTimeStamp) - System.currentTimeMillis();
		EMLog.d(TAG,"checkTokenAvailability, delay="+delay);
        if (delay <= 0) {
            connectionListener.onTokenNotification(EMError.TOKEN_EXPIRED);
        } else if (delay <= getTokenAvailablePeriod() * 20 / 100) {
            connectionListener.onTokenNotification(EMError.TOKEN_WILL_EXPIRE);
        }
    }

    synchronized void startCountDownTokenAvailableTime(EMAConnectionListener connectionListener) {
        String tokenTimeStamp = getTokenTimeStamp();
        if (TextUtils.isEmpty(tokenTimeStamp)||connectionListener==null) {
            return ;
        }
        long tokenAvailablePeriod = getTokenAvailablePeriod();
        if (timer != null) {
            timer.cancel();
        }
        timer=new Timer();
        TimerTask task1 = new TimerTask() {
            @Override
            public void run() {
                checkTokenAvailability(connectionListener);
            }
        };
        long delay = Long.valueOf(tokenTimeStamp) - System.currentTimeMillis();
        if(delay<=0) {
			//进入这里表明token已经过期，立马通知外界过期
            connectionListener.onTokenNotification(EMError.TOKEN_EXPIRED);
        }else if(delay<=tokenAvailablePeriod * 20/100) {
			//进入这里表明有效期小于20%这个时间点，需要立马通知外界即将过期，且开启一个任务在有效期到期时，启动检查来通知过期
            EMLog.d(TAG,"timer.schedule task1, delay="+delay);
            timer.schedule(task1, delay);
            connectionListener.onTokenNotification(EMError.TOKEN_WILL_EXPIRE);
        }else{
			//进入这里表明有效期大于20%这个时间点，暂时不用通知外界。
			//只需要开启两个任务：1、在20%这个时间点通知外界即将过期(task2) 2、在有效期到期时通知外界过期(task1)
            EMLog.d(TAG,"timer.schedule task1、task2, delay= "+delay);
            TimerTask task2 = new TimerTask() {
                @Override
                public void run() {
                    checkTokenAvailability(connectionListener);
                }
            };
			//在有效期到期这个时间点，启动检查，通知外界已经过期
            timer.schedule(task1, delay);
			//在有效期过了80%这个时间点，启动检查，通知外界即将过期
            timer.schedule(task2, delay - tokenAvailablePeriod *20/100);
        }
    }

    void clearLoginWithTokenData(){
        try {
            tokenAvailablePeriod = 0;
            tokenTimeStamp="";
            SharedPreferences setting = PreferenceManager.getDefaultSharedPreferences(appContext);
            Editor editor = setting.edit();
            editor.putLong(PREF_KEY_TOKEN_AVAILABLE_PERIOD, tokenAvailablePeriod);
            editor.putString(PREF_KEY_TOKEN_TIMESTAMP, tokenTimeStamp);
            editor.putBoolean(PREF_KEY_LOGIN_WITH_TOKEN, false);
            editor.commit();
            if (timer != null) {
                timer.cancel();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    byte[] getSecretKey(boolean replace) {
		String deviceId = getDeviceId(replace);
		String lastLoginUser = getLastLoginUser();
		return mSessionManager.getEncryptionKey(lastLoginUser, deviceId);
	}

	/**
	 * Get data from sp, not save deviceUuid to bak file
	 * @param replace
	 * @return
	 */
    String getDeviceId(boolean replace) {
		DeviceUuidFactory factory = new DeviceUuidFactory(appContext);
		String deviceUuid = factory.getDeviceUuid().toString();

		if(replace) {
		    saveDeviceId(deviceUuid);
		}else {
			String bakDeviceId = DeviceUuidFactory.getBakDeviceId(appContext);
			// If bakDeviceId is not null and deviceUuid is not equals with it, use bakDeviceId
			if(!TextUtils.isEmpty(bakDeviceId) && !TextUtils.equals(deviceUuid, bakDeviceId)) {
				return bakDeviceId;
			}
		}
		return deviceUuid;
	}

	void saveDeviceId(String deviceId) {
		DeviceUuidFactory.saveBakDeviceId(appContext, deviceId);
	}
}