package com.hyphenate.chat;


import com.hyphenate.EMCallBack;
import com.hyphenate.EMPresenceListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAPresence;
import com.hyphenate.chat.adapter.EMAPresenceManager;
import com.hyphenate.chat.adapter.EMAPresenceManagerListener;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * \~chinese
 * 在线状态管理类，提供在线状态管理功能。
 *
 * \~english
 * The Manager that defines how to manage presence states.
 */
public class EMPresenceManager {
    private final String TAG = getClass().getSimpleName();

    EMAPresenceManager emaObject;
    private List<EMPresenceListener> listeners = new CopyOnWriteArrayList<EMPresenceListener>();
    private EMAPresenceManagerListener listenerImpl = new EMAPresenceManagerListener() {
        @Override
        public void onPresenceUpdated(List<EMAPresence> presences) {

            synchronized (listeners) {
                for (EMPresenceListener listener : listeners) {
                    try {
                        listener.onPresenceUpdated(convertToEMPresence(presences));
                    } catch (Exception e) {
                        e.printStackTrace();
                        EMLog.e(TAG, "EMPresenceManager->listenerImpl:onPresenceUpdated() Error:" + e.getMessage());
                    }
                }

            }
        }
    };

    public EMPresenceManager(EMAPresenceManager presenceManager) {
        emaObject = presenceManager;
        emaObject.addListener(listenerImpl);
    }

    /**
     * \~chinese
     * 发布自定义在线状态。
     *
     * @param customStatus   自定义在线状态。可通过 {@link EMPresence#getExt()}来获取自定义状态。
     * @param callBack       该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Publishes a custom presence state.
     * @param customStatus   The custom presence status.The custom presence status can be obtained by {@link EMPresence#getExt()}.
     * @param callBack       The completion block, which contains the error message if this method fails.
     */
    public void publishPresence( String customStatus, EMCallBack callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                emaObject.publishPresence( customStatus, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess();
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 订阅指定用户的在线状态。订阅成功后，在线状态变更时订阅者会收到回调通知。
     * @param members  要订阅在线状态的用户 ID 数组。
     * @param expiry   订阅到期时间。单位秒，最长30天。
     * @param callBack 该方法完成调用的回调。如果该方法调用成功，会返回订阅用户的当前状态，调用失败，会包含调用失败的原因。
     *
     * \~english
     * Subscribes to a user's presence states. If the subscription succeeds, the subscriber will receive the callback when the user's presence state changes.
     * @param members  The array of IDs of users whose presence states you want to subscribe to.
     * @param expiry   The expiration time of the presence subscription.
     * @param callBack The completion block, which contains the error message if the method fails.
     */
    public void subscribePresences(List<String> members, long expiry, EMValueCallBack<List<EMPresence>> callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<EMAPresence> presences = emaObject.subscribePresences(members, expiry, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess(convertToEMPresence(presences));
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 取消订阅指定用户的在线状态。
     * @param members  要取消订阅在线状态的用户 ID 数组。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     *\~english
     * Unsubscribes from a user's presence states.
     * @param members  The array of IDs of users whose presence states you want to unsubscribe from.
     * @param callBack The completion block, which contains the error message if the method fails.
     */
    public void unsubscribePresences(List<String> members, EMCallBack callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                emaObject.unsubscribePresences(members, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess();
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });

    }

    /**
     * \~chinese
     * 分页查询当前用户订阅了哪些用户的在线状态。
     * @param pageNum  当前页码，从 1 开始。
     * @param pageSize 每页的订阅用户的数量。
     * @param callBack 完成回调，返回订阅的在线状态所属的用户 ID。若当前未订阅任何用户的在线状态，返回空值。
     *
     *\~english
     * Uses pagination to get a list of users whose presence states you have subscribed to.
     * @param pageNum  The current page number, starting from 0.
     * @param pageSize The number of subscribed users on each page.
     * @param callBack The completion block, which contains IDs of users whose presence states you have subscribed to. Returns nil if you subscribe to no user's presence state.
     */
    public void fetchSubscribedMembers(int pageNum, int pageSize, EMValueCallBack<List<String>> callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<String> subscribedMembers = emaObject.fetchSubscribedMembers(pageNum, pageSize, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess(subscribedMembers);
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 查询指定用户的当前在线状态。
     * @param members  用户 ID 数组，指定要查询哪些用户的在线状态。
     * @param callBack 完成回调，返回用户的在线状态。
     *
     * \~english
     * Gets the current presence state of users.
     * @param members  The array of IDs of users whose current presence state you want to check.
     * @param callBack The completion block, which contains the users whose presence state you have subscribed to.
     */
    public void fetchPresenceStatus(List<String> members, EMValueCallBack<List<EMPresence>> callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<EMAPresence> presenceStatus = emaObject.fetchPresenceStatus(members, error);
                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    callBack.onSuccess(convertToEMPresence(presenceStatus));
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 添加监听。
     * @param listener {@link EMPresenceListener} 要添加的监听。
     *
     * \~english
     * Adds a listener.
     * @param listener {@link EMPresenceListener} The listener to be added.
     */
    public void addListener(EMPresenceListener listener) {
        if (!listeners.contains(listener)) {
            listeners.add(listener);
        }
    }

    /**
     * \~chinese
     * 移除监听。
     * @param listener {@link EMPresenceListener} 要移除的监听。
     *
     *\~english
     * Removes a listener.
     * @param listener {@link EMPresenceListener} The listener to be removed.
     */
    public void removeListener(EMPresenceListener listener) {
        if (listener != null) {
            listeners.remove(listener);
        }
    }

    /**
     * \~chinese
     * 清空所有监听者
     *
     * \~english
     * Removes all listeners.
     */
    public void clearListeners() {
        listeners.clear();
    }

    private List<EMPresence> convertToEMPresence(List<EMAPresence> presences) {
        List<EMPresence> presenceList = new ArrayList<>();
        for (int i = 0; i < presences.size(); i++) {
            presenceList.add(new EMPresence(presences.get(i)));
        }
        return presenceList;
    }
}
