package com.hyphenate.cloud;

import android.content.Context;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;

import com.hyphenate.chat.EMClient;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.ProtocolException;
import java.net.Proxy;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * 作为http请求控制类
 */
class HttpClientController {
    private static final String TAG = HttpClientController.class.getSimpleName();
    private final Context mContext;
    private URL mURL;
    private HttpURLConnection mConn;
    private boolean isHttps;
    /* valid HTTP methods */
    private static final String[] methods = {
            "GET", "POST", "HEAD", "OPTIONS", "PUT", "DELETE", "TRACE"
    };
    private static int EM_DEFAULT_TIMEOUT = 60 * 1000;
    private static int EM_DEFAULT_READ_TIMEOUT = 60 * 1000;
    private static final String BOUNDARY = java.util.UUID.randomUUID().toString();
    private static final String TWO_HYPHENS = "--";
    private static final String LINE_END = "\r\n";
    private static final String EASEMOB_PLATFORM = "Android";

    public HttpClientController(Context mContext) {
        this.mContext = mContext;
    }

    /**
     * 设置URL
     * @param url
     * @throws IOException
     */
    public void setURL(String url) throws IOException {
        setURL(url, -1);
    }
    /**
     * 设置URL
     * @param url
     * @param port
     * @throws IOException
     */
    public void setURL(String url, int port) throws IOException {
        setURL(url, port,0);
    }

    /**
     * 设置URL
     * @param url
     * @param port
     * @param fpaPort
     * @throws IOException
     */
    public void setURL(String url, int port,int fpaPort) throws IOException {
        EMLog.d(TAG,"setURL()-> fpaPort="+fpaPort);
        url = HttpClientConfig.getFileRemoteUrl(url);
        url = HttpClientConfig.processUrl(url);
        //设置http的默认端口号为80
        URL originUrl = new URL(url);
        String protocol = originUrl.getProtocol();
        int originPort = originUrl.getPort();
        //默认接口为-1，且originPort不为-1
        if(originPort != -1) {
            port = originPort;
        }

        if(fpaPort>0) {
            //走fpa代理逻辑
            mURL = new URL(protocol, originUrl.getHost(), originUrl.getFile());
            mConn = (HttpURLConnection) mURL.openConnection(new Proxy(Proxy.Type.HTTP,
                    new InetSocketAddress("127.0.0.1",fpaPort)));
        }else{
            mURL = new URL(protocol, originUrl.getHost(), port, originUrl.getFile());
            mConn = (HttpURLConnection) mURL.openConnection();
        }
    }

    /**
     * 设置请求方法
     * @param requestMethod
     * @throws ProtocolException
     */
    public void setRequestMethod(String requestMethod) throws ProtocolException {
        mConn.setRequestMethod(requestMethod);
    }

    /**
     * 设置连接过期时间
     * @param timeout
     */
    public void setConnectTimeout(int timeout) {
        if (timeout <= 0) {
            timeout = EM_DEFAULT_TIMEOUT;
        }
        mConn.setConnectTimeout(timeout);
    }

    public void setReadTimeout(int timeout) {
        if (timeout <= 0) {
            timeout = EM_DEFAULT_READ_TIMEOUT;
        }
        mConn.setReadTimeout(timeout);
    }

    /**
     * 是否重定向，默认进行重定向
     * @param followRedirect 默认为true
     */
    public void setFollowRedirect(boolean followRedirect) {
        mConn.setInstanceFollowRedirects(followRedirect);
    }

    /**
     * 为请求头加上token
     */
    public void setToken() {
        mConn.setRequestProperty("Authorization", "Bearer " + EMClient.getInstance().getOptions().getAccessToken());
    }

    /**
     * 添加默认的请求头
     */
    public void setDefaultProperty() {
        mConn.setRequestProperty("User-agent", HttpClientConfig.getDefaultUserAgent());
        mConn.setRequestProperty("Connection", "Keep-Alive");
    }

    /**
     * 添加上传文件的请求头
     */
    public void setUploadProperty() {
        mConn.setRequestProperty("Charset", "UTF-8");
        mConn.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);
        //mConn.setRequestProperty("Expect", "100-Continue");
    }

    /**
     * 设置长传文件的大小
     */
    public void setUploadSet() {
        mConn.setChunkedStreamingMode(0);
    }

    /**
     * Get请求时的设置
     */
    public void setGetConnection() {
        mConn.setDoInput(true);
    }

    /**
     * Post等请求时的设置
     */
    public void setPostConnection() {
        mConn.setDoOutput(true);
        mConn.setDoInput(true);
        mConn.setUseCaches(false);
    }


    /**
     * Delete等请求时的设置(android 4.4版本以上才支持setDoOutput方法)
     */
    public void setDeleteConnection() {
        if(android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
            mConn.setDoOutput(true);
            mConn.setDoInput(true);
            mConn.setUseCaches(false);
        }
    }

    public HttpURLConnection getHttpURLConnection() {
        return mConn;
    }

    /**
     * 添加请求头
     * @param headers
     */
    public void addHeader(Map<String, String> headers) {
        if (headers != null && headers.size() > 0) {
            for (Map.Entry<String, String> item : headers.entrySet()) {
                mConn.setRequestProperty(item.getKey(), item.getValue());
            }
        }
    }

    /**
     * 添加参数
     * @param params
     * @param out
     * @throws IOException
     */
    public void addParams(Map<String, String> params, OutputStream out) throws IOException {
        //EMLog.d(TAG, "request params = "+params.toString());
        if (params == null || params.size() <= 0) {
            return;
        }
        String paramsString = getParamsString(params);
        if (TextUtils.isEmpty(paramsString)) {
            return;
        }
        out.write(paramsString.getBytes());
        out.flush();
    }

    /**
     * 添加参数
     * @param params
     * @param out
     * @throws IOException
     */
    public void addParams(String params, OutputStream out) throws IOException {
        //EMLog.d(TAG, "request params = "+params);
        if(TextUtils.isEmpty(params)) {
            return;
        }
        out.write(params.getBytes());
        out.flush();
    }

    /**
     * 上传文件时添加文件
     * @param fileUri
     * @param out
     * @param callback
     * @throws IOException
     */
    public void addFile(Uri fileUri, OutputStream out, HttpCallback callback) throws IOException {
        addFile("file", fileUri, null, out, callback);
    }

    /**
     * 上传文件时添加文件
     * @param fileKey
     * @param fileUri
     * @param filename
     * @param out
     * @param callback
     * @throws IOException
     */
    public void addFile(String fileKey, Uri fileUri, String filename, OutputStream out, HttpCallback callback) throws IOException {
        if(TextUtils.isEmpty(fileKey)) {
            fileKey = "file";
        }
        if (!EMFileHelper.getInstance().isFileExist(fileUri)) {
            throw new FileNotFoundException("file not exist");
        }
        String fileName = EMFileHelper.getInstance().getFilename(fileUri);
        String mimeType = EMFileHelper.getInstance().getFileMimeType(fileUri);
        long fileLength = EMFileHelper.getInstance().getFileLength(fileUri);
        if(!TextUtils.isEmpty(filename)) {
            fileName = filename;
        }
        String filePath = EMFileHelper.getInstance().getFilePath(fileUri);
        InputStream in = null;
        try {
            if (!TextUtils.isEmpty(filePath) && new File(filePath).exists()) {
                in = new FileInputStream(new File(filePath));
            } else {
                in = mContext.getContentResolver().openInputStream(fileUri);
            }
            int available = in.available();
            // 检查文件大小
            if(available > fileLength) {
                fileLength = available;
            }
            //添加文件头
            out.write(getFileHeaderParamsString(fileKey, fileName, mimeType, fileLength).getBytes());
            //添加文件体
            byte[] tmp = new byte[2048];
            int l;
            long sum = 0;
            while ((l = in.read(tmp)) != -1) {
                out.write(tmp, 0, l);
                sum += l;
                // 检查sum的值，不能大于fileLength
                if(sum > fileLength) {
                    sum = fileLength;
                }
                if (callback != null) {
                    callback.onProgress(fileLength, sum);
                }
            }
        } finally {
            if (in != null) {
                in.close();
            }
        }
        //添加文件尾
        out.write(getFileEndString().getBytes());
        out.flush();
    }

    /**
     * 连接
     * @throws IOException
     */
    public HttpURLConnection connect() throws IOException {
        printRequestInfo(false);
        mConn.connect();
        return mConn;
    }

    /**
     * 打印请求信息
     * @param showInfo
     * @throws IllegalStateException
     */
    private void printRequestInfo(boolean showInfo) throws IllegalStateException{
        if(showInfo && mConn != null) {
            //EMLog.d(TAG, "request start =========================== ");
            //EMLog.d(TAG, "request url = "+mConn.getURL());
            //EMLog.d(TAG, "request method = "+mConn.getRequestMethod());
            //EMLog.d(TAG, "request header = "+mConn.getRequestProperties().toString());
            //EMLog.d(TAG, "request end =========================== ");
        }
    }

    /**
     * 获取请求相应
     * @return
     * @throws IOException
     */
    public HttpResponse getHttpResponse() throws IOException {
        HttpResponse response = new HttpResponse();
        response.code = mConn.getResponseCode();
        if(response.code == HttpURLConnection.HTTP_OK) {
            response.contentLength = mConn.getContentLength();
            response.inputStream = mConn.getInputStream();
            response.content = parseStream(response.inputStream);
        }else {
            response.errorStream = mConn.getErrorStream();
            response.content = parseStream(response.errorStream);
        }
        printResponseInfo(false, response);
        return response;
    }

    /**
     * 获取下载文件时的相应
     * @return
     * @throws IOException
     */
    public HttpResponse getFileHttpResponse() throws IOException{
        HttpResponse response = new HttpResponse();
        response.code = mConn.getResponseCode();
        response.contentLength = mConn.getContentLength();
        response.inputStream = mConn.getInputStream();
        response.errorStream = mConn.getErrorStream();
        if(response.code != HttpURLConnection.HTTP_OK) {
            response.content = parseStream(response.errorStream);
        }
        printResponseInfo(false, response);
        return response;
    }

    /**
     * 打印相应信息
     * @param showInfo
     * @param response
     */
    private void printResponseInfo(boolean showInfo, HttpResponse response) {
        if(mConn == null || response == null) {
            return;
        }
        if(showInfo) {//展示详细信息
            //EMLog.d(TAG, "response ==========================start =================");
            //EMLog.d(TAG, "content: "+response.content);
            //EMLog.d(TAG, "url: "+mConn.getURL().toString());
            //EMLog.d(TAG, "headers: "+mConn.getHeaderFields().toString());
            //EMLog.d(TAG, "response ==========================end =================");
        }else {//只展示相应码及错误信息
            EMLog.d(TAG, "response code: "+response.code);
            if(response.code != HttpURLConnection.HTTP_OK) {
                EMLog.d(TAG, "error message: "+response.content);
            }
        }
    }

    /**
     * 获取Exception response
     * @param e
     * @return
     * @throws IOException
     */
    public HttpResponse getExceptionResponse(Exception e) throws IOException{
        HttpResponse response = new HttpResponse();
        if(mConn != null) {
            response.code = mConn.getResponseCode();
            response.contentLength = mConn.getContentLength();
            response.errorStream = mConn.getErrorStream();
            mConn.disconnect();
        }
        response.exception = e;
        return response;
    }

    /**
     * 获取重定向地址
     * @return
     */
    private String getRedirectionUrl() {
        return mConn.getHeaderField("Location");
    }

    /**
     * 解析流信息
     * @param is
     * @return
     */
    private String parseStream(InputStream is) {
        String buf;
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(is, "UTF-8"));
            StringBuilder sb = new StringBuilder();
            String line = "";
            while ((line = reader.readLine()) != null) {
                sb.append(line + "\n");
            }
            buf = sb.toString();
            return buf;

        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 上传文件时得到拼接的参数字符串
     */
    private String getParamsString(Map<String, String> paramsMap) {
        if(paramsMap == null || paramsMap.size() <= 0) {
            return null;
        }
        StringBuffer strBuf = new StringBuffer();
        for (String key : paramsMap.keySet()){
            strBuf.append(TWO_HYPHENS);
            strBuf.append(BOUNDARY);
            strBuf.append(LINE_END);
            strBuf.append("Content-Disposition: form-data; name=\"" + key + "\"");
            strBuf.append(LINE_END);

            strBuf.append("Content-Type: " + "text/plain" );
            strBuf.append(LINE_END);
            strBuf.append("Content-Length: "+paramsMap.get(key).length());
            strBuf.append(LINE_END);
            strBuf.append(LINE_END);
            strBuf.append(paramsMap.get(key));
            strBuf.append(LINE_END);
        }
        return strBuf.toString();
    }

    /**
     * 拼接文件头
     * @param fileKey
     * @param filename
     * @param mimeType
     * @param fileLength
     * @return
     */
    private String getFileHeaderParamsString(String fileKey, String filename, String mimeType, long fileLength) {
        StringBuffer strBuf = new StringBuffer();
        strBuf.append(LINE_END);
        strBuf.append(TWO_HYPHENS);
        strBuf.append(BOUNDARY);
        strBuf.append(LINE_END);
        strBuf.append("Content-Disposition: form-data; name=\"" + fileKey + "\"; filename=\"" + filename + "\"");
        strBuf.append(LINE_END);
        strBuf.append("Content-Type: " + mimeType );
        strBuf.append(LINE_END);
        strBuf.append("Content-Length: "+ fileLength);
        strBuf.append(LINE_END);
        strBuf.append(LINE_END);
        return strBuf.toString();
    }

    /**
     * 获取文件结束标记位
     * @return
     */
    private String getFileEndString() {
        return LINE_END + TWO_HYPHENS + BOUNDARY + TWO_HYPHENS + LINE_END;
    }

    private void checkAndProcessSSL(String url) {
        HttpClientConfig.checkAndProcessSSL(url, mConn);
    }

    public static class HttpParams {
        public final Context mContext;
        public String mRequestMethod;
        public int mPort = -1;//https默认443，http默认80
        public int fpaPort = 0;//适配fpa代理专用
        public int mConnectTimeout;
        public int mReadTimeout;
        public boolean followRedirect = true;//重定向, 默认开启
        public boolean canRetry;//是否可以重试
        public int mRetryTimes;//重试次数

        public Map<String, String> mHeaders = new HashMap<>();
        public Map<String, String> mParams = new HashMap<>();
        public String mParamsString;
        public String mUrl;

        public String mLocalFileUri;
        public String mDownloadPath;
        public String mFilename;
        public String mFileKey;
        public boolean isUploadFile;
        public boolean isDownloadFile;
        public boolean isCheckSSL;
        public boolean isNotUseToken;//是否使用token
        public boolean isTokenExceeded;//token是否过期
        public boolean isDefaultRetry;//是否是指定的默认重试的情况

        public HttpParams(Context mContext) {
            this.mContext = mContext;
        }

        public void apply(HttpClientController controller) throws IOException{
            if(fpaPort>0) {
                controller.setURL(mUrl, mPort,fpaPort);
            }else{
                if(mPort != -1) {
                    controller.setURL(mUrl, mPort);
                }else {
                    controller.setURL(mUrl);
                }
            }
            controller.setRequestMethod(mRequestMethod);

            if("GET".equalsIgnoreCase(mRequestMethod)) {
                controller.setGetConnection();
            }else if("DELETE".equalsIgnoreCase(mRequestMethod)){
                controller.setDeleteConnection();
            }else{
                controller.setPostConnection();
            }

            controller.setConnectTimeout(mConnectTimeout);
            controller.setReadTimeout(mReadTimeout);
            controller.setDefaultProperty();
            if(!isNotUseToken) {
                controller.setToken();
            }
            controller.setFollowRedirect(followRedirect);
            if(isUploadFile) {
                controller.setUploadProperty();
                controller.setUploadSet();
            }
            if(isDownloadFile) {
                if(TextUtils.isEmpty(mDownloadPath)) {
                    throw new FileNotFoundException("file download path is empty");
                }
                checkDownloadProperty();
            }
            controller.checkAndProcessSSL(mUrl);
            checkToken();
            controller.addHeader(mHeaders);
        }

        public void addFile(HttpClientController controller, OutputStream out, HttpCallback callback) throws IOException {
            if(isUploadFile) {
                controller.addFile(mFileKey, EMFileHelper.getInstance().formatInUri(mLocalFileUri), mFilename, out, callback);
            }
        }

        public HttpResponse getResponse(HttpClientController controller) throws IOException {
            return isDownloadFile ? controller.getFileHttpResponse() : controller.getHttpResponse();
        }

        public String getRedirectionUrl(HttpClientController controller) {
            return controller.getRedirectionUrl();
        }

        public HttpResponse getExceptionResponse(HttpClientController controller, IOException e) throws IOException {
            if(controller != null) {
                return controller.getExceptionResponse(e);
            }
            return null;
        }

        public void checkDownloadProperty() {
            if(isDownloadFile) {
                mHeaders.put("Authorization", "Bearer " + EMClient.getInstance().getOptions().getAccessToken());
                mHeaders.put("Accept", "application/octet-stream");
            }
        }

        public void checkToken() {
            if(mHeaders.keySet().contains("Authorization")) {
                if(TextUtils.isEmpty(mHeaders.get("Authorization"))) {
                    mHeaders.put("Authorization", "Bearer " + EMClient.getInstance().getOptions().getAccessToken());
                }
            }
        }
    }

}
