/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Build;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;

import com.hyphenate.EMConnectionListener;
import com.hyphenate.EMMessageListener;
import com.hyphenate.chat.EMClient.AppStateListener;
import com.hyphenate.chat.adapter.EMAChatClient.EMANetwork;
import com.hyphenate.chat.adapter.EMAHeartBeatCustomizedParams;
import com.hyphenate.monitor.EMNetworkMonitor;
import com.hyphenate.push.EMPushHelper;
import com.hyphenate.push.EMPushType;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.NetUtils;
import com.hyphenate.util.Utils;

import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * ＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝＝
 * 基本规则：
 * o表示成功
 * x表示失败
 *
 * ->表示interval增加一个步长
 * <-表示interval减少一个步长
 *
 * 状态主要是eval，re-eval，hit
 * eval下，ping-pong成功就跳下一个点(增加interval)，继续ping-pong
 *        不成功就会跳，状态切换到re-eval
 * re-eval下，ping-pong成功救确认hit
 *        不成功，继续回跳(减小interval)
 * hit表示成功
 *
 * 基本流程：
 * 这个一个确认流程，到达顶点后失败，返回并到达hit状态
 *  o -> o -> x  [eval]
 *         <-    [re-eva]
 *       o       [hit]
 *
 * 这是一个到达最大值，到达hit状态的流程
 *  o -> o -> o  [max hit]
 *
 * 下面是处理断开
 * 1. 到达hit状态后，如果断开，进入eval
 *           [hit]
 *      disconnect
 *         <-
 *       [eval]
 *
 * 2. 非hit状态后，如果断开，进入re-eval
 *           [no-hit]
 *      disconnect
 *         <-
 *       [re-eval]
 *
 *
 *
 * eval评估的时候断网，就使用最后一次成功的interval，缺陷是没有找到最大点
 * 这是因为从程序逻辑上无法判断上次失败是因为正常断网，还是因为到达心跳间隔最大点
 * 这个缺陷可以接受，确认一次hit状态以一次断网为代价，开销也很大。
 *
 */
class EMSmartHeartBeat {
    private static final String TAG = "smart ping";
    static boolean Debug = false;
    private int fixedInterval = -1;

    abstract class IParams {

        abstract int getMinInterval();

        abstract int getMaxInterval();

        abstract int getDefaultInterval();

        abstract int getPingPongTimeout();

        abstract int getPingPongCheckInterval();

        int getNextInterval(final int interval, final boolean increase) {
            if (fixedInterval != -1) {
                EMLog.d(TAG, "use fixed interval: " + fixedInterval);
                return fixedInterval;
            }

            int result = interval;
            int sign = increase ? 1 : -1;
            if (sign < 0) {
                if (interval <= 60 * 1000) {
                    result = interval + sign * 10 * 1000;
                } else if (interval <= 120 * 1000) {
                    result = interval + sign * 30 * 1000;
                } else {
                    result = interval + sign * 45 * 1000;
                }
            } else {
                if (interval < 60 * 1000) {
                    result = interval + sign * 10 * 1000;
                } else if (interval < 120 * 1000) {
                    result = interval + sign * 30 * 1000;
                } else {
                    result = interval + sign * 45 * 1000;
                }
            }

            if (result > getMaxInterval()) {
                result = getMaxInterval();
            }
            if (result < getMinInterval()) {
                result = getMinInterval();
            }
            return result;
        }
    }

    // quick test
    class EMParamsQuickTest extends IParams {
        final static int MOBILE_DEFAULT_INTERVAL = 20 * 1000;
        final static int WIFI_DEFAULT_INTERVAL = 20 * 1000;

        /**
         * heartbeat最小间隔时间 30s
         */
        final static int MIN_INTERVAL = 10 * 1000;
        /**
         * max heartbeat interval 4.5min,因为服务器5分钟判断离线的规则，这个时间不能大于5分钟
         */
        final static int MAX_INTERVAL = (int) (30 * 1000);

        final static int MAX_MIN_INTERVAL_COUNTER = 3;

        final static int PING_PONG_TIMEOUT = 8 * 1000;
        final static int PING_PONG_CHECK_INTERVAL = 15 * 60 * 1000; //15min

        @Override
        int getDefaultInterval() {
            int heartbeatInterval = 0;
            if (NetUtils.isWifiConnected(mContext))
            {
                heartbeatInterval = WIFI_DEFAULT_INTERVAL;
            } else {
                heartbeatInterval = MOBILE_DEFAULT_INTERVAL;
            }
            return heartbeatInterval;
        }

        @Override
        int getMinInterval() {
            return MIN_INTERVAL;
        }

        @Override
        int getMaxInterval() {
            return MAX_INTERVAL;
        }

        @Override
        int getPingPongTimeout() {
            return PING_PONG_TIMEOUT;
        }

        @Override
        int getPingPongCheckInterval() {
            return PING_PONG_CHECK_INTERVAL;
        }

        int getNextInterval(final int interval, final boolean increase) {
            int result = interval;
            int sign = increase ? 1 : -1;
            result = interval + sign * 5 * 1000;

            if (result > getMaxInterval()) {
                result = getMaxInterval();
            }
            if (result < getMinInterval()) {
                result = getMinInterval();
            }
            return result;
        }

    }

    // 国内使用
    class EMParams extends IParams {
        final static int MOBILE_DEFAULT_INTERVAL = 180 * 1000;
        final static int WIFI_DEFAULT_INTERVAL = 120 * 1000;

        /**
         * heartbeat最小间隔时间 30s
         */
        final static int MIN_INTERVAL = 30 * 1000;
        /**
         * max heartbeat interval 4.5min,因为服务器5分钟判断离线的规则，这个时间不能大于5分钟
         */
        final static int MAX_INTERVAL = (int) (4.5 * 60 * 1000);

        final static int MAX_MIN_INTERVAL_COUNTER = 3;

        final static int PING_PONG_TIMEOUT = 8 * 1000;
        final static int PING_PONG_CHECK_INTERVAL = 15 * 60 * 1000; //15min

        @Override
        int getDefaultInterval() {
            if (fixedInterval != -1) return fixedInterval;

            int heartbeatInterval = 0;
            if (NetUtils.isWifiConnected(mContext))
            {
                heartbeatInterval = WIFI_DEFAULT_INTERVAL;
            } else {
                heartbeatInterval = MOBILE_DEFAULT_INTERVAL;
            }
            return heartbeatInterval;
        }

        @Override
        int getMinInterval() {
            return MIN_INTERVAL;
        }

        @Override
        int getMaxInterval() {
            return MAX_INTERVAL;
        }

        @Override
        int getPingPongTimeout() {
            return PING_PONG_TIMEOUT;
        }

        @Override
        int getPingPongCheckInterval() {
            return PING_PONG_CHECK_INTERVAL;
        }

    }

    // = = = = = = = = = = = = = = = = = = = = = =
    /**
     * support customize
     */
    class EMParamsCustomized extends IParams {

        private final int defaultInterval;
        private final int minInterval;
        private final int maxInterval;


        final static int PING_PONG_TIMEOUT = 8 * 1000;

        static final int PING_PONG_CHECK_INTERVAL = 15 * 60 * 1000; //15min

        EMParamsCustomized(EMAHeartBeatCustomizedParams params) {
            defaultInterval = params.defaultInterval;
            minInterval = params.minInterval;
            maxInterval = params.maxInterval;
        }

        @Override
        int getDefaultInterval() {
            if (fixedInterval != -1) return fixedInterval;

            EMLog.d(TAG, "get customized default: " + defaultInterval);
            return defaultInterval;
        }

        @Override
        int getMinInterval() {
            return minInterval;
        }

        @Override
        int getMaxInterval() {
            return maxInterval;
        }

        @Override
        int getPingPongTimeout() {
            return PING_PONG_TIMEOUT;
        }

        @Override
        int getPingPongCheckInterval() {
            return PING_PONG_CHECK_INTERVAL;
        }
    }

    boolean useCustomizedParams = false;
    EMAHeartBeatCustomizedParams mCustomizedWifiParams = null;
    EMAHeartBeatCustomizedParams mCustomizedMobileParams = null;

    public void setCustomizedParams(EMAHeartBeatCustomizedParams wifiParams, EMAHeartBeatCustomizedParams mobileParams) {
        if (wifiParams == null || mobileParams == null) {
            return;
        }
        useCustomizedParams = true;
        mCustomizedWifiParams = wifiParams;
        mCustomizedMobileParams = mobileParams;
    }

    public void setFixedInterval(int interval) {
        fixedInterval = interval * 1000;
    }
    // = = = = = = = = = = = = = = = = = = = = = =

    private Context mContext;
    

    private final static int PING_PONG_TIMEOUT = 8 * 1000;

    /**
     * 成功的心跳间隔
     */
    private int succeededInterval;
    
    /**
     * 当前的心跳间隔
     */
    private int currentInterval;
    
    private boolean dataReceivedDuringInterval = false;

    IParams params;
    
    /**
     * 上一次发送成功的ping-pong时间
     */

    private EMHeartBeatReceiver alarmIntentReceiver = null;  
    private PendingIntent     alarmIntent = null;  
    
    private EMConnectionListener cnnListener = null;
    private EMMessageListener messageListener = null;

    private WakeLock wakeLock;    
    private Object stateLock = new Object();
    private boolean inited = false;

    private boolean prevWifi = false;

    private Timer disconnectTimer = null;
    private TimerTask disconnectTask = null;
    private AlarmManager alarmManager;
    
    private boolean isInBackground = false;
    
    
//    private final static int GCM_PINGPONG_INTERVAL = 20 * 60 * 1000; //GCM心跳前台固定间隔20min
    private final static int GCM_DISCONNECT_CHECK_INTERVAL = 3 * 60 * 1000;
    
    private enum EMSmartPingState{
    	EMReady,
    	EMEvaluating,
    	EMReevaluating,
    	EMHitted,
    	EMStopped
    }
    
    private EMSmartPingState pingState = EMSmartPingState.EMReady;
    
    
    // manager msg count thread pool
    ExecutorService threadPool;

    private EMSmartHeartBeat(Context context) {
        mContext = context;
    }
    
    public static EMSmartHeartBeat create(Context context){
    	return new EMSmartHeartBeat(context);
    }
    
    public void onInit(){
        
        threadPool = Executors.newSingleThreadExecutor();
        
        changeState(EMSmartPingState.EMEvaluating);
        
        reset();

        disconnectTimer = new Timer();
        alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
        
        if(cnnListener == null){
            cnnListener = new EMConnectionListener() {
                
                @Override
                public void onDisconnected(int error) {
                    EMLog.d(TAG, " onDisconnected ..." + error);
                }

                @Override
                public void onConnected() {
                    EMLog.d(TAG, " onConnectred ...");
                    if(EMPushHelper.getInstance().getPushType() == EMPushType.FCM && isInBackground){
                        //最好是onesdk接到网络改变通知后，如果发现是gcm已经app在后台状态，不重连
                        EMClient.getInstance().disconnect();
                        return;
                    }
                    // start to send heatbeat after login to keep the long lived connection
                    calcDisconnectedInterval();
                    scheduleNextAlarm();
                }
            };
        }
        
        if(messageListener == null){
            messageListener = new EMMessageListener(){

                @Override
                public void onMessageReceived(List<EMMessage> messages) {
                    dataReceivedDuringInterval = true;
                    scheduleNextAlarm();
                }

                @Override
                public void onCmdMessageReceived(List<EMMessage> messages) {
                    dataReceivedDuringInterval = true;
                    scheduleNextAlarm();
                }

                @Override
                public void onMessageRead(List<EMMessage> messages)
                {
                    dataReceivedDuringInterval = true;
                    scheduleNextAlarm();
                }

                @Override
                public void onMessageDelivered(List<EMMessage> messages)
                {
                    dataReceivedDuringInterval = true;
                    scheduleNextAlarm();

                }
                @Override
                public void onMessageRecalled(List<EMMessage> messages) 
                {
                    dataReceivedDuringInterval = true;
                    scheduleNextAlarm();
                }

                @Override
                public void onMessageChanged(EMMessage message, Object change) {
                }

                @Override
                public void onGroupMessageRead(List<EMGroupReadAck> groupReadAcks) {

                }

                @Override
                public void onReadAckForGroupMessageUpdated() {

                }
            };
        }
        
        // add a connection listener
        // in case that we switched to a new network connection, we need to reset the smart heart
        EMClient.getInstance().addConnectionListener(cnnListener);
        
        EMClient.getInstance().chatManager().addMessageListener(messageListener);
        
        //get wake lock here
        PowerManager pm = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);
        if (wakeLock == null) {
            wakeLock = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "heartbeatlock");
        }
        
        if(Utils.isSdk14()){
            //gcm环境下，切到后台跟ios类似3分钟后断掉长连接
            EMClient.getInstance().setAppStateListener(new AppStateListener() {
                
                @Override
                public void onForeground() {
                    EMLog.d(TAG, "app onForeground");                    
                    isInBackground = false;
                    new Thread(new Runnable() {
                        
                        @Override
                        public void run() {
                            if(EMPushHelper.getInstance().getPushType() == EMPushType.FCM){
                                if(disconnectTask != null){
                                    disconnectTask.cancel();
                                }
                            }
                            if (NetUtils.hasDataConnection(mContext)) {
                                if(!EMClient.getInstance().isConnected()){
                                    EMClient.getInstance().onNetworkChanged();
                                } else {
                                    sendPingCheckConnection();
                                }
                            }
                        }
                    }).start();
                }
                
                @Override
                public void onBackground() {
                    isInBackground = true;
                    EMLog.d(TAG, "app onBackground");
                    new Thread(new Runnable() {
                        
                        @Override
                        public void run() {
                            if(EMPushHelper.getInstance().getPushType() == EMPushType.FCM){
                                //disconnect after in background 3 mins
                                disconnectTask = getTask();
                                try {
                                    disconnectTimer.schedule(disconnectTask, GCM_DISCONNECT_CHECK_INTERVAL);
                                    EMLog.d(TAG, "schedule disconnect task");
                                } catch (Exception e) {
                                }
                            }
                        }
                    }).start();
                }
            });
        }
        
        inited = true;
    }

    void sendPingCheckConnection(){
        boolean success =  EMClient.getInstance().sendPing(true, 3 * 1000);
        EMClient.getInstance().checkTokenAvailability();
        EMLog.d(TAG, "send check Ping " + (success ? "success" : "timeout"));
        if (success) {
            dataReceivedDuringInterval = true;
            scheduleNextAlarm();
        } else {
            EMClient.getInstance().forceReconnect();
        }
    }
    
    private TimerTask getTask(){
        return new TimerTask() {
            
            @Override
            public void run() {
                EMLog.d(TAG, "enter the disconnect task");
                if(EMClient.getInstance().isConnected()){
                    EMClient.getInstance().disconnect();
                }
                try {
                    //cancel heartbeat alarm
                    alarmManager.cancel(alarmIntent);
                    mContext.unregisterReceiver(alarmIntentReceiver);
                    alarmIntentReceiver = null;
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        };
    }
    
    private Runnable heartBeatRunnable = new Runnable() {

        @Override
        public void run() {
            EMLog.d(TAG, "has network connection:" + NetUtils.hasNetwork(mContext)
                       + " has data conn:" + NetUtils.hasDataConnection(mContext) +
                         " isConnected to hyphenate server : " + EMClient.getInstance().isConnected());
                        
            if(hasDataConnection()){
                //保证发送心跳包时系统不休眠
                try {
                    wakeLock.acquire();
                } catch (Exception e) {
                    EMLog.e(TAG, e.getMessage());
                }
                EMLog.d(TAG, "acquire wake lock");
                //3.0下所有的ping都是有response，
                //这样可以每次都去check是否成功,减少失败后重连的间隔时间
                checkPingPong();
                
                releaseWakelock();
                
            } else {
                EMLog.d(TAG, "....no connection to server");
                if(!NetUtils.hasDataConnection(mContext) && EMClient.getInstance().isConnected()){
                    if(Utils.isSdk14()){
                        EMLog.d(TAG, "no data connection but im connection is connected, reconnect");
                        EMClient.getInstance().onNetworkChanged(EMANetwork.NETWORK_NONE);
                    }else{
                        //14以下不能知道前后台的切换，用重连这种更耗电的处理方式
                        EMClient.getInstance().forceReconnect();
                    }
                }
            }
            
            //collect traffic statics in heart beat
            EMNetworkMonitor.collectTrafficStats();
            
            //start next alarm
            scheduleNextAlarm();
        }

    };
    
    
    private void checkPingPong() {
    	EMLog.d(TAG, "check pingpong ...");
        boolean successed = false; 
        //失败最多尝试failTries次
        /*
      心跳增长或减少步长,40s
     */
        int heartbeatStep = 30 * 1000;
        /*
      失败尝试次数
     */
        int failTries = 3;
        for(int i = 0; i < failTries; i++){
        	// in case the thread was interrupted
        	// we should return immediately
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                // We received an interrupt
                // This only happens if we should stop pinging
                EMLog.e(TAG, "heartbeat thread be interrupt");
                Thread.currentThread().interrupt();
                return;
            }
            
            try {
            	if(dataReceivedDuringInterval){
                    return;
                }
            	
                successed = sendPingPong();
                
                if(successed){
                	EMLog.d(TAG, "success to send ping pong ... with current heartbeat interval : " + EMCollector.timeToString(currentInterval));
                    
                	succeededInterval = currentInterval;
                    EMLog.d(TAG, "send ping-pong successes");
                    
                    if(pingState == EMSmartPingState.EMHitted){
                        EMLog.d(TAG,"that's already in the EMHitted state, just return...");
                        return;
                    }
                    
                    if(succeededInterval == params.getMaxInterval() || pingState == EMSmartPingState.EMReevaluating){
                    	if(succeededInterval == params.getMaxInterval()){
                        	EMLog.d(TAG, "Find the best interval, interval is the max interval");
                        }
                        
                        if(pingState == EMSmartPingState.EMReevaluating){
                        	EMLog.d(TAG, "success to pingping and current state is EMSmartPingState.EMReevaluating, so use current interval as final interval");
                        }
                        
                        EMLog.d(TAG, "enter the ping state : " + pingState);
                        changeState(EMSmartPingState.EMHitted);
                        
                        return;
                    }

                    currentInterval = params.getNextInterval(currentInterval, true);
                    break;
                }
            } catch (Exception e) {
                return;
            }
        }

        if(!successed){//重试几次都失败
        	EMLog.d(TAG, "failed to send ping pong ... with current heartbeat interval : " + EMCollector.timeToString(currentInterval));

            if(hasDataConnection()) {//重试几次都失败
                EMLog.d(TAG, "failed to send ping pong ... with current heartbeat interval : " + EMCollector.timeToString(currentInterval));
                if(pingState == EMSmartPingState.EMEvaluating || pingState == EMSmartPingState.EMHitted){
                    EMLog.d(TAG, "send ping-pong failed, but has success interval candidate with ping state : " + pingState + " enter EMSmartPingState.EMReevaluating");
                    changeState(EMSmartPingState.EMReevaluating);
                }
                succeededInterval = 0;
                EMClient.getInstance().forceReconnect();
            }
        }        
    }
    
    private boolean hasDataConnection(){
    	return NetUtils.hasDataConnection(mContext) && EMClient.getInstance().isConnected();
    }
    
    /**
     * release wake lock
     */
    private void releaseWakelock(){
        // user report two releaseWakelock compete at the same time
        // they are: logout and heartbeat alarm receiver
        synchronized (this) {
            if (wakeLock != null && wakeLock.isHeld()) {
                try {
                    wakeLock.release();
                } catch (Exception e) {
                    EMLog.e(TAG, e.getMessage());
                }
                EMLog.d(TAG, "released the wake lock");
            }
        }
    }


    private boolean sendPingPong(){
        EMLog.d(TAG, "send ping-pong type heartbeat");
        
        if(!EMClient.getInstance().isConnected()){
        	return false;
        }
        EMClient.getInstance().checkTokenAvailability();
        return EMClient.getInstance().sendPing(true, PING_PONG_TIMEOUT);        
    }
    
    
    /**
     * 启动发送心跳包runnable
     */
    public void start() {
    	if(pingState ==  EMSmartPingState.EMStopped){
    		return;
    	}
        
    	if(!EMClient.getInstance().isConnected() && NetUtils.hasDataConnection(mContext)){
    	    EMClient.getInstance().onNetworkChanged();
    	}

        if(!EMClient.getInstance().isConnected() || !NetUtils.hasNetwork(mContext)){
        	
        	if(dataReceivedDuringInterval){
        		dataReceivedDuringInterval = false;
            }
        	
        	scheduleNextAlarm();
            return;
        }

        if (dataReceivedDuringInterval) {
            dataReceivedDuringInterval = false;
            EMLog.d(TAG, "receiving packet...");
            return;
        }

        EMLog.d(TAG, "post heartbeat runnable");
        
        synchronized(this) {
            if (threadPool != null && !threadPool.isShutdown()) {
                threadPool.execute(heartBeatRunnable);
            }
        }
    }

    public void scheduleNextAlarm() {
        try {
            EMLog.d(TAG, "schedule next alarm");
            EMLog.d(TAG, "current heartbeat interval : " + EMCollector.timeToString(currentInterval) + " smart ping state : " + pingState);
            
            dataReceivedDuringInterval = false;
            
            if (alarmIntent == null) {
                Intent intentForService = new Intent("hyphenate.chat.heatbeat." + EMClient.getInstance().getChatConfigPrivate().getAppKey());
                intentForService.setPackage(mContext.getPackageName());
                if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
                    alarmIntent = PendingIntent.getBroadcast(mContext, 0, intentForService, PendingIntent.FLAG_IMMUTABLE);
                } else {
                    alarmIntent = PendingIntent.getBroadcast(mContext, 0, intentForService, 0);
                }
            }

            if (alarmIntentReceiver == null) {
                alarmIntentReceiver = new EMHeartBeatReceiver(this);
                IntentFilter filter = new IntentFilter("hyphenate.chat.heatbeat." + EMClient.getInstance().getChatConfigPrivate().getAppKey());
                mContext.registerReceiver(alarmIntentReceiver, filter);
            }
            
            long triggerTime = System.currentTimeMillis() + 3*60*1000;
            
            if(hasDataConnection()){
                if(currentInterval <= 0){
                    currentInterval = params.getDefaultInterval();
                
                    EMLog.d(TAG, "current heartbeat interval is not set, use default interval : " + EMCollector.timeToString(currentInterval));
                }    
                
                triggerTime = System.currentTimeMillis() + currentInterval;
            }else{
                triggerTime = System.currentTimeMillis() + 3*60*1000;
                EMLog.d(TAG,"is not connected to server, so use idle interval : 3 mins");
            }

            if (Build.VERSION.SDK_INT >= 31) {
                if (alarmManager.canScheduleExactAlarms()) {
                    alarmManager.setExactAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerTime, alarmIntent);
                } else {
                    alarmManager.setAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerTime, alarmIntent);
                }
            }else if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
                // Wakes up the device in Doze Mode
                alarmManager.setExactAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerTime, alarmIntent);
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                alarmManager.setExact(AlarmManager.RTC_WAKEUP,  triggerTime, alarmIntent);
            } else {
                alarmManager.set(AlarmManager.RTC_WAKEUP,  triggerTime, alarmIntent);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public void stop() {
        EMLog.d(TAG, "stop heart beat timer");
        
        if(!inited){
            EMLog.w(TAG,"smart heartbeat is not inited!");
            return;
        }
        
        changeState(EMSmartPingState.EMStopped);
        
        synchronized(this) {
            threadPool.shutdownNow();
        }
        
        reset();
        releaseWakelock();
        disconnectTimer.cancel();
        
        if(cnnListener != null){
        	EMClient.getInstance().removeConnectionListener(cnnListener);
        }
        
        if(messageListener != null){
        	EMClient.getInstance().chatManager().removeMessageListener(messageListener);
            messageListener = null;
        }

        try {
            alarmManager.cancel(alarmIntent);
            mContext.unregisterReceiver(alarmIntentReceiver);
            alarmIntentReceiver = null;
        } catch (Exception e) {
            if (e.getMessage().contains("Receiver not registered")) {
                // ignore;
            } else {
                // print other error to check
                e.printStackTrace();
            }
        }
    }

    private void changeState(EMSmartPingState toState){
    	EMLog.d(TAG, "change smart ping state from : " + pingState + " to : " + toState);
    	
    	synchronized(stateLock){
    		pingState = toState;
    	}
    }

    private boolean isSameNet(boolean isWifi) {
        EMLog.d(TAG, "prevWifi:" + prevWifi + " isWifi:" + isWifi);
        return prevWifi == isWifi;
    }
    
    private void reset() {
    	EMLog.d(TAG, "reset interval...");
        currentInterval = 0;
        succeededInterval = 0;
        dataReceivedDuringInterval = false;
        changeState(EMSmartPingState.EMEvaluating);
    }

    private void calcDisconnectedInterval() {
        EMLog.d(TAG, "reset interval...");
        boolean isWifi =  NetUtils.isWifiConnected(mContext);
        boolean isEthernet = NetUtils.isEthernetConnected(mContext);

        if (Debug) {
            params = new EMParamsQuickTest();
        } else {
            if (useCustomizedParams) {
                params = isEthernet ? new EMParams() :  new EMParamsCustomized( isWifi ? mCustomizedWifiParams : mCustomizedMobileParams );
            } else {
                params = new EMParams();
            }
        }

        boolean sameNet = isSameNet(isWifi);
        prevWifi = isWifi;
        if (!sameNet || currentInterval == 0) {
            currentInterval = params.getDefaultInterval();
            succeededInterval = 0;
            changeState(EMSmartPingState.EMEvaluating);
        } else {
            currentInterval = params.getNextInterval(currentInterval, false);
            if (pingState == EMSmartPingState.EMHitted) {
                changeState(EMSmartPingState.EMEvaluating);
            } else {
                changeState(EMSmartPingState.EMReevaluating);
            }
            succeededInterval = 0;
        }
        dataReceivedDuringInterval = false;

        EMLog.d(TAG, "reset currentInterval:" + EMCollector.timeToString(currentInterval));
    }
}
