/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.text.TextUtils;

import com.hyphenate.EMCallBack;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAConversation.EMASearchDirection;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * \~chinese
 * 会话类，用于定义单聊会话、群聊会话和聊天室会话。
 *
 * 每类会话中包含发送和接收的消息。
 *
 * 例如，从会话中获取未读消息数：
 * ```java
 *     // conversationId 可以为聊天的另一方 ID，群组 ID 或者聊天室 ID。
 *     EMConversation conversation = EMClient.getInstance().chatManager().getConversation(conversationId);
 *     int unread = conversation.getUnreadMsgCount();
 * ```
 * @version 3.0
 * 
 * \~english
 * The conversation class, which defines one-to-one conversations, group conversations, and chat room conversations.
 *
 * Each type of conversation involves messages that are sent and received.
 *
 * The following code shows how to get the number of the unread messages from the conversation.
 * ```java
 *     // ConversationId can be the ID of the peer user, the group ID, or the chat room ID:
 *     EMConversation conversation = EMClient.getInstance().chatManager().getConversation(conversationId);
 *     int unread = conversation.getUnreadMsgCount();
 * ```
 *
*/

public class EMConversation extends EMBase<EMAConversation> {

    private static final String TAG = "conversation";
    private static final int LIST_SIZE = 512;

    EMConversation(EMAConversation conversation) {
        emaObject = conversation;
    }

    /**
     * \~chinese
     * 会话类型。
     *
     * \~english
     * The conversation types.
     */
    public enum EMConversationType {
        /**
         *\~chinese 
         * 单聊会话。
         *
         *\~english
         * One-to-one chat.
         */
        Chat,

        /**
         *\~chinese
         * 群聊会话。
         *
         *\~english
         * Group chat.
         */
        GroupChat,

        /**
         *\~chinese
         *聊天室会话。
         *\~english
         *Chat room.
         */
        ChatRoom,

        /**
         *\~chinese
         * 讨论组会话。暂不支持。
         *\~english
         * Discussion group. Currently, this type of conversation is not available.

         */
        DiscussionGroup,

        /**
         * \~chinese
         * 客服会话。
         *
         * \~english
         * Help desk.
         */
        HelpDesk
    }

    /**
     *  \~chinese
     *  消息搜索方向。
     *
     *  消息搜索基于消息中包含的 Unix 时间戳。每个消息中包含两个 Unix 时间戳：
     * - 消息创建的 Unix 时间戳；
     * - 服务器接收消息的 Unix 时间戳。
     * 消息搜索基于哪个 Unix 时间戳取决于 {@link EMOptions#setSortMessageByServerTime(boolean)}} 的设置。
     *
     *  \~english
     *  The message search direction.
     *
     * The message research is based on the Unix timestamp included in messages. Each message contains two Unix timestamps:
     * - The Unix timestamp when the message is created;
     * - The Unix timestamp when the message is received by the server.
     *
     * Which Unix timestamp is used for message search depends on the setting of {@link EMOptions#setSortMessageByServerTime(boolean)}.
     *
     */
    public enum EMSearchDirection {
        /**
         *\~chinese
         * 按照消息中的 Unix 时间戳的逆序搜索。
         *
         *\~english
         * Messages are retrieved in the descending order of the Unix timestamp included in them.
         */
        UP,     

        /**
         *\~chinese
         *  按照消息中的时间戳的正序搜索。
         *
         *\~english
         * Messages are retrieved in the ascending order of the Unix timestamp included in them.
         */
        DOWN    
    }

    /**
     * \~chinese
     * 会话 ID，取决于会话类型。
     * - 单聊/help desk：会话 ID 为对方的用户 ID。
     * - 群聊：会话 ID 为群组 ID。
     * - 聊天室：会话 ID 为聊天室的 ID。
     *
     * @return 会话 ID。
     *
     * \~english
     * The conversation ID, which depends on the conversation type.
     * - One-to-one chat/help desk: The conversation ID is the user ID of the peer user.
     * - Group chat: The conversation ID is the group ID.
     * - Chat room: The conversation ID is the chat room ID.
     *
     * @return The conversation ID.
     */
    public String conversationId() {
        return emaObject.conversationId();
    }

    /**
     * \~chinese
     * 获取会话类型。
     *
     * @return 会话类型。
     *  
     * \~english
     * Gets the conversation type.

     * @return  The conversation type.
     */
    public EMConversationType getType() {
        EMAConversationType t = emaObject._getType();
        if (t == EMAConversationType.CHAT) {
            return EMConversationType.Chat;
        }
        if (t == EMAConversationType.GROUPCHAT) {
            return EMConversationType.GroupChat;
        }
        if (t == EMAConversationType.CHATROOM) {
            return EMConversationType.ChatRoom;
        }
        if (t == EMAConversationType.DISCUSSIONGROUP) {
            return EMConversationType.DiscussionGroup;
        }
        if (t == EMAConversationType.HELPDESK) {
            return EMConversationType.HelpDesk;
        }
        return EMConversationType.Chat;
    }

    /**
     * \~chinese
     * 获取会话中未读的消息数量。
     *
     * @return 未读的消息数量。
     *
     * \~english
     * Gets the number of unread messages in the conversation.
     *
     * @return The unread message count in the conversation.
     */
    public int getUnreadMsgCount() {
        return emaObject.unreadMessagesCount();
    }

    /**
     *  \~chinese
     *  将所有未读消息设置为已读。
     *
     *  \~english
     *  Marks all unread messages as read.
     */
    public void markAllMessagesAsRead() {
        emaObject.markAllMessagesAsRead(true);
    }

    /**
     * \~chinese
     * 获取 SDK 本地数据库中会话的全部消息数。
     *
     * @return 会话的全部消息数量。
     *
     * \~english
     * Gets the number of all unread messages in the conversation in the local database.
     *
     * @return The number of all unread messages in the conversation.
     */
    public int getAllMsgCount() {
        return emaObject.messagesCount();
    }

    /**
     * \~chinese
     * 查看当前会话是否是子区会话。
     *
     * @return 返回是否是子区会话的结果。
     *         - `true` 表示是子区会话。
     *         - `false` 表示不是子区会话。
     *
     * \~english
     * Checks whether the current conversation is a thread conversation.
     *
     * @return Whether the conversation is a chat thread conversation.
     *         - `true`: Yes.
     *         - `false`: No.
     */
    public boolean isChatThread() {
        return emaObject.isChatThread();
    }

    /**
     * \~chinese
     * 从 SDK 本地数据库中分页加载消息。
     *
     * 加载的消息会基于消息中的时间戳放入当前会话的缓存中，调用 {@link #getAllMessages()} 时会返回所有加载的消息。
     *
     * @param startMsgId    查询的起始消息 ID。SDK 从该消息 ID 开始按消息时间戳的逆序加载。如果传入消息的 ID 为空，SDK 从最新消息开始按消息时间戳的逆序获取。
     * @param pageSize      每页期望加载的消息数。取值范围为 [1,400]。
     * @return              消息列表（不包含查询起始 ID 的消息）。
     *
     * \~english
     * Loads messages from the local database.
     *
     * The loaded messages will be put in the conversation in the memory according to the timestamp in them and will be returned when you call {@link #getAllMessages()}.
     *
     * @param startMsgId    The starting message ID for query. The SDK loads messages, starting from the specified one, in the descending order of the timestamp included in them.
     *                      If this parameter is set as `null` or an empty string, the SDK retrieves messages, from the latest one, according to the descending order of the timestamp included in them.
     * @param pageSize      The number of messages that you expect to get on each page. The value range is [1,400].
     * @return              The list of retrieved messages (excluding the one with the starting ID).
     */
    public List<EMMessage> loadMoreMsgFromDB(String startMsgId, int pageSize) {
        return loadMoreMsgFromDB(startMsgId, pageSize, EMSearchDirection.UP);
    }

    /**
     * \~chinese
     * 从指定消息 ID 开始分页加载数据库中的消息。
     *
     * 加载到的消息会加入到当前会话的消息中。
     *
     * @param startMsgId    查询的起始消息 ID。该参数设置后，SDK 从指定的消息 ID 开始按消息检索方向加载。
     *                      如果传入消息的 ID 为空，SDK 忽略该参数，按搜索方向查询消息：
 *                          - 若 `direction` 为 `UP`，SDK 从最新消息开始，按消息时间戳的倒序获取；
 *                          - 若 `direction` 为 `DOWN`，SDK 从最早消息开始，按消息时间戳的正序获取。
     * @param pageSize      每页期望加载的消息数。取值范围为 [1,40]。
     * @param direction     消息搜索方向。详见 {@link EMSearchDirection}。
	 *                       - `UP`：按照消息中的时间戳的逆序查询；
	 * 						 - `DOWN`：按照消息中的时间戳的正序查询；
     * @return              消息列表（不包含查询起始时间戳对应的消息）。
     *
     * \~english
     * Loads the messages from the local database, starting from a specific message ID.
     *
     * The loaded messages will be put in the conversation in the memory according to the timestamp included in them.
     *
     * @param startMsgId    The starting message ID for query. After this parameter is set, the SDK retrieves messages, from the specified one, according to the message search direction.
     *                      If this parameter is set as `nil`, the SDK retrieves messages according to the search direction while ignoring this parameter.
 *                          - If `direction` is set as `UP`, the SDK retrieves messages, starting from the latest one, in the descending order of the timestamp included in them.
 *                          - If `direction` is set as `DOWN`, the SDK retrieves messages, starting from the oldest one, in the ascending order of the timestamp included in them.
     * @param pageSize      The number of messages that you expect to get on each page. The value range is [1,400].
     * @param direction     The message search direction. See {@link EMSearchDirection}.
	 *                       - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					     - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.

     * @return              The list of retrieved messages (excluding the one with the starting ID).
     */
    public List<EMMessage> loadMoreMsgFromDB(String startMsgId, int pageSize, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;
        List<EMAMessage> msgs = emaObject.loadMoreMessages(startMsgId, pageSize, d);
        List<EMMessage> result = new ArrayList<EMMessage>();
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        getCache().addMessages(result);
        return result;
    }

    /**
     * \~chinese
     * 基于 Unix 时间戳搜索本地数据库中的消息。
     *
     * @param timeStamp  查询的起始消息 Unix 时间戳，单位为毫秒。该参数设置后，SDK 从指定时间戳的消息开始，按消息搜索方向获取。
	 *                   如果该参数设置为负数，SDK 从最新消息获取。
     * @param maxCount   每次获取的最大消息数。取值范围为 [1,400]。
     * @param direction  消息搜索方向。详见 {@link EMSearchDirection}。
	 *                    - `UP`：按照消息中的时间戳的逆序查询；
	 * 				      - `DOWN`：按照消息中的时间戳的正序查询。
     * @return           消息列表（不包含查询起始时间戳对应的消息）。
     *
     * \~english
     * Retrieves messages in the local database based on the Unix timestamp included in them.
     *
     * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If you set this parameter as a negative value, the SDK retrieves messages, starting from the current time, in the descending order of the timestamp included in them.
     * @param maxCount   The maximum number of message to retrieve each time. The value range is [1,400].
     * @param direction  The message search direction. See {@link EMSearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     *
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     *
     */
    public List<EMMessage> searchMsgFromDB(long timeStamp, int maxCount, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(timeStamp, maxCount, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 从本地数据库获取指定会话的一定数量的特定类型的消息。
     *
     * @param type       消息类型。详见 {@link Type}。
     * @param timeStamp  查询的起始消息 Unix 时间戳，单位为毫秒。该参数设置后，SDK 从指定时间戳的消息开始，按消息搜索方向获取。
     * @param maxCount   每次获取的最大消息数。取值范围为 [1,400]。
     * @param from       单聊或群聊中的消息发送方的用户 ID。若设置为 `null` 或空字符串，SDK 将在整个会话中搜索消息。
     * @param direction  消息搜索方向。详见 {@link EMSearchDirection}。
	 *                    - `UP`：按照消息中的时间戳的逆序查询；
	 * 				      - `DOWN`：按照消息中的时间戳的正序查询。
     * @return           消息列表（不包含查询起始时间戳对应的消息）。
     *
     * \~english
     * Gets messages of certain types that a specified user sends in the conversation.
     *
     * @param type       The message type. See {@link Type}.
     * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If this parameter is set as a negative value, the SDK retrieves from the latest message.
     * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
     * @param from       The user ID of the message sender in one-to-one chat or group chat.
     *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
     * @param direction  The message search direction. See {@link EMSearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     */
    public List<EMMessage> searchMsgFromDB(EMMessage.Type type, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 从本地数据库获取会话中的指定用户发送的包含特定关键词的消息。
     *
     * @param keywords   查询的关键字。
     * @param timeStamp  查询的起始消息 Unix 时间戳，单位为毫秒。该参数设置后，SDK 从指定时间戳的消息开始，按消息搜索方向获取。
	 *                   如果该参数设置为负数，SDK 从当前时间开始搜索。
     * @param maxCount   每次获取的最大消息数。取值范围为 [1,400]。
     * @param from       单聊或群聊中的消息发送方的用户 ID。若设置为 `null` 或空字符串，SDK 将在整个会话中搜索消息。
     * @param direction  消息搜索方向。详见 {@link EMSearchDirection}。
	 *                    - `UP`：按照消息中的时间戳的逆序查询；
	 * 				      - `DOWN`：按照消息中的时间戳的正序查询。
     * @return           消息列表（不包含查询起始时间戳对应的消息）。
     *
     * \~english
     * Gets messages with keywords that the specified user sends in the conversation.
     *
     * @param keywords   The keywords for query.
     * @param timeStamp  The starting Unix timestamp for search. The unit is millisecond.
     *                   If this parameter is set as a negative value, the SDK retrieves from the current time.
     * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
     * @param from       The user ID of the message sender in one-to-one chat or group chat.
     *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
     * @param direction  The message search direction. See {@link EMSearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     */
    public List<EMMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 从本地数据库中搜索指定时间段内发送或接收的一定数量的消息。
     *
     * @param startTimeStamp   搜索的起始时间戳。单位为毫秒。详见 {@link EMOptions#setSortMessageByServerTime}。
     * @param endTimeStamp     搜索的结束时间戳。单位为毫秒。详见 {@link EMOptions#setSortMessageByServerTime}。
     * @param maxCount         每次获取的最大消息数量。取值范围为 [1,400]。
     * @return                 消息列表（不包含搜索起始时间戳和结束时间戳对应的消息）。
     *
     * \~english
     * Gets a certain quantity of messages sent or received in a certain period from the local database.
     *
     * @param startTimeStamp    The starting Unix timestamp in the message for query. The unit is millisecond. See {@link EMOptions#setSortMessageByServerTime}.
     * @param endTimeStamp      The ending Unix timestamp in the message for query. The unit is millisecond. See {@link EMOptions#setSortMessageByServerTime}.
     * @param maxCount          The maximum number of messages to retrieve each time. The value range is [1,400].
     * @return                  The list of retrieved messages (excluding the ones with the starting timestamp and ending timestamp).
     */
    public List<EMMessage> searchMsgFromDB(long startTimeStamp, long endTimeStamp, int maxCount) {
        List<EMAMessage> msgs = emaObject.searchMessages(startTimeStamp, endTimeStamp, maxCount);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 从本地数据库获取会话中的指定用户发送的包含特定关键词的自定义消息。
     *
     * @param keywords   搜索的关键词。
     * @param timeStamp  查询的起始消息 Unix 时间戳，单位为毫秒。该参数设置后，SDK 从指定时间戳的消息开始，按消息搜索方向获取。
	 *                   如果该参数设置为负数，SDK 从当前时间开始搜索。
     * @param maxCount   每次获取的最大消息数。取值范围为 [1,400]。
     * @param from       单聊或群聊中的消息发送方的用户 ID。若设置为 `null` 或空字符串，SDK 将在整个会话中搜索消息。
     * @param direction  消息加载方向。详见 {@link EMSearchDirection}。
     * @return           消息列表（不包含查询起始时间戳对应的消息）。
     *
     * \~english
     * Gets custom messages with keywords that the specified user sends in the conversation.
     *
     * Note: Be cautious about memory usage when the maxCount is large.
     *
     * @param keywords   The keywords for search.
     * @param timeStamp  TThe starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If this parameter is set as a negative value, the SDK retrieves from the current time.
     * @param maxCount   The maximum number of messages to retrieve. The value range is [1,400].
     * @param from       The user ID of the message sender in one-to-one chat or group chat.
     *                   If this parameter is set to `null` or an empty string, the SDK searches for messages in the entire conversation.
     * @param direction  The message search direction. See {@link EMSearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
     * @return           The list of retrieved messages (excluding the one with the starting timestamp).
     */
    public List<EMMessage> searchCustomMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchCustomMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据消息 ID 获取已读的消息。
     *
     * SDK 首先在内存中查找消息，若在内存中未找到，SDK 会在本地数据库查询并加载。
     * 
     * @param  messageId    消息 ID。
     * @param  markAsRead   是否将获取的消息标为已读。
     *                      - `true`：是。
     *                      - `false`：否
     * @return              获取到的消息实例。
     *
     * \~english
     * Gets the message with message ID.
     *
     * The SDK first retrieves the message from the memory. If no message is found, the SDK will retrieve it from the local database and load it.
     *
     * @param  messageId     The message ID.
     * @param  markAsRead    Whether to mark the retrieved message as read.
     *                       - `true`: Yes.
     *                       - `false`: No.
     * @return message       The retrieved message instance.
     */
    public EMMessage getMessage(String messageId, boolean markAsRead) {
        EMLog.d(TAG, "getMessage messageId: "+messageId+" markAsRead: "+markAsRead);
        EMMessage msg = getCache().getMessage(messageId);
        if (msg == null) {
            EMAMessage emaMsg = emaObject.loadMessage(messageId);
            if (emaMsg == null) {
                return null;
            }
            msg = new EMMessage(emaMsg);
        }
        emaObject.markMessageAsRead(messageId, markAsRead);
        return msg;
    }

    /**
     * \~chinese
     * 设置指定消息为已读。
     *
     * 也可调用 {@link EMMessage#setUnread(boolean)} 将消息置为已读。
     *
     * @param messageId 消息 ID。
     *
     * \~english
     * Marks a message as read.
     *
     * You can also call {@link EMMessage#setUnread(boolean)} to mark the message as read.
     *
     * @param messageId The message ID.
     */
    public void markMessageAsRead(String messageId) {
        emaObject.markMessageAsRead(messageId, true);
    }

    /**
     * \~chinese
     * 获取该会话当前内存中的所有消息。
     *
     * 如果内存中为空，SDK 再从本地数据库中加载最近一条消息。
     *
     * @return 消息列表。
     *
     * \~english
     * Gets all messages of the conversation in the local memory.
     *
     * If no message is found in the memory, the SDK will load the latest message from the local database.
     *
     * @return The message list.
     */
    public List<EMMessage> getAllMessages() {
        if (getCache().isEmpty()) {
            EMAMessage lastMsg = emaObject.latestMessage();
            List<EMMessage> msgs = new ArrayList<EMMessage>();
            if (lastMsg != null) {
                msgs.add(new EMMessage(lastMsg));
            }
            getCache().addMessages(msgs);
        }
        return getCache().getAllMessages();
    }

    /**
     * \~chinese
     * 删除本地数据库中的一条指定消息。
     *
     * @param messageId     待删除消息的 ID。
     *
     * \~english
     * Deletes a message in the local database.
     *
     *
     * @param messageId    The ID of the message to delete.
     */
    public void removeMessage(String messageId) {
        EMLog.d(TAG, "remove msg from conversation: " + messageId);
        emaObject._removeMessage(messageId);
        getCache().removeMessage(messageId);
    }

    /**
     * \~chinese
     * 获取会话中的最新一条消息。
     *
     * 此操作不会改变未读消息计数。
     *
     * SDK 首先在内存中查找消息，若在内存中未找到，SDK 会在本地数据库查询并加载。
     *
     * @return 消息。
     *
     * \~english
     * Gets the latest message in the conversation.
     *
     * A call to this method has no impact on the number of unread messages.
     *
     * The SDK first retrieves the message from the memory. If no message is found, the SDK will retrieve it from the local database and load it.
     *
     * @return  The message instance.
     */
    public EMMessage getLastMessage() {
        if (getCache().isEmpty()) {
            EMAMessage _msg = emaObject.latestMessage();
            EMMessage msg = _msg == null ? null : new EMMessage(_msg);
            getCache().addMessage(msg);
            return msg;
        } else {
            return getCache().getLastMessage();
        }
    }

    /**
     * \~chinese
     * 获取会话中收到的最新一条消息。
     *
     * @return 消息。
     *
     * \~english
     * Gets the latest received message in the conversation.
     *
     * @return  The message instance.
     */
    public EMMessage getLatestMessageFromOthers() {
        EMAMessage _msg = emaObject.latestMessageFromOthers();
        EMMessage msg = _msg == null ? null : new EMMessage(_msg);
        getCache().addMessage(msg);
        return msg;
    }

    /**
     * \~chinese
     * 清除会话中的所有消息。
     *
     * 只清除内存的，不清除本地数据库的消息。
     *
     * 在退出会话的时候清除内存缓存，减小内存消耗。
     *
     * \~english
     * Deletes all the messages in the conversation.
     *
     * This method only deletes all the messages in a conversation in the memory, but not the messages in the local database.
     *
     * When exiting a conversation, you need to clear the memory to reduce the memory consumption.
     */
    public void clear() {
        getCache().clear();
    }

    /**
     * \~chinese
     * 清除内存和数据库中指定会话中的消息。
     *
     * \~english
     * Deletes all the messages in the conversation from the memory and local database.
     */
    public void clearAllMessages() {
        emaObject.clearAllMessages();
        getCache().clear();
    }

    /**
     * \~chinese
     * 设置会话的扩展字段。
     *
     * 扩展字段只在本地数据库中保存，不同步到服务器。
     *
     * 子区会话的扩展字段不在本地数据库中保存。
     *
     * @param ext       会话的扩展字段。
     *
     * \~english
     * Sets the extension field of the conversation.
     *
     * The extension field is only stored in the local database, but not synchronized to the server.
     *
     * The extension field of a chat thread conversation is not saved in the local database.
     *
     * @param ext       The extension field of the conversation.
     */
    public void setExtField(String ext) {
        if(!isChatThread()) {
            emaObject._setExtField(ext);
        }
    }

    /**
     * \~chinese
     * 获取会话的扩展字段。
     *
     * 扩展字段只在本地数据库中保存，不同步到服务器。
     *
     * 子区会话的扩展字段不在本地数据库中保存。
     *
     * @return      会话的扩展字段。
     *
     * \~english
     * Gets the extension field of the conversation.
     *
     * The extension field is only stored in the local database, but not synchronized to the server.
     *
     * The extension field of a chat thread conversation is not saved in the local database.
     *
     * @return The extension content.
     */
    public String getExtField() {
        return emaObject.extField();
    }

    /**
     * \~chinese
     * 将消息类型转化为会话类型。
     *
     * @param       id   消息 ID。
     * @param       type 消息类型。
     * @return      会话类型。
     *
     * \~english
     * Converts a message type to a conversation type.
     *
     * @param       id    The message ID.
     * @param       type  The message type.
     * @return      The conversation type.
     */
    public static EMConversationType msgType2ConversationType(String id, EMMessage.ChatType type) {
        switch (type) {
            case GroupChat:
                return EMConversationType.GroupChat;
            case ChatRoom:
                return EMConversationType.ChatRoom;
            default:
                return EMConversationType.Chat;
        }
    }

    /**
     * \~chinese
     * 获取是否是群组或者聊天室会话。
     *
     * @return 群组和聊天室类型都会返回 `true`，其他类型返回 `false`。
     *
     * \~english
     * Checks whether it is a group conversation or a chat room conversation, or a conversation of another type.
     *
     * @return  Whether it is a group conversation or a chat room conversation, or a conversation of another type.
     *          - `true`: It is a group conversation or a chat room conversation.
     *          - `false`: It is not a group conversation or a chat room conversation.
     */
    public boolean isGroup() {
        EMConversationType type = getType();
        // 需要排除Thread会话
        return !isChatThread() && (type == EMConversationType.GroupChat ||
                type == EMConversationType.ChatRoom);
    }

    /**
     *  \~chinese
     *
     * 在本地数据库的会话中插入一条消息。
     *
     * 消息的会话 ID 应与会话的 ID 一致。
     *
     * 消息会根据消息里的 Unix 时间戳插入本地数据库，SDK 会更新会话的 `latestMessage` 等属性。
     *
     *  @param msg 消息实例。
     *
     *  \~english
     * Inserts a message to the conversation in the local database。
     *
     * To insert the message correctly, ensure that the conversation ID of the message is the same as that of the conversation.
     *
     * The message will be inserted based on the Unix timestamp included in it. Upon message insertion, the SDK will automatically update attributes of the conversation, including `latestMessage`.
     *
     *  @param msg  The message instance.
     */
    public boolean insertMessage(EMMessage msg) {
        EMLog.d(TAG, "insertMessage msg: {msgId:"+msg.getMsgId()+" conversation:"+msg.conversationId()+" unread:"+msg.isUnread()+"}");
        boolean result = emaObject.insertMessage(msg.emaObject);
        if(result) {
            getCache().addMessage(msg);
        }
        return result;
    }

    /**
     *  \~chinese
     * 在本地数据库中会话的尾部插入一条消息。
     *
     * 消息的会话 ID 应与会话的 ID 一致。
     *
     * 消息插入后，SDK 会自动更新会话的 `latestMessage` 等属性。
     *
     *  @param msg 消息实例。
     *
     *  \~english
     * Inserts a message to the end of the conversation in the local database.
     *
     * To insert the message correctly, ensure that the conversation ID of the message is the same as that of the conversation.
     *
     * After a message is inserted, the SDK will automatically update attributes of the conversation, including `latestMessage`.
     *
     *  @param msg The message instance.
     */
	public boolean appendMessage(EMMessage msg) {
        EMLog.d(TAG, "appendMessage msg: {msgId:"+msg.getMsgId()+" conversation:"+msg.conversationId()+" unread:"+msg.isUnread()+"}");
	    boolean result = emaObject.appendMessage(msg.emaObject);
	    if(result) {
            getCache().addMessage(msg);
	    }
	    return result;
	}

    /**
     *  \~chinese
     * 更新本地数据库的指定消息。
     *
     * 消息更新时，消息 ID 不会修改。
     *
     * 消息更新后，SDK 会自动更新会话的 `latestMessage` 等属性。
     *
     *  @param msg 要更新的消息。
     *
     *  \~english
     * Updates a message in the local database.
     *
     * After you update a message, the message ID remains unchanged and the SDK automatically updates attributes of the conversation, like `latestMessage`.
     *
     *  @param msg  The message to update.
     */
    public boolean updateMessage(EMMessage msg) {
        EMLog.d(TAG, "updateMessage msg{ msgId:"+msg.getMsgId()+" conversation:"+msg.conversationId()+" unread:"+msg.isUnread()+"}");
        boolean updateMessage = emaObject.updateMessage(msg.emaObject);
        if(updateMessage) {
            getCache().addMessage(msg);
        }
        return updateMessage;
    }

    /**
     * \~chinese
     * 获取会话对应的附件的存储路径。
     *
     * 该方法适用于清理会话数据前，如果不确定该路径一定存在，请调用该方法前查询路径。如有必要，添加异常保护。
     *
     * @return 附件存储路径。
     *
     * \~english
     * Gets the path to save attachments in the conversation.
     *
     * Before clearing conversation data, you can call this method to check the path to save attachments related to the conversation. If necessary, implement exception protection.
     *
     * @return The path to save attachments in the conversation.
     */
    public String getMessageAttachmentPath() {
        String downloadPath = EMClient.getInstance().getChatConfigPrivate().getDownloadPath();
        return downloadPath + "/" + EMClient.getInstance().getCurrentUser()
                + "/" + conversationId();
    }

    /**
     * \~chinese
     * 单向删除漫游消息（根据msgId删除）
     *
     * @param msgIdList 		msgId列表
     * @param callBack			处理结果回调,详见 {@link EMCallBack}。
     *
     * \~english
     * @param msgIdList 		msgId list
     * @param callBack			The result callback which contains the error information if the method fails. See {@link EMCallBack}.
     */
    public void removeMessagesFromServer(List<String> msgIdList, EMCallBack callBack){
        EMClient.getInstance().chatManager().removeMessagesFromServer(conversationId(),getType(),msgIdList,callBack);
    }

    /**
     * \~chinese
     * 单向删除漫游消息（根据时间节点删除 只支持按开始时间 向上删除）
     *
     * @param beforeTimeStamp	 开始时间 单位毫秒
     * @param callBack			 处理结果回调,详见 {@link EMCallBack}。
     *
     * \~english
     * One way delete roaming message (Only upward deletion is available according to time node deletion)
     *
     * @param beforeTimeStamp	 Time Node（millisecond）
     * @param callBack			 The result callback which contains the error information if the method fails. See {@link EMCallBack}.
     */
    public void removeMessagesFromServer(long beforeTimeStamp, EMCallBack callBack){
        EMClient.getInstance().chatManager().removeMessagesFromServer(conversationId(),getType(),beforeTimeStamp,callBack);
    }



    // ====================================== Message cache ======================================

    MessageCache getCache() {
        MessageCache cache;
        synchronized (EMClient.getInstance().chatManager().caches) {
            cache = EMClient.getInstance().chatManager().caches.get(emaObject.conversationId());
            if (cache == null) {
                cache = new MessageCache();
            }
            EMClient.getInstance().chatManager().caches.put(emaObject.conversationId(), cache);
        }
        return cache;
    }

    static class MessageCache {

        TreeMap<Long, Object> sortedMessages = new TreeMap<Long, Object>(new MessageComparator());
        Map<String, EMMessage> messages = new HashMap<String, EMMessage>();
        Map<String, Long> idTimeMap = new HashMap<String, Long>();
        boolean hasDuplicateTime = false;

        final boolean sortByServerTime = EMClient.getInstance().getChatConfigPrivate().getOptions().isSortMessageByServerTime();

        class MessageComparator implements Comparator<Long> {

            @Override
            public int compare(Long time0, Long time1) {
                long val = time0 - time1;
                if (val > 0) {
                    return 1;
                } else if (val == 0) {
                    return 0;
                } else {
                    return -1;
                }
            }
        }

        public synchronized EMMessage getMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return null;
            }
            return messages.get(msgId);
        }

        public synchronized void addMessages(List<EMMessage> msgs) {
            for (EMMessage msg : msgs) {
                addMessage(msg);
            }
        }

        public synchronized void addMessage(EMMessage msg) {
            if (msg == null || msg.emaObject == null || msg.getMsgTime() == 0 || msg.getMsgTime() == -1 || msg.getMsgId() == null
                    || msg.getMsgId().isEmpty() || msg.getType() == Type.CMD) {
                return;
            }
            String id = msg.getMsgId();
            //remove old message to avoid duplicate message
            Iterator<EMMessage> messageIterator = messages.values().iterator();
            while (messageIterator.hasNext()) {
                EMMessage message = messageIterator.next();
                if(TextUtils.equals(id, message.getMsgId())) {
                    Long time = idTimeMap.get(id);
                    if(time != null) {
                        removeMessageByTime(id, time);
                        idTimeMap.remove(id);
                    }
                    messageIterator.remove();
                }
            }
            // messages share same time stamp
            long time = sortByServerTime ? msg.getMsgTime() : msg.localTime();
            if (sortedMessages.containsKey(time)) {
                hasDuplicateTime = true;
                Object v = sortedMessages.get(time);
                if (v != null) {
                    if (v instanceof EMMessage) {
                        List<EMMessage> msgs = new LinkedList<>();
                        msgs.add((EMMessage) v);
                        msgs.add(msg);
                        sortedMessages.put(time, msgs);
                    } else if (v instanceof List) {
                        List<EMMessage> msgs = (List<EMMessage>) v;
                        msgs.add(msg);
                    }
                }
            } else {
                sortedMessages.put(time, msg);
            }
            messages.put(id, msg);
            idTimeMap.put(id, time);
        }

        public synchronized void removeMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return;
            }
            EMMessage msg = messages.get(msgId);
            if (msg != null) {
                Long time = idTimeMap.get(msgId);
                if (time != null) {
                    removeMessageByTime(msgId, time);
                    idTimeMap.remove(msgId);
                }
                messages.remove(msgId);
            }
        }

        private synchronized void removeMessageByTime(String msgId, long time) {
            if (msgId == null || msgId.isEmpty()) {
                return;
            }
            if (hasDuplicateTime && sortedMessages.containsKey(time)) {
                Object v = sortedMessages.get(time);
                if (v != null && v instanceof List) {
                    List<EMMessage> msgs = (List)v;
                    for (EMMessage m : msgs) {
                        if (m != null && m.getMsgId() != null && m.getMsgId().equals(msgId)) {
                            msgs.remove(m);
                            break;
                        }
                    }
                } else {
                    sortedMessages.remove(time);
                }
            } else {
                sortedMessages.remove(time);
            }
        }

        public synchronized List<EMMessage> getAllMessages() {
            List<EMMessage> list = new ArrayList<EMMessage>();
            if (hasDuplicateTime == false) {
                for (Object v : sortedMessages.values()) {
                    list.add((EMMessage)v);
                }
            } else {
                for (Object v : sortedMessages.values()) {
                    if (v != null) {
                        if (v instanceof List) {
                            list.addAll((List<EMMessage>)v);
                        } else {
                            list.add((EMMessage)v);
                        }
                    }
                }
            }
            return list;
        }

        public synchronized EMMessage getLastMessage() {
            if (sortedMessages.isEmpty()) {
                return null;
            }

            Object o = sortedMessages.lastEntry().getValue();
            if (o == null) {
                return null;
            }
            if (o instanceof EMMessage) {
                return (EMMessage)o;
            } else if (o instanceof List){
                List<EMMessage> msgs = (List<EMMessage>)o;
                if (msgs.size() > 0) {
                    return msgs.get(msgs.size() - 1);
                }
                return null;
            }
            return null;
        }

        public synchronized void clear() {
            //no need to keep the last message
            sortedMessages.clear();
            messages.clear();
            idTimeMap.clear();
        }

        public synchronized boolean isEmpty() {
            return sortedMessages.isEmpty();
        }
    }
}
