/************************************************************
 *  * hyphenate CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 hyphenate Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of hyphenate Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from hyphenate Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMContactListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAContactListener;
import com.hyphenate.chat.adapter.EMAContactManager;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * \~chinese
 * EMContactManager 是用来记录，查询，修改用户的联系人/好友列表的类。
 *
 * \~english
 * The `EMContactManager` is used to record, query, and modify contacts.

 */
public class EMContactManager {

    EMAContactManager emaObject;
    EMClient mClient;
    private List<EMContactListener> contactListeners = Collections.synchronizedList(new ArrayList<EMContactListener>());
    private EMAContactListenerImpl contactImpl = new EMAContactListenerImpl();

    EMContactManager(EMClient client, EMAContactManager contactManager) {
        mClient = client;
        emaObject = new EMAContactManager(contactManager);
        emaObject.registerContactListener(contactImpl);
    }

    class EMAContactListenerImpl extends EMAContactListener {

        @Override
        public void onContactAdded(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactAdded(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactDeleted(String username) {
            EMClient.getInstance().chatManager().caches.remove(username);

            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactDeleted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactInvited(String username, String reason) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactInvited(username, reason);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactAgreed(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onFriendRequestAccepted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactRefused(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onFriendRequestDeclined(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void handleError(EMAError error)  throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    /**
     * \~chinese
     * 添加联系人。
     * 
     * 同步方法，会阻塞当前线程。
     * 
     * 参考：
     * 异步方法见 {@link #asyncAddContact(String, String, EMCallBack)}。
     *
     * @param username		用户名。
     * @param reason		添加好友的原因。选填参数，可以设置为 `null` 或 `""`。
     * @throws HyphenateException   如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Adds a new contact.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * Reference：
     * For the asynchronous method, see {@link #asyncAddContact(String, String, EMCallBack)}.
     * 
     * @param username		The user to be added.
     * @param reason     	(optional) The invitation message. Set the parameter as null if you want to ignore the information.
     * @throws HyphenateException    A description of the issue that caused this exception.
     */
    public void addContact(String username, String reason) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.inviteContact(username, reason, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 添加联系人。
     * 
     * 异步方法。
     * 
     * 同步方法见 {@link #addContact(String, String)}。
     *
     * @param username  用户名。
     * @param reason    原因、理由（optional)。
     * @param callback  返回调用结果，失败会返回失败原因。
     *
     * \~english
     * Adds a new contact.
     * 
     * This is an asynchronous method.
     * 
     * Reference：
     * For the synchronous method, see {@link #addContact(String, String)}
     *
     * @param username  The user ID of the contact to be added.
     * @param reason    The message for adding contact (optional).
     * @param callback  The result of the method, which contains the error information if the method fails.
     */
    public void asyncAddContact(final String username, final String reason, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addContact(username, reason);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 删除联系人及其相关的会话。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username 要删除的联系人用户名。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Deletes a contact and all the conversations associated.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The contact to be deleted.
     * @throws HyphenateException A description of the issue that caused the exception.
     */
    public void deleteContact(String username) throws HyphenateException {
        deleteContact(username, false);
    }

    /**
     * \~chinese
     * 删除好友。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username
     * @param keepConversation  是否保留会话和消息。`true` 表示保留，`false` 表示删除。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Deletes a contact.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The user ID.
     * @param keepConversation  Whether to keep the associated conversation and messages.
     * @throws HyphenateException  A description of the issue that caused the exception.
     */
    public void deleteContact(String username, boolean keepConversation) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.deleteContact(username, error, keepConversation);
        EMClient.getInstance().chatManager().caches.remove(username);
        handleError(error);
    }

    /**
     * \~chinese
     * 删除好友。
     * 
     * 异步方法。
     * 
     * 参考：
     * 同步方法见 {@link #deleteContact(String)}。
     *
     * @param username  要删除的用户 ID。
     * @param callback  `Success` 表示调用成功，`Error` 表示调用失败，会包含失败原因。
     *
     * \~english
     * Deletes a contact.
     * 
     * This is an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #deleteContact(String)}.
     *
     * @param username   The contact‘s user ID to be deleted.
     * @param callback   The result of the method, which contains the error information if the method fails.
     */
    public void asyncDeleteContact(final String username,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    deleteContact(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从服务器获取所有的好友。
     * 
     * 同步方法，会阻塞当前线程。
     * 
     * 参考：
     * 异步方法见{@link #asyncGetAllContactsFromServer(EMValueCallBack)}
     *
     * @return 联系人列表。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Get all contacts from the server.
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncGetAllContactsFromServer(EMValueCallBack)}.
     *
     * @return The list of contacts.
     * @throws HyphenateException A description of the exception.
     */
    public List<String> getAllContactsFromServer() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> contacts = emaObject.getContactsFromServer(error);
        handleError(error);
        return contacts;
    }

    /**
     * \~chinese
     * 从服务器获取所有的好友。
     * 
     * 参考：
     * 同步方法见 {@link #getAllContactsFromServer()}。
     *
     * @param callback 包含联系人列表。
     *
     * \~english
     * Get all contacts from the server.
     * 
     * Reference:
     * This is a synchronous method see {@link #getAllContactsFromServer()}.
     *
     * @param callback If the method call succeeds, returns the list of contacts; if the method call fails, returns the error information.

     */
    public void asyncGetAllContactsFromServer(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> allContacts = getAllContactsFromServer();
                    callback.onSuccess(allContacts);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    //region contactListener
    /**
     * \~chinese
     * 注册联系人监听。
     * 
     * 参考：
     * 调用{@link #removeContactListener(EMContactListener)}移除监听。
     *
     * @param contactListener 要注册的联系人监听。
     *
     * \~english
     * Registers a new contact listener.
     * 
     * Reference:
     * Call {@link #removeContactListener(EMContactListener)} to remove the listener.
     *
     * @param contactListener  The contact listener to be registered.

     */
    public void setContactListener(EMContactListener contactListener) {
        if (!contactListeners.contains(contactListener)) {
            contactListeners.add(contactListener);
        }
    }

    /**
     * \~chinese
     * 移除联系人监听。
     * 
     * 参考：
     * 添加联系人监听调用{@link #setContactListener(EMContactListener)}。
     *
     * \~english
     * Removes the contact listener.
     * 
     * Reference:
     * Adds the contact listener by calling {@link #setContactListener(EMContactListener)}.
     */
    public void removeContactListener(EMContactListener contactListener) {
        contactListeners.remove(contactListener);
    }
    //endregion


    /**
     * \~chinese
     * 把指定用户加入到黑名单中。
     * 可以给黑名单的中用户发消息，但是接收不到对方发送的消息。
     * 
     * 参考：
     * 异步方法见{@link #asyncAddUserToBlackList(String, boolean, EMCallBack)}。
     * 批量加入黑名单见{@link #saveBlackList(List)}。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  此用户 ID。
     * @param both      此参数已废弃。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Adds a user to blocklist.
     * You can send message to the user in blocklist, but you can not receive the message sent by the other.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncAddUserToBlackList(String, boolean, EMCallBack)}.
     * Adds the batch blocklist by calling {@link #saveBlackList(List)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The user to be blocked.
     * @param both      This parameter is deprecated.
     * @throws HyphenateException A description of the issue that caused the exception.
     */
    public void addUserToBlackList(String username, boolean both) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.addToBlackList(username, both, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 把指定用户加入到黑名单中。
     * 可以给黑名单的中用户发消息，但是接收不到对方发送的消息。
     * 
     * 参考：
     * 同步方法见 {@link #addUserToBlackList(String, boolean)}。
     * 批量加入黑名单见 {@link #asyncSaveBlackList(List, EMCallBack)}。
     *
     * @param username  此用户 ID。
     * @param both      此参数已废弃。
     * @param callback  该方法完成的回调。
     * - `Success`表示加入成功；
     * - `Error`表示失败，会包含失败原因。
     *
     * \~english
     * Adds the user to blocklist.
     * You can send message to the user in the blocklist, but you can not receive the message sent by the user in the blocklist.
     * 
     * Reference:
     * For the synchronous method, see {@link #addUserToBlackList(String, boolean)}.
     * If you want to add multiple users in a batch to the blocklist, see {@link #asyncSaveBlackList(List, EMCallBack)}.
     *
     * @param username  The user to be blocked.
     * @param both      This parameter is deprecated.
     * @param callback  The callback completion.
     * - `Success`: The user is successfully added to the blocklist. 
     * - `Error`: The user fails to be added to the blocklist. In this case, an error message is returned.
     */
    public void asyncAddUserToBlackList(final String username,
                                        final boolean both,
                                        final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addUserToBlackList(username, both);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 上传黑名单列表到服务器。
     * 
     * 参考：
     * 异步方法见 {@link #asyncSaveBlackList(List, EMCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param blackList  黑名单的列表。
     *
     * \~english
     * Uploads the blocklist to server.
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncSaveBlackList(List, EMCallBack)}.
     *
     * @param blackList The blocklist.
     */
    public void saveBlackList(List<String> blackList) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.saveBlackList(blackList, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 上传黑名单列表到服务器。
     * 
     * 参考：
     * 同步方法见 {@link #saveBlackList(List)}。
     *
     * @param blackList 黑名单列表。
     * @param callback  该方法完成的回调。
     * - `Success`表示上传成功；
     * - `Error`表示失败，会包含失败原因。
     *
     * \~english
     * Uploads the blocklist to server.
     * 
     * Reference:
     * For the synchronous method, see {@link #saveBlackList(List)}.
     *
     * @param blackList  The blocklist.
     * @param callback   The completion callback.  
     * - `Success`: The blocklist is uploaded to the server.
     * - `Error`: The blocklist fails to be uploaded to the server. The reason for the failure will be returned.
     */
    public void asyncSaveBlackList(final List<String> blackList,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    saveBlackList(blackList);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 将用户从黑名单中移除。
     * 
     * 参考：
     * 异步方法见{@link #asyncRemoveUserFromBlackList(String, EMCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  要移除的用户名。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Removes the contact from the blocklist.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncRemoveUserFromBlackList(String, EMCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The user to be removed from the blocklist.
     * @throws HyphenateException   A description of the exception.
     */
    public void removeUserFromBlackList(String username) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.removeFromBlackList(username, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 将用户从黑名单中移除。
     * 
     * 参考：
     * 同步方法见{@link #removeUserFromBlackList(String)}
     *
     * @param username  要移除的用户名。
     * @param callback  该方法完成的回调。
     * - `Success`表示移除成功；
     * - `Error`表示失败，会包含失败原因。
     *
     * \~english
     * Removes the contact from the blocklist.
     * 
     * Reference:
     * For the synchronous method, see {@link #removeUserFromBlackList(String)}.
     *
     * @param username  The user to be removed from the blocklist.
     * @param callback  The completion callback.
     * - `Success`: The user is successfully removed from the blocklist.
     * - `Error`: The user fails to be removed from the blocklist. The reason for the failure will be returned.
     */
    public void asyncRemoveUserFromBlackList(final String username,
                                             final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    removeUserFromBlackList(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从本地获取黑名单列表。
     *
     * @return The blocklist.
     *
     * \~english
     * Gets the local blocklist.
     *
     * @return The blocklist.
     */
    public List<String> getBlackListUsernames() {
        EMAError error = new EMAError();
        return emaObject.getBlackListFromDB(error);
    }

    /**
     * \~chinese
     * 从服务器获取黑名单列表。
     * 
     * 参考：
     * 异步方法见 {@link #asyncGetBlackListFromServer(EMValueCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @return  该方法调用成功会返回黑名单列表。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Gets the blocklist from the server.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncGetBlackListFromServer(EMValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return  Returns the blocklist if the method succeeds.
     * @throws HyphenateException A description of the issue that caused the exception.
     */
    public List<String> getBlackListFromServer() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> blacklist = emaObject.getBlackListFromServer(error);
        handleError(error);
        return blacklist;
    }

    /**
     * \~chinese
     * 从服务器获取黑名单列表。
     * 
     * 参考：
     * 同步方法见{@link #getBlackListFromServer()}
     *
     * @param callback 该方法调用成功会返回黑名单列表，失败时返回错误原因。
     *
     * \~english
     * Gets the blocklist from the server.
     * 
     * Reference:
     * For the synchronous method, see {@link #getBlackListFromServer()}.
     *
     * @param callback Returns the blocklist if the call succeed and the description of the cause if the call fails.
     */
    public void asyncGetBlackListFromServer(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> blackList = getBlackListFromServer();
                    callback.onSuccess(blackList);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 接受加好友的邀请。
     * 
     * 同步方法，会阻塞当前线程。
     * 
     * 参考：
     * 异步方法见 {@link #asyncAcceptInvitation(String, EMCallBack)}。
     *
     * @param username  发起好友申请的用户 ID。
     *
     * \~english
     *
     * Accepts a friend invitation。
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncAcceptInvitation(String, EMCallBack)}
     *
     * @param username  The user who initiates the friend request.
     */
    public void acceptInvitation(String username) throws HyphenateException
    {
        EMAError error = new EMAError();
        emaObject.acceptInvitation(username, error);
        handleError(error);
    }


    /**
     * \~chinese
     * 接受加好友的邀请。
     *      
     * 异步方法。
     * 
     * 参考：
     * 同步方法见{@link #acceptInvitation(String)}。
     *
     * @param username  发起好友请求的用户 ID。
     * @param callback  该方法完成的回调。
     * - `Success`表示添加成功；
     * - `Error`表示失败，会包含失败原因。
     *
     * \~english
     * Accepts a friend invitation.
     *
     * This an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #acceptInvitation(String)}.
     *
     * @param username  The user who initiates the friend request.
     * @param callback  Returns `Success` if the call succeed and the description of the cause if the call fails.
     */
    public void asyncAcceptInvitation(final String username,
                                      final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    acceptInvitation(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 拒绝加好友的邀请。
     *     
     * 同步方法，会阻塞当前线程。
     * 
     * 参考：
     * 异步方法见{@link #asyncDeclineInvitation(String, EMCallBack)}
     *
     * @param username 发起好友请求的用户 ID。
     *
     * \~english
     * Declines a friend invitation.
     *      
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncDeclineInvitation(String, EMCallBack)}.
     *
     * @param username The user who initiates the invitation.
     */
    public void declineInvitation(String username) throws HyphenateException  {
        EMAError error = new EMAError();
        emaObject.declineInvitation(username, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 拒绝加好友的邀请。
     *      
     * 异步方法。
     * 
     * 参考：
     * 同步方法见{@link #declineInvitation(String)}
     *
     * @param username  发起好友申请的用户 ID。
     * @param callback  该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Declines a friend invitation.
     *      
     * This an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #declineInvitation(String)}.
     *
     * @param username The user who initiates the friend request.
     * @param callback  Returns `Success` if the call succeed and the description of the cause if the call fails.
     */
    public void asyncDeclineInvitation(final String username,
                                       final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    declineInvitation(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从数据库获取好友列表。
     *
     * @return 调用成功会返回好友列表。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Gets the contact list from the local database.
     *
     * @return The contact list.
     * @throws HyphenateException A description of the issue that caused the exception.
     */
    public List<String> getContactsFromLocal() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> contacts = getContactsFromDB(error);
        handleError(error);
        return contacts;
    }

    /**
     * \~chinese
     * 从数据库获取好友列表。
     *
     * @param error  方法调用失败会返回 error，包含错误信息。
     * @return 调用成功会返回好友列表。
     *
     * \~english
     * Get the contact list from the local database.
     *
     * @param error  A description of the issue that caused this call to fail.
     * @return  Returns the contact list if the method succeeds.
     */
    List<String> getContactsFromDB(EMAError error) {
        return emaObject.getContactsFromDB(error);
    }

    void onLogout() {
    }

    /**
     * \~chinese
     * 从服务器获取登录用户在其他设备上登录的设备 ID，这里的 ID 是由 username + "/" + resource 组成的。
     *      
     * 同步方法，会阻塞当前线程。
     * 
     * 参考：
     * 异步方法见 {@link #asyncGetSelfIdsOnOtherPlatform(EMValueCallBack)}。
     *
     * @return  该方法调用成功会返回 ID 列表。
     * @throws HyphenateException  如果有方法调用的异常会在这里抛出，可以看到具体错误原因。
     *
     * \~english
     * Get the unique IDs of current user on the other devices. The ID is username + "/" + resource.
     *      
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncGetSelfIdsOnOtherPlatform(EMValueCallBack)}.
     *
     * @return   Returns the unique device ID list on the other devices if the method succeeds.
     * @throws HyphenateException A description of the issue that caused this call to fail.
     */
    public List<String> getSelfIdsOnOtherPlatform() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> selfIds = emaObject.getSelfIdsOnOtherPlatform(error);
        handleError(error);
        return selfIds;
    }

    /**
     * \~chinese
     * 从服务器获取登录用户在其他设备上登录的 ID。这里 ID 是一个组合值，username + "/" + resource(设备的识别号)。
     * 
     * 异步方法。
     * 
     * 参考：
     * 同步方法见 {@link #getSelfIdsOnOtherPlatform()}。
     *
     * @param callback  该方法完成的回调，成功会包含用户在其他设备上登录的 ID；失败则包含失败原因。
     *
     * \~english
     * Get the unique selfID list of the current user on the other devices. The selfID contains username + "/" + resource(the device ID).
     *      
     * This is an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #getSelfIdsOnOtherPlatform()}.
     *
     * @param callback  - Returns the unique selfID list on the other devices if the method succeeds.
     *                  - Returns the description of the cause of the error if the method fails.

     */
    public void asyncGetSelfIdsOnOtherPlatform(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> selfIds = getSelfIdsOnOtherPlatform();
                    callback.onSuccess(selfIds);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
}
