/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.hyphenate.EMError;
import com.hyphenate.EMResultCallBack;
import com.hyphenate.EMCallBack;
import com.hyphenate.EMChatRoomChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatRoom;
import com.hyphenate.chat.adapter.EMAChatRoomManager;
import com.hyphenate.chat.adapter.EMAChatRoomManagerListener;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * \~chinese
 * 聊天室管理类，负责聊天室加入和退出、聊天室列表获取以及成员权限管理等。
 * 比如，加入聊天室：
 * ```java
 *  EMClient.getInstance().chatroomManager().joinChatRoom(conversationId, new EMValueCallBack<EMChatRoom>() {
 *
 *     	public void onSuccess(EMChatRoom value) {
 *     	    //成功加入聊天室后的处理逻辑。
 *     	}
 *
 *     	public void onError(int error, String errorMsg) {
 *     	    //加入聊天室失败后的处理逻辑。
 *     	}
 * 	});
 * ```
 *
 * \~english
 * The chat room manager, which manages users joining and existing the chat room and getting the chat room list, and manages member privileges.
 * For example, joining a chat room:
 * ```java
 *   EMClient.getInstance().chatroomManager().joinChatRoom(conversationId, new EMValueCallBack<EMChatRoom>() {
 *    	public void onSuccess(EMChatRoom value) {
 *    	    //Processing logic for successful chat room joining.
 *        }
 *    	public void onError(int error, String errorMsg) {
 *    	    //Processing logic for chat room joining failures.
 *        }
 * 	});
 * ```
 */
public class EMChatRoomManager {
    EMAChatRoomManager emaObject;
	private List<EMChatRoomChangeListener> chatRoomListeners = Collections.synchronizedList(new ArrayList<EMChatRoomChangeListener>());
	private ExecutorService threadPool = null;
	private List<EMChatRoom> chatRooms = Collections.synchronizedList(new ArrayList<EMChatRoom>());
	
	EMClient mClient;
	public EMChatRoomManager(EMClient client, EMAChatRoomManager manager) {
	    emaObject = manager;
	    emaObject.addListener(chatRoomListenerImpl);
		mClient = client;
		threadPool = Executors.newCachedThreadPool();
	}

	/**
	 * \~chinese
	 * 注册聊天室事件监听对象。
	 *
	 * @note
	 * 聊天室被销毁、成员的加入和退出、禁言和加入白名单等操作均可通过设置 {@link EMChatRoomChangeListener} 进行监听。
	 *
	 * 利用本方法注册聊天室事件监听对象后，可调用 {@link #removeChatRoomListener(EMChatRoomChangeListener)} 将其移除。
	 *
	 * @param listener 聊天室事件监听对象，详见 {@link EMChatRoomChangeListener}。
	 *
	 * \~english
	 * Adds a chat room event listener.
	 * 
	 * @note
	 * Chat room destruction, member entry and exit, mute, and allow list changes among other chat room operations can be listened for by setting
	 * {@link EMChatRoomChangeListener}.
	 *
	 * Chat room event listeners added with this method can be removed by calling {@link #removeChatRoomListener(EMChatRoomChangeListener)}.
	 *
	 * @param listener A chat room event listener. See {@link EMChatRoomChangeListener}.
	 */
	public void addChatRoomChangeListener(EMChatRoomChangeListener listener) {
		chatRoomListeners.add(listener);
	}

    /**
     * \~chinese
     * 移除聊天室事件监听对象。
	 * 
	 * @note
	 * 利用 {@link #addChatRoomChangeListener(EMChatRoomChangeListener)} 注册聊天室事件监听对象后，可调用本方法将其移除。目前，推荐使用
	 * {@link #removeChatRoomListener(EMChatRoomChangeListener)} 移除聊天室事件监听对象。
	 *
	 * @param listener 	要移除的聊天室监听对象。
	 * @deprecated  	已废弃，请用 {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} 代替。
	 *
	 * \~english
	 * Removes a chat room event listener.
	 * 
	 * @note
	 * This method removes a chat room event listener added with {@link #addChatRoomChangeListener(EMChatRoomChangeListener)}.
	 * 
	 * Currently, {@link #removeChatRoomListener(EMChatRoomChangeListener)} is recommended to remove a chat room event listener.
	 *
	 * @param listener	The chat room listener to be removed.
	 * @deprecated  	Deprecated. Please use {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} instead.
	 */
    @Deprecated
	public void removeChatRoomChangeListener(EMChatRoomChangeListener listener) {
	    removeChatRoomListener(listener);
    }

	/**
	 * \~chinese
	 * 移除聊天室事件监听对象。
	 * 
	 * @note
	 * 利用 {@link #addChatRoomChangeListener(EMChatRoomChangeListener)} 注册聊天室事件监听对象后，可调用本方法将其移除。
	 *
	 * @param listener	要移除的聊天室事件监听对象。
	 *
	 * \~english
	 * Removes the chat room event listener.
	 * 
	 * @note
	 * This method removes the chat room event listener added with {@link #addChatRoomChangeListener(EMChatRoomChangeListener)}.
	 *
	 * @param listener	The chat room event listener to be removed.
	 */
	public void removeChatRoomListener(EMChatRoomChangeListener listener)
	{
		chatRoomListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 加入聊天室。
	 * 
	 * @note
	 * 退出聊天室调用 {@link #leaveChatRoom(String)}。
	 *
	 * 异步方法。
	 *
	 * @param roomId 	聊天室 ID。
	 * @param callback	聊天室加入结果回调。加入成功，回调 {@link EMValueCallBack#onSuccess(Object)} 并返回聊天室对象。
	 *                  加入失败，回调 {@link EMValueCallBack#onError(int, String)}，其中第一个参数为错误参数，第二个参数为错误信息。
	 * 
	 * \~english
     * Joins a chat room.
	 * 
	 * @note
	 * To exit the chat room, call {@link #leaveChatRoom(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param roomId 	The ID of the chat room to join.
	 * @param callback 	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the chat room object;
	 *                  if this call fails, calls {@link EMValueCallBack#onError(int, String)}, where the first parameter is the error code and
	 *                  the second is the error message.
	 */
	public void joinChatRoom(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMAChatRoom r = EMChatRoomManager.this.emaObject.joinChatRoom(roomId, error);
				EMChatRoom room = new EMChatRoom(r);
				if (error.errCode() == EMAError.EM_NO_ERROR || error.errCode() == EMAError.CHATROOM_ALREADY_JOINED) {
					callback.onSuccess(room);
				} else {
					callback.onError(error.errCode(), error.errMsg());
				}
			}
		});

	}
	
	/**
	 * \~chinese
	 * 退出聊天室。
	 * 
	 * @note
	 * 利用 {@link #joinChatRoom(String, EMValueCallBack)} 加入聊天室后，离开时调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param roomId 	聊天室 ID。
	 *
	 * \~english
	 * Exits a chat room.
	 * 
	 * @note
	 * A user that joins a chat room using {@link #joinChatRoom(String, EMValueCallBack)} can call the leaveChatRoom method to exit the chat room.
	 *
	 * This is an asynchronous method.
	 *
	 * @param roomId 	The ID of the chat room to exit.
	 */
	public void leaveChatRoom(final String roomId) {
		EMChatRoom room = getChatRoom(roomId);
		if (room == null) {
		    return;
		}

        EMOptions options = EMClient.getInstance().getOptions();

        boolean allowed = options.isChatroomOwnerLeaveAllowed();
        String owner = room.getOwner();
        if(!allowed && owner.equals(EMSessionManager.getInstance().getLastLoginUser())){
            return;
        }

        // The following is a temporary workaround to delete the conversation in case of leaving the room,
        // but deleteConversation could be a time-consuming operation.
		boolean delete = options.isDeleteMessagesAsExitChatRoom();
        if(delete) {
			EMClient.getInstance().chatManager().deleteConversation(roomId,true);
        }
		threadPool.submit(new Runnable() {

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMChatRoomManager.this.emaObject.leaveChatRoom(roomId, error);
			}
		});
	}

	/**
	 * \~chinese
	 * 以分页的方式从服务器获取聊天室数据。
	 * 
	 * 同步方法，会阻塞当前线程。
	 *
	 * 也可使用异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}。
	 *
	 *
	 * @param pageNum 				当前页码，从 1 开始。
	 * @param pageSize 				每页期望返回的记录数。取值范围为 [1,50]。
	 * @return  					分页获取结果 {@link EMPageResult}。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets chat room data from the server with pagination.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * The asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} can be used.
	 *
	 * @param pageNum 				The page number, starting from 1.
	 * @param pageSize 				The number of records that you expect to get on each page. The value range is [1,50].
	 * @return      				Chat room data. See {@link EMPageResult}.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 *
	 */
	public EMPageResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMPageResult<EMAChatRoom> result = emaObject.fetchChatroomsWithPage(pageNum, pageSize, error);
		handleError(error);

		List<EMAChatRoom> rooms = result.getData();
		int count = result.getPageCount();

		EMPageResult<EMChatRoom> ret = new EMPageResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : rooms) {
			data.add(new EMChatRoom(room));
		}
		ret.setPageCount(count);
		ret.setData(data);

		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}

	/**
	 * \~chinese
	 * 以分页的方式从服务器获取聊天室数据。
	 * 
	 * @note
	 * 返回的结果中，当 {@link EMCursorResult#getCursor()} 为空字符串 ("") 时，表示数据获取完毕。
	 * 
	 * 此方法为耗时操作，可以调用异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param pageSize 				每页返回的记录数。取值范围为 [1,50]。
	 * @param cursor 				从游标位置开始取数据。
	 * @return   					分页获取结果 {@link EMCursorResult}。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 * @deprecated 					已废弃，请用 {@link #fetchPublicChatRoomsFromServer(int, int)} 代替。
	 *
	 * 
	 * 
	 * 
	 * \~english
	 * Gets chat room data from the server with pagination.
	 * 
	 * @note
	 * When {@link EMCursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 * 
	 * As this method is time-consuming, the asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)} can be used.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageSize 				The number of records per page. The value range is [1,50].
	 * @param cursor 				The cursor position from which to start getting data.
	 * @return 					    Chat room data. See {@link EMCursorResult}.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 * @deprecated 					Deprecated. Please use {@link #fetchPublicChatRoomsFromServer(int, int)} instead.
	 *
	 */
	@Deprecated
	public EMCursorResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAChatRoom> cursorResult = emaObject.fetchChatroomsWithCursor(cursor, pageSize, error);
		handleError(error);
		
		EMCursorResult<EMChatRoom> ret = new EMCursorResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : cursorResult.getData()) {
			data.add(new EMChatRoom(room));
		}
		ret.setCursor(cursorResult.getCursor());
		ret.setData(data);
		
		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}
	
	/**
     * \~chinese
	 * 以分页的方式从服务器获取聊天室。
	 * 
	 * @note
	 * 返回的结果中，当 {@link EMCursorResult#getCursor()} 为空字符串 ("") 时，表示没有更多数据。
	 *
	 * 异步方法。
	 *
	 * @param pageSize 	每页期望返回的记录数。取值范围为 [1,50]。
	 * @param cursor 	从游标位置开始取数据。
	 * @param callback	结果回调，成功回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * @deprecated 		使用 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} 代替。
	 *
	 * \~english
	 * Gets chat room data from the server with pagination.
	 *
	 * @note
	 * When {@link EMCursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 *
	 * This is an asynchronous method.
	 *
	 * @param pageSize 	The number of records that you expect to get on each page. The value range is [1,50].
	 * @param cursor 	The cursor position from which to start getting data.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                  If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 *
	 * @deprecated 		Deprecated. Please use {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} instead.
	 * 
	 */
	@Deprecated
	public void asyncFetchPublicChatRoomsFromServer(final int pageSize, final String cursor, final EMValueCallBack<EMCursorResult<EMChatRoom>> callback) {
	    EMClient.getInstance().execute(new Runnable() {
	        
	        @Override
            public void run() {
                try {
                    EMCursorResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageSize, cursor);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
    
    /**
     * \~chinese
     * 以分页的方式从服务器获取聊天室。
	 *
     * 异步方法。
	 *
     * @param pageNum 	当前页数，从 1 开始。
     * @param pageSize 	每页返回的记录数。取值范围为 [1,50]。
     * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  失败时回调 {@link EMValueCallBack#onError(int, String)}。
     *
     * \~english
     * Gets chat room data from the server with pagination.
	 *
     * This is an asynchronous method.
     *
     * @param pageNum 	The page number, starting from 1.
     * @param pageSize 	The number of records that you expect to get on each page. The value range is [1,50].
     * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)}.
	 *                  If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack<EMPageResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMPageResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageNum, pageSize);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 从服务器获取聊天室详情。
	 * 
	 * @note
	 * 该方法不获取取成员列表。
	 *
	 * 同步方法，会阻塞当前线程。
	 * 
	 * 异步方法见 {@link #asyncFetchChatRoomFromServer(String, EMValueCallBack)}。
	 *
	 * @param roomId				聊天室 ID。
	 * @return EMChatRoom			返回聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，错误码详见 {@link EMError}。
	 *
	 * \~english
	 * Gets details of a chat room from the server.
	 * 
	 * @note
	 * This method does not get the member list.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * For the asynchronous method, see {@link #asyncFetchChatRoomFromServer(String, EMValueCallBack)}.
	 *
	 * @param roomId				The chat room ID.
	 * @return EMChatRoom			The chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMChatRoom fetchChatRoomFromServer(String roomId) throws HyphenateException {
		return fetchChatRoomFromServer(roomId, false);
	}
	
	/**
	 * \~chinese
	 * 从服务器获取聊天室详情。
	 * 
	 * @note
	 * 如果需要取成员列表，默认最多取 200 个成员，超出部分，请调用 {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param roomId 		        聊天室 ID。
	 * @param fetchMembers 			是否需要获取聊天室成员。
	 * @return EMChatRoom			返回的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets details of a chat room from the server.
	 * 
	 * @note
	 * The member list, if required, can contain at most 200 members by default. For more members, call {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param roomId 				The chat room ID.
	 * @param fetchMembers 			Whether to get chat room members.
	 * @return EMChatRoom			The chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
 	public EMChatRoom fetchChatRoomFromServer(String roomId, boolean fetchMembers)throws HyphenateException {
 	   EMAError error = new EMAError();
       EMAChatRoom room = emaObject.fetchChatroomSpecification(roomId, error, fetchMembers);
       handleError(error);
       return new EMChatRoom(room);
 	}
	
	/**
     * \~chinese
     * 从服务器获取聊天室详情。
	 * 
	 * @note
	 * 该方法不获取聊天室成员列表。
	 * 
     * 异步方法。
	 *
	 * 同步方法见 {@link #fetchChatRoomFromServer(String)}。
	 *
     * @param roomId	聊天室 ID。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  失败时回调 {@link EMValueCallBack#onError(int, String)}。
     *
     * \~english
     * Gets details of a chat room from the server.
	 * 
	 * @note
	 * This method does not get the chat room member list.
	 *	 
	 * This is an asynchronous method.
	 *
     * For the synchronous method, see {@link #fetchChatRoomFromServer(String)}.
	 *
     * @param roomId	The chat room ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                  if this call fails, calls{@link EMValueCallBack#onError(int, String)}.
     */
    public void asyncFetchChatRoomFromServer(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMChatRoom chatRoom = fetchChatRoomFromServer(roomId);
                    callback.onSuccess(chatRoom);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });   
    }
	
	/**
	 * \~chinese
	 * 从内存中获取聊天室。
	 *
	 * @param roomId		聊天室 ID。
     * @return				返回聊天室对象。如果内存中不存在聊天室对象，返回 null。
	 *
	 * \~english
	 * Gets the chat room in the memory.
	 *
	 * @param roomId		The chat room ID.
	 * @return 				The chat room instance. The SDK returns null if the chat room is not found in the memory.
	 */
	public EMChatRoom getChatRoom(String roomId) {
		EMAChatRoom room = emaObject.getChatroom(roomId);
		if(room == null){
			return null;
		}
		return new EMChatRoom(room);
	}
	
	/**
	 * \~chinese
	 * 获取当前内存的聊天室列表。
	 * 
	 * @note
	 * 调用以下任何一种方法后，调用该方法才能有返回数据：
	 * 
	 * （1）{@link #fetchPublicChatRoomsFromServer(int, int)} 或其异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 * （2）{@link #fetchPublicChatRoomsFromServer(int, String)} 或其异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 *   
	 *   返回的数据是以上方法最后一次分页获取的数据。
	 *
	 * @return	返回 EMChatRoomManager 维护的聊天室列表。
	 * @deprecated 已废弃。
	 * 
	 * \~english
	 * Gets the list of chat rooms in the memory.
	 * 
	 * @note
	 * Data can be returned upon the call of this method only after one of the following methods is called:
	 * - {@link #fetchPublicChatRoomsFromServer(int, int)} or its asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 * - {@link #fetchPublicChatRoomsFromServer(int, String)} or its asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 *  The returned data is the data last obtained by using one of the preceding methods.
	 * @return	The list of chat rooms maintained by EMChatRoomManager.
	 * @deprecated Deprecated.
	 */
	@Deprecated
	public List<EMChatRoom> getAllChatRooms() {
		return Collections.unmodifiableList(chatRooms);
	}

	EMAChatRoomManagerListener chatRoomListenerImpl = new EMAChatRoomManagerListener() {

		@Override
		public void onLeaveChatRoom(EMAChatRoom chatroom, int leaveReason) {
            EMClient.getInstance().chatManager().caches.remove(chatroom.getId());
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            if (leaveReason == DESTROYED) {
			                listener.onChatRoomDestroyed(chatroom.getId(), chatroom.getName());
			            } else {
							listener.onRemovedFromChatRoom(leaveReason, chatroom.getId(), chatroom.getName(), EMClient.getInstance().getCurrentUser());
						}
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberJoinedChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
				    for (EMChatRoomChangeListener listener : chatRoomListeners) {
					    listener.onMemberJoined(chatroom.getId(), member);
				    }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberLeftChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            listener.onMemberExited(chatroom.getId(), chatroom.getName(), member);
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		public void onAddMuteList(EMAChatRoom chatRoom, List<String> mutes, long expireTime) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListAdded(chatRoom.getId(), mutes, expireTime);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveMutes(EMAChatRoom chatRoom, List<String> mutes) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListRemoved(chatRoom.getId(), mutes);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListAdded(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListAdded(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListRemoved(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListRemoved(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAllMemberMuteStateChanged(EMAChatRoom chatRoom, final boolean isMuted) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAllMemberMuteStateChanged(chatRoom.getId(), isMuted);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAddAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminAdded(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminRemoved(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onOwnerChanged(EMAChatRoom chatRoom, String newOwner, String oldOwner) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onOwnerChanged(chatRoom.getId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAChatRoom chatRoom, String announcement) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAnnouncementChanged(chatRoom.getId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateSpecificationFromChatroom(EMAChatRoom chatroom) {
			synchronized (chatRoomListeners) {
				for (EMChatRoomChangeListener listener : chatRoomListeners) {
					try {
						if (null != chatroom) {
							listener.onSpecificationChanged(new EMChatRoom(chatroom));
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onAttributesUpdate(String chatRoomId, String attributes, String from) {
			synchronized (chatRoomListeners) {
				for (EMChatRoomChangeListener listener : chatRoomListeners) {
					Map<String,String> attributeMap = parseJsonUpdate(attributes);
					try {
						if (null != attributeMap && attributeMap.size() > 0)
						listener.onAttributesUpdate(chatRoomId,attributeMap,from);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onAttributesRemoved(String chatRoomId, String attributes, String from) {
			synchronized (chatRoomListeners) {
				for (EMChatRoomChangeListener listener : chatRoomListeners) {
					List<String> keyList = parseJsonRemove(attributes);
					try {
						if (null != keyList && keyList.size() > 0)
							listener.onAttributesRemoved(chatRoomId,keyList,from);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}
	};


	/**
	 * \~chinese
	 * 创建聊天室。
	 * 
	 * @Note
	 * 异步方法见 {@link #asyncCreateChatRoom(String, String, String, int, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param subject           	聊天室名称。
	 * @param description       	聊天室描述。
	 * @param welcomeMessage    	邀请用户加入聊天室的消息。
	 * @param maxUserCount      	聊天室的最大成员数。
	 * @param members           	邀请加入聊天室的用户列表。
	 * @return EMChatRoom 		    创建成功的聊天室对象。
     * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Creates a chat room.
	 * 
	 * @Note
	 * For the asynchronous method, see {@link #asyncCreateChatRoom(String, String, String, int, List, EMValueCallBack)}. 
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param subject           	The chat room name.
	 * @param description       	The chat room description.
	 * @param welcomeMessage    	A welcome message inviting users to join the chat room.
	 * @param maxUserCount      	The maximum number of members allowed in the chat room.
	 * @param members           	The list of users invited to join the chat room.
	 * @return EMChatRoom		    The created chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom createChatRoom(String subject, String description, String welcomeMessage,
	                                 int maxUserCount, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.createChatRoom(subject, description, welcomeMessage, EMChatRoom.EMChatRoomStyle.EMChatRoomStylePublicOpenJoin.ordinal(), maxUserCount, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 创建聊天室。
	 *
	 * 同步方法见 {@link #createChatRoom(String, String, String, int, List)}。
	 *
	 * 异步方法。
	 *
	 * @param subject           聊天室名称。
	 * @param description       聊天室描述。
	 * @param welcomeMessage    用户加入聊天室后收到的欢迎消息。
	 * @param maxUserCount      聊天室的最大成员数。
	 * @param members           邀请加入聊天室的用户列表。
	 * @param callBack          结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)},
	 *                          失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Creates a chat room.
	 *
	 * For the synchronous method, see {@link #createChatRoom(String, String, String, int, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param subject           The chat room name.
	 * @param description       The chat room description.
	 * @param welcomeMessage    A welcome message for inviting users to join the chat room.
	 * @param maxUserCount      The maximum number of members allowed in the chat room.
	 * @param members           The list of users invited to join the chat room.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                          if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncCreateChatRoom(final String subject, final String description, final String welcomeMessage,
	                                final int maxUserCount, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(createChatRoom(subject, description, welcomeMessage, maxUserCount, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解散聊天室。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncDestroyChatRoom(String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Destroys a chat room.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncDestroyChatRoom(String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public void destroyChatRoom(String chatRoomId)
			throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyChatroom(chatRoomId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 解散聊天室。
	 *
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 同步方法见 {@link #destroyChatRoom(String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param callBack	    结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 *                      失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Destroys a chat room.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #destroyChatRoom(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *                      if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncDestroyChatRoom(final String chatRoomId, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyChatRoom(chatRoomId);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改聊天室名称。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncChangeChatRoomSubject(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param newSubject			修改后的聊天室名称。
	 * @return			            返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Changes the chat room name.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncChangeChatRoomSubject(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param newSubject			The modified chat room name.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}
     */
	public EMChatRoom changeChatRoomSubject(String chatRoomId, String newSubject)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomSubject(chatRoomId, newSubject, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改聊天室名称。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 同步方法见 {@link #changeChatRoomSubject(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param newSubject	修改后的聊天室名称。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Changes the chat room name.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #changeChatRoomSubject(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	A chat room ID.
	 * @param newSubject	The modified chat room name.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeChatRoomSubject(final String chatRoomId, final String newSubject, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatRoomSubject(chatRoomId, newSubject));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改聊天室描述信息。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncChangeChatroomDescription(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param newDescription		修改后的聊天室描述。
	 * @return						修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见{@link EMError}。
	 *
	 * \~english
	 * Modifies the chat room description.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncChangeChatroomDescription(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param newDescription		The modified chat room description.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom changeChatroomDescription(String chatRoomId, String newDescription)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomDescription(chatRoomId, newDescription, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改聊天室描述信息。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 同步方法见 {@link #changeChatroomDescription(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param newDescription	修改后的聊天室描述。
	 * @param callBack   		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                          失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Modifies the chat room description.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #changeChatroomDescription(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param newDescription	The modified chat room description.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                          if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeChatroomDescription(final String chatRoomId, final String newDescription, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatroomDescription(chatRoomId, newDescription));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表。
	 * 
	 * @note
	 * 返回的结果中，当 {@link EMCursorResult#getCursor()} 为空字符串 ("") 时，表示没有更多数据。
	 * 
	 * 异步方法见 {@link #asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param cursor				从该游标位置开始取数据。首次调用 cursor 传空值，从最新数据开始获取。
	 * @param pageSize				每页期望获取的成员数。取值范围为 [1,50]。
	 * @return						聊天室成员列表和用于下次获取的 cursor。详见 {@link EMCursorResult}。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 * 
	 * \~english
	 * Gets the chat room member list.
	 * 
	 * @note
	 * When {@link EMCursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param cursor				The cursor position from which to start getting data. At the first call, if you set the cursor as "null", the SDK gets chat room members in the reverse chronological order of when members joined the chat room.
	 * @param pageSize				The number of members that you expect to get on each page. The value range is [1,50].
	 * @return						The list of chat room members and the cursor for the next query. See {@link EMCursorResult}.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMCursorResult<String> fetchChatRoomMembers(String chatRoomId, String cursor, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult result = emaObject.fetchChatroomMembers(chatRoomId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表。
	 * 
	 * @note
	 * 返回的结果中，当 {@link EMCursorResult#getCursor()} 为空字符串 ("") 时，表示没有更多数据。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param cursor		从该游标位置开始取数据。首次调用 cursor 传空值，从最新数据开始获取。
	 * @param pageSize		每页期望获取的成员数。取值范围为 [1,50]。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 * 
	 * \~english
	 * Gets the chat room member list.
	 * 
	 * @note
	 * When {@link EMCursorResult#getCursor()} is an empty string ("") amid the result, all data is fetched.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param cursor		The cursor position from which to start getting data. At the first call, if you set the cursor as "null", the SDK gets chat room members in the reverse chronological order of when members joined the chat room.
	 * @param pageSize		The number of members that you expect to get on each page. The value range is [1,50].
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomMembers(final String chatRoomId, final String cursor, final int pageSize, final EMValueCallBack<EMCursorResult<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMembers(chatRoomId, cursor, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将一组成员禁言。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncMuteChatRoomMembers(String, List, long, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param muteMembers 			禁言的成员列表。
	 * @param duration 				禁言时长，单位为毫秒，如果是 `-1` 代表永久。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Mutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncMuteChatRoomMembers(String, List, long, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param muteMembers 			The list of members to be muted.
	 * @param duration 				The mute duration in milliseconds. The value `-1` indicates that the member is muted permanently.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom muteChatRoomMembers(String chatRoomId, List<String> muteMembers, long duration)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.muteChatroomMembers(
				chatRoomId, muteMembers, duration, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 将一组成员禁言。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 同步方法见 {@link #muteChatRoomMembers(String, List, long)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param muteMembers   要禁言的用户列表。
	 * @param duration 		禁言时长，单位为毫秒，如果是 `-1` 代表永久。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Mutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #muteChatRoomMembers(String, List, long)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param muteMembers 	The list of members to be muted.
	 * @param duration 		The mute duration in milliseconds. The value `-1` indicates that the member is muted permanently. 
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncMuteChatRoomMembers(final String chatRoomId, final List<String> muteMembers, final long duration, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(muteChatRoomMembers(chatRoomId, muteMembers, duration));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除禁言。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncUnMuteChatRoomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param members				解除禁言的成员列表。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Unmutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncUnMuteChatRoomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be unmuted.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom unMuteChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unmuteChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 解除禁言。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 同步方法见 {@link #unMuteChatRoomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param members       解除禁言的用户列表。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Unmutes members in a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #unMuteChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	A chat room ID.
	 * @param members		The list of members to be unmuted.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncUnMuteChatRoomMembers(final String chatRoomId, final List<String> members,
	                                             final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unMuteChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncChangeOwner(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param newOwner				聊天室的新所有者。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Transfers the chat room ownership.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncChangeOwner(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			A chat room ID.
	 * @param newOwner				The new owner of the chat room.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMChatRoom changeOwner(String chatRoomId, String newOwner)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.transferChatroomOwner(chatRoomId, newOwner, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 同步方法见 {@link #changeOwner(String, String)}
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param newOwner		聊天室的新所有者。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。 
	 *
	 * \~english
	 * Transfers the chat room ownership.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #changeOwner(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param newOwner		The new owner of the chat room.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeOwner(final String chatRoomId, final String newOwner, final EMValueCallBack<EMChatRoom> callBack)
			throws HyphenateException {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeOwner(chatRoomId, newOwner));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加聊天室管理员。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncAddChatRoomAdmin(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param admin					要设置为管理员的成员。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Adds a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncAddChatRoomAdmin(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param admin					The member to be set as the chat room admin.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom addChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.addChatroomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 添加聊天室管理员。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 同步方法见 {@link #addChatRoomAdmin(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param admin			要设置为管理员的成员。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Adds a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #addChatRoomAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param admin			The member to be set as the chat room admin.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncAddChatRoomAdmin(final String chatRoomId, final String admin, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(addChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 移除聊天室管理员权限。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncRemoveChatRoomAdmin(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param admin					要移除管理员权限的 ID。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes administrative privileges of a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncRemoveChatRoomAdmin(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param admin					The chat room admin whose administrative privileges are to be removed.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom removeChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 移除聊天室管理员权限。
	 * 
	 * @note
	 * 仅聊天室所有者可调用此方法。
	 * 
	 * 同步方法见 {@link #removeChatRoomAdmin(String, String)}}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param admin			要移除管理员权限的 ID。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes administrative privileges of a chat room admin.
	 * 
	 * @note
	 * Only the chat room owner can call this method.
	 * 
	 * For the synchronous method, see {@link #removeChatRoomAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param admin			The chat room admin whose administrative privileges are to be removed.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomAdmin(final String chatRoomId, final String admin,
	                                final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室禁言列表。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param pageNum				当前页码，从 1 开始。
	 * @param pageSize				每页期望返回的禁言成员数。取值范围为 [1,50]。
	 * @return  					返回的包含禁言成员 ID 及其禁言时长的 Map，其中每一项的 key 是禁言的成员 ID，value 是禁言时长，单位是毫秒。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 * 
	 * \~english
	 * Gets the list of muted chat room members from the server.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The number of muted members that you expect to get on each page. The value range is [1,50].
	 * @return 						A map which contains the muted member ID and mute duration, where the key of each entry
	 * 								is the ID of a muted member and the value is the mute duration in milliseconds.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public Map<String, Long> fetchChatRoomMuteList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, Long> members = emaObject.fetchChatRoomMuteList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~chinese
	 * 获取聊天室禁言列表。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 同步方法见 {@link #fetchChatRoomMuteList(String, int, int)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param pageNum		当前页码，从 1 开始。
	 * @param pageSize		每页期望返回的禁言成员数。取值范围为 [1,50]。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the list of muted chat room members from the server.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #fetchChatRoomMuteList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param pageNum		The page number, starting from 1.
	 * @param pageSize		The number of muted members that you expect to get on each page. The value range is [1,50].
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomMuteList(final String chatRoomId, final int pageNum, final int pageSize,
	                                       final EMValueCallBack<Map<String, Long>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMuteList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员移出聊天室。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncRemoveChatRoomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param members			要移出聊天室的成员列表。
	 * @return			        返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes members from a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncRemoveChatRoomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members			The list of members to be removed.
	 * @return				The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom removeChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 将成员移出聊天室。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 同步方法见 {@link #removeChatRoomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID
	 * @param members		要移出的成员列表。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes members from a chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #removeChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be removed.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomMembers(final String chatRoomId, final List<String> members,
	                                       final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员添加到聊天室黑名单。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * - 成员添加到黑名单的同时，将被服务器移出聊天室。
	 * - 可通过 {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} 回调通知，
	 * 第一个参数是具体原因，加入黑名单返回的原因是 {@link EMAChatRoomManagerListener#BE_KICKED}。
	 * - 添加到黑名单的成员禁止再次加入到聊天室。
	 * 
	 * 异步方法见 {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param members				要加入黑名单的成员列表。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Adds members to the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 *
	 * - A member, once added to the chat room block list, will be removed from the chat room by the server.
	 * - The method {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} is triggered, 
	 *  where the first parameter is the reason which is {@link EMAChatRoomManagerListener#BE_KICKED}.  
	 * - Members added to the block list are banned from rejoining the chat room until they are removed from the block list.
	 * 
	 * For the asynchronous method, see {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be added to block list.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom blockChatroomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.blockChatroomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 将成员添加到聊天室黑名单。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * - 成员加入黑名单的同时，将被服务器移出聊天室。
	 * - 可通过 {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} 回调通知，
	 * 第一个参数是具体的原因，加入黑名单返回的原因是 {@link EMAChatRoomManagerListener#BE_KICKED}。
	 * - 加入黑名单的成员禁止再次加入到聊天室。
	 * 同步方法见{@link #blockChatroomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param members		要加入黑名单的成员列表。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Adds members to the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * - A member, once added to the chat room block list, will be removed from the chat room by the server.
	 * - The method {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} can be called to send a callback notification,
	 *    where the first parameter is the reason which is {@link EMAChatRoomManagerListener#BE_KICKED}. 
	 * - Members added to the block list are banned from rejoining the chat room until they are removed from the block list.
	 * For the synchronous method, see {@link #blockChatroomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be added to block list.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncBlockChatroomMembers(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(blockChatroomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从聊天室黑名单中移除成员。
	 * 
	 * @note
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param members				要移除黑名单的成员列表。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes members from the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be removed from the block list.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     */
	public EMChatRoom unblockChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unblockChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 从聊天室黑名单中移除成员。
	 * 
	 * @note
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 
	 * 同步方法见 {@link #unblockChatRoomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param members		要移除黑名单的成员列表。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes members from the block list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #unblockChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be removed from the block list.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncUnBlockChatRoomMembers(final String chatRoomId, final List<String> members,
	                                        final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unblockChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 以分页的形式获取聊天室黑名单列表。
	 * 
	 * @note
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param pageNum				当前页码，从 1 开始。
	 * @param pageSize				每页期望获取的黑名单中的成员数。取值范围为 [1,50]。
	 * @return						返回聊天室黑名单列表。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the chat room block list with pagination.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The number of members on the block list that you expect to get on each page. The value range is [1,50].
	 * @return						The block list of the chat room.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
     * 
	 * 
     */
	public List<String> fetchChatRoomBlackList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		List<String> blockList = emaObject.fetchChatRoomBlackList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return blockList;
	}

	/**
	 * \~chinese
	 * 以分页的形式获取聊天室黑名单列表。
	 * 
	 * @note
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 
	 * 同步方法见 {@link #fetchChatRoomBlackList(String, int, int)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param pageNum		当前页码，从 1 开始。
	 * @param pageSize		每页期望获取的黑名单中的成员数。取值范围为 [1,50]。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回聊天室黑名单列表；
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the chat room block list with pagination.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #fetchChatRoomBlackList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param pageNum		The page number, starting from 1.
	 * @param pageSize		The number of members on the block list that you expect to get on each page. The value range is [1,50].
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the block list of the chat room;
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
     * 		
	 */
	public void asyncFetchChatRoomBlackList(final String chatRoomId, final int pageNum, final int pageSize,
	                                        final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomBlackList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员添加到白名单。
	 * 
	 * @note
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 
	 * 聊天室所有者或者管理员执行 {@link #muteAllMembers(String, EMValueCallBack)} 时，加入白名单的成员不受影响。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @param members 				加入白名单的成员列表。
	 * @param callBack				结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 								失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Adds members to the allow list of the chat room.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * When {@link #muteAllMembers(String, EMValueCallBack)} executed by the chat room owner or admin, does not work for members on the allow list.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param members 				The list of members to be added to the allowlist.
	 * @param callBack				The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 								if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void addToChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.addToWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员从白名单移除。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 成员从白名单移除后，将受到 {@link #muteAllMembers(String, EMValueCallBack)} 功能的影响。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @param members 				移除白名单的用户列表。
	 * @param callBack 				结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 								失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes members from the chat room block list.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For members removed from the block list, {@link #muteAllMembers(String, EMValueCallBack)} works.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param members 				The list of members to be removed from the chat room block list.
	 * @param callBack              The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)}; 
	 *                              if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void removeFromChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.removeFromWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 检查成员自己是否加入了白名单。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack 		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回是否在白名单中；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Checks whether the current member is on the chat room block list.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} to show whether the member is on the block list;
	 * 						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void checkIfInChatRoomWhiteList(final String chatRoomId, EMValueCallBack<Boolean> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~chinese
	 * 检查成员自己是否被禁言。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack 		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回是否被禁言中；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Checks whether the current member is muted or not.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} to show whether the member is be muted or not;
	 * 						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncCheckIfInMuteList(final String chatRoomId, EMValueCallBack<Boolean> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfBeMuted(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取白名单列表。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the chat room allow list from the server.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void fetchChatRoomWhiteList(final String chatRoomId, final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchChatRoomWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 设置全员禁言。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 聊天室所有者、管理员及加入白名单的用户不受影响。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack 		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Mutes all members.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * This method does not work for the chat room owner, admin, and members on the chat room block list.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId  	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void muteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.muteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除所有成员的禁言状态。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack 		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Unmutes all members.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void unmuteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.unmuteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 更新聊天室公告。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 异步方法见 {@link #asyncUpdateChatRoomAnnouncement(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @param announcement 			更新后的公告内容。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Updates the chat room announcement.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the asynchronous method, see {@link #asyncUpdateChatRoomAnnouncement(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param announcement 			The modified announcement content.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void updateChatRoomAnnouncement(String chatRoomId, String announcement) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.updateChatRoomAnnouncement(chatRoomId, announcement, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 更新聊天室公告。
	 * 
	 * @note
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 
	 * 同步方法见 {@link #updateChatRoomAnnouncement(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param announcement 	更新后的公告内容。
	 * @param callBack 		结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 						失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Updates the chat room announcement.
	 * 
	 * @note
	 * Only the chat room owner or admin can call this method.
	 * 
	 * For the synchronous method, see {@link #updateChatRoomAnnouncement(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID
	 * @param announcement 	The modified announcement content.
	 * @param callBack 		The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 						if this call fails, calls {@link EMCallBack#onError(int, String)}. 
	 */
	public void asyncUpdateChatRoomAnnouncement(final String chatRoomId, final String announcement, final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateChatRoomAnnouncement(chatRoomId, announcement);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容。
	 * 
	 * 异步方法见 {@link #asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @return						聊天室公告。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the chat room announcement from the server.
	 * 
	 * For the asynchronous method, see {@link #asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * @param chatRoomId 			The chat room ID.
	 * @return						The chat room announcement.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}
	 */
	public String fetchChatRoomAnnouncement(String chatRoomId) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.fetchChatRoomAnnouncement(chatRoomId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容。
	 * 
	 * 同步方法见 {@link #fetchChatRoomAnnouncement(String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack  	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回聊天室公告；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the chat room announcement from the server.
	 * 
	 * For the synchronous method, see {@link #fetchChatRoomAnnouncement(String)}.
	 * 
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack  	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the chat room announcement;
	 * 						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomAnnouncement(final String chatRoomId, final EMValueCallBack<String> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomAnnouncement(chatRoomId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 设置多个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法不支持覆盖其他成员设置的属性。
	 * 
	 * 同步方法，会阻塞当前线程。
	 * 
	 * 异步方法见 {@link #asyncSetChatroomAttributes(String,Map,boolean, EMResultCallBack)}。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param attributeMap		聊天室自定义属性集合，为键值对（key-value）结构。在键值对中，key 为属性名，不超过 128 字符；value 为属性值，不超过 4096 字符。
	 *                          每个聊天室最多有 100 个属性，每个应用的聊天室属性总大小不超过10 GB。
	 * @param autoDelete		当前成员退出聊天室是否自动删除该聊天室中其设置的所有聊天室自定义属性。
	 * - （默认）`true`:是。
	 * - `false`:否。
	 *
	 * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Adds custom chat room attributes.
	 * 
	 * @note
	 * This method does not overwrite attributes set by other members.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncSetChatroomAttributes(String,Map,boolean, EMResultCallBack)}
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The map of new chat room attributes in key-value format. 
	 *                          In a key-value pair, the key is the attribute name that can contain 128 characters at most; the value is the attribute value that cannot exceed 4096 characters. 
	 *                          A chat room can have a maximum of 100 custom attributes and the total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`: Yes.
	 * - `false`: No.
	 *
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
	 */
	private Map<String,Integer> setChatroomAttribute(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete) throws HyphenateException{
		EMAError error = new EMAError();
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),false,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> setChatroomAttributes(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete,EMAError error) {
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),false,error);
		return parseCodeJson(result);
	}

	/**
	 * \~chinese
	 * 设置多个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法不支持覆盖其他成员设置的属性。
	 *
	 * 异步方法。
	 * 
	 * @param chatRoomId		聊天室 ID。
	 * @param attributeMap		聊天室自定义属性集合，为键值对（key-value）结构。在键值对中，key 为属性名，不超过 128 字符；value 为属性值，不超过 4096 字符。
	 *                          每个聊天室最多有 100 个属性，每个应用的聊天室属性总大小不超过 10 GB。
	 * @param autoDelete		当前成员退出聊天室是否自动删除该聊天室中其设置的所有聊天室自定义属性。
	 * - （默认）`true`:是。
	 * - `false`:否。
	 * @param callBack			结果回调 {@link EMResultCallBack#onResult(int, Object)}，
	 *							举例：Object 为 Map={"k1":703} 详细可以查看{@link EMError}
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 *
	 * \~english
	 * Adds custom chat room attributes.
	 * 
	 * @note
	 * This method does not overwrite attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The new chat room attributes Key-Value. A single key cannot exceed 128 characters; the total number of keys in a chat room cannot exceed 100. A single Value can not exceed 4096 characters. The total attribute can not exceed 10 GB.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. calls {@link EMResultCallBack#onResult(int, Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link EMError}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncSetChatroomAttributes(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete ,@NonNull final EMResultCallBack<Map<String,Integer>> callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				if (null == attributeMap || attributeMap.size() == 0){
					callBack.onResult(EMError.INVALID_PARAM,new HashMap<>());
					return;
				}
				EMAError error = new EMAError();
				Map<String,Integer> resultMap = setChatroomAttributes(chatRoomId,attributeMap,autoDelete,error);
				callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~chinese
	 * 设置聊天室单个自定义属性。
	 * 
	 * @note
	 * 该方法不允许覆盖其他成员设置的属性。
	 * 
	 * 异步方法。
	 * 
	 * @param chatRoomId		聊天室 ID。
	 * @param key				聊天室属性 key，指定属性名。属性名不能超过 128 字符。每个聊天室最多可有 100 个属性。Key 支持以下字符集：
     * - 26 个小写英文字母 a-z；
     * - 26 个大写英文字母 A-Z；
     * - 10 个数字 0-9；
     * - “_”, “-”, “.”。
	 * @param value				聊天室属性 value，属性值。每个属性值不超过 4096 字符，每个应用的聊天室属性总大小不能超过 10 GB。
	 * @param autoDelete		当前成员退出聊天室是否自动删除该聊天室中其设置的所有聊天室自定义属性。
	 * - （默认）`true`:是。
	 * - `false`:否。
	 * @param callBack			结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 							失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Sets a custom chat room attribute.
	 * 
	 * @note
	 * This method does not overwrite attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param key				The chat room attribute key that specifies the attribute name. The attribute name can contain 128 characters at most. 
	 *                          A chat room can have a maximum of 100 custom attributes. The following character sets are supported:
     * - 26 lowercase English letters (a-z)
     * - 26 uppercase English letters (A-Z)
     * - 10 numbers (0-9)
     * - "_", "-", "."
	 * @param value				The chat room attribute value. The attribute value can contain a maximum of 4096 characters. The total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attribute set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 							if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncSetChatroomAttribute(String chatRoomId,String key,String value,boolean autoDelete,@NonNull final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key) ){
						callBack.onError(EMError.INVALID_PARAM,"add Attributes key Cannot be an empty string");
						return;
					}
					Map<String,String> attributeMap = new HashMap<>();
					attributeMap.put(key,value);
					setChatroomAttribute(chatRoomId,attributeMap,autoDelete);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 强制设置多个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法会覆盖其他成员设置的属性。
	 * 
	 * 异步方法见 {@link #asyncSetChatroomAttributesForced(String, Map, boolean, EMResultCallBack)} }
	 *
	 * 同步方法，会阻塞当前线程。
	 * 
	 * @param chatRoomId		聊天室 ID。
	 * @param attributeMap		聊天室自定义属性集合，为键值对（key-value）结构。在键值对中，key 为属性名，不超过 128 字符；value 为属性值，不超过 4096 字符。
	 *                          每个聊天室最多有 100 个属性，每个应用的聊天室属性总大小不超过 10 GB。
	 * @param autoDelete		当前成员退出聊天室是否自动删除该聊天室中其设置的所有聊天室自定义属性。
	 * - （默认）`true`:是。
	 * - `false`:否。
	 * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Sets custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method overwrites attributes set by other members.
	 * 
	 * For the synchronous method, see {@link #asyncSetChatroomAttributesForced(String, Map, boolean, EMResultCallBack)}}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The map of new custom chat room attributes in key-value format. 
	 *                          In a key-value pair, the key is the attribute name that can contain 128 characters at most; the value is the attribute value that cannot exceed 4096 characters. 
	 *                          A chat room can have a maximum of 100 custom attributes and the total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 *
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
	 */
	private Map<String,Integer> setChatroomAttributeForced(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),true,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> setChatroomAttributesForced(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete,EMAError error){
		String result = emaObject.setChatroomAttributes(chatRoomId,toJsonString(attributeMap,autoDelete),true,error);
		return parseCodeJson(result);
	}

	/**
	 * \~chinese
	 * 强制设置多个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法会覆盖其他成员设置的属性。
	 *
	 * 异步方法。
	 * 
	 * @param chatRoomId		聊天室 ID。
	 * @param attributeMap		聊天室自定义属性集合，为键值对（key-value）结构。在键值对中，key 为属性名，不超过 128 字符；value 为属性值，不超过 4096 字符。
	 *                          每个聊天室最多有 100 个属性，每个应用的聊天室属性总大小不超过 10 GB。
	 * @param autoDelete		成员退出聊天室是否自动删除该聊天室中其设置的所有聊天室自定义属性。
	 * - （默认）`true`:是。
	 * - `false`:否。
	 * @param callBack			结果回调{@link EMResultCallBack#onResult(int,Object)}，
	 *							举例：Object 为 Map={"k1":703} 详细可以查看{@link EMError}
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 *
	 * \~english
	 * Sets custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method overwrites attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param attributeMap		The map of new chat room attributes in key-value format. 
	 *                          In a key-value pair, the key is the attribute name that can contain 128 characters at most; the value is the attribute value that cannot exceed 4096 characters. 
	 *                          A chat room can have a maximum of 100 custom attributes and the total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. calls {@link EMResultCallBack#onResult(int,Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link EMError}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncSetChatroomAttributesForced(String chatRoomId, Map<String,String> attributeMap ,boolean autoDelete ,@NonNull final EMResultCallBack<Map<String,Integer>> callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
					if (null == attributeMap || attributeMap.size() == 0 ){
						callBack.onResult(EMError.INVALID_PARAM,new HashMap<>());
						return;
					}
					EMAError error = new EMAError();
					Map<String,Integer> resultMap = setChatroomAttributesForced(chatRoomId,attributeMap,autoDelete,error);
					callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~chinese
	 * 强制设置单个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法会覆盖其他成员设置的属性。
	 *
	 * 异步方法。
	 * 
	 * @param chatRoomId		聊天室 ID。
	 * @param key				聊天室属性 key，指定属性名。属性名不能超过 128 字符。每个聊天室最多可有 100 个属性。Key 支持以下字符集：
     * • 26 个小写英文字母 a-z；
     * • 26 个大写英文字母 A-Z；
     * • 10 个数字 0-9；
     * • “_”, “-”, “.”。
	 * @param value				聊天室属性值。每个属性值不超过 4096 字符，每个应用的聊天室属性总大小不能超过 10 GB。
	 * @param autoDelete		当前成员退出聊天室是否自动删除该聊天室中其设置的所有聊天室自定义属性。
	 * - （默认）`true`:是。
	 * - `false`:否。
	 * @param callBack			结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 							失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Sets a custom chat room attribute forcibly.
	 * 
	 * @note
	 * This method overwrites attributes set by other members.
	 *
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param key				The chat room attribute key that specifies the attribute name. The attribute name can contain 128 characters at most. 
	 *                          A chat room can have a maximum of 100 custom attributes. The following character sets are supported:
     * - 26 lowercase English letters (a-z)
     * - 26 uppercase English letters (A-Z)
     * - 10 numbers (0-9)
     * - "_", "-", "."
	 * @param value				The chat room attribute value. The attribute value can contain a maximum of 4096 characters. The total length of custom chat room attributes cannot exceed 10 GB for each app.
	 * @param autoDelete		Whether to delete the chat room attributes set by the member when he or she exits the chat room.
	 * - (Default)`true`:Yes.
	 * - `false`: No.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 							if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncSetChatroomAttributeForced(String chatRoomId,String key,String value,boolean autoDelete ,@NonNull final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key) ){
						callBack.onError(EMError.INVALID_PARAM,"add Attributes key Cannot be an empty string");
						return;
					}
					Map<String,String> attributeMap = new HashMap<>();
					attributeMap.put(key,value);
					setChatroomAttributeForced(chatRoomId,attributeMap,autoDelete);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 根据聊天室属性 key 列表获取属性列表。
	 * 
	 * 同步方法，会阻塞当前线程。
	 * 
	 * 异步方法见 {@link #asyncFetchChatroomAttributesFromServer(String,List,EMValueCallBack)}。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param keyList				聊天室属性 key 的列表，传 `null` 返回所有自定义属性。
	 * @throws HyphenateException 	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the list of custom chat room attributes based on the attribute key list.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncFetchChatroomAttributesFromServer(String,List,EMValueCallBack)}.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param keyList				The list of chat room attribute keys. If you pass `null`, the SDK returns all custom attributes of the chat room.
	 * @throws HyphenateException 	A description of the exception. See {@link EMError}.
	 */
	private Map<String,String> fetchChatroomAttributes(String chatRoomId , List<String> keyList) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.fetchChatroomAttributes(chatRoomId,keyList,error);
		handleError(error);
		return parseJson(result);
	}

	/**
	 * \~chinese
	 * 根据聊天室属性 key 列表获取属性列表。
	 * 
	 * 异步方法。
	 * 
	 * 同步方法见 {@link #fetchChatroomAttributes(String,List)}。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param keyList			聊天室属性 key 列表，传 `null` 返回所有自定义属性。
	 * @param callBack			结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 							失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the list of custom chat room attributes based on the attribute key list.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #fetchChatroomAttributes(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param keyList			The list of chat room attribute keys. If you pass `null`, the SDK returns all custom attributes of the chat room.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 							if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatroomAttributesFromServer(String chatRoomId,List<String> keyList,@NonNull final EMValueCallBack<Map<String,String>> callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatroomAttributes(chatRoomId,keyList));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室所有自定义属性。
	 * 
	 * 异步方法。
	 * 
	 * 同步方法见 {@link #fetchChatroomAttributes(String,List)}。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param callBack			结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 							失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets all the custom attributes of a chat room.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #fetchChatroomAttributes(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 							if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomAllAttributesFromServer(String chatRoomId,@NonNull final EMValueCallBack<Map<String,String>> callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatroomAttributes(chatRoomId,null));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 根据聊天室 ID 和属性 key 列表删除聊天室自定义属性。
	 * 
	 * @note
	 * 该方法不支持删除其他成员设置的属性。
	 * 
	 * 同步方法，会阻塞当前线程。
	 * 
	 * 异步方法见 {@link #asyncRemoveChatRoomAttributesFromServer(String,List, EMResultCallBack)}。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param keyList				要删除的聊天室自定义属性的 key 列表。
	 * @throws HyphenateException 	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes custom chat room attributes by chat room ID and attribute key list.
	 * 
	 * @note
	 * This method does not remove attributes set by other members.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncRemoveChatRoomAttributesFromServer(String,List, EMResultCallBack)}.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param keyList				The key list of the chat room attributes to remove. If you pass `null`, the SDK returns all the custom attributes of the chat room.
	 * @throws HyphenateException 	A description of the exception. See {@link EMError}
	 */
	private Map<String,Integer> removeChatroomAttribute(String chatRoomId, List<String> keyList) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,false,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> removeChatroomAttributes(String chatRoomId, List<String> keyList,EMAError error) {
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,false,error);
		return parseCodeJson(result);
	}

	/**
	 * \~chinese
	 * 根据聊天室自定义属性 key 列表批量删除自定义聊天室属性。
	 * 
	 * @note
	 * 该方法不删除其他成员设置的属性。
	 * 
	 * 异步方法。
	 * 
	 * @param chatRoomId		聊天室 ID。
	 * @param keyList			要删除的聊天室自定义属性的 key 列表。
	 * @param callBack			结果回调{@link EMResultCallBack#onResult(int, Object)}，
	 *							举例：Object 为 Map={"k1":703} 详细可以查看{@link EMError}
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 *
	 * \~english
	 * Removes custom chat room attributes by the attribute key list.
	 * 
	 * @note
	 * This method does not remove attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId		The chat room ID.
	 * @param keyList			The key list of the custom chat room attributes to remove.
	 * @param callBack			The completion callback. calls {@link EMResultCallBack#onResult(int,Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link EMError}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncRemoveChatRoomAttributesFromServer(String chatRoomId, List<String> keyList,@NonNull final EMResultCallBack<Map<String,Integer>> callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
					if ( null == keyList || keyList.size() == 0 ){
						callBack.onResult(EMError.INVALID_PARAM,new HashMap<>());
						return;
					}
					EMAError error = new EMAError();
					Map<String,Integer> resultMap = removeChatroomAttributes(chatRoomId,keyList,error);
					callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~chinese
	 * 删除单个聊天室属性。
	 * 
	 * @note
	 * 该方法不删除其他成员设置的属性。
	 * 
	 * 异步方法。
	 * 
	 * 同步方法见 {@link #removeChatroomAttribute(String,List)}。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param key				要删除的聊天室自定义属性的 key。
	 * @param callBack			结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 							失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes a custom chat room attribute.
	 * 
	 * @note
	 * This method does not remove attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #removeChatroomAttribute(String,List)}
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param key				The key of the custom chat room attribute to remove.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 							if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomAttributeFromServer(String chatRoomId, String key,@NonNull final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key)){
						callBack.onError(EMError.INVALID_PARAM,"remove Attribute key Cannot be an empty string");
						return;
					}
					List<String> keyList = new ArrayList<>();
					keyList.add(key);
					removeChatroomAttribute(chatRoomId,keyList);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 强制删除多个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法支持删除其他成员设置的自定义属性。
	 * 
	 * 同步方法，会阻塞当前线程。
	 * 
	 * 异步方法见 {@link #asyncRemoveChatRoomAttributesFromServerForced(String,List, EMResultCallBack)}
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param keyList				要删除的聊天室自定义属性的 key 列表。
	 * @throws HyphenateException 	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method removes attributes set by other members.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * For the synchronous method, see {@link #asyncRemoveChatRoomAttributesFromServerForced(String,List, EMResultCallBack)}.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param keyList				The key list of the custom chat room attributes to remove.
	 * @throws HyphenateException 	A description of the exception. See {@link EMError}.
	 */
	private Map<String,Integer> removeChatroomAttributeForced(String chatRoomId, List<String> keyList) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,true,error);
		handlePartialError(error);
		return parseCodeJson(result);
	}

	private Map<String,Integer> removeChatroomAttributesForced(String chatRoomId, List<String> keyList,EMAError error) {
		String result = emaObject.removeChatroomAttributes(chatRoomId,keyList,true,error);
		return parseCodeJson(result);
	}


	/**
	 * \~chinese
	 * 强制删除多个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法支持删除其他成员设置的自定义属性。
	 * 
	 * 异步方法。
	 * 
	 * 同步方法见 {@link #removeChatroomAttributeForced(String, List)}}。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param keyList			要删除的聊天室自定义属性的 key 列表。
	 * @param callBack			结果回调{@link EMResultCallBack#onResult(int, Object)} (int,Object)}，
	 *							举例：Object 为 Map={"k1":703} 详细可以查看{@link EMError}
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 *
	 * \~english
	 * Removes custom chat room attributes forcibly.
	 * 
	 * @note
	 * This method removes attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #removeChatroomAttributeForced(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param keyList			The key list of the custom chat room attributes to remove.
	 * @param callBack			The completion callback. calls {@link EMResultCallBack#onResult(int, Object)} (int,Object)};
	 * 							return Object is Map={"k1":703}, A description of the exception. See {@link EMError}.
	 *							EXCEED_SERVICE_LIMIT = 4;
	 *							INVALID_PARAM = 110;
	 *							QUERY_PARAM_REACHES_LIMIT = 112;
	 *							CHATROOM_PERMISSION_DENIED = 703;
	 */
	public void asyncRemoveChatRoomAttributesFromServerForced(String chatRoomId, List<String> keyList,@NonNull final EMResultCallBack<Map<String,Integer>> callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
					if (null == keyList || keyList.size() == 0 ){
						callBack.onResult(EMError.INVALID_PARAM,new HashMap<>());
						return;
					}
					EMAError error = new EMAError();
					Map<String,Integer> resultMap = removeChatroomAttributesForced(chatRoomId,keyList,error);
					callBack.onResult(error.errCode(),resultMap);
			}
		});
	}

	/**
	 * \~chinese
	 * 强制删除单个聊天室自定义属性。
	 * 
	 * @note
	 * 该方法支持删除其他成员设置的自定义属性。
	 * 
	 * 异步方法。
	 * 
	 * 同步方法见 {@link #removeChatroomAttributeForced(String, List)}}。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param key				要删除的聊天室自定义属性的 key。
	 * @param callBack			结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 							失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes a custom chat room attribute forcibly.
	 * 
	 * @note
	 * This method removes attributes set by other members.
	 * 
	 * This is an asynchronous method.
	 * 
	 * For the synchronous method, see {@link #removeChatroomAttributeForced(String,List)}.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param key				The key of the custom chat room attribute to remove.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 							if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomAttributeFromServerForced(String chatRoomId, String key,@NonNull final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					if (TextUtils.isEmpty(key)){
						callBack.onError(EMError.INVALID_PARAM,"remove Attribute key Cannot be an empty string");
						return;
					}
					List<String> keyList = new ArrayList<>();
					keyList.add(key);
					removeChatroomAttributeForced(chatRoomId,keyList);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}



	// ============================= chatRoom_reform new add api end
	private void handleError(EMAError error)  throws HyphenateException {
	    if (error.errCode() != EMAError.EM_NO_ERROR) {
	        throw new HyphenateException(error);
	    }
	}

	private void handlePartialError(EMAError error)  throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR && error.errCode() != EMAError.PARTIAL_SUCCESS) {
			throw new HyphenateException(error);
		}
	}

	private Map<String,String> parseJson(String jsonString) throws HyphenateException {
		if (TextUtils.isEmpty(jsonString)) return null;
		Map<String, String> resultMap = new HashMap<>();
		try {
			JSONObject resultJson = new JSONObject(jsonString);
			Iterator<String> iterator = resultJson.keys();
			while (iterator.hasNext()) {
				String k = iterator.next();
				String v = resultJson.getString(k);
				resultMap.put(k,v);
			}
			return resultMap;
		} catch (JSONException e) {
			e.printStackTrace();
			throw new HyphenateException(EMError.SERVER_UNKNOWN_ERROR,"Unknown server error");
		}
	}

	private Map<String,Integer> parseCodeJson(String jsonString){
		Map<String, Integer> resultMap = new HashMap<>();
		if (TextUtils.isEmpty(jsonString)) return resultMap;
		try {
			JSONObject resultJson = new JSONObject(jsonString);
			Iterator<String> iterator = resultJson.keys();
			while (iterator.hasNext()) {
				String k = iterator.next();
				int v = resultJson.getInt(k);
				resultMap.put(k,v);
			}
		} catch (JSONException e) {
			EMLog.e("parseCodeJson",e.getMessage());
		}
		return resultMap;
	}

	private Map<String,String> parseJsonUpdate(String jsonString){
		if (TextUtils.isEmpty(jsonString)) return null;
		Map<String, String> resultMap = new HashMap<>();
		try {
			List<String> successList = new ArrayList<>();
			JSONObject resultJson = new JSONObject(jsonString);
			JSONObject result = resultJson.getJSONObject("result");
			JSONArray jsonArray = result.getJSONArray("successKeys");
			for (int i = 0; i < jsonArray.length(); i++) {
				successList.add(jsonArray.getString(i));
			}
			JSONObject jsonObject = resultJson.getJSONObject("properties");
			Iterator<String> iterator = jsonObject.keys();
			while (iterator.hasNext()) {
				String k = iterator.next();
				String v = jsonObject.getString(k);
				if (successList.contains(k)){
					resultMap.put(k,v);
				}
			}
			return resultMap;
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	private List<String> parseJsonRemove(String jsonString){
		if (TextUtils.isEmpty(jsonString)) return null;
		try {
			List<String> successList = new ArrayList<>();
			JSONObject resultJson = new JSONObject(jsonString);
			JSONObject result = resultJson.getJSONObject("result");
			JSONArray jsonArray = result.getJSONArray("successKeys");
			for (int i = 0; i < jsonArray.length(); i++) {
				successList.add(jsonArray.getString(i));
			}
			return successList;
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	private String toJsonString(Map<String,String> extJson,boolean autoDelete){
		JSONObject resultJson = null;
		try {
			resultJson = new JSONObject();
			resultJson.put("metaData",new JSONObject(extJson));
			if(autoDelete){
				resultJson.put("autoDelete","DELETE");
			}else {
				resultJson.put("autoDelete","NO_DELETE");
			}
			return resultJson.toString();
		} catch (JSONException e) {
			e.printStackTrace();
			return "";
		}
	}
	
	void onLogout() {
	    chatRoomListeners.clear();
	}
}
