package com.hyphenate.push;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.support.annotation.NonNull;
import android.util.Pair;

import com.hyphenate.EMError;
import com.hyphenate.chat.EMClient;
import com.hyphenate.chat.core.EMPreferenceUtils;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.push.common.PushUtil;
import com.hyphenate.push.platform.IPush;
import com.hyphenate.push.platform.fcm.EMFCMPush;
import com.hyphenate.push.platform.hms.EMHMSPush;
import com.hyphenate.push.platform.meizu.EMMzPush;
import com.hyphenate.push.platform.mi.EMMiPush;
import com.hyphenate.push.platform.normal.EMNormalPush;
import com.hyphenate.push.platform.oppo.EMOppoPush;
import com.hyphenate.push.platform.vivo.EMVivoPush;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;

import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.Random;

/**
 * \~chinese
 * 推送帮助类。
 *
 * \~english
 * The push help class.
 */
public class EMPushHelper {
    private static final String TAG = "EMPushHelper";

    private static final int TIMES_RETRY = 3;
    private static final int MSG_WHAT_REGISTER = 0;
    private static final int MSG_WHAT_UNREGISTER = 1;

    private Context context;
    private EMPushConfig pushConfig;
    private IPush pushClient;
    private Handler handler;

    private EMPushType pushType;
    private String pushToken;
    private boolean unregisterSuccess;

    private final Object bindLock = new Object();
    private final Object unbindLock = new Object();

    private PushListener pushListener;

    private boolean isClientInit = false;

    private EMPushHelper() {
        final HandlerThread handlerThread = new HandlerThread("token-uploader");
        handlerThread.start();

        handler = new Handler(handlerThread.getLooper()) {
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what) {
                    case MSG_WHAT_REGISTER:
                        synchronized (bindLock) { // 确保task在执行时没有新的tasks加入，避免该task执行完后把新加入的tasks清空。
                            String token = (String) msg.obj;
                            boolean registerSuccess = uploadTokenInternal(pushClient.getNotifierName(), token);
                            if (registerSuccess) {
                                removeMessages(MSG_WHAT_REGISTER);
                                return;
                            }
                            // 当所有task执行完且还没有上传成功时，更换为NORMAL push type。
                            boolean allTasksExecuted = !hasMessages(MSG_WHAT_REGISTER);
                            if (allTasksExecuted) {
                                EMPushHelper.this.onErrorResponse(pushType, EMError.PUSH_BIND_FAILED);
                                register(EMPushType.NORMAL);
                            }
                        }
                        break;
                    case MSG_WHAT_UNREGISTER:
                        unregisterSuccess = uploadTokenInternal(pushClient.getNotifierName(), "");
                        if (!unregisterSuccess) {
                            EMPushHelper.this.onErrorResponse(pushType, EMError.PUSH_UNBIND_FAILED);
                        }

                        synchronized (unbindLock) {
                            unbindLock.notifyAll();
                        }
                        break;
                    default:
                        super.handleMessage(msg);
                        break;
                }
            }
        };
    }

    /**
     * \~chinese
     * 获取 EMPushHelper 的单例。
     * @return EMPushHelper 的单例。
     *
     * \~chinese
     * Gets a singleton of EMPushHelper.
     * @return The singleton of EMPushHelper.
     */
    public static EMPushHelper getInstance() {
        return InstanceHolder.INSTANCE;
    }

    /**
     * \~chinese
     * 设置推送监听。
     * 可重写 {@link PushListener#isSupportPush(EMPushType, EMPushConfig)}，设置支持的推送类型。
     * SDK 支持的推送类型为 EMPushType.FCM，EMPushType.MIPUSH，EMPushType.HMSPUSH，EMPushType.MEIZUPUSH，EMPushType.OPPOPUSH，EMPushType.VIVOPUSH 这几种。如果开发者定义的支持推送类型多于这几种，将永远不会被匹配到，SDK 不会报错，但会选择推送类型为 EMPushType.NORMAL。
     * @param callback  自定义推送监听。
     *
     * \~english
     * Sets the push listener.
     * You can override {@link PushListener#isSupportPush(EMPushType, EMPushConfig)} to set the push types. Supported push types are EMPushType.FCM，EMPushType.MIPUSH，EMPushType.HMSPUSH，EMPushType.MEIZUPUSH，EMPushType.OPPOPUSH，EMPushType.VIVOPUSH. If you set a push type beyond this scope, the SDK will choose the Normal push of the chat service without error information without reporting error information.
     * @param callback  The push listener.
     */
    public void setPushListener(PushListener callback) {
        this.pushListener = callback;
    }

    /**
     * \~chinese
     * 初始化 EMPushHelper。
     * @param context   上下文。
     * @param config    推送配置。
     *
     * \~english
     * Initializes the EMPushHelper.
     * @param context  The context of Android Activity or Application.
     * @param config   The Push configurations.
     */
    public void init(Context context, EMPushConfig config) {
        EMLog.e(TAG, TAG + " init, config: " + config.toString());

        if (context == null || config == null) {
            throw new IllegalArgumentException("Null parameters, context=" + context + ", config=" + config);
        }

        this.context = context.getApplicationContext();
        this.pushConfig = config;
    }

    /**
     * \~chinese
     * 注册推送。
     * 登录成功后，由 SDK 调用，用户无需调用。
     *
     * \~english
     * Registers the push service.
     * After successful login, the SDK calls the method instead of you.
     */
    public void register() {
        if (context == null || pushConfig == null) {
            EMLog.e(TAG, "EMPushHelper#init(Context, EMPushConfig) method not call previously.");
            return;
        }

        EMPushType pushType = getPreferPushType(pushConfig);
        EMLog.e(TAG, TAG + " register, prefer push type: " + pushType);
        register(pushType);
    }

    /**
     * \~chinese
     * 解注册推送。
     * 调用 {@link EMClient#logout(boolean)} 时，由 SDK 调用。
     * @param unbindToken  是否解绑设备。
     * @return  解注册是否成功。
     *
     * \~english
     * Unregisters the push service.
     * This method is called by SDK during the call of {@link EMClient#logout(boolean)}}.  
     * @param unbindToken   Whether to unbind the device.
     * @return Whether the unregistering push succeeds.
     */
    public boolean unregister(boolean unbindToken) {
        EMLog.e(TAG, TAG + " unregister, unbind token: " + unbindToken);

        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered previously, return true directly.");
            return true;
        }

        this.isClientInit = false;
        pushClient.unregister(context);

        // 停止 Token 上传操作。
        handler.removeMessages(MSG_WHAT_REGISTER);

        if (!unbindToken) {
            pushType = null;
            return true;
        }

        deleteToken();
        // Wait for token delete result.
        synchronized (unbindLock) {
            try {
                unbindLock.wait();
            } catch (InterruptedException e) {
            }
        }
        if (unregisterSuccess) {
            pushType = null;
        }
        EMLog.e(TAG, "Push type after unregister is " + pushType);
        return unregisterSuccess;
    }

    private boolean isTokenChanged(EMPushType type, final String token) {
        String savedToken = EMPushHelper.getInstance().getPushTokenWithType(type);

        if ((savedToken == null) || !savedToken.equals(token)) {
            EMPushHelper.getInstance().setPushTokenWithType(type, token);
            return true;
        }
        return false;
    }

    /**
     * \~chinese
     * 接收并上传设备 token。
     * @param type  推送类型。
     * @param token 设备 token。
     *
     * \~english
     * Receives and uploads the device token。
     * @param type  The push type.
     * @param token  The device token.
     */
    public void onReceiveToken(EMPushType type, final String token) {
        EMLog.e(TAG, "onReceiveToken: " + type + " - " + token);
        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered, abort token upload action.");
            return;
        }
        this.pushToken = token;
        if (isTokenChanged(type, token)) {
            EMLog.d(TAG, "push token changed, upload to server");
            uploadToken(token);
            return;
        }
        if (EMClient.getInstance().getChatConfigPrivate().isNewLoginOnDevice()) {
            EMLog.d(TAG, "push token not change, but last login is not on this device, upload to server");
            uploadToken(token);
        } else {
            EMLog.e(TAG, TAG + " not first login, ignore token upload action.");
        }
    }

    /**
     * \~chinese
     * 发生推送错误回调。当推送发生错误时，如绑定推送令牌失败时、解绑推送令牌失败时、不支持用户设置的推送类型等 SDK 会触发该回调。你可以通过 resultCode 了解具体的错误类型。
     * 如果自定义的 PushListener 不为空，则将错误信息设置给 {@link PushListener#onError(EMPushType, long)}。
     * @param type          推送类型。
     * @param resultCode    错误码。
     *
     * \~english
     * Occurs when a push error occurs.
     * The SDK triggers this callback when a push error, such as push token binding failure (PUSH_BIND_FAILED), push token unbinding failure (PUSH_UNBIND_FAILED), or unsupported custom types (PUSH_NOT_SUPPORT), occurs. You can have more detailed information on the error from the returned resultCode.
     * If the customized PushListener is not null, pass error info to {@link PushListener#onError(EMPushType, long)}.
     * @param type          The push type.
     * @param resultCode    The error code.
     */
    public void onErrorResponse(EMPushType type, long resultCode) {
        EMLog.e(TAG, "onErrorResponse: " + type + " - " + resultCode);
        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered, abort error response action.");
            return;
        }

        if (resultCode == EMError.PUSH_NOT_SUPPORT) {
            register(EMPushType.NORMAL);
        }

        if (pushListener != null) pushListener.onError(type, resultCode);
    }

    /**
     * \~chinese
     * 获取推送类型。
     * @return  推送类型。
     *
     * \~english
     * Gets the push type.
     * @return  The push type.
     */
    public EMPushType getPushType() {
        return pushType;
    }

    /**
     * \~chinese
     * 获取推送的设备 token。
     * @return  推送的设备 token。
     *
     * \~english
     * Gets the push token.
     * @return  The push token.
     */
    public String getPushToken() {
        return pushToken;
    }

    /**
     * \~chinese
     * 获取 FCM 推送的设备 token。
     * @return  FCM 推送的设备 token。
     *
     * \~english
     * Gets the FCM's push token.
     * @return  The FCM's push token.
     */
    public String getFCMPushToken() {
        return EMPreferenceUtils.getInstance().getFCMPushToken();
    }

    /**
     * \~chinese
     * 保存 FCM 推送的设备 token。
     * @return  FCM 推送的设备 token。
     *
     * \~english
     * Saves the FCM's push token.
     * @return  The FCM's push token.
     */
    public void setFCMPushToken(String token) {
        EMPreferenceUtils.getInstance().setFCMPushToken(token);
    }

     /**
     * \~chinese
     * 根据推送类型获取推送 token。
     * 
     * @param type 推送类型。
     * @return 推送 token。
     * 
     * \~english
     * Gets the push token with the push type.
     * 
     * @param type The push type.
     * @return The push token.
     */
    public String getPushTokenWithType(EMPushType type) {
        return EMPreferenceUtils.getInstance().getPushTokenWithType(type);
    }

     /**
     * \~chinese
     * 根据推送类型设置推送 token。
     * 
     * @param type 推送类型。
     * @param token 推送 token。
     * 
     * \~english
     * Sets the push token with the push type.
     * 
     * @param type The push type.
     * @param token The push token.
     */
    public void setPushTokenWithType(EMPushType type, final String token) {
        EMPreferenceUtils.getInstance().setPushTokenWithType(type, token);
    }

    private void register(@NonNull EMPushType pushType) {
        if (this.pushType == pushType) {
            EMLog.e(TAG, "Push type " + pushType + " no change, return. ");
            return;
        }

        if (this.pushClient != null) {
            EMLog.e(TAG, pushClient.getPushType() + " push already exists, unregister it and change to " + pushType + " push.");
            pushClient.unregister(context);
        }

        this.pushType = pushType;

        switch (pushType) {
            case FCM:
                pushClient = new EMFCMPush();
                break;
            case MIPUSH:
                pushClient = new EMMiPush();
                break;
            case OPPOPUSH:
                pushClient = new EMOppoPush();
                break;
            case VIVOPUSH:
                pushClient = new EMVivoPush();
                break;
            case MEIZUPUSH:
                pushClient = new EMMzPush();
                break;
            case HMSPUSH:
                pushClient = new EMHMSPush();
                break;
            case NORMAL:
            default:
                pushClient = new EMNormalPush();
                break;
        }

        this.isClientInit = true;
        pushClient.register(context, pushConfig, pushListener);
    }

    private void uploadToken(String token) {
        // Cancel all previous upload tasks first.
        handler.removeMessages(MSG_WHAT_REGISTER);

        synchronized (bindLock) { // 确保把tasks放入queue时没有task正在执行。以避免task执行完后把刚放进的tasks清空。
            for (int i = -1; i < TIMES_RETRY; i++) {
                Message msg = handler.obtainMessage(MSG_WHAT_REGISTER, token);
                if (i == -1) {
                    handler.sendMessage(msg);
                } else {
                    int delaySeconds = randomDelay(i);
                    EMLog.i(TAG, "Retry upload after " + delaySeconds + "s if failed.");
                    handler.sendMessageDelayed(msg, delaySeconds * 1000);
                }
            }
        }
    }

    private void deleteToken() {
        handler.obtainMessage(MSG_WHAT_UNREGISTER).sendToTarget();
    }

    private boolean uploadTokenInternal(String notifierName, String token) {
        String remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl(true, false) + "/users/"
                + EMClient.getInstance().getCurrentUser();
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(EMClient.getInstance().getContext());
        JSONObject json = new JSONObject();
        try {
            json.put("device_token", token);
            json.put("notifier_name", notifierName);
            json.put("device_id", deviceFactory.getDeviceUuid().toString());
        } catch (Exception e) {
            EMLog.e(TAG, "uploadTokenInternal put json exception: " + e.toString());
        }
        int retry_times = 2;
        do {
            try {
                EMLog.e(TAG, "uploadTokenInternal, token=" + token + ", url=" + remoteUrl
                        + ", notifier name=" + notifierName);

                Pair<Integer, String> response = EMHttpClient.getInstance().sendRequestWithToken(remoteUrl,
                        json.toString(), EMHttpClient.PUT);
                int statusCode = response.first;
                String content = response.second;

                if (statusCode == HttpURLConnection.HTTP_OK) {
                    EMLog.e(TAG, "uploadTokenInternal success.");
                    return true;
                }

                EMLog.e(TAG, "uploadTokenInternal failed: " + content);
                remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + EMClient.getInstance().getCurrentUser();
            } catch (Exception e) {
                EMLog.e(TAG, "uploadTokenInternal exception: " + e.toString());
                remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + EMClient.getInstance().getCurrentUser();
            }
        } while (--retry_times > 0);

        return false;
    }

    private EMPushType getPreferPushType(EMPushConfig pushConfig) {
        EMPushType[] supportedPushTypes = new EMPushType[]{
                EMPushType.FCM,
                EMPushType.MIPUSH,
                EMPushType.HMSPUSH,
                EMPushType.MEIZUPUSH,
                EMPushType.OPPOPUSH,
                EMPushType.VIVOPUSH,
        };

        ArrayList<EMPushType> enabledPushTypes = pushConfig.getEnabledPushTypes();

        for (EMPushType pushType : supportedPushTypes) {
            if (enabledPushTypes.contains(pushType) && isSupportPush(pushType, pushConfig)) {
                return pushType;
            }
        }

        return EMPushType.NORMAL;
    }

    private boolean isSupportPush(EMPushType pushType, EMPushConfig pushConfig) {
        boolean support;
        if (pushListener != null) {
            support = pushListener.isSupportPush(pushType, pushConfig);
        } else {
            support = PushUtil.isSupportPush(pushType, pushConfig);
        }
        EMLog.i(TAG, "isSupportPush: " + pushType + " - " + support);
        return support;
    }

    public int randomDelay(int attempts) {
        if (attempts == 0) {
            return new Random().nextInt(5) + 1;
        }

        if (attempts == 1) {
            return new Random().nextInt(54) + 6;
        }

        return new Random().nextInt(540) + 60;
    }

    private static class InstanceHolder {
        static EMPushHelper INSTANCE = new EMPushHelper();
    }
}
