package com.hyphenate.push;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;

import com.hyphenate.util.EMLog;

import java.util.ArrayList;

/**
 * \~chinese
 * 推送配置类。
 *
 * \~english
 * The push configuration class.
 */
public class EMPushConfig {
    private static final String TAG = "EMPushConfig";

    private String fcmSenderId;
    private String hwAppId;
    private String miAppId;
    private String miAppKey;
    private String mzAppId;
    private String mzAppKey;
    private String oppoAppKey;
    private String oppoAppSecret;
    private String vivoAppId;
    private String vivoAppKey;
    private ArrayList<EMPushType> enabledPushTypes;

    private EMPushConfig() {
    }

    /**
     * \~chinese
     * 获取 FCM 的 Sender ID。
     * @return  FCM 的 Sender ID。
     *
     * \~english
     * Gets the sender ID of FCM.
     * @return  The sender ID of FCM.
     */
    public String getFcmSenderId() {
        return fcmSenderId;
    }

    /**
     * \~chinese
     * 获取华为推送的 App ID。
     * @return  华为推送的 App ID。
     *
     * \~english
     * Gets the app ID of Huawei push.
     * @return  The app ID of Huawei push.
     */
    public String getHwAppId() {
        return hwAppId;
    }

    /**
     * \~chinese
     * 获取小米推送的 App ID。
     * @return  小米推送的 App ID。
     *
     * \~english
     * Gets the app ID of Mi push.
     * @return  The app ID of Mi push.
     */
    public String getMiAppId() {
        return this.miAppId;
    }

    /**
     * \~chinese
     * 获取小米推送的 App Key。
     * @return  小米推送的 App Key。
     *
     * \~english
     * Gets the app key of Mi push.
     * @return  The app key of Mi push.
     */
    public String getMiAppKey() {
        return this.miAppKey;
    }

    /**
     * \~chinese
     * 获取魅族推送的 App ID。
     * @return 魅族推送的 App ID。
     *
     * \~english
     * Gets the app ID of Meizu push.
     * @return The app ID of Meizu push.
     */
    public String getMzAppId() {
        return this.mzAppId;
    }

    /**
     * \~chinese
     * 获取魅族推送的 App Key。
     * @return 魅族推送的 App Key。
     *
     * \~english
     * Gets the app key of Meizu push.
     * @return The app key of Meizu push.
     */
    public String getMzAppKey() {
        return this.mzAppKey;
    }

    /**
     * \~chinese
     * 获取 Oppo 推送的 App Key。
     * @return Oppo 推送的 App Key。
     *
     * \~english
     * Gets the app key of Oppo push.
     * @return The app key of Oppo push.
     */
    public String getOppoAppKey() {
        return this.oppoAppKey;
    }

    /**
     * \~chinese
     * 获取 Oppo 推送的 secret。
     * @return Oppo 推送的 secret。
     *
     * \~english
     * Gets the secret of Oppo push.
     * @return The secret of Oppo push.
     */
    public String getOppoAppSecret() {
        return this.oppoAppSecret;
    }

    /**
     * \~chinese
     * 获取 Vivo 推送的 App ID。
     * @return Vivo 推送的 App ID。
     *
     * \~english
     * Gets the app ID of Vivo push.
     * @return The app ID of Vivo push.
     */
    public String getVivoAppId() {
        return vivoAppId;
    }

    /**
     * \~chinese
     * 获取 Vivo 推送的 App Key。
     * @return Vivo 推送的 App Key。
     *
     * \~english
     * Gets the app key of Vivo push.
     * @return The app key of Vivo push.
     */
    public String getVivoAppKey() {
        return vivoAppKey;
    }

    /**
     * \~chinese
     * 获取支持的推送类型。
     * @return 支持的推送类型列表。
     *
     * \~english
     * Gets the supported push types.
     * @return The list of supported push types.

     */
    public ArrayList<EMPushType> getEnabledPushTypes() {
        return this.enabledPushTypes;
    }

    @Override
    public String toString() {
        return "EMPushConfig{" +
                "fcmSenderId='" + fcmSenderId + '\'' +
                ", hwAppId='" + hwAppId + '\'' +
                ", miAppId='" + miAppId + '\'' +
                ", miAppKey='" + miAppKey + '\'' +
                ", mzAppId='" + mzAppId + '\'' +
                ", mzAppKey='" + mzAppKey + '\'' +
                ", oppoAppKey='" + oppoAppKey + '\'' +
                ", oppoAppSecret='" + oppoAppSecret + '\'' +
                ", vivoAppId='" + vivoAppId + '\'' +
                ", vivoAppKey='" + vivoAppKey + '\'' +
                ", enabledPushTypes=" + enabledPushTypes +
                '}';
    }

    public static class Builder {
        private Context context;

        private String fcmSenderId;
        private String hwAppId;
        private String miAppId;
        private String miAppKey;
        private String mzAppId;
        private String mzAppKey;
        private String oppoAppKey;
        private String oppoAppSecret;
        private String vivoAppId;
        private String vivoAppKey;
        private ArrayList<EMPushType> enabledPushTypes = new ArrayList();

        /**
         * \~chinese
         * 构建一个 EMPushConfig 的 Builder 对象。
         * @param context 上下文。
         *
         * \~english
         * Creates an instance of the EMPushConfig.
         * @param context The context of Android Activity or Application.
         */
        public Builder(Context context) {
            this.context = context.getApplicationContext();
        }

        /**
         * \~chinese
         * 构建一个 EMPushConfig 的 Builder 对象。
         * @param context 上下文。
         * @param config  EMPushConfig 对象。
         *
         * \~english
         * Creates an instance of the EMPushConfig.
         * @param context The context of Android Activity or Application.
         * @param config    The instance of EMPushConfig.
         */
        public Builder(Context context, EMPushConfig config) {
            this(context);

            if (config == null) {
                return;
            }

            if (config.enabledPushTypes.contains(EMPushType.MIPUSH)) {
                enableMiPush(config.miAppId, config.miAppKey);
            }

            if (config.enabledPushTypes.contains(EMPushType.HMSPUSH)) {
                enableHWPush();
            }

            if (config.enabledPushTypes.contains(EMPushType.VIVOPUSH)) {
                enableVivoPush();
            }

            if (config.enabledPushTypes.contains(EMPushType.OPPOPUSH)) {
                enableOppoPush(config.oppoAppKey, config.oppoAppSecret);
            }

            if (config.enabledPushTypes.contains(EMPushType.MEIZUPUSH)) {
                enableMeiZuPush(config.mzAppId, config.mzAppKey);
            }

            if (config.enabledPushTypes.contains(EMPushType.FCM)) {
                enableFCM(config.fcmSenderId);
            }
        }

        /**
         * \~chinese
         * 传入小米推送的 App ID 和 App Key，以使小米推送在小米设备上可用。
         * @return 当前 Builder 实例。
         *
         * \~english
         * Passes the app ID and app key of Mi push to enable Mi push on Mi devices.
         * 
         * @return The instance of Builder.
         */
        public Builder enableMiPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.miAppId = appId;
                this.miAppKey = appKey;
                this.enabledPushTypes.add(EMPushType.MIPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MI push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 使华为推送在华为设备上可用。
         * 需要在 AndroidManifest 内配置 App ID 或者配置 agconnect-services.json。
         * @return 当前 Builder 对象。
         *
         * \~english
         * Enables Huawei push on Huawei devices.
         * Be sure to set app ID in AndroidManifest or to set agconnect-services.json.
         * @return  The instance of Builder.
         */
        public Builder enableHWPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                hwAppId = appInfo.metaData.getString("com.huawei.hms.client.appid");

                if(hwAppId != null && hwAppId.contains("=")) {
                    hwAppId = hwAppId.split("=")[1];
                } else {
                    // compatible for user set "value='xxxx'" directly
                    int id = appInfo.metaData.getInt("com.huawei.hms.client.appid");
                    hwAppId = String.valueOf(id);
                }
                this.enabledPushTypes.add(EMPushType.HMSPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                EMLog.e(TAG, "Huawei push must config meta-data: com.huawei.hms.client.appid in AndroidManifest.xml.");
            } catch (ArrayIndexOutOfBoundsException e) {
                EMLog.e(TAG, "Huawei push meta-data: com.huawei.hms.client.appid value must be like this 'appid=xxxxxx'.");
            }
            return this;
        }

        /**
         * \~chinese
         * 传入 FCM 的 Sender ID，以使支持 FCM 推送的设备上可用。
         * @return 当前 Builder 对象。
         *
         * \~english
         * Passes the sender ID of FCM push to enable FCM push on FCM devices.

         * @return The instance of Builder.
         */
        public Builder enableFCM(String senderId) {
            if (!TextUtils.isEmpty(senderId)) {
                this.fcmSenderId = senderId;
                this.enabledPushTypes.add(EMPushType.FCM);
                return this;
            } else {
                EMLog.e(TAG, "senderId can't be empty when enable FCM push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 传入魅族推送的 App ID 和 App Key，以使魅族推送在魅族设备上可用。
         * @return 当前 Builder 对象。
         *
         * \~english
         * Passes the app ID and app key of Meizu push to enable Meizu push on Meizu devices.
         * @return The instance of Builder.
         */
        public Builder enableMeiZuPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.mzAppId = appId;
                this.mzAppKey = appKey;
                this.enabledPushTypes.add(EMPushType.MEIZUPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MEIZU push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 传入 Oppo 推送的 App ID 和 App Key，以使 Oppo 推送在 Oppo 设备上可用。
         * @return  当前 Builder 对象。
         *
         * \~english
         * Passes the app ID and app key of Oppo push to enable Oppo push on Oppo devices.
         * The app ID and app key of Oppo push are passed in to make Oppo push available on Oppo devices.
         * @return  The instance of Builder.
         */
        public Builder enableOppoPush(String appKey, String appSecret) {
            if (!TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(appSecret)) {
                this.oppoAppKey = appKey;
                this.oppoAppSecret = appSecret;
                this.enabledPushTypes.add(EMPushType.OPPOPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appKey or appSecret can't be empty when enable OPPO push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 使 Vivo 推送在 Vivo 设备上可用，需要在 AndroidManifest 中配置 App ID 和 app Key。
         * @return 当前 Builder 对象。
         *
         * \~english
         * Be sure to set the app ID and app key in AndroidManifest in order to make Vivo push available on Vivo devices.
         * @return The instance of Builder.
         */
        public Builder enableVivoPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                vivoAppId = appInfo.metaData.getInt("com.vivo.push.app_id") + "";
                vivoAppKey = appInfo.metaData.getString("com.vivo.push.api_key");
                this.enabledPushTypes.add(EMPushType.VIVOPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                EMLog.e(TAG, "NameNotFoundException: " + e.getMessage());
            }
            return this;
        }

        /**
         * \~chinese
         * 返回 EMPushConfig 对象。
         * @return  EMPushConfig 对象。
         *
         * \~english
         * Returns the instance of EMPushConfig.
         * @return  The EMPushConfig instance.
         */
        public EMPushConfig build() {
            EMPushConfig pushConfig = new EMPushConfig();
            pushConfig.fcmSenderId = this.fcmSenderId;
            pushConfig.hwAppId = this.hwAppId;
            pushConfig.miAppId = this.miAppId;
            pushConfig.miAppKey = this.miAppKey;
            pushConfig.mzAppId = this.mzAppId;
            pushConfig.mzAppKey = this.mzAppKey;
            pushConfig.oppoAppKey = this.oppoAppKey;
            pushConfig.oppoAppSecret = this.oppoAppSecret;
            pushConfig.vivoAppId = this.vivoAppId;
            pushConfig.vivoAppKey = this.vivoAppKey;
            pushConfig.enabledPushTypes = this.enabledPushTypes;
            return pushConfig;
        }
    }
}
