/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.text.TextUtils;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAPushConfigs;
import com.hyphenate.chat.adapter.EMAPushManager;
import com.hyphenate.exceptions.HyphenateException;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * \~chinese
 * 离线消息推送管理类，针对 GCM、小米、华为等离线推送以及 APNS。
 *
 * \~english
 * The message push configuration options.
 */
public class EMPushManager {
    private static final String TAG = EMPushManager.class.getSimpleName();

    /**
     * \~chinese
     * 推送消息展示样式：SimpleBanner 表示展示简单消息，MessageSummary 表示展示消息内容。
     *
     * \~english
     * The push message presentation style: SimpleBanner represents the presentation of a simple message,
     * and MessageSummary represents the presentation of message content.
     */
    public enum DisplayStyle {
        SimpleBanner, MessageSummary
    }

    EMClient mClient;
    EMAPushManager emaObject;
    EMPushManager(EMClient client, EMAPushManager pushManager) {
        emaObject = pushManager;
        mClient = client;
    }


    /**
     * \~chinese
     * 开启离线消息推送。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @throws HyphenateException
     *
     * \~english
     * Turns on the push notification.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @throws HyphenateException
     */
    public void enableOfflinePush() throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.enableOfflineNotification(error);
        handleError(error);
    }

    /**
     * \~chinese
     * 在指定的时间段(24 小时制)内，不推送离线消息。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param start 开始时间。
     * @param end 结束时间。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     *
     * \~english
     * Do not push the offline messages within the specified time period (24-hour clock).

     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param start The start hour.
     * @param end The end hour.
     * @throws HyphenateException A description of the cause of the exception.
     */
    public void disableOfflinePush(int start, int end) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.disableOfflineNotification(start, end, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 从缓存获取推送配置信息。
     * @return 推送配置信息。
     *
     * \~english
     * Get the push configs from cache.
     * @return The push configs.
     */
    public EMPushConfigs getPushConfigs(){
        EMAPushConfigs pushConfigs = emaObject.getPushConfigs();
        if(pushConfigs == null){
            return null;
        }
        return new EMPushConfigs(pushConfigs);
    }

    /**
     * \~chinese
     * 从服务器获取推送配置信息。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @return 推送配置信息。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     *
     * \~english
     * Get the push configs from the server.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return The push configs.
     * @throws HyphenateException A description of the cause of the exception.
     */
    public EMPushConfigs getPushConfigsFromServer() throws HyphenateException{
        EMAError error = new EMAError();
        EMAPushConfigs pushConfigs = emaObject.getPushConfigsFromServer(error);
        handleError(error);

        return new EMPushConfigs(pushConfigs);
    }


    /**
     * \~chinese
     * 设置指定的群组是否接受离线消息推送。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param groupIds 要设置的群组列表。
     * @param noPush - `true`：不接收离线消息推送；
     *               - `false`：接收推送。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     *
     * \~english
     * Sets wether to turn on or turn off the push notification for the the specified groups.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param groupIds The list of groups to be set.
     * @param noPush - `true`: Turns off the notification;
     *               - `false`: Turns on the notification.
     * @throws HyphenateException A description of the cause of the exception.
     */
    public void updatePushServiceForGroup(List<String> groupIds, boolean noPush) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.updatePushServiceForGroup(groupIds, noPush, error);
        handleError(error);
    }
    /**
     * \~chinese
     * 设置指定的用户是否接收离线消息推送。
     * @param userIds 要设置的用户列表。
     * @param noPush - `true`：不接收离线消息推送；
     *               - `false`：接收推送。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     *
     * \~english
     * Sets whether the specified group accepts the offline message notification.
     * @param userIds The list of users to be set.
     * @param noPush - `true`：turn off the notification;
     *               - `false`：turn on the notification.
     * @throws HyphenateException A description of the cause of the exception.
     */
    public void updatePushServiceForUsers(List<String> userIds, boolean noPush) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.updatePushServiceForUsers(userIds, noPush, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 获取关闭了离线消息推送的群组。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @return 群组列表。
     *
     * \~english
     * Gets the list of groups which have blocked the push notification.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return The list of groups that blocked the push notification.
     */
    public List<String> getNoPushGroups(){
        return emaObject.getNoPushGroups();
    }
    /**
     * \~chinese
     * 从缓存中获取关闭了离线消息推送的用户。
     * 
     * 注意：
     * 如果需要获取最新的数据可先调用 {@link EMPushManager#getPushConfigsFromServer()} 后，再调用本方法。
     * @return 关闭了离线消息推送的用户列表。
     *
     * \~english
     * Gets the list of user ID which have blocked the push notification from the cache.
     * 
     * Note:
     * If you needs to get the latest data, call {@Link EmpushManager# getPushConfigsFromServer()} before calling this method.
     * @return  The list of users who have blocked the push notification.
     */
    public List<String> getNoPushUsers(){
        return emaObject.getNoPushUsers();
    }

    /**
	 * \~chinese
	 * 更新当前用户的推送昵称。
     * 离线消息推送的时候可以显示推送昵称而不是用户 ID。
     * 当用户更改昵称（可通过 {@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)} 或者
     * {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)} 修改）时，
     * 务必也调用此方法更新到 Chat 服务器，防止显示差异。
     *
     * 参考：
     * 异步方法见 {@link #asyncUpdatePushNickname(String, EMCallBack)}
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param nickname 推送昵称，需要与用户属性中的昵称区分开。
	 *
	 * \~english
     * Updates the push display nickname of the current user.
	 * This method can be used to set a push display nickname, the push display nickname will be used to show for offline push notification.
     * When the app user changes the nickname in the user profile(use {@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)
     * or {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)} to set},
     * be sure to also call this method to update to prevent the display differences.
     *
     * Reference:
     * The asynchronous method see {@link #asyncUpdatePushNickname(String, EMCallBack)}
	 *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param nickname The push display nickname, which is different from the nickname in the user profile.
	 */
    public boolean updatePushNickname(String nickname) throws IllegalArgumentException, HyphenateException {
        if (TextUtils.isEmpty(nickname)) {
            throw new IllegalArgumentException("nick name is null or empty");
        }
        String currentUser = EMClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = EMClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushNickname(nickname, error);
        handleError(error);
        return true;
    }

    /**
	 * \~chinese
	 * 更新当前用户的推送昵称。
     * 
     * 异步方法。
     * 
     * 离线消息推送的时候可以显示推送昵称而不是用户 ID。
     * 当用户更改昵称（可通过 {@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)} 或者
     * {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)} 修改）时，

     * 务必也调用此方法更新到环信服务器，防止显示差异。
     * 
     * 参考：
     * 同步方法见 {@link #updatePushNickname(String)}。
	 *
	 * @param nickname 推送昵称，需要与用户属性中的昵称区分开。
	 *
	 * \~english
	 * Update the push display nickname of the current user.
     * 
     * This is an asynchronous method.
     * 
     * This method can be used to set a push nickname, the push nickname will be used for offline push notification.
     * When the user changes the nickname in the user profile(use {@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)
     * or {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)} to set),
     * be sure to also call this method to update the display nickname to prevent the display differences.
     *
     * Reference:
     * The synchronous method see {@link #updatePushNickname(String)}.
     *
	 * @param nickname  The push nickname, which is different from the nickname in user profiles.
	 */
    public void asyncUpdatePushNickname(String nickname, EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushNickname(nickname);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
	 * \~chinese
	 * 更新推送消息样式，默认是 {@link DisplayStyle#SimpleBanner}。
     * 
     * 参考：
     * 异步方法见 {@link #asyncUpdatePushDisplayStyle(DisplayStyle, EMCallBack)}。
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param style 推送消息样式。
	 *
	 * \~english
	 * Update the push message style. The default value is {@link DisplayStyle#SimpleBanner}.
     * 
     * Reference:
     * The asynchronous method see {@link #asyncUpdatePushDisplayStyle(DisplayStyle, EMCallBack)}
     *
     * This is a synchronous method and blocks the current thread.
	 *
	 * @param style The push message display style.
	 */
    public void updatePushDisplayStyle(DisplayStyle style) throws IllegalArgumentException, HyphenateException
    {
        String currentUser = EMClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = EMClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushDisplayStyle(style.ordinal(), error);
        handleError(error);
    }

    /**
	 * \~chinese
	 * 更新推送消息样式，默认是 {@link DisplayStyle#SimpleBanner}。
     * 
     * 异步方法。
     * 
     * 参考：
     * 同步方法见 {@link #updatePushDisplayStyle(DisplayStyle)}。
	 *
	 * @param style 推送消息样式。
	 *
	 * \~english
	 * Update the push message style. The default value is {@link DisplayStyle#SimpleBanner}.
     *   
     * This is an asynchronous method.
     * 
     * Reference:
     * The synchronous method see {@link #updatePushDisplayStyle(DisplayStyle)}
	 *
	 * @param style The push message style.
	 */
    public void asyncUpdatePushDisplayStyle(DisplayStyle style, EMCallBack callback)
    {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushDisplayStyle(style);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    private void reportPushAction(String parameters) throws IllegalArgumentException, HyphenateException
    {
        String currentUser = EMClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = EMClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.reportPushAction(parameters, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 上报推送事件。
     *
     * 异步方法。
     *
     * @param taskId 任务ID
     * @param provider 厂商通道名称
     * @param action 推送事件
     *
     * \~english
     * Reports the push events.
     *
     * This is an asynchronous method.
     *
     * @param taskId Task ID
     * @param provider Vendor Channel name
     * @param action Push event
     */
    public void asyncReportPushAction(String taskId, String provider, EMPushAction action, EMCallBack callback)
    {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    JSONObject parameters = new JSONObject();
                    if(!taskId.isEmpty()){
                        JSONObject report = new JSONObject();
                        report.put("task_id", taskId);
                        parameters.put("report", report);
                    }
                    parameters.put("provider", provider);
                    parameters.put("action", action.name);
                    reportPushAction(parameters.toString());
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        });
    }

    private void handleError(EMAError error) throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    /**
     * \~chinese
     * 推送事件枚举类。
     *
     * \~english
     * The push actions.
     */
    public enum EMPushAction{
        ARRIVE("arrive"),/** \~chinese 送达事件。  \~english Arrived event. */
        CLICK("click");/** \~chinese 点击事件。  \~english Clicked event. */

        private String name;

        EMPushAction(String name){
            this.name = name;
        }
    }
}
