/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.chat.EMMessage.ChatType;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAConversation.EMASearchDirection;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * \~chinese
 * 会话类，表示和一个用户/群组/聊天室的对话，包含发送和接收的消息。
 * 
 * 例如，从会话中获取未读消息数：
 * ```java
 *     // conversationId 可以为聊天的另一方 ID，群组 ID 或者聊天室 ID。
 *     EMConversation conversation = EMClient.getInstance().chatManager().getConversation(conversationId);
 *     int unread = conversation.getUnreadMsgCount();
 * ```
 * @version 3.0
 * 
 * \~english
 * The conversation class, which represents a conversation with a user/group/chat room and contains the messages that are sent and received.
 * 
 * The following code shows how to get the number of the unread messages from the conversation.
 * ```java
 *     // ConversationId can be the other party id, the group id, or the chat room id
 *     EMConversation conversation = EMClient.getInstance().chatManager().getConversation(conversationId);
 *     int unread = conversation.getUnreadMsgCount();
 * ```
 * 
*/

public class EMConversation extends EMBase<EMAConversation> {

    private static final String TAG = "conversation";
    private static final int LIST_SIZE = 512;

    EMConversation(EMAConversation conversation) {
        emaObject = conversation;
    }

    /**
     * \~chinese
     * 会话类型。
     *
     * \~english
     * The conversation types.
     */
    public enum EMConversationType {
        /**
         *\~chinese 
         * 单聊会话。
         *
         *\~english
         * One-to-one chat.
         */
        Chat,

        /**
         *\~chinese
         * 群聊会话。
         *
         *\~english
         * Group chat.
         */
        GroupChat,

        /**
         *\~chinese
         *聊天室会话。
         *\~english
         *Chat room.
         */
        ChatRoom,

        /**
         *\~chinese
         * 讨论组会话。暂不支持。
         *\~english
         * Discussion group. Reserved for future use.

         */
        DiscussionGroup,

        /**
         * \~chinese
         * 客服会话。
         *
         * \~english
         * Help desk.
         */
        HelpDesk
    }

    /**
     *  \~chinese
     *  消息搜索方向。
     *
     *  \~english
     *  The message search direction.
     */
    public enum EMSearchDirection {
        /**
         *\~chinese
         *  向上搜索。 
         *
         *\~english
         *  Searches up, means searching from the newer messages to the older messages.
         */
        UP,     

        /**
         *\~chinese
         *  向下搜索。
         *
         *\~english
         *  Searches down, means searching from the older messages to the newer messages.
         */
        DOWN    
    }

    /**
     * \~chinese
     * 会话 ID。
     * 对于单聊类型，会话 ID 同时也是对方用户的名称。
     * 对于群聊类型，会话 ID 同时也是对方群组的 ID，并不同于群组的名称。
     * 对于聊天室类型，会话 ID 同时也是聊天室的 ID，并不同于聊天室的名称。
     * 对于 HelpDesk 类型，会话 ID 与单聊类型相同，是对方用户的名称。
     * @return 会话 ID。
     *
     * \~english
     * The conversation ID.
     * 
     * For one-to-one chat，the conversation ID is the same with the other side's name.
     * For group chat, the conversation ID is the group ID, different with group name.
     * For chat room, the conversation ID is the chatroom ID, different with chat room name.
     * For help desk, it is the same with one-to-one chat, the conversation ID is also the other chat user's name.
     * 
     * @return The conversation ID.
     */
    public String conversationId() {
        return emaObject.conversationId();
    }

    /**
     * \~chinese
     * 获取会话类型。
     * @return 会话类型。
     *  
     * \~english
     * Gets the conversation type.

     * @return  The conversation type.
     */
    public EMConversationType getType() {
        EMAConversationType t = emaObject._getType();
        if (t == EMAConversationType.CHAT) {
            return EMConversationType.Chat;
        }
        if (t == EMAConversationType.GROUPCHAT) {
            return EMConversationType.GroupChat;
        }
        if (t == EMAConversationType.CHATROOM) {
            return EMConversationType.ChatRoom;
        }
        if (t == EMAConversationType.DISCUSSIONGROUP) {
            return EMConversationType.DiscussionGroup;
        }
        if (t == EMAConversationType.HELPDESK) {
            return EMConversationType.HelpDesk;
        }
        return EMConversationType.Chat;
    }

    /**
     * \~chinese
     * 获取此对话中未读的消息数量。
     * @return 未读的消息数量。
     *
     * \~english
     * Gets the number of unread messages of the conversation.
     * @return The unread message count of the conversation.
     */
    public int getUnreadMsgCount() {
        return emaObject.unreadMessagesCount();
    }

    /**
     *  \~chinese
     *  将所有未读消息设置为已读。
     *
     *  \~english
     *  Marks all messages as read.
     */
    public void markAllMessagesAsRead() {
        emaObject.markAllMessagesAsRead(true);
    }

    /**
     * \~chinese
     * 获取 SDK 本地数据库中会话的全部消息数目。
     * @return 会话的全部消息数量。
     *
     * \~english
     * Gets all messages count in this conversation in the local database.
     * @return The count of all the messages in this conversation.
     */
    public int getAllMsgCount() {
        return emaObject.messagesCount();
    }

    /**
     * \~chinese
     * 从 SDK 本地数据库中加载更多消息。
     * 根据传入的参数从本地数据库加载 startMsgId 之前(存储顺序)指定数量的消息，
     * 加载到的 messages 会加入到当前会话的缓存中，通过 {@link #getAllMessages()} 将会返回所有加载的消息。
     *
     * @param startMsgId    加载这个 ID 之前的 message，如果传入""或者 null，将从最近的消息开始加载。
     * @param pageSize      一页加载的条数。
     * @return              消息列表。
     *
     * \~english
     * Loads more messages from the local database.
     * Loads messages from the local database before the specified message.
     * The messages will also be stored in to current conversation's memory cache.
     * So when next time calling {@link #getAllMessages()}, the result will contain those messages.
     *
     * @param startMsgId    The specified message ID. If the `startMsgId` is set as "" or null, the SDK will load latest messages in database.
     * @param pageSize      The number of records in a page.
     * @return              The message list.
     */
    public List<EMMessage> loadMoreMsgFromDB(String startMsgId, int pageSize) {
        return loadMoreMsgFromDB(startMsgId, pageSize, EMSearchDirection.UP);
    }

    /**
     * \~chinese
     * 根据传入的参数从 SDK 本地数据库加载 startMsgId 之前或之后(存储顺序)指定数量的 message，
     * 加载到的 messages 会加入到当前 conversation 的 messages 里。
     *
     * @param startMsgId    加载这个 ID 之前的 message，即作为起始点的 message。
     * @param pageSize      每页加载多少条。
     * @param direction     消息加载的方向。
     * @return              消息列表。
     *
     * \~english
     * Loads messages starting from the specified message id from the local database. 
     * The messages will also be stored in to the memory cache.
     * So the next time calling getAllMessages(), the result will contain those messages.
     *
     * @param startMsgId    The specified message ID. If the `startMsgId` is set as "" or null, the SDK will load latest messages in database.
     * @param pageSize      The number of records in a page.
     * @param direction     The direction in which the message is loaded: EMMessageSearchDirection. 
                            - `EMMessageSearchDirectionUp`: get aCount of messages before the timestamp of the specified message ID; 
                            - `EMMessageSearchDirectionDown`: get aCount of messages after the timestamp of the specified message ID.
     * @return              The message list.
     */
    public List<EMMessage> loadMoreMsgFromDB(String startMsgId, int pageSize, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;
        List<EMAMessage> msgs = emaObject.loadMoreMessages(startMsgId, pageSize, d);
        List<EMMessage> result = new ArrayList<EMMessage>();
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        getCache().addMessages(result);
        return result;
    }

    /**
     * \~chinese
     * 根据搜索消息时间点、一次搜索结果的最大条数和搜索方向从 SDK 本地数据库中搜索指定数量的消息。
     * 
     * 注意：当 maxCount 非常大时，需要考虑内存消耗。
     *
     * @param timeStamp  搜索消息的时间点。
     * @param maxCount   一次搜索结果的最大条数。
     * @param direction     消息加载的方向。
     * @return           消息列表。
     *
     * \~english
     * Searches messages from the local database according the following parameters.
     * 
     * Note: Be cautious about the memory usage when the maxCount is large.
     *
     * @param timeStamp  The Unix timestamp when searching.
     * @param maxCount   The max number of message to search.
     * @return           The message list.
     * @param direction     The direction in which the message is loaded: EMMessageSearchDirection. 
     *                      - `EMMessageSearchDirectionUp`: get aCount of messages before the timestamp of the specified message ID; 
     *                      - `EMMessageSearchDirectionDown`: get aCount of messages after the timestamp of the specified message ID.
     */
    public List<EMMessage> searchMsgFromDB(long timeStamp, int maxCount, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(timeStamp, maxCount, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据消息类型、搜索消息的时间点、搜索结果的最大条数、搜索来源和搜索方向从 SDK 本地数据库中搜索指定数量的消息。
     * 
     * 注意：当 maxCount 非常大时，需要考虑内存消耗。
     *
     * @param type       消息类型，文本、图片、语音等等。
     * @param timeStamp  搜索消息的时间点。
     * @param maxCount   搜索结果的最大条数。
     * @param from       搜索来自某人的消息，也可用于搜索群组里的消息。
     * @param direction  消息加载的方向。
     * @return           消息列表。
     *
     * \~english
     * Searches messages from the local database according the following parameters: the message type, the Unix timestamp, maxcount, sender.
     * 
     * Note: 
     * Be cautious about the memory usage when the maxCount is large.
     *
     * @param type       The message type, including TXT、VOICE、IMAGE and so on.
     * @param timeStamp  The Unix timestamp for search.
     * @param maxCount   The max number of message to search.
     * @param from       The sender of the message. The param can also be used to search in group chat.
     * @param direction     The direction in which the message is loaded: EMMessageSearchDirection. 
     *                      - `EMMessageSearchDirectionUp`: get aCount of messages before the timestamp of the specified message ID; 
     *                      - `EMMessageSearchDirectionDown`: get aCount of messages after the timestamp of the specified message ID.
     * 
     * @return           The message list.
     */
    public List<EMMessage> searchMsgFromDB(EMMessage.Type type, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据消息中的关键词、搜索消息的时间点、搜索结果的最大条数、搜索来源和搜索方向从 SDK 本地数据库中搜索指定数量的消息。
     * 注意：当 maxCount 非常大时，需要考虑内存消耗。
     *
     * @param keywords   搜索消息中的关键词。
     * @param timeStamp  搜索消息的时间点。
     * @param maxCount   搜索结果的最大条数。
     * @param from       搜索来自某人的消息，也可用于搜索群组里的消息。
     * @param direction  消息加载的方向。
     * @return           消息列表。
     *
     * \~english
     * Searches messages from the local database by the following parameters: keywords, timestamp, maxcount, sender, search direction.
     * 
     * Note: Be cautious about memory usage when the maxCount is large.
     *
     * @param keywords   The keywords in message.
     * @param timeStamp  The timestamp for search.
     * @param maxCount   The max number of message to search.
     * @param from       The message sender. The param can also be used to search in group chat.
     * @param direction     The direction in which the message is loaded: EMMessageSearchDirection. 
     *                      - `EMMessageSearchDirectionUp`: get aCount of messages before the timestamp of the specified message ID; 
     *                      - `EMMessageSearchDirectionDown`: get aCount of messages after the timestamp of the specified message ID.
     *
     * @return           The list of searched messages.
     */
    public List<EMMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 从 SDK 本地数据库中搜索指定时间段内指定数量的消息。
     * 注意：当 maxCount 非常大时，需要考虑内存消耗。
     *
     * @param startTimeStamp   搜索的起始时间。
     * @param endTimeStamp     搜索的结束时间。
     * @param maxCount         搜索结果的最大条数。
     * @return                 消息列表。
     *
     * \~english
     * Searches specifying messages from the local database by the following parameters.
     * Note: Be cautious about memory usage when the maxCount is large.
     *
     * @param startTimeStamp    The start Unix timestamp to search.
     * @param endTimeStamp      The end Unix timestamp to search.
     * @param maxCount          The max number of message to search.
     * @return                  The message list.
     */
    public List<EMMessage> searchMsgFromDB(long startTimeStamp, long endTimeStamp, int maxCount) {
        List<EMAMessage> msgs = emaObject.searchMessages(startTimeStamp, endTimeStamp, maxCount);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据搜索消息中的关键词、搜索消息的时间点、搜索结果的最大条数、搜索来源和搜索方向从 SDK 本地数据库中搜索自定义消息中指定数量的消息。
     * 
     * 注意：当 maxCount 非常大时，需要考虑内存消耗。
     *
     * @param keywords   搜索消息中的关键词。
     * @param timeStamp  搜索消息的时间点。
     * @param maxCount   搜索结果的最大条数。
     * @param from       搜索来自某人的消息，也可用于搜索群组里的消息。
     * @param direction  消息加载的方向。
     * @return           消息列表。
     *
     * \~english
     * Searches specifying custom messages from the local database based the parameters.
     * 
     * Note: Be cautious about memory usage when the maxCount is large.
     *
     * @param keywords   The keywords in message.
     * @param timeStamp  The Unix timestamp to search.
     * @param maxCount   The max number of message to search.
     * @param from       The message sender, the param can also be used to search in group chat.
     * @param direction     The direction in which the message is loaded: EMMessageSearchDirection. 
     *                      - `EMMessageSearchDirectionUp`: get aCount of messages before the timestamp of the specified message ID; 
     *                      - `EMMessageSearchDirectionDown`: get aCount of messages after the timestamp of the specified message ID.
     *
     * @return           The message list.
     */
    public List<EMMessage> searchCustomMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchCustomMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据 messageId 获取已读的消息。
     * 优先从内存中加载，如果内存中没有则从数据库中加载，并将其插入到内存中。
     * 
     * @param  messageId    消息 ID。
     * @param  markAsRead   是否获取消息的同时标记消息为已读，如果标记为已读，则会发送已读标记到服务器。
     * @return              获取到的消息实例。
     *
     * \~english
     * Gets the message with message ID.
     * 
     * If the message already loaded into the memory cache, the message will be directly returned,
     * otherwise the message will be loaded from the local database, and be set into the cache.
     *
     * @param  messageId     The message ID.
     * @param  markAsRead    Whether to mark the message as read while getting it.
     *                       - `true`: The SDK will mark the message as read and send a read receipt to the server;
     *                       - `false`: Do not mark the message as read.
     * @return message       Returns the message instance.
     */
    public EMMessage getMessage(String messageId, boolean markAsRead) {
        EMMessage msg = getCache().getMessage(messageId);
        if (msg == null) {
            EMAMessage emaMsg = emaObject.loadMessage(messageId);
            if (emaMsg == null) {
                return null;
            }
            msg = new EMMessage(emaMsg);
        }
        emaObject.markMessageAsRead(messageId, markAsRead);
        return msg;
    }

    /**
     * \~chinese
     * 加载一组消息。
     * 如果缓存不存在会去本地数据库查询并加载。
     *
     * @param msgIds    一组消息 ID。
     * @return          返回一组消息，如果找不到返回 null。
     *
     * \~english
     * Loads a list of messages.
     * If those messages don't exists in the memory cache, the memory cache will load message from the local database.
     *
     * @param  msgIds   The message IDs to be loaded.
     * @return          Returns the message list, if the messages are not found, the SDK will return null.
     */
    @Deprecated
    public List<EMMessage> loadMessages(List<String> msgIds) {

        List<EMMessage> msgs = new ArrayList<EMMessage>();

        for (String msgId : msgIds) {
            EMAMessage msg = emaObject.loadMessage(msgId);
            if (msg != null) {
                msgs.add(new EMMessage(msg));
            }
        }
        getCache().addMessages(msgs);
        return msgs;
    }

    /**
     * \~chinese
     * 设置指定的消息为已读。
     * 
     * 将消息置为已读，也可用 {@link EMMessage#setUnread(boolean)}。
     *
     * @param messageId 消息 ID。 
     *
     * \~english
     * Marks a message as read.
     * 
     * To mark the message read, you can also call {@link EMMessage#setUnread(boolean)}.
     * 
     * @param messageId The message ID.
     */
    public void markMessageAsRead(String messageId) {
        emaObject.markMessageAsRead(messageId, true);
    }

    /**
     * \~chinese
     * 获取此 conversation 当前内存所有的 message。
     * 如果内存中为空，再从本地数据库中加载最近的一条消息。
     *
     * @return 消息列表。
     *
     * \~english
     * Gets all messages in the local cache.
     * 
     * If no message is found in cache, then the SDK will load the latest message from the local database.
     *
     * @return The message list.
     */
    public List<EMMessage> getAllMessages() {
        if (getCache().isEmpty()) {
            EMAMessage lastMsg = emaObject.latestMessage();
            List<EMMessage> msgs = new ArrayList<EMMessage>();
            if (lastMsg != null) {
                msgs.add(new EMMessage(lastMsg));
            }
            getCache().addMessages(msgs);
        }
        return getCache().getAllMessages();
    }

    /**
     * \~chinese
     * 删除一条指定的消息。
     * 注：只会对本地数据进行操作。
     *
     * @param messageId     待删除消息的 ID。
     *
     * \~english
     * Deletes a message in the local database.
     * 
     * Note: Operates only on the local database.
     *
     * @param messageId    The message to be deleted.
     */
    public void removeMessage(String messageId) {
        EMLog.d(TAG, "remove msg from conversation: " + messageId);
        emaObject._removeMessage(messageId);
        getCache().removeMessage(messageId);
    }

    /**
     * \~chinese
     * 获取消息队列中的最后一条消息。
     * 此操作不会改变未读消息计数。
     * 优先从缓存中获取，如果缓存中没有，则从数据库中加载最近的一条消息。
     *
     * @return 消息。
     *
     * \~english
     * Gets the last message from the conversation.
     * 
     * The operation does not change the unread message count.
     * 
     * Gets from the cache first, if no message is found, loads from the local database and then put it in the cache.
     *
     * @return  The message instance.
     */
    public EMMessage getLastMessage() {
        if (getCache().isEmpty()) {
            EMAMessage _msg = emaObject.latestMessage();
            EMMessage msg = _msg == null ? null : new EMMessage(_msg);
            getCache().addMessage(msg);
            return msg;
        } else {
            return getCache().getLastMessage();
        }
    }

    /**
     * \~chinese
     * 获取会话接收到的最后一条消息。
     * @return 消息。
     *
     * \~english
     * Gets the latest message from the conversation.
     * @return  The message instance.
     */
    public EMMessage getLatestMessageFromOthers() {
        EMAMessage _msg = emaObject.latestMessageFromOthers();
        EMMessage msg = _msg == null ? null : new EMMessage(_msg);
        getCache().addMessage(msg);
        return msg;
    }

    /**
     * \~chinese
     * 清除对话中的所有消息。
     * 
     * 只清除内存的，不清除本地数据库的消息。
     * 在退出会话的时候清除内存缓存，减小内存消耗。
     *
     * \~english
     * Deletes all the messages in this conversation.
     * 
     * NOTE:
     * The call only deletes all the messages in the memory cache, but not the messages in the local database.
     * 
     * You can reduce the memory consumption by clearing the memory cache when exiting a conversation.
     */
    public void clear() {
        getCache().clear();
    }

    /**
     * \~chinese
     * 删除会话中所有消息，同时清除内存和数据库中的消息。
     *
     * \~english
     * Deletes all the messages of the conversation from the memory cache and local database.
     */
    public void clearAllMessages() {
        emaObject.clearAllMessages();
        getCache().clear();
    }

    /**
     * \~chinese
     * 设置会话扩展字段。
     * 
     * 该字段只保存在本地，不进行网络同步。
     *
     * @param ext       会话对应扩展字段的内容。
     *
     * \~english
     * Sets the extension of the conversation.
     * 
     * The extend field only stored in local database, not updated to the server.
     *
     * @param ext       The extension string.
     */
    public void setExtField(String ext) {
        emaObject._setExtField(ext);
    }

    /**
     * \~chinese
     * 获取会话扩展字段。
     * 
     * 该字段只保存在本地，不进行网络同步。
     *
     * @return      会话对应扩展字段的内容。
     *
     * \~english
     * Gets the extension of the conversation.
     * 
     * The extend field only stored in the local database, not updated to the server.
     *
     * @return The extension content.
     */
    public String getExtField() {
        return emaObject.extField();
    }

    /**
     * \~chinese
     * 从消息类型到会话类型的转化。
     *
     * @param       id   消息 ID，用来区分客服和单聊，对于其他类型，这个参数没有影响。
     * @param       type 消息类型。
     * @return      会话类型。
     *
     * \~english
     * Provides a transformation from a message type to a conversation type.
     *
     * @param       id    The message ID, used to distinguish one-to-one chat and the help desk chat, has no effect on other chat type.
     * @param       type  The message type.
     * @return      The conversation type.
     */
    public static EMConversationType msgType2ConversationType(String id, EMMessage.ChatType type) {
        if (type == ChatType.Chat) {
//			if(EMCustomerService.getInstance().isCustomServiceAgent(id)){
//				return EMConversationType.HelpDesk;
//			}
            return EMConversationType.Chat;
        } else if (type == ChatType.GroupChat) {
            return EMConversationType.GroupChat;
        } else if (type == ChatType.ChatRoom) {
            return EMConversationType.ChatRoom;
        }

        return EMConversationType.Chat;
    }

    /**
     * \~chinese
     * 获取是否是群组或者聊天室会话。
     *
     * @return 群组和聊天室类型都会返回 `true`，其他类型返回 `false`。
     *
     * \~english
     * To check whether it is a group or chat room conversation.
     *
     * @return  If the conversation type is group or chatroom chat, the SDK returns `true`. Otherwise, the SDK returns `false`.
     */
    public boolean isGroup() {
        EMConversationType type = getType();
        return type == EMConversationType.GroupChat ||
                type == EMConversationType.ChatRoom;
    }

    /**
     *  \~chinese
     *  插入一条消息在 SDK 本地数据库，消息的 conversation ID 应该和会话的 conversation ID 一致，消息会根据消息里的时间戳被插入 SDK 本地数据库，并且更新会话的 latestMessage 等属性。
     *
     *  @param msg 消息实例。
     *
     *  \~english
     *  Inserts a message to a conversation in local database and SDK will update the last message automatically.
     * 
     *  The conversation ID of the message should be the same as conversation ID of the conversation in order to insert the message into the conversation correctly. The inserting message will be inserted based on timestamp.
     * 
     *  @param msg  The message instance.
     */
    public boolean insertMessage(EMMessage msg) {
        boolean result = emaObject.insertMessage(msg.emaObject);
        if(result) {
            getCache().addMessage(msg);
        }
        return result;
    }

    /**
     *  \~chinese
     *  插入一条消息到会话尾部。
     * 
     *  消息的 conversationId 应该和会话的 conversationId 一致，消息会被插入 SDK 本地数据库，并且更新会话的 latestMessage 等属性。
     *
     *  @param msg 消息实例。
     *
     *  \~english
     *  Inserts a message to the end of a conversation in the local database.
     * 
     *  The `conversationId` of the message should be the same as the `conversationId` of the conversation in order to insert the message into the conversation correctly. And the `latestMessage` and other properties of the session should be updated.
     *
     *  @param msg The message instance.
     */
	public boolean appendMessage(EMMessage msg) {
	    boolean result = emaObject.appendMessage(msg.emaObject);
	    if(result) {
            getCache().addMessage(msg);
	    }
	    return result;
	}

    /**
     *  \~chinese
     *  更新 SDK 本地数据库的消息。
     *  不能更新消息 ID，消息更新后，会话的 latestMessage 等属性进行相应更新。
     *
     *  @param msg 要更新的消息。
     *
     *  \~english
     *  Uses this method to update a message in local database. Changing properties will affect data in database.
     *  
     *  The latestMessage of the conversation and other properties will be updated accordingly. The messageID of the message cannot be updated.
     *
     *  @param msg  The message to be updated.
     */
    public boolean updateMessage(EMMessage msg) {
        boolean updateMessage = emaObject.updateMessage(msg.emaObject);
        if(updateMessage) {
            getCache().addMessage(msg);
        }
        return updateMessage;
    }

    /**
     * \~chinese
     * 返回会话对应的附件在 sdcard 中的存储路径。
     * 
     * 该方法适用于清理该会话磁盘存储，不确保该路径一定存在，请在删除对应路径前加以判断，并加上异常保护。
     * 
     * @return Sdcard 中的附件存储路径。
     *
     * \~english
     * Returns the attachment path associated with the conversation.
     * 
     * The path can be used to erase the conversation related files from the local storage.
     * 
     * Not ensure the path exists, please handle IOException when deleting directory.
     *
     * @return The attachment path in the sdcard.
     */
    public String getMessageAttachmentPath() {
        String downloadPath = EMClient.getInstance().getChatConfigPrivate().getDownloadPath();
        return downloadPath + "/" + EMClient.getInstance().getCurrentUser()
                + "/" + conversationId();
    }

    /**
     * 单向删除漫游消息（根据msgId删除）
     * @param msgIdList 		msgId列表
     * @param callBack			处理结果回调
     */
    public void removeMessagesFromServer(List<String> msgIdList, EMCallBack callBack){
        EMClient.getInstance().chatManager().removeMessagesFromServer(conversationId(),getType(),msgIdList,callBack);
    }

    /**
     * 单向删除漫游消息（根据时间节点删除 只支持按开始时间 向上删除）
     * @param beforeTimeStamp	 开始时间 单位毫秒
     * @param callBack			 处理结果回调
     */
    public void removeMessagesFromServer(long beforeTimeStamp, EMCallBack callBack){
        EMClient.getInstance().chatManager().removeMessagesFromServer(conversationId(),getType(),beforeTimeStamp,callBack);
    }



    // ====================================== Message cache ======================================

    MessageCache getCache() {
        MessageCache cache;
        synchronized (EMClient.getInstance().chatManager().caches) {
            cache = EMClient.getInstance().chatManager().caches.get(emaObject.conversationId());
            if (cache == null) {
                cache = new MessageCache();
            }
            EMClient.getInstance().chatManager().caches.put(emaObject.conversationId(), cache);
        }
        return cache;
    }

    static class MessageCache {

        TreeMap<Long, Object> sortedMessages = new TreeMap<Long, Object>(new MessageComparator());
        Map<String, EMMessage> messages = new HashMap<String, EMMessage>();
        Map<String, Long> idTimeMap = new HashMap<String, Long>();
        boolean hasDuplicateTime = false;

        final boolean sortByServerTime = EMClient.getInstance().getChatConfigPrivate().getOptions().isSortMessageByServerTime();

        class MessageComparator implements Comparator<Long> {

            @Override
            public int compare(Long time0, Long time1) {
                long val = time0 - time1;
                if (val > 0) {
                    return 1;
                } else if (val == 0) {
                    return 0;
                } else {
                    return -1;
                }
            }
        }

        public synchronized EMMessage getMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return null;
            }
            return messages.get(msgId);
        }

        public synchronized void addMessages(List<EMMessage> msgs) {
            for (EMMessage msg : msgs) {
                addMessage(msg);
            }
        }

        public synchronized void addMessage(EMMessage msg) {
            if (msg == null || msg.emaObject == null || msg.getMsgTime() == 0 || msg.getMsgTime() == -1 || msg.getMsgId() == null
                    || msg.getMsgId().isEmpty() || msg.getType() == Type.CMD) {
                return;
            }
            String id = msg.getMsgId();
            // override message
            if (messages.containsKey(id)) {
                long time = idTimeMap.get(id);
                sortedMessages.remove(time);
                messages.remove(id);
                idTimeMap.remove(id);
            }
            // messages share same time stamp
            long time = sortByServerTime ? msg.getMsgTime() : msg.localTime();
            if (sortedMessages.containsKey(time)) {
                hasDuplicateTime = true;
                Object v = sortedMessages.get(time);
                if (v != null) {
                    if (v instanceof EMMessage) {
                        List<EMMessage> msgs = new LinkedList<>();
                        msgs.add((EMMessage) v);
                        msgs.add(msg);
                        sortedMessages.put(time, msgs);
                    } else if (v instanceof List) {
                        List<EMMessage> msgs = (List<EMMessage>) v;
                        msgs.add(msg);
                    }
                }
            } else {
                sortedMessages.put(time, msg);
            }
            messages.put(id, msg);
            idTimeMap.put(id, time);
        }

        public synchronized void removeMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return;
            }
            EMMessage msg = messages.get(msgId);
            if (msg != null) {
                Long time = idTimeMap.get(msgId);
                if (time != null) {
                    if (hasDuplicateTime && sortedMessages.containsKey(time)) {
                        Object v = sortedMessages.get(time);
                        if (v != null && v instanceof List) {
                            List<EMMessage> msgs = (List)v;
                            for (EMMessage m : msgs) {
                                if (m != null && m.getMsgId() != null && m.getMsgId().equals(msgId)) {
                                    msgs.remove(m);
                                    break;
                                }
                            }
                        } else {
                            sortedMessages.remove(time);
                        }
                    } else {
                        sortedMessages.remove(time);
                    }
                    idTimeMap.remove(msgId);
                }
                messages.remove(msgId);
            }
        }

        public synchronized List<EMMessage> getAllMessages() {
            List<EMMessage> list = new ArrayList<EMMessage>();
            if (hasDuplicateTime == false) {
                for (Object v : sortedMessages.values()) {
                    list.add((EMMessage)v);
                }
            } else {
                for (Object v : sortedMessages.values()) {
                    if (v != null) {
                        if (v instanceof List) {
                            list.addAll((List<EMMessage>)v);
                        } else {
                            list.add((EMMessage)v);
                        }
                    }
                }
            }
            return list;
        }

        public synchronized EMMessage getLastMessage() {
            if (sortedMessages.isEmpty()) {
                return null;
            }

            Object o = sortedMessages.lastEntry().getValue();
            if (o == null) {
                return null;
            }
            if (o instanceof EMMessage) {
                return (EMMessage)o;
            } else if (o instanceof List){
                List<EMMessage> msgs = (List<EMMessage>)o;
                if (msgs.size() > 0) {
                    return msgs.get(msgs.size() - 1);
                }
                return null;
            }
            return null;
        }

        public synchronized void clear() {
            //no need to keep the last message
            sortedMessages.clear();
            messages.clear();
            idTimeMap.clear();
        }

        public synchronized boolean isEmpty() {
            return sortedMessages.isEmpty();
        }
    }
}
