/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

import com.hyphenate.chat.EMGroupReadAck;
import com.hyphenate.chat.EMMessage;

import java.util.List;

/**
 * \~chinese
 * 消息事件监听器。
 * 用于监听消息接收情况，消息成功发送到对方手机后会有回执（需开启送达回执，详见 {@link com.hyphenate.chat.EMOptions#setRequireDeliveryAck(boolean)}）。
 * 对方阅读了这条消息也会收到回执（需开启允许已读回执，详见 {@link com.hyphenate.chat.EMOptions#setRequireAck(boolean)}）。
 * 发送消息过程中，消息 ID 会从最初本地生成的 uuid 变更为服务器端生成的全局唯一 ID，该 ID 在使用 SDK 的所有设备上均唯一。
 * 应用需实现此接口监听消息变更状态。
 * 
 *     添加消息监听：
 *     ```java
 *     EMClient.getInstance().chatManager().addMessageListener(mMessageListener);
 *     ```
 *
 *     移除消息监听：
 *     ```java
 *     EMClient.getInstance().chatManager().removeMessageListener(mMessageListener);
 *     ```
 * 
 *
 * \~english
 * The message event listener.
 * This listener is used to check whether messages are received. If messages are sent successfully, a delivery receipt will be returned (delivery receipt needs to be enabled: {@link com.hyphenate.chat.EMOptions#setRequireDeliveryAck(boolean)}.
 * If the peer reads the received message, a read receipt will be returned (read receipt needs to be enabled: {@link com.hyphenate.chat.EMOptions#setRequireAck(boolean)})
 * During message delivery, the message ID will be changed from a local uuid to a global unique ID that is generated by the server to uniquely identify a message on all devices using the SDK.
 * This API should be implemented in the app to listen for message status changes.
 * 
 *     Adds the message listener:
 *     ```java
 *     EMClient.getInstance().chatManager().addMessageListener(mMessageListener);
 *     ```
 *
 *     Removes the message listener:
 *     ```java
 *     EMClient.getInstance().chatManager().removeMessageListener(mMessageListener);
 *     ```
 * 
 */
public interface EMMessageListener
{
	/**
     * \~chinese
	 * 收到消息。
	 * 在收到文本、图片、视频、语音、地理位置和文件等消息时，通过此回调通知用户。
     *
     * \~english
	 * Occurs when a message is received.
	 * This callback is triggered to notify the user when a message such as texts or an image, video, voice, location, or file is received.
	 */
	void onMessageReceived(List<EMMessage> messages);

	/**
     * \~chinese
	 * 收到命令消息。
	 * 与 {@link #onMessageReceived(List)} 不同, 这个回调只包含命令的消息，命令消息通常不对用户展示。
	 *
     * \~english
	 * Occurs when a command message is received.
	 * Unlike {@link EMMessageListener#onMessageReceived(List)}, this callback only contains a command message body that is usually invisible to users.
	 *
	 */
	default void onCmdMessageReceived(List<EMMessage> messages) {
	}

    /**
     * \~chinese
     * 收到消息的已读回执。
     *
     * \~english
     * Occurs when a read receipt is received for a message. 
     */
	default void onMessageRead(List<EMMessage> messages) {}

	/**
	 * \~chinese
	 * 收到群组消息的已读回执。
	 *
	 * \~english
	 * Occurs when a read receipt is received for a group message.
	 */
	default void onGroupMessageRead(List<EMGroupReadAck> groupReadAcks) {}

	/**
	 * \~chinese
	 * 收到群组消息的读取状态更新。
	 *
	 * \~english
	 * Occurs when the update for the group message read status is received.
	 */
	default void onReadAckForGroupMessageUpdated() {}

    /**
     * \~chinese
     * 收到消息的送达回执。
     *
     * \~english
     * Occurs when a delivery receipt is received.
     */
	default void onMessageDelivered(List<EMMessage> messages){}

	/**
 	 * \~chinese
 	 * 撤回收到的消息。
 	 *
 	 * \~english
 	 * Occurs when a received message is recalled.
 	 */
	default void onMessageRecalled(List<EMMessage> messages){}
    
	/**
     * \~chinese
     * 收到消息变化通知，包括消息 ID 的变化。
     * @param message	修改后的消息。
     * @param change    消息的变化。
	 * @deprecated 已废弃，请用 {@link EMMessage#setMessageStatusCallback(EMCallBack)} 代替。
     *
     * \~english
     * Occurs when a message change notification is received, including the message ID change.
     *
     * @param message   The changed message.
     * @param change    The message change.
	 * @deprecated Please use {@link EMMessage#setMessageStatusCallback(EMCallBack)} instead.
     */
	@Deprecated
	default void onMessageChanged(EMMessage message, Object change) {}
}
