package com.hyphenate.push;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.support.annotation.NonNull;
import android.util.Pair;

import com.hyphenate.EMError;
import com.hyphenate.chat.EMClient;
import com.hyphenate.chat.core.EMPreferenceUtils;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.push.common.PushUtil;
import com.hyphenate.push.platform.IPush;
import com.hyphenate.push.platform.fcm.EMFCMPush;
import com.hyphenate.push.platform.hms.EMHMSPush;
import com.hyphenate.push.platform.meizu.EMMzPush;
import com.hyphenate.push.platform.mi.EMMiPush;
import com.hyphenate.push.platform.normal.EMNormalPush;
import com.hyphenate.push.platform.oppo.EMOppoPush;
import com.hyphenate.push.platform.vivo.EMVivoPush;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;

import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.Random;

/**
 * \~chinese
 * 推送帮助类
 *
 * \~english
 * Push help class
 */
public class EMPushHelper {
    private static final String TAG = "EMPushHelper";

    private static final int TIMES_RETRY = 3;
    private static final int MSG_WHAT_REGISTER = 0;
    private static final int MSG_WHAT_UNREGISTER = 1;

    private Context context;
    private EMPushConfig pushConfig;
    private IPush pushClient;
    private Handler handler;

    private EMPushType pushType;
    private String pushToken;
    private boolean unregisterSuccess;

    private final Object bindLock = new Object();
    private final Object unbindLock = new Object();

    private PushListener pushListener;

    private boolean isClientInit = false;

    private EMPushHelper() {
        final HandlerThread handlerThread = new HandlerThread("token-uploader");
        handlerThread.start();

        handler = new Handler(handlerThread.getLooper()) {
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what) {
                    case MSG_WHAT_REGISTER:
                        synchronized (bindLock) { // 确保task在执行时没有新的tasks加入，避免该task执行完后把新加入的tasks清空。
                            String token = (String) msg.obj;
                            boolean registerSuccess = uploadTokenInternal(pushClient.getNotifierName(), token);
                            if (registerSuccess) {
                                removeMessages(MSG_WHAT_REGISTER);
                                return;
                            }
                            // 当所有task执行完且还没有上传成功时，更换为NORMAL push type。
                            boolean allTasksExecuted = !hasMessages(MSG_WHAT_REGISTER);
                            if (allTasksExecuted) {
                                EMPushHelper.this.onErrorResponse(pushType, EMError.PUSH_BIND_FAILED);
                                register(EMPushType.NORMAL);
                            }
                        }
                        break;
                    case MSG_WHAT_UNREGISTER:
                        unregisterSuccess = uploadTokenInternal(pushClient.getNotifierName(), "");
                        if (!unregisterSuccess) {
                            EMPushHelper.this.onErrorResponse(pushType, EMError.PUSH_UNBIND_FAILED);
                        }

                        synchronized (unbindLock) {
                            unbindLock.notifyAll();
                        }
                        break;
                    default:
                        super.handleMessage(msg);
                        break;
                }
            }
        };
    }

    /**
     * \~chinese
     * 获取EMPushHelper的单例
     * @return EMPushHelper的单例
     *
     * \~chinese
     * Gets a singleton of EMPushHelper
     * @return Singleton of EMPushHelper
     */
    public static EMPushHelper getInstance() {
        return InstanceHolder.INSTANCE;
    }

    /**
     * \~chinese
     * 设置推送监听
     * 可重写{@link PushListener#isSupportPush(EMPushType, EMPushConfig)}，设置支持的推送类型，
     * 但不能多余SDK支持的推送类型
     * @param callback  自定义推送监听
     *
     * \~english
     * Set custom push listener
     * You can override {@link PushListener#isSupportPush(EMPushType, EMPushConfig)} to set the push type supported,
     * but not more than the push type supported by the SDK
     * @param callback  Custom push listener
     */
    public void setPushListener(PushListener callback) {
        this.pushListener = callback;
    }

    /**
     * \~chinese
     * 初始化EMPushHelper
     * @param context   上下文
     * @param config    推送配置
     *
     * \~english
     * Initialize EMPushHelper
     * @param context
     * @param config    Push configs
     */
    public void init(Context context, EMPushConfig config) {
        EMLog.e(TAG, TAG + " init, config: " + config.toString());

        if (context == null || config == null) {
            throw new IllegalArgumentException("Null parameters, context=" + context + ", config=" + config);
        }

        this.context = context.getApplicationContext();
        this.pushConfig = config;
    }

    /**
     * \~chinese
     * 注册推送
     * 登录成功后，由SDK调用，用户无需调用
     *
     * \~english
     * Register push
     * After successful login, the SDK calls, and the user does not need to call
     */
    public void register() {
        if (context == null || pushConfig == null) {
            EMLog.e(TAG, "EMPushHelper#init(Context, EMPushConfig) method not call previously.");
            return;
        }

        EMPushType pushType = getPreferPushType(pushConfig);
        EMLog.e(TAG, TAG + " register, prefer push type: " + pushType);
        register(pushType);
    }

    /**
     * \~chinese
     * 解注册推送
     * 调用{@link EMClient#logout(boolean)}时，由SDK调用
     * @param unbindToken  是否解绑设备
     * @return  解注册是否成功
     *
     * \~english
     * Unregister push
     * When called {@link EMClient#logout(boolean)}}, invoked by  SDK
     * @param unbindToken   Whether unbind device
     * @return Whether unregister push
     */
    public boolean unregister(boolean unbindToken) {
        EMLog.e(TAG, TAG + " unregister, unbind token: " + unbindToken);

        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered previously, return true directly.");
            return true;
        }

        this.isClientInit = false;
        pushClient.unregister(context);

        // 停止Token上传操作。
        handler.removeMessages(MSG_WHAT_REGISTER);

        if (!unbindToken) {
            pushType = null;
            return true;
        }

        deleteToken();
        // Wait for token delete result.
        synchronized (unbindLock) {
            try {
                unbindLock.wait();
            } catch (InterruptedException e) {
            }
        }
        if (unregisterSuccess) {
            pushType = null;
        }
        EMLog.e(TAG, "Push type after unregister is " + pushType);
        return unregisterSuccess;
    }

    /**
     * \~chinese
     * 接收并上传设备token
     * @param type  推送类型
     * @param token 设备token
     *
     * \~english
     * Receive and upload device token
     * @param type  Push type
     * @param token Device token
     */
    public void onReceiveToken(EMPushType type, final String token) {
        EMLog.e(TAG, "onReceiveToken: " + type + " - " + token);
        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered, abort token upload action.");
            return;
        }
        this.pushToken = token;
        uploadToken(token);
    }

    /**
     * \~chinese
     * 接收到推送错误
     * 如果自定义的PushListener不为空，则将错误信息设置给{@link PushListener#onError(EMPushType, long)}
     * @param type          推送类型
     * @param resultCode    错误码
     *
     * \~english
     * Push error received
     * If custom PushListener is not null, then set error info to {@link PushListener#onError(EMPushType, long)}
     * @param type          Push type
     * @param resultCode    Error code
     */
    public void onErrorResponse(EMPushType type, long resultCode) {
        EMLog.e(TAG, "onErrorResponse: " + type + " - " + resultCode);
        if (!isClientInit) {
            EMLog.e(TAG, TAG + " is not registered, abort error response action.");
            return;
        }

        if (resultCode == EMError.PUSH_NOT_SUPPORT) {
            register(EMPushType.NORMAL);
        }

        if (pushListener != null) pushListener.onError(type, resultCode);
    }

    /**
     * \~chinese
     * 获取推送类型
     * @return  推送类型
     *
     * \~english
     * Get push type
     * @return  Push type
     */
    public EMPushType getPushType() {
        return pushType;
    }

    /**
     * \~chinese
     * 获取推送的设备token
     * @return  推送的设备token
     *
     * \~english
     * Get push token
     * @return  Push token
     */
    public String getPushToken() {
        return pushToken;
    }

    /**
     * \~chinese
     * 获取FCM推送的设备token
     * @return  FCM推送的设备token
     *
     * \~english
     * Get FCM's push token
     * @return  FCM's push token
     */
    public String getFCMPushToken() {
        return EMPreferenceUtils.getInstance().getFCMPushToken();
    }

    /**
     * \~chinese
     * 保存FCM推送的设备token
     * @return  FCM推送的设备token
     *
     * \~english
     * Save FCM's push token
     * @return  FCM's push token
     */
    public void setFCMPushToken(String token) {
        EMPreferenceUtils.getInstance().setFCMPushToken(token);
    }

    private void register(@NonNull EMPushType pushType) {
        if (this.pushType == pushType) {
            EMLog.e(TAG, "Push type " + pushType + " no change, return. ");
            return;
        }

        if (this.pushClient != null) {
            EMLog.e(TAG, pushClient.getPushType() + " push already exists, unregister it and change to " + pushType + " push.");
            pushClient.unregister(context);
        }

        this.pushType = pushType;

        switch (pushType) {
            case FCM:
                pushClient = new EMFCMPush();
                break;
            case MIPUSH:
                pushClient = new EMMiPush();
                break;
            case OPPOPUSH:
                pushClient = new EMOppoPush();
                break;
            case VIVOPUSH:
                pushClient = new EMVivoPush();
                break;
            case MEIZUPUSH:
                pushClient = new EMMzPush();
                break;
            case HMSPUSH:
                pushClient = new EMHMSPush();
                break;
            case NORMAL:
            default:
                pushClient = new EMNormalPush();
                break;
        }

        this.isClientInit = true;
        pushClient.register(context, pushConfig, pushListener);
    }

    private void uploadToken(String token) {
        // Cancel all previous upload tasks first.
        handler.removeMessages(MSG_WHAT_REGISTER);

        synchronized (bindLock) { // 确保把tasks放入queue时没有task正在执行。以避免task执行完后把刚放进的tasks清空。
            for (int i = -1; i < TIMES_RETRY; i++) {
                Message msg = handler.obtainMessage(MSG_WHAT_REGISTER, token);
                if (i == -1) {
                    handler.sendMessage(msg);
                } else {
                    int delaySeconds = randomDelay(i);
                    EMLog.i(TAG, "Retry upload after " + delaySeconds + "s if failed.");
                    handler.sendMessageDelayed(msg, delaySeconds * 1000);
                }
            }
        }
    }

    private void deleteToken() {
        handler.obtainMessage(MSG_WHAT_UNREGISTER).sendToTarget();
    }

    private boolean uploadTokenInternal(String notifierName, String token) {
        String remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl() + "/users/"
                + EMClient.getInstance().getCurrentUser();
        try {
            EMLog.e(TAG, "uploadTokenInternal, token=" + token + ", url=" + remoteUrl
                    + ", notifier name=" + notifierName);

            DeviceUuidFactory deviceFactory = new DeviceUuidFactory(EMClient.getInstance().getContext());

            JSONObject json = new JSONObject();
            json.put("device_token", token);
            json.put("notifier_name", notifierName);
            json.put("device_id", deviceFactory.getDeviceUuid().toString());

            Pair<Integer, String> response = EMHttpClient.getInstance().sendRequestWithToken(remoteUrl,
                    json.toString(), EMHttpClient.PUT);
            int statusCode = response.first;
            String content = response.second;

            if (statusCode == HttpURLConnection.HTTP_OK) {
                EMLog.e(TAG, "uploadTokenInternal success.");
                return true;
            }

            EMLog.e(TAG, "uploadTokenInternal failed: " + content);
        } catch (Exception e) {
            EMLog.e(TAG, "uploadTokenInternal exception: " + e.toString());
        }

        return false;
    }

    private EMPushType getPreferPushType(EMPushConfig pushConfig) {
        EMPushType[] supportedPushTypes = new EMPushType[]{
                EMPushType.FCM,
                EMPushType.MIPUSH,
                EMPushType.HMSPUSH,
                EMPushType.MEIZUPUSH,
                EMPushType.OPPOPUSH,
                EMPushType.VIVOPUSH,
        };

        ArrayList<EMPushType> enabledPushTypes = pushConfig.getEnabledPushTypes();

        for (EMPushType pushType : supportedPushTypes) {
            if (enabledPushTypes.contains(pushType) && isSupportPush(pushType, pushConfig)) {
                return pushType;
            }
        }

        return EMPushType.NORMAL;
    }

    private boolean isSupportPush(EMPushType pushType, EMPushConfig pushConfig) {
        boolean support;
        if (pushListener != null) {
            support = pushListener.isSupportPush(pushType, pushConfig);
        } else {
            support = PushUtil.isSupportPush(pushType, pushConfig);
        }
        EMLog.i(TAG, "isSupportPush: " + pushType + " - " + support);
        return support;
    }

    public int randomDelay(int attempts) {
        if (attempts == 0) {
            return new Random().nextInt(5) + 1;
        }

        if (attempts == 1) {
            return new Random().nextInt(54) + 6;
        }

        return new Random().nextInt(540) + 60;
    }

    private static class InstanceHolder {
        static EMPushHelper INSTANCE = new EMPushHelper();
    }
}
