package com.hyphenate.push;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;

import com.hyphenate.util.EMLog;

import java.util.ArrayList;

/**
 * \~chinese
 * 推送配置类
 *
 * \~english
 * Push config class
 */
public class EMPushConfig {
    private static final String TAG = "EMPushConfig";

    private String fcmSenderId;
    private String hwAppId;
    private String miAppId;
    private String miAppKey;
    private String mzAppId;
    private String mzAppKey;
    private String oppoAppKey;
    private String oppoAppSecret;
    private String vivoAppId;
    private String vivoAppKey;
    private ArrayList<EMPushType> enabledPushTypes;

    private EMPushConfig() {
    }

    /**
     * \~chinese
     * 获取FCM的senderId
     * @return  FCM的senderId
     *
     * \~english
     * Get the senderId of FCM
     * @return  SenderId of FCM
     */
    public String getFcmSenderId() {
        return fcmSenderId;
    }

    /**
     * \~chinese
     * 获取华为推送的appId
     * @return  华为推送的appId
     *
     * \~english
     * Get the appId of Huawei push
     * @return  AppId of Huawei push
     */
    public String getHwAppId() {
        return hwAppId;
    }

    /**
     * \~chinese
     * 获取小米推送的appId
     * @return  小米推送的appId
     *
     * \~english
     * Get the appId of Mi push
     * @return  AppId of Mi push
     */
    public String getMiAppId() {
        return this.miAppId;
    }

    /**
     * \~chinese
     * 获取小米推送的appKey
     * @return  小米推送的appKey
     *
     * \~english
     * Get the appKey of Mi push
     * @return  AppKey of Mi push
     */
    public String getMiAppKey() {
        return this.miAppKey;
    }

    /**
     * \~chinese
     * 获取魅族推送的appId
     * @return 魅族推送的appId
     *
     * \~english
     * Get the appId of Meizu push
     * @return AppId of Meizu push
     */
    public String getMzAppId() {
        return this.mzAppId;
    }

    /**
     * \~chinese
     * 获取魅族推送的appKey
     * @return 魅族推送的appKey
     *
     * \~english
     * Get the appKey of Meizu push
     * @return AppKey of Meizu push
     */
    public String getMzAppKey() {
        return this.mzAppKey;
    }

    /**
     * \~chinese
     * 获取Oppo推送的appKey
     * @return Oppo推送的appKey
     *
     * \~english
     * Get the appKey of Oppo push
     * @return AppKey of Oppo push
     */
    public String getOppoAppKey() {
        return this.oppoAppKey;
    }

    /**
     * \~chinese
     * 获取Oppo推送的secret
     * @return Oppo推送的secret
     *
     * \~english
     * Get the secret of Oppo push
     * @return Secret of Oppo push
     */
    public String getOppoAppSecret() {
        return this.oppoAppSecret;
    }

    /**
     * \~chinese
     * 获取Vivo推送的appId
     * @return Vivo推送的appId
     *
     * \~english
     * Get the appId of Vivo push
     * @return AppId of Vivo push
     */
    public String getVivoAppId() {
        return vivoAppId;
    }

    /**
     * \~chinese
     * 获取Vivo推送的appKey
     * @return Vivo推送的appKey
     *
     * \~english
     * Get the appKey of Vivo push
     * @return AppKey of Vivo push
     */
    public String getVivoAppKey() {
        return vivoAppKey;
    }

    /**
     * \~chinese
     * 获取支持的推送类型
     * @return 支持的推送类型列表
     *
     * \~english
     * Get the enabled push types
     * @return List of enabled push types
     */
    public ArrayList<EMPushType> getEnabledPushTypes() {
        return this.enabledPushTypes;
    }

    @Override
    public String toString() {
        return "EMPushConfig{" +
                "fcmSenderId='" + fcmSenderId + '\'' +
                ", hwAppId='" + hwAppId + '\'' +
                ", miAppId='" + miAppId + '\'' +
                ", miAppKey='" + miAppKey + '\'' +
                ", mzAppId='" + mzAppId + '\'' +
                ", mzAppKey='" + mzAppKey + '\'' +
                ", oppoAppKey='" + oppoAppKey + '\'' +
                ", oppoAppSecret='" + oppoAppSecret + '\'' +
                ", vivoAppId='" + vivoAppId + '\'' +
                ", vivoAppKey='" + vivoAppKey + '\'' +
                ", enabledPushTypes=" + enabledPushTypes +
                '}';
    }

    public static class Builder {
        private Context context;

        private String fcmSenderId;
        private String hwAppId;
        private String miAppId;
        private String miAppKey;
        private String mzAppId;
        private String mzAppKey;
        private String oppoAppKey;
        private String oppoAppSecret;
        private String vivoAppId;
        private String vivoAppKey;
        private ArrayList<EMPushType> enabledPushTypes = new ArrayList();

        /**
         * \~chinese
         * 构建一个EMPushConfig的Builder对象
         * @param context 上下文
         *
         * \~english
         * Constructor of EMPushConfig's Builder
         * @param context
         */
        public Builder(Context context) {
            this.context = context.getApplicationContext();
        }

        /**
         * \~chinese
         * 构建一个EMPushConfig的Builder对象
         * @param context 上下文
         * @param config  EMPushConfig对象
         *
         * \~english
         * Constructor of EMPushConfig's Builder
         * @param context
         * @param config    Instance of EMPushConfig
         */
        public Builder(Context context, EMPushConfig config) {
            this(context);

            if (config == null) {
                return;
            }

            if (config.enabledPushTypes.contains(EMPushType.MIPUSH)) {
                enableMiPush(config.miAppId, config.miAppKey);
            }

            if (config.enabledPushTypes.contains(EMPushType.HMSPUSH)) {
                enableHWPush();
            }

            if (config.enabledPushTypes.contains(EMPushType.VIVOPUSH)) {
                enableVivoPush();
            }

            if (config.enabledPushTypes.contains(EMPushType.OPPOPUSH)) {
                enableOppoPush(config.oppoAppKey, config.oppoAppSecret);
            }

            if (config.enabledPushTypes.contains(EMPushType.MEIZUPUSH)) {
                enableMeiZuPush(config.mzAppId, config.mzAppKey);
            }

            if (config.enabledPushTypes.contains(EMPushType.FCM)) {
                enableFCM(config.fcmSenderId);
            }
        }

        /**
         * \~chinese
         * 传入小米推送的appId和appKey，以使小米推送在小米设备上可用
         * @return  Builder
         *
         * \~english
         * AppId and appKey of Mi Push are passed in to make Mi Push available on Mi devices
         * @return  Builder
         */
        public Builder enableMiPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.miAppId = appId;
                this.miAppKey = appKey;
                this.enabledPushTypes.add(EMPushType.MIPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MI push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 使华为推送在华为设备上可用
         * 需要在AndroidManifest内配置appId或者配置agconnect-services.json
         * @return  Builder
         *
         * \~english
         * Make Huawei push enabled on Huawei devices
         * Need to set appId in AndroidManifest or to set agconnect-services.json
         * @return  Builder
         */
        public Builder enableHWPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                hwAppId = appInfo.metaData.getString("com.huawei.hms.client.appid");

                if(hwAppId != null && hwAppId.contains("=")) {
                    hwAppId = hwAppId.split("=")[1];
                } else {
                    // compatible for user set "value='xxxx'" directly
                    int id = appInfo.metaData.getInt("com.huawei.hms.client.appid");
                    hwAppId = String.valueOf(id);
                }
                this.enabledPushTypes.add(EMPushType.HMSPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                EMLog.e(TAG, "Huawei push must config meta-data: com.huawei.hms.client.appid in AndroidManifest.xml.");
            } catch (ArrayIndexOutOfBoundsException e) {
                EMLog.e(TAG, "Huawei push meta-data: com.huawei.hms.client.appid value must be like this 'appid=xxxxxx'.");
            }
            return this;
        }

        /**
         * \~chinese
         * 传入FCM的senderId，以使支持FCM推送的设备上可用
         * @return  Builder
         *
         * \~english
         * SenderId of FCM Push are passed in to make FCM Push available on the devices which are supported
         * @return  Builder
         */
        public Builder enableFCM(String senderId) {
            if (!TextUtils.isEmpty(senderId)) {
                this.fcmSenderId = senderId;
                this.enabledPushTypes.add(EMPushType.FCM);
                return this;
            } else {
                EMLog.e(TAG, "senderId can't be empty when enable FCM push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 传入魅族推送的appId和appKey，以使魅族推送在魅族设备上可用
         * @return  Builder
         *
         * \~english
         * AppId and appKey of Meizu Push are passed in to make Meizu Push available on Meizu devices
         * @return  Builder
         */
        public Builder enableMeiZuPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.mzAppId = appId;
                this.mzAppKey = appKey;
                this.enabledPushTypes.add(EMPushType.MEIZUPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MEIZU push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 传入Oppo推送的appId和appKey，以使Oppo推送在Oppo设备上可用
         * @return  Builder
         *
         * \~english
         * AppId and appKey of Oppo Push are passed in to make Oppo Push available on Oppo devices
         * @return  Builder
         */
        public Builder enableOppoPush(String appKey, String appSecret) {
            if (!TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(appSecret)) {
                this.oppoAppKey = appKey;
                this.oppoAppSecret = appSecret;
                this.enabledPushTypes.add(EMPushType.OPPOPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appKey or appSecret can't be empty when enable OPPO push !");
                return this;
            }
        }

        /**
         * \~chinese
         * 使Vivo推送在Vivo设备上可用
         * 需要在AndroidManifest中配置appId和appKey
         * @return  Builder
         *
         * \~english
         * Need to set appId and appKey in AndroidManifest
         * @return  Builder
         */
        public Builder enableVivoPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                vivoAppId = appInfo.metaData.getInt("com.vivo.push.app_id") + "";
                vivoAppKey = appInfo.metaData.getString("com.vivo.push.api_key");
                this.enabledPushTypes.add(EMPushType.VIVOPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                EMLog.e(TAG, "NameNotFoundException: " + e.getMessage());
            }
            return this;
        }

        /**
         * \~chinese
         * 返回EMPushConfig对象
         * @return  EMPushConfig
         *
         * \~english
         * Return the instance of EMPushConfig
         * @return  EMPushConfig
         */
        public EMPushConfig build() {
            EMPushConfig pushConfig = new EMPushConfig();
            pushConfig.fcmSenderId = this.fcmSenderId;
            pushConfig.hwAppId = this.hwAppId;
            pushConfig.miAppId = this.miAppId;
            pushConfig.miAppKey = this.miAppKey;
            pushConfig.mzAppId = this.mzAppId;
            pushConfig.mzAppKey = this.mzAppKey;
            pushConfig.oppoAppKey = this.oppoAppKey;
            pushConfig.oppoAppSecret = this.oppoAppSecret;
            pushConfig.vivoAppId = this.vivoAppId;
            pushConfig.vivoAppKey = this.vivoAppKey;
            pushConfig.enabledPushTypes = this.enabledPushTypes;
            return pushConfig;
        }
    }
}
