package com.hyphenate.notification;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.support.annotation.IdRes;
import android.support.v4.app.NotificationCompat;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.widget.RemoteViews;

import java.lang.ref.SoftReference;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.Date;

public class EMNotificationBuilder {

    private Context appContext;
    private boolean sound;
    private boolean vibrate;
    private String channelId;
    private String channelName;
    private int channelLevel;
    private boolean autoCancel = true;
    private String title;
    private String content;
    private String formatedDate;
    private SoftReference<Bitmap> iconBitmapRef;
    private EMNotificationDefaultStyle style = new EMNotificationDefaultStyle();
    private PendingIntent pendingIntent;
    private int smallIcon = android.R.drawable.sym_def_app_icon;
    private String ticker;
    private PendingIntent fullScreenIntent;


    public EMNotificationBuilder(Context context) {
        if (context == null) {
            throw new IllegalArgumentException("context cannot is null");
        }
        appContext = context.getApplicationContext();
    }

    public EMNotificationBuilder setTitle(String title) {
        this.title = title;
        return this;
    }

    public EMNotificationBuilder setContent(String content) {
        this.content = content;
        return this;
    }

    public EMNotificationBuilder setFormatedDate(String formatedDate) {
        this.formatedDate = formatedDate;
        return this;
    }

    public EMNotificationBuilder setIcon(@IdRes int res) {
        Bitmap tempBitmap = BitmapFactory.decodeResource(appContext.getResources(), res);
        if (tempBitmap != null) {
            iconBitmapRef = new SoftReference<>(tempBitmap);
        }
        return this;
    }

    public EMNotificationBuilder setIcon(Bitmap iconBitmap) {
        if (iconBitmap != null) {
            iconBitmapRef = new SoftReference<>(iconBitmap);
        }
        return this;
    }

    public EMNotificationBuilder setSmallIcon(@IdRes int smallIcon) {
        this.smallIcon = smallIcon;
        return this;
    }

    public EMNotificationBuilder setStyle(EMNotificationDefaultStyle style) {
        this.style = style;
        return this;
    }

    public EMNotificationBuilder setPendingIntent(PendingIntent pendingIntent) {
        this.pendingIntent = pendingIntent;
        return this;
    }

    public EMNotificationBuilder setFullScreenIntent(PendingIntent fullScreenIntent) {
        this.fullScreenIntent = fullScreenIntent;
        return this;
    }

    public EMNotificationBuilder setSound(boolean enable) {
        this.sound = enable;
        return this;
    }

    public EMNotificationBuilder setVibrate(boolean enable) {
        this.vibrate = enable;
        return this;
    }

    public EMNotificationBuilder setLevel(int level) {
        this.channelLevel = level;
        return this;
    }

    public EMNotificationBuilder setTicker(String ticker) {
        this.ticker = ticker;
        return this;
    }

    public EMNotificationBuilder setChannelId(String channelId) {
        if (!TextUtils.isEmpty(channelId)) {
            this.channelId = channelId;
        }
        return this;
    }

    public EMNotificationBuilder setChannelName(String channelName) {
        if (!TextUtils.isEmpty(channelName)) {
            this.channelName = channelName;
        }
        return this;
    }

    public EMNotificationBuilder setAutoCancel(boolean autoCancel) {
        this.autoCancel = autoCancel;
        return this;
    }


    public Notification build() {

        if (channelId == null) {
            channelId = "chat";
        }

        if (channelName == null) {
            channelName = "消息";
        }

        createNotificationChannel(appContext, vibrate, channelId, channelName, channelLevel);

        if (TextUtils.isEmpty(formatedDate)) {
            formatedDate = new SimpleDateFormat("HH:mm").format(new Date()).toString();
        }

        RemoteViews remoteViews = getRemoteView(appContext, title, content, iconBitmapRef, formatedDate);
        if (remoteViews == null) {
            printLog("remoteViews create failed!");
            return null;
        }

        RemoteViews bigRemoteViews = getBigRemoteView(appContext, title, content, iconBitmapRef, formatedDate, style);
        if (bigRemoteViews == null) {
            printLog("bigRemoteViews create failed!");
            return null;
        }

        int defaults = 0;

        if (sound) {
            defaults = Notification.DEFAULT_SOUND;
        }
        if (vibrate) {
            defaults = Notification.DEFAULT_VIBRATE;
        }

        if (sound && vibrate) {
            defaults = Notification.DEFAULT_ALL;
        }

        NotificationCompat.Builder builder = new NotificationCompat.Builder(appContext);

        try {
            Method method = NotificationCompat.Builder.class.getDeclaredMethod("setChannelId", String.class
            );
            method.setAccessible(true);
            method.invoke(builder, channelId);
        }catch (Exception ignored){

        }

        // 设置小图标
        builder.setSmallIcon(smallIcon);

        if (defaults != 0) {
            builder.setDefaults(defaults);
        }

        if (iconBitmapRef != null && iconBitmapRef.get() != null) {
            builder.setLargeIcon(iconBitmapRef.get());
        }

        builder.setTicker(ticker)
                .setPriority(channelLevel - 3)
                .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
                // 点击后自动清除
                .setAutoCancel(autoCancel)
                .setCustomContentView(remoteViews)
                .setCustomBigContentView(bigRemoteViews)
                // 设置通知标题
                .setContentTitle(title)
                // 设置通知内容
                .setContentText(content)
                // 设置通知的动作
                .setContentIntent(pendingIntent)
                // 设置通知时间，默认为系统发出通知的时间
                .setWhen(System.currentTimeMillis());

        if (fullScreenIntent != null) {
            builder.setFullScreenIntent(fullScreenIntent, true);
        }

        if (vibrate) {
            // 添加自定义震动提醒
            // 延迟100ms后震动300ms, 再延迟400ms后震动500ms
            long[] pattern = new long[]{100, 200, 300, 400, 500};
            builder.setVibrate(pattern);
        }

        Notification notification = builder.build();
        // 锁屏时显示通知
        builder.setPublicVersion(notification);
        return notification;
    }


    /**
     * 创建通知渠道
     */
    private void createNotificationChannel(Context context, boolean vibrate, String channelId, String channelName, int channelLevel) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            int importance = NotificationManager.IMPORTANCE_DEFAULT;
            if (channelLevel >= 0 && channelLevel <= 5) {
                importance = channelLevel;
            }
            NotificationChannel channel = new NotificationChannel(channelId, channelName, importance);
            if (vibrate) {
                channel.enableVibration(true);
                channel.setVibrationPattern(new long[]{100, 200, 300, 400, 500});
            }
            NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
            notificationManager.createNotificationChannel(channel);
        }
    }


    private RemoteViews getRemoteView(Context context, String title, String content, SoftReference<Bitmap> bitmapRef, String notificationDate) {
        String packageName = context.getPackageName();

        // 获取布局文件资源的ID
        int layoutId = context.getResources().getIdentifier("em_simple_notification", "layout", packageName);
        if (layoutId <= 0) {
            printLog("layout id is not exist");
            return null;
        }

        // 获取ImageView的ID
        int iconId = context.getResources().getIdentifier("em_notification_icon", "id", packageName);
        if (iconId <= 0) {
            printLog("icon id is not exist");
            return null;
        }

        // 获取Title的ID
        int titleId = context.getResources().getIdentifier("em_notification_title", "id", packageName);
        if (titleId <= 0) {
            printLog("title id is not exist");
            return null;
        }

        // 获取Content的ID
        int contentId = context.getResources().getIdentifier("em_notification_content", "id", packageName);
        if (contentId <= 0) {
            printLog("content id is not exist");
            return null;
        }

        // 获取Date的ID
        int dateId = context.getResources().getIdentifier("em_notification_date", "id", packageName);
        if (dateId <= 0) {
            printLog("date id is not exist");
            return null;
        }

        RemoteViews remoteViews = new RemoteViews(context.getPackageName(), layoutId);
        remoteViews.setTextViewText(titleId, title);
        remoteViews.setTextViewText(contentId, content);
        if (bitmapRef != null && bitmapRef.get() != null) {
            remoteViews.setImageViewBitmap(iconId, bitmapRef.get());
        } else {
            remoteViews.setViewVisibility(iconId, View.GONE);
        }
        remoteViews.setTextViewText(dateId, notificationDate);
        EMNotificationTextColorCompat.byAuto(context).setContentTitleColor(remoteViews, titleId)
                .setContentTextColor(remoteViews, contentId)
                .setContentTextColor(remoteViews, dateId);
        return remoteViews;
    }


    private RemoteViews getBigRemoteView(Context context, String title, String content, SoftReference<Bitmap> bitmapRef, String notificationDate, EMNotificationDefaultStyle style) {
        RemoteViews remoteViews = getRemoteView(context, title, content, bitmapRef, notificationDate);
        if (remoteViews != null) {
            // 获取bigView 的TextView
            int bigTxtId = context.getResources().getIdentifier("em_bigview_TextView", "id", context.getPackageName());
            // 获取bigView 的ImageView
            int bigPicId = context.getResources().getIdentifier("em_bigview_imageView", "id", context.getPackageName());
            remoteViews.setViewVisibility(bigTxtId, View.GONE);
            remoteViews.setViewVisibility(bigPicId, View.GONE);

            if (style instanceof EMNotificationBigTextStyle) {

                String bigTxt = ((EMNotificationBigTextStyle) style).getBigTxt();

                if (bigTxtId <= 0) {
                    printLog("bigTxt id is not exist");
                    return null;
                }
                remoteViews.setViewVisibility(bigTxtId, View.VISIBLE);
                remoteViews.setTextViewText(bigTxtId, bigTxt);
                EMNotificationTextColorCompat.byAuto(context).setContentTitleColor(remoteViews, bigTxtId);
            } else if (style instanceof EMNotificationBigPicStyle) {
                Bitmap bigPic = ((EMNotificationBigPicStyle) style).getBigPic();
                if (bigPicId <= 0) {
                    printLog("bigPic id is not exist");
                    return null;
                }
                if (bigPic != null) {
                    remoteViews.setImageViewBitmap(bigPicId, bigPic);
                    remoteViews.setViewVisibility(bigPicId, View.VISIBLE);
                } else {
                    remoteViews.setViewVisibility(bigPicId, View.GONE);
                }
            }
        }
        return remoteViews;
    }


    public static class EMNotificationDefaultStyle {
    }

    public static class EMNotificationBigTextStyle extends EMNotificationDefaultStyle {
        private String bigTxt;

        public String getBigTxt() {
            return bigTxt;
        }

        public EMNotificationBigTextStyle setBigTxt(String bigTxt) {
            this.bigTxt = bigTxt;
            return this;
        }
    }

    public static class EMNotificationBigPicStyle extends EMNotificationDefaultStyle {
        private SoftReference<Bitmap> bitmapRef;

        public Bitmap getBigPic() {
            if (bitmapRef != null && bitmapRef.get() != null) {
                return bitmapRef.get();
            }
            return null;
        }

        public EMNotificationBigPicStyle setBigPic(Bitmap bigPic) {
            if (bigPic != null) {
                bitmapRef = new SoftReference<>(bigPic);
            }
            return this;
        }
    }


    private void printLog(String message) {
        Log.e("notif_helper", message);
    }


}
