/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;

/**
 * \~chinese
 * 视频消息体
 * <pre>
 *     EMVideoMessageBody body = new EMVideoMessageBody(videoFilePath,thumbPath, duration, filelength);
 * </pre>
 *
 * \~english
 * Video message body
 * <pre>
 *     EMVideoMessageBody body = new EMVideoMessageBody(videoFilePath, thumbPath, duration, filelength);
 * </pre>
 */
public class EMVideoMessageBody extends EMFileMessageBody implements Parcelable {
    private static final String TAG = EMVideoMessageBody.class.getSimpleName();

    public EMVideoMessageBody() {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
    }

    public EMVideoMessageBody(EMAVideoMessageBody body) {
        super(body);
    }

    /**
     * \~chinese
     * 创建一个视频消息体
     *
     * @param videoFilePath 视频文件路径
     * @param thumbPath     预览图路径
     * @param duration      视频时长，单位是秒
     * @param filelength    视频文件大小
     *
     * \~english
     * Create a video message body
     *
     * @param videoFilePath The path of the video file
     * @param thumbPath     The path of the video first frame
     * @param duration      Video duration, in seconds
     * @param filelength    The length of the video file
     */
    public EMVideoMessageBody(String videoFilePath, String thumbPath, int duration, long filelength) {
        this(EMFileHelper.getInstance().formatInUri(videoFilePath), EMFileHelper.getInstance().formatInUri(thumbPath), duration, filelength);
        EMLog.d(EMClient.TAG, "EMVideoMessageBody thumbPath = "+thumbPath);
    }

    /**
     * \~chinese
     * 创建一个视频消息体
     *
     * @param videoFilePath 视频文件的资源标识符（路径）
     * @param thumbPath     预览图的资源标识符（路径）
     * @param duration      视频时间，单位是秒
     * @param filelength    视频文件大小
     *
     * \~english
     * Create a video message body
     *
     * @param videoFilePath The uri of the video
     * @param thumbPath     The uri of the video first frame
     * @param duration      Video duration, in seconds
     * @param filelength    The length of the video file
     */
    public EMVideoMessageBody(Uri videoFilePath, Uri thumbPath, int duration, long filelength) {
        super(videoFilePath, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(thumbPath));
        ((EMAVideoMessageBody) emaObject).setDuration(duration);
        ((EMAVideoMessageBody) emaObject).setDisplayName(EMFileHelper.getInstance().getFilename(videoFilePath));
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
        EMLog.d("videomsg", "create video, message body for:" + videoFilePath + " filename = "+EMFileHelper.getInstance().getFilename(videoFilePath));
        EMLog.d(EMClient.TAG, "EMVideoMessageBody thumbPath = "+thumbPath);
    }

    /**
     * create video message body from incoming xmpp message json
     */
    EMVideoMessageBody(String fileName, String remoteUrl, String thumbnailUrl, int length) {
        super(fileName, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setLocalPath(fileName);
        ((EMAVideoMessageBody) emaObject).setRemotePath(remoteUrl);
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setFileLength(length);

    }

    /**
     * \~chinese
     * 获取视频文件的大小
     *
     * @return 视频文件的大小
     *
     * \~english
     * Get video file length, in bytes
     *
     * @return Video file length
     */
    public long getVideoFileLength() {
        return ((EMAVideoMessageBody) emaObject).fileLength();
    }

    /**
     * \~chinese
     * 设置视频文件的文件大小
     * 用户在做数据库迁移的时候可能使用
     * @param filelength 视频文件的大小
     *
     * \~english
     * Set video file length, in bytes
     * Might need it during database migration
     * @param filelength Video file length
     */
    public void setVideoFileLength(long filelength) {
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
    }

    /**
     *  \~chinese
     *  获取缩略图在服务器的路径
     * @return 缩略图在服务器的路径
     *
     *  \~english
     *  Get thumbnail's remote url
     * @return Thumbnail's remote url
     */
    public String getThumbnailUrl() {
        return ((EMAVideoMessageBody) emaObject).thumbnailRemotePath();
    }

    /**
     *  \~chinese
     *  设置缩略图在服务器的路径
     * @param thumbnailUrl  缩略图在服务器的路径
     *
     *  \~english
     *  Set server url path of thumbnail
     *  @param thumbnailUrl Server url path of thumbnail
     */
    public void setThumbnailUrl (String thumbnailUrl) {
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    /**
     * \~chinese
     * 设置视频缩略图尺寸
     * @param width     视频缩略图宽度
     * @param height    视频缩略图高度
     *
     * \~english
     * Set video thumbnail's size
     * @param width     Thumbnail's width
     * @param height    Thumbnail's height
     */
    public void setThumbnailSize(int width, int height) {
        ((EMAVideoMessageBody)emaObject).setSize(width, height);
    }
    /**
     * \~chinese
     * 获取视频缩略图的宽度
     * @return  视频缩略图的宽度
     *
     * \~english
     * Get video thumbnail width
     * @return  Width of video thumbnail
     */
    public int getThumbnailWidth() {
        return ((EMAVideoMessageBody)emaObject).width();
    }

    /**
     * \~chinese
     * 获取视频缩略图的高度
     * @return  视频缩略图的高度
     *
     * \~english
     * Get video thumbnail height
     * @return  Height of video thumbnail
     */
    public int getThumbnailHeight() {
        return ((EMAVideoMessageBody)emaObject).height();
    }

    /**
     *  \~chinese
     *  获取视频缩略图的本地绝对路径或者资源标识符（路径）的字符串形式
     *
     *  @return 视频缩略图的本地路径
     *
     *  \~english
     *  Get local absolute path of thumbnail or the string from of Uri
     *
     *  @return Local path of thumbnail
     */
    public String getLocalThumb() {
        return EMFileHelper.getInstance().formatOutLocalUrl(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
    }

    /**
     *  \~chinese
     *  获取缩略图的本地资源标识符（路径）
     *
     *  @return 缩略图的本地资源标识符（路径）
     *
     *  \~english
     *  Get local uri of thumbnail
     *
     *  @return Local uri of thumbnail
     */
    public Uri getLocalThumbUri() {
        return EMFileHelper.getInstance().formatOutUri(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
    }

    /**
     * \~chinese
     * 设置视频预览图路径
     * @param localThumbPath  视频预览图路径，可以是绝对路径，也可以是文件的资源标识符
     *
     * \~english
     * Set path of video thumbnail
     * @param localThumbPath Path of video thumbnail, which can be either the absolute path or the Uri of file
     */
    public void setLocalThumb(String localThumbPath) {
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localThumbPath));
    }

    /**
     * \~chinese
     * 设置视频预览图路径
     * @param localThumbPath    文件的资源标识符
     *
     * \~english
     * Set path of video thumbnail
     * @param localThumbPath    Uri of video thumbnail
     */
    public void setLocalThumb(Uri localThumbPath) {
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localThumbPath));
    }

    /**
     * \~chinese
     * 获得视频时长, 单位为秒
     *
     * @return 视频时长, 单位为秒
     *
     * \~english
     * Get video duration, in seconds
     *
     * @return  Video duration, in seconds
     */
    public int getDuration() {
        return (int) ((EMAVideoMessageBody) emaObject).duration();
    }

    public String toString() {
        return "video: " + ((EMAVideoMessageBody) emaObject).displayName() +
                ", localUrl: " + ((EMAVideoMessageBody) emaObject).getLocalUrl() +
                ", remoteUrl: " + ((EMAVideoMessageBody) emaObject).getRemoteUrl() +
                ", thumbnailUrl: " + ((EMAVideoMessageBody) emaObject).thumbnailLocalPath() +
                ", length: " + ((EMAVideoMessageBody) emaObject).fileLength();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        // members from FileMessageBody
        dest.writeString(((EMAVideoMessageBody) emaObject).displayName());
        dest.writeString(((EMAVideoMessageBody) emaObject).getLocalUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).getRemoteUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeInt(((EMAVideoMessageBody) emaObject).duration());
        dest.writeLong(((EMAVideoMessageBody) emaObject).fileLength());
        dest.writeInt(((EMAVideoMessageBody) emaObject).width());
        dest.writeInt(((EMAVideoMessageBody) emaObject).height());
    }

    public static final Parcelable.Creator<EMVideoMessageBody> CREATOR = new Creator<EMVideoMessageBody>() {

        @Override
        public EMVideoMessageBody[] newArray(int size) {
            return new EMVideoMessageBody[size];
        }

        @Override
        public EMVideoMessageBody createFromParcel(Parcel in) {

            return new EMVideoMessageBody(in);
        }
    };

    private EMVideoMessageBody(Parcel in) {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
        // members from FileMessageBody
        ((EMAVideoMessageBody) emaObject).setDisplayName(in.readString());
        ((EMAVideoMessageBody) emaObject).setLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setDuration(in.readInt());
        ((EMAVideoMessageBody) emaObject).setFileLength(in.readLong());
        int width = in.readInt();
        int height = in.readInt();
        ((EMAVideoMessageBody) emaObject).setSize(width, height);
    }

    /**
     *  \~chinese
     *  获取视频预览图的密钥（访问信令）
     *  下载视频预览图时需要密匙做校验
     * @return 视频预览图的密钥（访问信令）
     *
     *  \~english
     *  Get secret key of thumbnail
     *  Required to download a thumbnail
     * @return Secret key (token) of thumbnail
     */
    public String getThumbnailSecret() {
        return ((EMAVideoMessageBody) emaObject).thumbnailSecretKey();
    }

    /**
     * \~chinese
     * 设置视频预览图的密钥（访问信令）
     * @param secret 视频预览图的密钥（访问信令）
     *
     * \~english
     * Set the secret key (token) of thumbnail
     * @param secret Secret key (token) of thumbnail
     */
    public void setThumbnailSecret(String secret) {
        ((EMAVideoMessageBody) emaObject).setThumbnailSecretKey(secret);
    }

    /**
     * \~chinese
     * 获取缩略图下载状态
     *
     * @return  缩略图下载状态
     *
     * \~english
     * Get download status of thumbnail
     *
     * @return  Download status of thumbnail
     */
    public EMDownloadStatus thumbnailDownloadStatus() {
        EMADownloadStatus _status = ((EMAVideoMessageBody) emaObject).thumbnailDownloadStatus();
        switch (_status) {
            case DOWNLOADING:
                return EMDownloadStatus.DOWNLOADING;
            case SUCCESSED:
                return EMDownloadStatus.SUCCESSED;
            case FAILED:
                return EMDownloadStatus.FAILED;
            case PENDING:
                return EMDownloadStatus.PENDING;
        }
        return EMDownloadStatus.SUCCESSED;
    }

    void setThumbnailDownloadStatus(EMDownloadStatus status) {
        ((EMAVideoMessageBody) emaObject).setThumbnailDownloadStatus(EMADownloadStatus.valueOf(status.name()));
    }
}
