/************************************************************
  *  * EaseMob CONFIDENTIAL 
  * __________________ 
  * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
  *  
  * NOTICE: All information contained herein is, and remains 
  * the property of EaseMob Technologies.
  * Dissemination of emaObject information or reproduction of emaObject material 
  * is strictly forbidden unless prior written permission is obtained
  * from EaseMob Technologies.
  */
package com.hyphenate.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.EMCallBack;
import com.hyphenate.chat.adapter.EMACallback;
import com.hyphenate.chat.adapter.message.EMACmdMessageBody;
import com.hyphenate.chat.adapter.message.EMACustomMessageBody;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMALocationMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.adapter.message.EMAMessage.EMADirection;
import com.hyphenate.chat.adapter.message.EMAMessage.EMAMessageStatus;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMATextMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.chat.adapter.message.EMAVoiceMessageBody;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * \~chinese
 * 消息对象，代表一条发送或接收到的消息
 * 
 * <p>
 * 构造一条文本发送消息
 * <pre>
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * </pre>
 * 
 * <p>
 * 构造一条图片消息
 * <pre>
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * </pre>
 * 
 * \~english
 * Message object, represent a sent/received message
 * <p>
 * Construct a new send text message
 * <pre>
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * </pre>
 * 
 * <p>
 * Construct a new receive text message
 * <pre>
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * </pre>
 */
public class EMMessage extends EMBase<EMAMessage> implements Parcelable, Cloneable{
    static final String ATTR_ENCRYPTED = "isencrypted";
    
    private final static String TAG = "msg";

    /**
     * \~chinese
     * 消息类型枚举类
     * 包含：文本，图片，视频，位置，语音，文件,透传, 用户自定义消息
     * 
     * \~english
     * Message type enum
     * Include:TXT, IMAGE, VIDEO, LOCATION, VOICE, FILE, CMD, CUSTOM
     */
    public enum Type {
        TXT, IMAGE, VIDEO, LOCATION, VOICE, FILE, CMD, CUSTOM
    }
    
    /**
     * \~chinese
     * 消息的方向类型枚举类
     * 区分是发送消息还是接收到的消息
     * 
     * \~english
     * Message direction enum
     * Send direction or receive direction
     */
    public enum Direct {
        SEND, RECEIVE
    }
    
    /**
     * \~chinese
     * 消息的发送/接收状态枚举类
     * 包括如下状态：成功，失败，发送/接收过程中，创建成功待发送
     * 
     * \~english
     * Message status enum
     * Includes the following states: success, failure, send/receive process, create success to be sent
     */
    public enum Status {
        SUCCESS, FAIL, INPROGRESS, CREATE
    }
    
    EMMessageBody body;
    
    /**
     * \~chinese
     * 获取消息的发送/接收状态
     * @return 消息的发送/接收状态
     * 
     * \~english
     * Get the send/receive status of message
     * @return  Send/receive status of message
     */
    public Status status() {
    	EMAMessage.EMAMessageStatus st = emaObject._status();
    	Status s = Status.CREATE;
    	if (st == EMAMessageStatus.NEW) {
    		s = Status.CREATE;
    	} else if (st == EMAMessageStatus.SUCCESS) {
    		s = Status.SUCCESS;
    	} else if (st == EMAMessageStatus.FAIL) {
    		s = Status.FAIL;
    	} else if (st == EMAMessageStatus.DELIVERING) {
    		s = Status.INPROGRESS;
    	}
    	return s;
    }
    
    /**
     * \~chinese
     * 聊天类型枚举类
     * 包括：单聊，群聊，聊天室
     *
     *\~english
     * Chat type enum
     * Include: private chat, group chat, chat room
     */
    public enum ChatType {
        /**
         * \~chinese
         * 单聊
         *
         * \~english
         * Private single chat
         */
        Chat, 
        /**
         * \~chinese
         * 群聊
         *
         * \~english
         * Group chat
         */
        GroupChat,
        /**
         * \~chinese
         * 聊天室
         *
         * \~english
         * Chat room
         */
        ChatRoom
    }
    
    public EMMessage(EMAMessage message) {
        emaObject = message;
    }
    
    /**
     * \~chinese
     * 设置消息的状态
     * @param status 息的状态
     * 
     * \~english
     * Set the status of the message
     * @param status Status of the message
     */
    public void setStatus(Status status) {
    	EMAMessage.EMAMessageStatus s = EMAMessageStatus.SUCCESS;
    	if (status == Status.CREATE) {
    		s = EMAMessageStatus.NEW;
    	} else if (status == Status.SUCCESS) {
    		s = EMAMessageStatus.SUCCESS;
    	} else if (status == Status.FAIL) {
    		s = EMAMessageStatus.FAIL;
    	} else if (status == Status.INPROGRESS) {
    		s = EMAMessageStatus.DELIVERING;
    	}
    	emaObject.setStatus(s.ordinal());
    }
    
    /**
     * \chinese
     * 获取消息类型
     * @return  消息类型
     * 
     * \~english
     * Get message chat type
     * @return Message chat type
     */
    public Type getType() {
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		int type = bodies.get(0).type();
    		//Log.d(TAG, "type:" + type);
    		if (type == Type.TXT.ordinal()) {
    			return Type.TXT;
    		} else if (type == Type.IMAGE.ordinal()) {
    			return Type.IMAGE;
    		} else if (type == Type.CMD.ordinal()) {
    			return Type.CMD;
    		} else if (type == Type.FILE.ordinal()) {
    			return Type.FILE;
    		} else if (type == Type.VIDEO.ordinal()) {
    			return Type.VIDEO;
    		} else if (type == Type.VOICE.ordinal()) {
    			return Type.VOICE;
    		} else if (type == Type.LOCATION.ordinal()) {
    			return Type.LOCATION;
    		} else if (type == Type.CUSTOM.ordinal()) {
    		    return Type.CUSTOM;
            }
    	}
        return Type.TXT;
    }
    
    /**
     * \~chinese
     * 获取消息体
     * @return  消息体
     * 
     * \~english
     * Get message body
     * @return  Message body
     */
    public EMMessageBody getBody() {
        if (body != null) {
            return body;
        }
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		EMAMessageBody emaBody = bodies.get(0);
    		if (emaBody instanceof EMATextMessageBody) {
    			body = new EMTextMessageBody((EMATextMessageBody)emaBody);
    		} else if (emaBody instanceof EMACmdMessageBody) {
    			body = new EMCmdMessageBody((EMACmdMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVideoMessageBody) {
    			body = new EMVideoMessageBody((EMAVideoMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVoiceMessageBody) {
    			body = new EMVoiceMessageBody((EMAVoiceMessageBody)emaBody);
    		} else if (emaBody instanceof EMAImageMessageBody) {
    			body = new EMImageMessageBody((EMAImageMessageBody)emaBody);
    		} else if (emaBody instanceof EMALocationMessageBody) {
    			body = new EMLocationMessageBody((EMALocationMessageBody)emaBody);
    		} else if (emaBody instanceof EMAFileMessageBody) {
    			body = new EMNormalFileMessageBody((EMAFileMessageBody)emaBody);
    		} else if (emaBody instanceof EMACustomMessageBody) {
    		    body = new EMCustomMessageBody((EMACustomMessageBody) emaBody);
            }
        	return body;
    	}

    	return null;
    }
    
    /**
     * \~chinese
     * 获取消息的时间戳(服务器时间)
     * @return  消息的时间戳(服务器时间)
     * 
     * \~english
     * Get message timestamp(server time)
     * @return  Message timestamp(server time)
     */
    public long getMsgTime() {
    	return emaObject.timeStamp();
    }
    
    
    /**
     * \~chinese
     * 设置消息的时间戳(服务器时间)
     * @param msgTime   消息的时间戳(服务器时间)
     * 
     * \~english
     * Set message timestamp (server time)
     * @param msgTime   Message timestamp (server time)
     */
    public void setMsgTime(long msgTime) {
		emaObject.setTimeStamp(msgTime);
	}
    
    /**
     * \~chinese
     * 获取消息的本地时间戳
     * @return  消息的本地时间戳
     * 
     * \~english
     * Get local timestamp
     * @return Local timestamp
     */
    public long localTime() {
        return emaObject.getLocalTime();
    }
    
    /**
     * \~chinese
     * 设置消息的本地时间戳
     * @param serverTime    消息的本地时间戳
     * 
     * \~english
     * Set message local time
     * @param serverTime    Message local time
     */
    public void setLocalTime(long serverTime) {
        emaObject.setLocalTime(serverTime);
    }

    /**
     * \~chinese
     * 消息是否需要群组已读回执
     * @return true：需要已读回执；false：不需要已读回执
     *
     * \~english
     * Message if need group read ack
     * @return True: need group read ack; False: not need group read ack
     */
    public boolean isNeedGroupAck() {return emaObject.isNeedGroupAck(); }


    /**
     * \~chinese
     * 设置消息是否需要群组已读回执
     * @param need
     *
     * \~english
     * Set message if need group read ack
     * @param need
     */
    public void setIsNeedGroupAck(boolean need) {emaObject.setIsNeedGroupAck(need); }

    /**
     * \~chinese
     * 群消息被读的人数
     * @return 消息被读的人数
     *
     * \~english
     * Read ack number of group message
     * @return Read ack number of group message
     */
    public int groupAckCount() { return emaObject.groupAckCount(); }

    /**
     * \~chinese
     * 设置群消息被读的人数
     * @param count
     *
     * \~english
     * Set read ack number of group message
     * @param count
     */
    public void setGroupAckCount(int count) { emaObject.setGroupAckCount(count); }


	/**
	 * \~chinese
     * 创建一个发送消息
     * @param type 消息类型
     * @return 消息对象
     * 
     * \~english
     * Create a new send message
     * @param type The message type
     * @return The message instance
     */
    public static EMMessage createSendMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createSendMessage(self(), "", null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个接收消息
     * @param type 消息类型
     * @return 消息对象
     * 
     * \~english
     * Create a new receive message
     * @param type The message type
     * @return The message instance
     */
    public static EMMessage createReceiveMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        msg.setTo(EMSessionManager.getInstance().currentUser.getUsername());
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个文本发送消息
     * @param content 文本内容
     * @param username 消息接收人或群id
     * @return 消息对象
     * 
     * \~english
     * Create a text send message
     * @param content Text content
     * @param username The recipient(user or group) id
     * @return The message instance
     */
    public static EMMessage createTxtSendMessage(String content, String username){
        if (content.length() > 0) {
        	EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
        	EMTextMessageBody txtBody = new EMTextMessageBody(content);
        	msg.addBody(txtBody);
        	msg.setTo(username);
            return msg;
        }
        EMLog.e(TAG, "text content size must be greater than 0");
        return null;
    }
    
    /**
     * \~chinese
     * 创建一个语音发送消息
     * @param filePath 语音文件路径
     * @param timeLength 语音时间长度(单位秒)
     * @param username 消息接收人或群id
     * @return 消息对象
     * 
     * \~english
     * Create a voice send message
     * @param filePath The path of the voice file
     * @param timeLength The time length of the voice(unit s)
     * @param username  The recipient(user or group) id
     * @return The message instance
     *
     */
    public static EMMessage createVoiceSendMessage(String filePath, int timeLength, String username){
        return createVoiceSendMessage(EMFileHelper.getInstance().formatInUri(filePath), timeLength, username);
    }

    /**
     * \~chinese
     * 创建一个语音发送消息
     * @param filePath 语音文件uri
     * @param timeLength 语音时间长度(单位秒)
     * @param username 消息接收人或群id
     * @return 消息对象
     *
     * \~english
     * Create a voice send message
     * @param filePath The uri of the voice file
     * @param timeLength The time length of the voice(unit s)
     * @param username  The recipient(user or group) id
     * @return The message instance
     */
    public static EMMessage createVoiceSendMessage(Uri filePath, int timeLength, String username){
        if (!EMFileHelper.getInstance().isFileExist(filePath)) {
            EMLog.e(TAG, "voice file does not exsit");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.VOICE);
        // 如果是群聊，设置chattype,默认是单聊
        EMVoiceMessageBody body = new EMVoiceMessageBody(filePath, timeLength);
        message.addBody(body);

        message.setTo(username);
        return message;
    }

    /**
     * \~chinese
     * 创建一个图片发送消息
     * @param filePath 图片路径
     * @param sendOriginalImage 是否发送原图(默认大于100K的图片sdk会进行压缩)
     * @param username 消息接收人或群id
     * @return 消息对象
     * 
     * \~english
     * Create a image send message
     * @param filePath The path of the image
     * @param sendOriginalImage Whether to send the original(if image greater than 100k sdk will be compressed)
     * @param username The recipient(user or group) id
     * @return The message instance
     *
     */
    public static EMMessage createImageSendMessage(String filePath, boolean sendOriginalImage, String username){
        return createImageSendMessage(EMFileHelper.getInstance().formatInUri(filePath), sendOriginalImage, username);
    }

    /**
     * \~chinese
     * 创建一个图片发送消息
     * @param imgUri 图片Uri
     * @param sendOriginalImage 是否发送原图(默认大于100K的图片sdk会进行压缩)
     * @param username 消息接收人或群id
     * @return 消息对象
     *
     * \~english
     * Create a image send message
     * @param imgUri The uri of the image
     * @param sendOriginalImage Whether to send the original(if image greater than 100k sdk will be compressed)
     * @param username The recipient(user or group) id
     * @return The message instance
     */
    public static EMMessage createImageSendMessage(Uri imgUri, boolean sendOriginalImage, String username){
        if(!EMFileHelper.getInstance().isFileExist(imgUri)) {
            EMLog.e(TAG, "image file does not exsit");
            return null;
        }
        // create and add image message in view
        final EMMessage message = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
        message.setTo(username);
        EMImageMessageBody body = new EMImageMessageBody(imgUri);
        // 默认超过100k的图片会压缩后发给对方，可以设置成发送原图
        body.setSendOriginalImage(sendOriginalImage);
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 创建一个视频发送消息
     * @param videofilePath 视频文件路径
     * @param imageThumbPath 视频第一帧图缩略图
     * @param timeLength 视频时间长度(单位秒)
     * @param username 消息接收人或群id
     * @return 消息对象
     * 
     * \~english
     * Create a video send message
     * @param videofilePath The path of the video file
     * @param imageThumbPath The path of the thumbnail
     * @param timeLength The length of the video time, unit s
     * @param username The recipient(user or group) id
     * @return The message instance
     *
     */
    public static EMMessage createVideoSendMessage(String videofilePath, String imageThumbPath, int timeLength, String username){
        return createVideoSendMessage(EMFileHelper.getInstance().formatInUri(videofilePath)
                , EMFileHelper.getInstance().formatInUri(imageThumbPath)
                , timeLength
                , username);
    }

    /**
     * \~chinese
     * 创建一个视频发送消息
     * @param videofilePath 视频文件路径
     * @param imageThumbPath 视频第一帧图缩略图
     * @param timeLength 视频时间长度(单位秒)
     * @param username 消息接收人或群id
     * @return 消息对象
     *
     * \~english
     * Create a video send message
     * @param videofilePath The path of the video file
     * @param imageThumbPath The path of the thumbnail
     * @param timeLength The length of the video time, unit s
     * @param username The recipient(user or group) id
     * @return The message instance
     */
    public static EMMessage createVideoSendMessage(Uri videofilePath, String imageThumbPath, int timeLength, String username){
        return createVideoSendMessage(videofilePath, EMFileHelper.getInstance().formatInUri(imageThumbPath), timeLength, username);
    }

    /**
     * \~chinese
     * 创建一个视频发送消息
     * @param videofilePath 视频文件路径
     * @param imageThumbPath 视频第一帧图缩略图
     * @param timeLength 视频时间长度(单位秒)
     * @param username 消息接收人或群id
     * @return 消息对象
     *
     * \~english
     * Create a video send message
     * @param videofilePath The path of the video file
     * @param imageThumbPath The path of the thumbnail
     * @param timeLength The length of the video time, unit s
     * @param username The recipient(user or group) id
     * @return  The message instance
     */
    public static EMMessage createVideoSendMessage(Uri videofilePath, Uri imageThumbPath, int timeLength, String username){
        if (!EMFileHelper.getInstance().isFileExist(videofilePath)) {
            EMLog.e(TAG, "video file does not exist");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.VIDEO);
        message.setTo(username);
        EMVideoMessageBody body = new EMVideoMessageBody(videofilePath, imageThumbPath, timeLength
                , EMFileHelper.getInstance().getFileLength(videofilePath));
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 创建一个位置发送消息
     * @param latitude 纬度
     * @param longitude 经度
     * @param locationAddress 位置详情
     * @param username 消息接收人或群id
     * @return 消息对象
     * 
     * \~english
     * Create a location send message
     * @param latitude The latitude
     * @param longitude The longitude
     * @param locationAddress Location details
     * @param username The recipient(user or group) id
     * @return The message instance
     */
    public static EMMessage createLocationSendMessage(double latitude, double longitude, String locationAddress, String username){
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.LOCATION);
        EMLocationMessageBody locBody = new EMLocationMessageBody(locationAddress, latitude, longitude);
        message.addBody(locBody);
        message.setTo(username);
        
        return message;
    }
    
    /**
     * \~chinese
     * 创建一个普通文件发送消息
     * @param filePath 文件路径
     * @param username 消息接收人或群id
     * @return 消息对象
     * 
     * \~english
     * Create a normal file send message
     * @param filePath The path of the file
     * @param username The recipient(user or group) id
     * @return The message instance
     */
    public static EMMessage createFileSendMessage(String filePath, String username){
        return createFileSendMessage(EMFileHelper.getInstance().formatInUri(filePath), username);
    }

    /**
     * \~chinese
     * 创建一个普通文件发送消息
     * @param filePath 文件路径
     * @param username 消息接收人或群id
     * @return 消息对象
     *
     * \~english
     * Create a normal file send message
     * @param filePath The path of the file
     * @param username The recipient(user or group) id
     * @return The message instance
     */
    public static EMMessage createFileSendMessage(Uri filePath, String username){
        if(!EMFileHelper.getInstance().isFileExist(filePath)){
            EMLog.e(TAG, "file does not exist");
            return null;
        }
        // 创建一个文件消息
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.FILE);

        message.setTo(username);
        // add message body
        EMNormalFileMessageBody body = new EMNormalFileMessageBody(filePath);
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 设置消息体
     * @param body
     *
     * \~english
     * set message body
     * @param body
     */
    public void setBody(EMMessageBody body) {
        if(this.body != body) {
            this.body = body;
            emaObject.clearBodies();
            emaObject.addBody(body.emaObject);
        }
    }

    /**
     * \~chinese
     * 添加消息体
     * 现在只支持添加一个消息体
     * @param body 消息体
     * 
     * \~english
     * Add a message body
     * Only support add one now
     * @param body The message body
     */
    public void addBody(EMMessageBody body) {
        this.body = body;
        emaObject.addBody(body.emaObject);
    }
    
    /**
     * \~chinese
     * 获取消息发送者的用户名
     * @return 用户名
     * 
     * \~english
     * Get the sender id
     * @return user id 
     */
    public String getFrom() {
        return emaObject.from();
    }
    
    /**
     * \~chinese
     * 设置消息发送者id
     * @param from  发送者id
     * 
     * \~english
     * Set message sender id
     * @param from Sender id
     */
    public void setFrom(String from) {
		emaObject.setFrom(from);
		// The reason for using "getTo ()" here is to prevent "from" as "conversationId" when the message is a group type
		if (!self().equals(from) && getTo() != null && getTo() != "" && getTo().equals(self())) {
			emaObject.setConversationId(from);
		}
	}

    /**
     * \~chinese
     * 设置消息的接收者id
     * @param to    收者id
     * 
     * \~english
     * Sets the receiver ID of the message
     * @param to    The receiver ID
     */
	public void setTo(String to) {
        emaObject.setTo(to);
        // Removing the logic to check if it's you is to prevent you from being able to message yourself
        emaObject.setConversationId(to);
	}

	/**
	 * \~chinese
     * 获取消息接收者的用户名
     * @return  接收者的用户名
     * 
     * \~english
     * Get receiver name
     * @return Receiver name
     */
    public String getTo() {
        return emaObject.to();
    }
    
    /**
     * \~chinese
     * 获取消息的id
     * @return 消息的id
     * 
     * \~english
     * Get message id
     * @return Message id
     */
    public String getMsgId() {
        return emaObject.msgId();
    }
    
    /**
     * \~chinese
     * 设置本地消息id
     * @param msgId 消息id
     * 
     * \~english
     * Set local message id
     * @param msgId Message id
     */
    public void setMsgId(String msgId){
    	emaObject.setMsgId(msgId);
    }

    /**
     * EMXXXListener doesn't need EMCallbackHolder, user will be responsible for add, remove listener reference.
     * Message's callback function is different from listener, user add callback, and won't remove the reference.
     * Old implementation for Message's callback is simply hold the reference, which leads to memory leak.
     * SDK caches hold reference of message, message hold callback, callback hold Activity's this reference, so activities will never be released.
     *
     * Solution:
     * 1. change the message's callback to weak reference.
     * 2. when sendMessage or downloadMessageAttachment make the reference be strong one.
     * 3. when callback be executed, onSuccess or onError, release the reference.
     *
     * Declaim:
     * 1. don't set message be strong when user call setMessageStatusCallback, user may just attach the callback, there will not be onError or onSuccess to be called
     */
    static class EMCallbackHolder implements EMCallBack {

        EMCallbackHolder(EMCallBack callback) {
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void update(EMCallBack callback) {
            if (strong != null) {
                strong = callback;
                return;
            }
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void makeItStrong() {
            if (strong != null) {
                return;
            }
            if (weak != null && weak.get() != null) {
                strong = weak.get();
            }
        }

        synchronized void release() {
            if (strong == null) {
                return;
            }
            weak = new WeakReference<EMCallBack>(strong);
            strong = null;
        }

        synchronized EMCallBack getRef() {
            if (strong != null) {
                return strong;
            }
            if (weak != null) {
                EMCallBack callBack = weak.get();
                if (callBack == null) {
                    EMLog.d(TAG, "getRef weak:" + callBack);
                }
                return callBack;
            }
            return null;
        }

        public void onSuccess() {
            if (innerCallback != null) {
                innerCallback.onSuccess();
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onSuccess();
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onError(int code, String error) {
            if (innerCallback != null) {
                innerCallback.onError(code, error);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onError(code, error);
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onProgress(int progress, String status) {
            if (innerCallback != null) {
                innerCallback.onProgress(progress, status);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onProgress(progress, status);
            }
        }

        // strong & weak are used for external callback reference
        // inner callback is used by chat manager sendMessage, to handle changes of msgId and msg time
        private EMCallBack strong;
        private WeakReference<EMCallBack> weak;
        EMCallBack innerCallback = null;
    }

    EMCallbackHolder messageStatusCallBack;

    synchronized void setInnerCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.innerCallback = callback;
        } else {
            messageStatusCallBack = new EMCallbackHolder(null);
            messageStatusCallBack.innerCallback = callback;
        }
        setCallback(messageStatusCallBack);
    }

    /**
     * \~chinese
     * 设置消息状态改变的回调
     * 设置消息状态回调，以便刷新界面
     * @param callback  消息状态改变的回调
     * 
     * \~english
     * Set message status callback
     * Your app should set emaObject callback to get message status and then refresh the ui accordingly
     * @param callback
     */
    public synchronized void setMessageStatusCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.update(callback);
        } else {
            messageStatusCallBack = new EMCallbackHolder(callback);
        }
        setCallback(messageStatusCallBack);
    }


    void setCallback(final EMCallbackHolder holder) {
        this.emaObject.setCallback(new EMACallback(new EMCallBack(){

            @Override
            public void onSuccess() {
                if(holder != null){
                    holder.onSuccess();
                }
            }

            @Override
            public void onProgress(int progress, String status) {
                if(holder != null){
                    holder.onProgress(progress, status);
                }
            }

            @Override
            public void onError(int code, String error) {
                if(holder != null){
                    holder.onError(code, error);
                }
            }
        }));
    }

    void makeCallbackStrong() {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.makeItStrong();
        }
    }


    public String toString() {
        String sb = "msg{from:" + getFrom() +
                ", to:" + getTo() +
                " body:" + getBody();
        return sb;
    }
    
    /**
     * \~chinese
     * 设置消息的boolean 类型扩展属性
     * @param attribute 属性名
     * @param value  属性值
     * 
     * \~english
     * Set a boolean type extra attributes of the message
     * @param attribute Attribute key
     * @param value Attribute value
     */
    public void setAttribute(String attribute, boolean value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 设置消息的int 类型扩展属性
     * @param attribute 属性名
     * @param value  属性值
     * 
     * \~english
     * Set a int type extra attributes of the message
     * @param attribute Attribute key
     * @param value Attribute value
     */
    public void setAttribute(String attribute, int value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }


    /**
     * \~chinese
     * 设置消息的long 类型扩展属性
     * @param attribute 属性名
     * @param value  属性值
     * 
     * \~english
     * Set a long type extra attributes of the message
     * @param attribute Attribute key
     * @param value Attribute value
     */
    public void setAttribute(String attribute, long value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    
    /**
     * \~chinese
     * 设置消息的JSONObject 类型扩展属性
     * @param attribute 属性名
     * @param value  属性值
     * 
     * \~english
     * Set a JSONObject type extra attributes of the message
     * @param attribute Attribute key
     * @param value Attribute value
     */
    public void setAttribute(String attribute, JSONObject value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的JSONArray 类型扩展属性
     * @param attribute 属性名
     * @param value  属性值
     * 
     * \~english
     * Set a JSONArray type extra attributes of the message
     * @param attribute Attribute key
     * @param value Attribute value
     */
    public void setAttribute(String attribute, JSONArray value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的string 类型扩展属性
     * @param attribute 属性名
     * @param value  属性值
     * 
     * \~english
     * Set a string type extra attributes of the message
     * @param attribute Attribute key
     * @param value Attribute value
     */
    public void setAttribute(String attribute, String value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 获取boolean 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * Get a boolean type extra attribute
     * @param attribute Attribute key
     * @return Attribute value
     * @throws HyphenateException
     */
    public boolean getBooleanAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取 boolean 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * Get a boolean type extra attribute
     * @param attribute Attribute key
     * @param defaultValue the default value you want
     * @return Attribute value
     */
    public boolean getBooleanAttribute(String attribute, boolean defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取 int 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * Get a int type extra attribute
     * @param attribute Attribute key
     * @param defaultValue The default value you want
     * @return Attribute value
     */
    public int getIntAttribute(String attribute,int defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取 long 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * Get a long type extra attribute
     * @param attribute Attribute key
     * @param defaultValue The default value you want
     * @return Attribute value
     */
    public long getLongAttribute(String attribute,long defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.longValue();
    }
    
    /**
     * \~chinese
     * 获取 int 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * Get a int type extra attribute
     * @param attribute Attribute key
     * @return Attribute value
     * @throws HyphenateException
     */
    public int getIntAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取 long 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * Get long type extra attribute
     * @param attribute Attribute key
     * @return Attribute value
     * @throws HyphenateException
     */
    public long getLongAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.longValue();
    }
    
    /**
     * \~chinese
     * 获取 String 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * Get a string type extra attribute
     * @param attribute Attribute key
     * @return Attribute value
     * @throws HyphenateException
     */
    public String getStringAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.toString();
    }
    
    /**
     * \~chinese
     * 获取 String 类型扩展属性
     * @param attribute 属性名
     * @param defaultValue 缺省值
     * @return 属性值
     * 
     * \~english
     * Get a string type extra attribute
     * @param attribute Attribute key
     * @param defaultValue The default value you want
     * @return Attribute value
     */
    public String getStringAttribute(String attribute, String defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            return defaultValue;
        }        
        return val.toString();
    }
    
    
    /**
     * \~chinese
     * 获取 JSONObject 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * Get a JSONObject type attribute
     * @param attribute Attribute key
     * @return Attribute value
     * @throws HyphenateException
     */
    public JSONObject getJSONObjectAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            return null;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "{}", val);
        if (exists) {
            try {
                JSONObject jsonObj = new JSONObject(val.toString());
                return jsonObj;
            } catch (JSONException e) {
            }
        }
        throw new HyphenateException("attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取 JSONArray 类型扩展属性
     * @param attribute 属性名
     * @return 属性值
     * @throws HyphenateException
     * 
     * \~english
     * Get a JSONArray type attribute
     * @param attribute Attribute key
     * @return Attribute value
     * @throws HyphenateException
     */
    public JSONArray getJSONArrayAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            return null;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "[]", val);
        if (exists) {
            try {
                JSONArray jsonArray = new JSONArray(val.toString());
                return jsonArray;
            } catch (JSONException e) {
            }
        }
        throw new HyphenateException("attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取聊天类型
     * @return ChatType
     * 
     * \~english
     * Get chat type
     * @return ChatType
     */
    public ChatType getChatType() {
    	EMAMessage.EMAChatType t = emaObject.chatType();
    	ChatType type = ChatType.Chat;
    	if (t == EMAMessage.EMAChatType.SINGLE) {
    		type = ChatType.Chat;
    	} else if (t == EMAMessage.EMAChatType.GROUP) {
    		type = ChatType.GroupChat;
    	} else {
    		type = ChatType.ChatRoom;
    	}
    	return type;
    }
    
    /**
     * \~chinese
     * 设置聊天类型
     * 缺省为单聊{@link ChatType#Chat}
     * @param chatType  聊天类型，详见{@link ChatType}
     *
     * \~english
     * Set chat type
     * The default is single chat {@link ChatType#Chat}
     * @param chatType see {@link ChatType}
     */
    public void setChatType(ChatType chatType) {
    	EMAMessage.EMAChatType t = EMAMessage.EMAChatType.SINGLE;
    	if (chatType == ChatType.Chat) {
    		t = EMAMessage.EMAChatType.SINGLE;
    	} else if (chatType == ChatType.GroupChat) {
    		t = EMAMessage.EMAChatType.GROUP;
    	} else {
    		t = EMAMessage.EMAChatType.CHATROOM;
    	}
    	emaObject.setChatType(t);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(getMsgId());
    }
    
    public static final Parcelable.Creator<EMMessage> CREATOR
        = new Parcelable.Creator<EMMessage>() {
        public EMMessage createFromParcel(Parcel in) {
            EMMessage msg = null;
            try {
                msg = new EMMessage(in);
            } catch (HyphenateException e) {
                e.printStackTrace();
            }
            return msg;
        }

        public EMMessage[] newArray(int size) {
            return new EMMessage[size];
        }
    };
    
	@SuppressWarnings("CloneDoesntCallSuperClone")
    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    /**
     * \~chinese
     * 对方是否已读
     * @return 是否已读
     * 
     * \~english
     * whether to read ack by the other
     * @return whether to read
     */
	public boolean isAcked() {
		return emaObject.isAcked();
	}

    /**
     * \~chinese
     * 设置对方是否已读
     * 开发者不应该调用此方法，由SDK进行调用
     * @param isAcked
     * 
     * \~english
     * Sets whether the other has been read
     * Not supposed to be called by app
     * @param isAcked
     */
	public void setAcked(boolean isAcked) {
		emaObject.setIsAcked(isAcked);
	}

    /**
     * \~chinese
     * 获取消息是否已到达对方
     * @return 消息是否已到达对方
     * 
     * \~english
     * Delivery Ack, check if the peer has received the message
     * @return Whether the peer has received the message
     */
	public boolean isDelivered() {
		return emaObject.isDeliverAcked();
	}

    /**
     * \~chinese
     * 设置消息是否已经送达对方
     * 开发者不要调用，由SDK内部调用
     * @param isDelivered
     *
     * \~english
     * Sets whether the message has been delivered
     * Not supposed to be called by app
     * @param isDelivered
     */
	public void setDelivered(boolean isDelivered) {
		emaObject.setIsDeliverAcked(isDelivered);
	}

    /**
     * \~chinese
     * 检查消息是否已读
     * @return 消息是否已读
     *
     * \~english
     * Check if the message has been read
     * @return Whether the message has been read
     */
	public boolean isUnread() {
		return !emaObject.isRead();
	}

    /**
     * \~chinese
     * 设置消息已读
     * 建议在会话中，使用{@link EMConversation#markAllMessagesAsRead()}
     * @param unread
     *
     * \~chinese
     * 设置消息已读
     * It is recommended to use {@link EMConversation#markAllMessagesAsRead()} in a conversation
     * @param unread
     */
	public void setUnread(boolean unread) {
		emaObject.setIsRead(!unread);
	}

	/**
	 * \~chinese
	 * 获取语音是否已听
	 * @return
	 * 
	 * \~english
	 * Get whether the message has been listened
     * @return
	 */
	public boolean isListened() {
		return emaObject.isListened();
	}

	/**
	 * \~chinese
	 * 设置语音是否已听
	 * @param isListened
	 * 
	 * \~english
	 * Sets whether the other has been listened
	 * @param isListened
	 */
	public void setListened(boolean isListened) {
	    emaObject.setListened(isListened);
	}
	
	/**
	 * \~chinese
	 * 获取通话对象
	 * @return 对方id
	 * 
	 * \~english
	 * Get the peer's id
     * @return Peer's id
	 */
	public String getUserName() {
		String username = "";
		if (getFrom() != null && getFrom().equals(EMClient.getInstance().getCurrentUser())) {
			username = getTo();
		} else {
			username = getFrom();
		}
		return username;
	}

    /**
     * \~chinese
     * 设置消息已送达标记
     * 开发者不要调用，由SDK内部调用
     * @param isDeliverAcked
     *
     * \~english
     * Set message deliver ack
     * Not supposed to be called by app
     * @param isDeliverAcked
     */
	public void setDeliverAcked(boolean isDeliverAcked) {
	    emaObject.setIsDeliverAcked(isDeliverAcked);
	}

    /**
     * \~chinese
     * 消息包含附件的上传或者下载进度，值的范围0-100
     * 消息附件的缩略图不涉及进度信息
     * @return 进度值
     *
     * \~english
     * Indicate message attachment upload or download progress, value ranges between 0-100
     * For message attachment's thumbnail, it doesn't has progress information
     * @return progress value
     */
	public int progress() {
	    return emaObject.progress();
	}

    /**
     * \~chinese
     * 设置消息包含附件的上传或者下载进度，值的范围0-100
     * 对于app开发者来说，通常不需要主动设置进度
     * @param progress
     *
     * \~english
     * Set message attachment upload or download progress, value ranges between 0-100
     * For app developing, it doesn't need to set progress
     * @param progress
     */
    public void setProgress(int progress) {
        emaObject.setProgress(progress);
    }

	/**
	 * \~chinese
	 * 消息方向
     * @return 详见 {@link Direct}
	 * 
	 * \~english
	 * The message direction
     * @return See {@link Direct}
	 */
	public Direct direct() {
	    if (emaObject.direction() == EMADirection.SEND) {
	        return Direct.SEND;
	    } else {
	        return Direct.RECEIVE;
	    }
	}
	
	/**
	 * \~chinese
	 * 设置消息的方向
	 * @param dir 消息的方向，详见{@link Direct}
	 * 
	 * \~english
	 * Set message direction
	 * @param dir Message direction, see {@link Direct}
	 */
	public void setDirection(Direct dir) {
	    emaObject.setDirection(dir.ordinal());
	}

    /**
     * \~chinese
     * 获取会话id
     * @return  会话id
     *
     * \~english
     * Get conversation id
     * @return Conversation id
     */
	public String conversationId() {
	    return emaObject.conversationId();
	}

    static String self() {
        String myself = EMClient.getInstance().getCurrentUser();
        if (myself == null) {
            //currentUser is only set after login successfully
            //since it is not when not logedin, use last login user instead
            myself = EMSessionManager.getInstance().getLastLoginUser();
        }
        return myself;
    }

    private EMMessage(Parcel in) throws HyphenateException {
        String msgId = in.readString();
        EMMessage msg = EMClient.getInstance().chatManager().getMessage(msgId);
        if (msg == null) {
            throw new HyphenateException("EMMessage constructed from parcel failed");
        }
        emaObject = msg.emaObject;
    }

    /**
     * \~chinese
     * 获取消息包含的全部扩展字段，并以Map形式返回
     *
     * @return  消息的扩展字段，以Map形式返回，返回Entry.value的具体类型有Boolean, Integer, Long, Float, Double, String
     *          输入JsonObject的属性，map结构返回时属性是String
     *
     * \~english
     * Get message extension, return type is Map&#60;String, Object&#62;
     *
     * @return  Return message extension, return type is Map&#60;String, Object&#62;
     *          Object can be Boolean, Integer, Long, Float, Double, String
     *          If the input is JsonObject or JsonArray, which use setAttribute(String attribute,
     *          JSONObject json) to passed in, the Map.Entry.value type is String
     */
    public Map<String, Object> ext() {
        return emaObject.ext();
    }
}
