/************************************************************
  *  * EaseMob CONFIDENTIAL 
  * __________________ 
  * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
  *  
  * NOTICE: All information contained herein is, and remains 
  * the property of EaseMob Technologies.
  * Dissemination of this information or reproduction of this material 
  * is strictly forbidden unless prior written permission is obtained
  * from EaseMob Technologies.
  */
package com.hyphenate.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.util.EMFileHelper;

import java.io.File;

/**
 * \~chinese
 * 图片消息体
 * <pre>
 *     EMImageMessageBody body = new EMImageMessageBody(imageFile);
 * </pre>
 *
 * \~english
 * Image message body class
 *
 * <pre>
 *     EMImageMessageBody body = new EMImageMessageBody(imageFile);
 * </pre>
 */
public class EMImageMessageBody extends EMFileMessageBody implements Parcelable {
	private static final String TAG = EMImageMessageBody.class.getSimpleName();
	
	private boolean sendOriginalImage = false;
    
    /**
     * \~chinese
     * 创建一个图片消息体
     * @param imageFile 图片文件
     *
     * \~english
     * Create a image message body
	 *
     * @param imageFile The image file
     */
    public EMImageMessageBody(File imageFile){
    	super(Uri.fromFile(imageFile), EMAMessageBody.EMAMessageBodyType_IMAGE);
    }

    /**
     * \~chinese
     * 创建一个图片消息体
     * @param imageUri 图片的资源标识符
     *
     * \~english
     * Create a image message body
	 *
     * @param imageUri The image uri
     */
    public EMImageMessageBody(Uri imageUri){
    	super(imageUri, EMAMessageBody.EMAMessageBodyType_IMAGE);
    	setFileName(EMFileHelper.getInstance().getFilename(imageUri));
    }

	/**
	 * \~chinese
	 * 创建一个图片消息体，包含附件和缩略图
	 *
	 * @param imageFile 		图片文件
	 * @param thumbnailFile 	缩略图文件
	 *
	 * \~english
	 * Create a image message body with original image, and also upload thumbnail to server
	 *
	 * @param imageFile 		The original image file
	 * @param thumbnailFile		The thumbnail image file
	 */
    public EMImageMessageBody(File imageFile, File thumbnailFile) {
        super(Uri.fromFile(imageFile), EMAMessageBody.EMAMessageBodyType_IMAGE);
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(thumbnailFile == null ?
				"" : Uri.fromFile(thumbnailFile).toString());
    }

	/**
	 * \~chinese
	 * 创建一个图片消息体，包含附件和缩略图
	 *
	 * @param imageLocalPath 	图片uri
	 * @param thumbnailPath 	缩略图uri
	 *
	 * \~english
	 * create a image message body with original image, but also upload thumbnail to server
	 *
	 * @param imageLocalPath 	The original image uri
	 * @param thumbnailPath		The thumbnail image uri
	 */
    public EMImageMessageBody(Uri imageLocalPath, Uri thumbnailPath) {
        super(imageLocalPath, EMAMessageBody.EMAMessageBodyType_IMAGE);
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(thumbnailPath != null ?
				thumbnailPath.toString() : "");
        setFileName(EMFileHelper.getInstance().getFilename(imageLocalPath));
    }

    public EMImageMessageBody(EMAImageMessageBody body) {
    	super(body);
	}
    
    /**
     * create EMImageMessageBody from incoming message
	 *
     * @param fileName
     * @param remoteUrl
     * @param thumbnailUrl
     */
    EMImageMessageBody(String fileName, String remoteUrl, String thumbnailUrl) {
    	super((Uri) null, EMAMessageBody.EMAMessageBodyType_IMAGE);
    	emaObject = new EMAImageMessageBody((Uri) null, (Uri) null);
    	((EMAImageMessageBody)emaObject).setDisplayName(fileName);
    	((EMAImageMessageBody)emaObject).setRemotePath(remoteUrl);
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    /**
     * \~chinese
     * 获取缩略图的服务器地址
     * 如果设置了{@link EMOptions#setAutoDownloadThumbnail(boolean)}，接受消息时，
	 * 接收方会自动根据thumbnailRemotePath下载缩略图，存储到本地
     * @return 缩略图的服务器地址
     * 
     * \~english
     * Server url path of thumbnail
     * If has set {@link EMOptions#setAutoDownloadThumbnail(boolean)}, image receiver will
	 * get thumbnail stored under thumbnailRemotePath after a successful download automatically
     * @return The thumbnail's remote url
     */
    public String getThumbnailUrl() {
		return ((EMAImageMessageBody) emaObject).thumbnailRemotePath();
    }
    
    /**
     * \~chinese
     * 设置缩略图的服务器路径
     * @param thumbnailUrl	缩略图的服务器路径
     * 
     * \~english
     * Set the thumbnail's remote url
     * @param thumbnailUrl The thumbnail's remote url
     */
    public void setThumbnailUrl(String thumbnailUrl) {
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    public String toString() {
        return "image: " + ((EMAImageMessageBody)emaObject).displayName() +
				", localurl: " + ((EMAImageMessageBody)emaObject).getLocalUrl() +
				", remoteurl: " + ((EMAImageMessageBody)emaObject).getRemoteUrl() +
				", thumbnail: " + ((EMAImageMessageBody)emaObject).thumbnailRemotePath();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        //members from FileMessageBody
        dest.writeString(((EMAImageMessageBody)emaObject).displayName());
        dest.writeString(((EMAImageMessageBody)emaObject).getLocalUrl());
        dest.writeString(((EMAImageMessageBody)emaObject).getRemoteUrl());
        dest.writeString(((EMAImageMessageBody)emaObject).thumbnailRemotePath());
        
        dest.writeInt(((EMAImageMessageBody)emaObject).width());
        dest.writeInt(((EMAImageMessageBody)emaObject).height());
        
    }
    
    public static final Parcelable.Creator<EMImageMessageBody> CREATOR
            = new Parcelable.Creator<EMImageMessageBody>() {
        public EMImageMessageBody createFromParcel(Parcel in) {
            return new EMImageMessageBody(in);
        }

        public EMImageMessageBody[] newArray(int size) {
            return new EMImageMessageBody[size];
        }
    };

    private EMImageMessageBody(Parcel in) {
    	super((Uri) null, EMAMessageBody.EMAMessageBodyType_IMAGE);
        //members from FileMessageBody
    	((EMAImageMessageBody)emaObject).setDisplayName(in.readString());
    	((EMAImageMessageBody)emaObject).setLocalPath(in.readString());
    	((EMAImageMessageBody)emaObject).setRemotePath(in.readString());
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(in.readString());
		int width = in.readInt();
		int height = in.readInt();
		((EMAImageMessageBody) emaObject).setSize(width, height);
    }

	/**
	 * \~chinese
	 * 设置缩略图的访问令牌
	 * @param secret 缩略图的访问令牌
	 *
	 * \~english
	 * Set the thumbnail's token
	 * @param secret Thumbnail's token
	 */
	public void setThumbnailSecret(String secret) {
    	((EMAImageMessageBody)emaObject).setThumbnailSecretKey(secret);
    }

	/**
	 * \~chinese
	 * 获取缩略图的访问令牌
	 * @return  缩略图的访问令牌
	 *
	 * \~english
	 * Get the thumbnail's token
	 * @return  Thumbnail's token
	 */
    public String getThumbnailSecret() {
    	return ((EMAImageMessageBody)emaObject).thumbnailSecretKey();
    }
    
    /**
     * \~chinese
     * 设置发送图片时，是否发送原图，默认是发送压缩过后的图(图片超过100k会被压缩)
     * @param sendOriginalImage	是否发送原图
     * 
     * \~english
     * Set whether to send original image when sending image,
     * the default is to send the figure after the compression(image greater than 100k will be compress)
     * @param sendOriginalImage Whether to send original image
     */
    public void setSendOriginalImage(boolean sendOriginalImage) {
    	this.sendOriginalImage = sendOriginalImage;
    }
    
    /**
     * \~chinese
     * 获取发送图片时，是否发送原图
     * @return	是否发送原图
     * 
     * \~english
     * Get whether to send original image
	 * @return Whether to send original image
     */
    public boolean isSendOriginalImage(){
    	return sendOriginalImage;
    }
    
    /**
     * \~chinese
     * 获取图片的宽度
     * @return 图片的宽度
     * 
     * \~english
     * Get image width
     * @return Image width
     */
    public int getWidth() {
		return ((EMAImageMessageBody)emaObject).width();
	}

    /**
     * \~chinese
     * 获取图片的高度
     * @return 图片的高度
     * 
     * \~english
     * Get image height
     * @return Image height
     */
	public int getHeight() {
		return ((EMAImageMessageBody)emaObject).height();
	}
	
	/**
     * \~chinese
     * 获取图片文件名字
     * @return 图片文件名字
     * 
     * \~english
     * Get image file name
     * @return Image file name
     */
	public String getFileName() {
		return ((EMAImageMessageBody)emaObject).displayName();
	}
	
	void setSize(int width, int height) {
		((EMAImageMessageBody)emaObject).setSize(width, height);
	}

	/**
	 * \~chinese
	 * 获取图片文件大小
	 * @return 图片文件大小
	 *
	 * \~english
	 * Get the image file length
	 * @return Image file length
	 */
	public long getFileSize() {
		return ((EMAImageMessageBody)emaObject).fileLength();
	}
	
	/**
     * \~chinese
     * 获取缩略图本地路径或者资源标识符的字符串类型
     * @return 缩略图本地路径或者资源标识符的字符串类型
     *
     * \~english
     * Get local path of thumbnail or the String from of Uri
     * @return Local path of thumbnail or the String from of Uri
     */
	public String thumbnailLocalPath() {
		return EMFileHelper.getInstance().formatOutLocalUrl(((EMAImageMessageBody)emaObject).thumbnailLocalPath());
	}

	/**
     * \~chinese
     * 获取缩略图本地资源标识符（路径）
     * @return 缩略图本地资源标识符（路径）
     *
     * \~english
     * Get local Uri of image thumbnail
     * @return Local Uri of image thumbnail
     */
	public Uri thumbnailLocalUri() {
		return EMFileHelper.getInstance().formatOutUri(((EMAImageMessageBody)emaObject).thumbnailLocalPath());
	}

	/**
     * \~chinese
     * 设置缩略图本地路径
	 * @param localPath	设置缩略图的本地路径
     *
     * \~english
     * Set local path of thumbnail
	 * @param localPath Local path of thumbnail
     */
    public void setThumbnailLocalPath(String localPath) {
		((EMAImageMessageBody)emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localPath));
    }

	/**
     * \~chinese
     * 设置缩略图本地路径
	 * @param localPath	设置缩略图的本地路径
     *
     * \~english
     * Set local Uri of thumbnail
	 * @param localPath	Local Uri of thumbnail
     */
    public void setThumbnailLocalPath(Uri localPath) {
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localPath));
    }

	/**
	 * \~chinese
	 * 设置缩略图的尺寸
	 * @param width 	缩略图的宽度
	 * @param height 	缩略图的高度
	 *
	 * \~english
	 * Set the width and height of the thumbnail image
	 * @param width		The width of thumbnail
	 * @param height	The height of thumbnail
	 */
    public void setThumbnailSize(int width, int height) {
		((EMAImageMessageBody)emaObject).setThumbnailSize(width, height);
	}

	/**
	 * \~chinese
	 * 获取缩略图的下载状态
	 * @return EMDownloadStatus
	 * 
	 * \~english
	 * Get the download status of the thumbnail
	 * @return EMDownloadStatus
	 */
	public EMDownloadStatus thumbnailDownloadStatus() {
	    EMADownloadStatus _status = ((EMAImageMessageBody)emaObject).thumbnailDownloadStatus();
	    switch (_status) {
	    case DOWNLOADING:
	        return EMDownloadStatus.DOWNLOADING;
	    case SUCCESSED:
	        return EMDownloadStatus.SUCCESSED;
	    case FAILED:
	        return EMDownloadStatus.FAILED;
	    case PENDING:
	        return EMDownloadStatus.PENDING;
	    }
	    return EMDownloadStatus.SUCCESSED;
	}
	
	void setThumbnailDownloadStatus(EMDownloadStatus status) {
		((EMAImageMessageBody)emaObject).setThumbnailDownloadStatus(EMADownloadStatus.valueOf(status.name()));
    }
}
