/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.EMGroupChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMACallback;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAGroup;
import com.hyphenate.chat.adapter.EMAGroup.EMGroupLeaveReason;
import com.hyphenate.chat.adapter.EMAGroupManager;
import com.hyphenate.chat.adapter.EMAGroupManagerListener;
import com.hyphenate.chat.adapter.EMAGroupSetting;
import com.hyphenate.chat.adapter.EMAMucShareFile;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;


/**
 * \~chinese
 * 群组管理, 用于管理群组的创建，删除，成员管理等操作
 *
 * \~english
 * Group management is used to manage group creation, deletion, member management and other operations
 */
public class EMGroupManager {

    /**
     * \~chinese
	 * 群组类型枚举类
	 *
	 * \~english
	 * Group style menu
     */
	public enum EMGroupStyle {

		/**
		 * \~chinese
		 * 私有群组，创建完成后，只允许Owner邀请用户加入
		 *
		 * \~english
		 * Private groups, after created, only allowing owner invite users to join
		 */
		EMGroupStylePrivateOnlyOwnerInvite,

		/**
		 * \~chinese
		 * 私有群组，创建完成后，只允许Owner和群成员邀请用户加入
		 *
		 * \~english
		 * Private groups, after created, only allowing owner and members invite users to join
		 */
		EMGroupStylePrivateMemberCanInvite,

		/**
		 * \~chinese
		 * 公开群组，创建完成后，只允许Owner邀请用户加入; 非群成员用户需发送入群申请，Owner同意后才能入组
		 *
		 * \~english
		 * Public groups, after created, only allowing owner invite users to join;
		 * Non-group members into the group must send an application to the owner after the Owner agree
		 */
		EMGroupStylePublicJoinNeedApproval,

		/**
		 * \~chinese
		 * 公开群组，创建完成后，允许非群组成员加入，不需要Owner同意
		 *
		 * \~english
		 * Public groups, after created, to allow non-members to join the group, does not require owner agrees
		 */
		EMGroupStylePublicOpenJoin
	}

	EMAGroupManager emaObject;

	private static String TAG = "group";

	List<EMGroupChangeListener> groupChangeListeners;

	EMAGroupManagerListener listenerImpl = new EMAGroupManagerListener() {

		@Override
		public void onReceiveInviteFromGroup(String groupId, String inviter, String inviteMessage) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationReceived(groupId, /*groupName*/ "", inviter, inviteMessage);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		/**
		 * \brief Callback user when the user accept to join the group.
		 *
		 * @param  invitee The group that invite the user.
		 * @return NA
		 */
		@Override
		public void onReceiveInviteAcceptionFromGroup(EMAGroup group, String invitee) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationAccepted(group.groupId(), invitee, "");
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		/**
		 * \brief Callback user when the user decline to join the group.
		 *
		 * @param  invitee The group that invite the user.
		 * @param  reason User's decline reason.
		 * @return NA
		 */
		@Override
		public void onReceiveInviteDeclineFromGroup(EMAGroup group, String invitee, String reason) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationDeclined(group.groupId(), invitee, "");
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAutoAcceptInvitationFromGroup(EMAGroup group, String inviter, String inviteMessage) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAutoAcceptInvitationFromGroup(group.groupId(), inviter, inviteMessage);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onLeaveGroup(EMAGroup group, int reason) {
			EMClient.getInstance().chatManager().caches.remove(group.groupId());
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						if (reason == EMGroupLeaveReason.BE_KICKED.ordinal()) {
							listener.onUserRemoved(group.groupId(), group.groupSubject());
						} else {
							listener.onGroupDestroyed(group.groupId(), group.groupSubject());
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveJoinGroupApplication(EMAGroup group, String from, String message) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinReceived(group.groupId(), group.groupSubject(), from, message);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveAcceptionFromGroup(EMAGroup group) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinAccepted(group.groupId(), group.groupSubject(), group.getOwner());
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveRejectionFromGroup(String groupId, String reason) {
			EMGroup group = EMGroupManager.this.getGroup(groupId);
			String groupName = group == null ? "" : group.groupSubject();
			String decliner = group == null ? "" : group.getOwner();

			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinDeclined(groupId, groupName, decliner, reason);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateMyGroupList(List<EMAGroup> groups) {

		}

		@Override
		public void onAddMutesFromGroup(EMAGroup group, final List<String> muteMembers, long muteExpire) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListAdded(group.groupId(), muteMembers, muteExpire);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveMutesFromGroup(EMAGroup group, final List<String> banPostList) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListRemoved(group.groupId(), banPostList);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

        public void onWhiteListAdded(EMAGroup group, List<String> members) {
            synchronized (groupChangeListeners) {
                try {
                    for(EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onWhiteListAdded(group.groupId(), members);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        public void onWhiteListRemoved(EMAGroup group, List<String> members) {
            synchronized (groupChangeListeners) {
                try {
                    for(EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onWhiteListRemoved(group.groupId(), members);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        public void onAllMemberMuteStateChanged(EMAGroup group, boolean isMuted) {
            synchronized (groupChangeListeners) {
                try {
                    for(EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onAllMemberMuteStateChanged(group.groupId(), isMuted);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

		@Override
		public void onAddAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAdminAdded(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAdminRemoved(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAssignOwnerFromGroup(EMAGroup group, String newOwner, String oldOwner) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onOwnerChanged(group.groupId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onMemberJoined(EMAGroup group, String member) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMemberJoined(group.groupId(), member);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onMemberExited(EMAGroup group, String member) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMemberExited(group.groupId(), member);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAGroup group, String announcement) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAnnouncementChanged(group.groupId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUploadShareFileFromGroup(EMAGroup group, EMAMucShareFile shareFile) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onSharedFileAdded(group.groupId(), new EMMucSharedFile(shareFile));
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onDeleteShareFileFromGroup(EMAGroup group, String fileId) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onSharedFileDeleted(group.groupId(), fileId);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	};

	EMClient mClient;
	EMGroupManager(EMClient client, EMAGroupManager groupManager) {
		emaObject = groupManager;
		mClient = client;
		groupChangeListeners = Collections.synchronizedList(new ArrayList<EMGroupChangeListener>());
		emaObject.addListener(listenerImpl);
		EMClient.getInstance().chatManager();
	}

	/**
	 * \~chinese
	 * 获取当前(内存)用户的所有群组
	 * 在调用此方法之前，可以在适当的时机调用{@link #loadAllGroups()}，提前从数据将群组加载到内存中；
	 * 如果没有调用过{@link #loadAllGroups()}，首次会从数据库中加载群组数据，之后再从内存中加载
	 *
	 * @return 群组列表
	 *
	 * \~english
	 * Get all groups from cache
	 * Before call this method, we suggest that you can call {@link #loadAllGroups()} at the appropriate time to load the
	 * group data from the DB into memory; If have not called {@link #loadAllGroups()}, this method will load data from database first
	 * and then load from cache
	 *
	 * @return Group List
	 */
	public List<EMGroup> getAllGroups() {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return Collections.unmodifiableList(result);
	}

	/**
	 * \~chinese
	 * 根据群组ID，从内存中获得群组对象
	 *
	 * @param groupId	群组ID
	 * @return 			返回群组对象。如果群组不存在，返回null
	 *
	 * \~english
	 * Get local group from cache by group id
	 *
	 * @param groupId	Group id to get
	 * @return 			Return EMGroup instance, return null if group is not exist
	 */
	public EMGroup getGroup(String groupId) {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		for (EMAGroup group : groups) {
			if (group.groupId().equals(groupId)) {
				return new EMGroup(group);
			}
		}
		return null;
	}

	/**
	 * \~chinese
	 * 创建一个群组
	 * 群组创建成功后，会更新内存及数据库中的数据，多端多设备会收到相应的通知事件，在接受到事件后，会将群组更新到内存及数据库中，
	 * 可通过设置{@link com.hyphenate.EMMultiDeviceListener}监听相关事件，事件回调函数为
	 * {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)},
	 * 第一个参数为事件，建群事件为{@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE}
	 * 异步方法见{@link #asyncCreateGroup(String, String, String[], String, EMGroupOptions, EMValueCallBack)}
	 *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupName     		群组的名字，非必要，不设置传null
     * @param desc          		群组描述，非必要，不设置传null
	 * @param allMembers    		群成员数组,不需要群主ID，非必要，不可为null
	 * @param reason        		邀请群成员加入时的邀请信息，非必要，不设置传null
	 * @param option        		群的设置，非必要，不可为null。详见{@link EMGroupOptions}
	 *                      		可以设置如下参数：
	 *                      		群最大成员数，默认值为200；
	 *                      		群类型，详见{@link EMGroupManager.EMGroupStyle}，默认为{@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite}；
	 *                      		邀请进群是否需要对方同意，默认为false，即直接被邀请后进群；
	 *                      		还可以设置群详情扩展
	 *
	 * @return 						返回创建成功的群对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Create a group
	 * After group is successful created, the data in memory and database will be updated; Multi-device will receive the notification event and
	 * then update the data in memory and database. You can set {@link com.hyphenate.EMMultiDeviceListener} to monitor the event, the callback function
	 * is {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)}, the first parameter is the event which
	 * is {@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE}
	 * Asynchronously method see {@link #asyncCreateGroup(String, String, String[], String, EMGroupOptions, EMValueCallBack)}
     *
     * Synchronization method will block the current thread
     *
	 * @param groupName     		Group name, not necessary, pass null if not set
     * @param desc          		Group description, not necessary, pass null if not set
	 * @param allMembers    		Array of members, no need include the owner of this group, not necessary, not null
	 * @param reason        		Invitation information when inviting members to join, not necessary, pass null if not set
	 * @param option        		Options when creating a group, not necessary, not null. See {@link EMGroupOptions}
	 *                      		Can set the following parameters:
	 *                      		Maximum number of group members, default is 200;
	 *                      		Group style, see {@link EMGroupStyle}, default is {@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite};
	 *                      		If it requires the content of invitee, default is false, invitee will auto join the group;
	 *                      		Also can set group ext
	 *
	 * @return 						Return the EMGroup instance created successful
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
	 */
	public EMGroup createGroup(String groupName, String desc, String[] allMembers, String reason, EMGroupOptions option) throws HyphenateException{
		int style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;

		switch(option.style) {
			case EMGroupStylePrivateOnlyOwnerInvite:
				style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;
				break;

			case EMGroupStylePrivateMemberCanInvite:
				style = EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE;
				break;

			case EMGroupStylePublicJoinNeedApproval:
				style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL;
				break;

			case EMGroupStylePublicOpenJoin:
				style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN;
				break;
		}

		return createGroup(style, groupName, desc, allMembers, option.maxUsers, reason, option.inviteNeedConfirm, option.extField);
	}

	/**
	 * \~chinese
	 * 创建一个群组
     * 群组创建成功后，会更新内存及数据库中的数据，多端多设备会收到相应的通知事件，在接受到事件后，会将群组更新到内存及数据库中，
	 * 可通过设置{@link com.hyphenate.EMMultiDeviceListener}监听相关事件，事件回调函数为
	 * {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)},
	 * 第一个参数为事件，建群事件为{@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE}
	 *
	 * 异步方法
	 *
	 * @param groupName     群组的名字，非必要，不设置传null
     * @param desc          群组描述，非必要，不设置传null
	 * @param allMembers    群成员数组,不需要群主ID，非必要，不可为null
     * @param reason        邀请群成员加入时的邀请信息，非必要，不设置传null
	 * @param option        群的设置，非必要，不可为null。详见{@link EMGroupOptions}
	 *                      可以设置如下参数：
	 *                      群最大成员数，默认值为200；
	 *                      群类型，详见{@link EMGroupManager.EMGroupStyle}，默认为{@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite}；
	 *                      邀请进群是否需要对方同意，默认为false，即直接被邀请后进群；
	 *                      还可以设置群详情扩展
	 *
     * @param callback		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，返回创建成功的群对象;
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Create a group
     * After group is successful created, the data in memory and database will be updated; Multi-device will receive the notification event and
	 * then update the data in memory and database. You can set {@link com.hyphenate.EMMultiDeviceListener} to monitor the event, the callback function
	 * is {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)}, the first parameter is the event which
	 * is {@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE}
	 *
	 * Asynchronously method
	 *
	 * @param groupName     Group name, not necessary, pass null if not set
     * @param desc          Group description, not necessary, pass null if not set
	 * @param allMembers    Array of members, no need include the owner of this group, not necessary, not null
	 * @param reason        Invitation information when inviting members to join, not necessary, pass null if not set
	 * @param option        Options when creating a group, not necessary, not null. See {@link EMGroupOptions}
	 *                      Can set the following parameters:
	 *                      Maximum number of group members, default is 200;
	 *                      Group style, see {@link EMGroupStyle}, default is {@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite};
	 *                      If it requires the content of invitee, default is false, invitee will auto join the group;
	 *                      Also can set group ext
	 *
     * @param callback		Result callback, success call {@link EMValueCallBack#onSuccess(Object)}, return the EMGroup instance created successful
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
     */
	public void asyncCreateGroup(final String groupName,
								 final String desc,
								 final String[] allMembers,
								 final String reason,
								 final EMGroupOptions option,
								 final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = createGroup(groupName, desc, allMembers, reason, option);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

    // Private method
    /**
     *  Create a group
     *
     *  @param style                Public or private
     *  @param groupName            Group name
     *  @param desc                 Group description
     *  @param allMembers           Group members, exclude owner/creator
     *  @param maxUsers             max group member capacity
     *  @param reason               Invitation message
     *  @param inviteNeedConfirm    if need invitation confirmation
     *  @param extension            group extension info
     *  @return                     Created group
     */
    private EMGroup createGroup(int style,
                                String groupName,
                                String desc,
                                String[] allMembers,
                                int maxUsers,
                                String reason,
                                boolean inviteNeedConfirm,
                                String extension) throws HyphenateException {
        EMAGroupSetting setting = new EMAGroupSetting(style, maxUsers, inviteNeedConfirm, extension);
        List<String> newMembers = new ArrayList<String>();
        Collections.addAll(newMembers, allMembers);
        EMAError error = new EMAError();
        EMAGroup group = emaObject.createGroup(groupName, desc, reason, setting, newMembers, inviteNeedConfirm, error);
        handleError(error);
        return new EMGroup(group);
    }

	/**
	 * \~chinese
	 * 从数据库中加载所有群组
	 *
	 * \~english
	 * load all local groups from database
	 */
	public synchronized void loadAllGroups() {
		emaObject.loadAllMyGroupsFromDB();
	}

	/**
	 * \~chinese
	 * 解散群组
	 * 需要群组owner权限
	 * 异步方法见{@link #asyncDestroyGroup(String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Destroy group
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncDestroyGroup(String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void destroyGroup(final String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyGroup(groupId, error);
		EMClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 解散群组
	 * 需要群组owner权限
	 * 同步方法见{@link #destroyGroup(String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
     * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
     *
	 * \~english
	 * Destroy group
	 * Owner‘s authority is required
	 * Synchronization method see {@link #destroyGroup(String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
     * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
     */
	public void asyncDestroyGroup(final String groupId, final EMCallBack callback){
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyGroup(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 向群组中添加新的成员
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncAddUsersToGroup(String, String[], EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId 				群组的ID
	 * @param newmembers			待添加的新成员数组
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Add users to the group
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncAddUsersToGroup(String, String[], EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @param newmembers			Array of new members to add
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void addUsersToGroup(String groupId, String[] newmembers) throws HyphenateException {
		addUsersToGroup(groupId, newmembers, "welcome");
	}

	/**
	 * \~chinese
	 * 向群组中添加新的成员
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncAddUsersToGroup(String, String[], EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId               群组的ID
	 * @param newmembers            待添加的新成员数组
	 * @param welcomeMessage   		邀请信息
	 * @throws HyphenateException   错误信息详见{@link EMError}
	 *
	 * \~english
	 * Add users to the group
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncAddUsersToGroup(String, String[], EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId                Group id
	 * @param newmembers             Array of new members to add
	 * @param welcomeMessage		 welcome message
	 * @throws HyphenateException    Error info, see {@link EMError}
	 */
	public void addUsersToGroup(String groupId, String[] newmembers, String welcomeMessage) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> newmembers2 = new ArrayList<String>();
		Collections.addAll(newmembers2, newmembers);
		emaObject.addGroupMembers(groupId, newmembers2, welcomeMessage, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 向群组中添加新的成员
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #addUsersToGroup(String, String[])}
	 *
	 * 异步方法
	 *
	 * @param groupId               群组的ID
	 * @param newmembers            待添加的新成员数组
	 * @param callback				结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 								失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Add users to the group
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #addUsersToGroup(String, String[])}
	 *
	 * Asynchronously method
	 *
	 * @param groupId		Group id
	 * @param newmembers	Array of new members to add
	 * @param callback		Result callback, success call {@link EMCallBack#onSuccess()},
	 * 						failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncAddUsersToGroup(final String groupId,
									 final String[] newmembers,
									 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					addUsersToGroup(groupId, newmembers);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从群组中删除成员
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncRemoveUserFromGroup(String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组的ID
	 * @param username				待删除成员的用户名
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Remove member from group
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncRemoveUserFromGroup(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @param username   			User id to be removed
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void removeUserFromGroup(final String groupId, final String username) throws HyphenateException {
		List<String> members = new ArrayList<String>();
		EMAError error = new EMAError();
		members.add(username);
		emaObject.removeGroupMembers(groupId, members, error);
		handleError(error);
		emaObject.fetchGroupSpecification(groupId, error, true);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 从群组中删除成员
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #removeUserFromGroup(String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组的ID
	 * @param username	待删除成员的用户名
     * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
     *
	 * \~english
	 * Remove member from group
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #removeUserFromGroup(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId   Group id
	 * @param username  User id to be removed
     * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
     */
	public void asyncRemoveUserFromGroup(final String groupId,
										 final String username,
										 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					removeUserFromGroup(groupId, username);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从群组中删除成员
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #removeUsersFromGroup(String, List)}}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @param members				待删除成员的用户名集合
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Remove members from group
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncRemoveUsersFromGroup(String, List, EMCallBack)}}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @param members   			User ids to be removed
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void removeUsersFromGroup(final String groupId, final List<String> members) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.removeGroupMembers(groupId, members, error);
		handleError(error);
		emaObject.fetchGroupSpecification(groupId, error, true);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 从群组中删除成员
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #removeUsersFromGroup(String, List)}
	 *
	 * 异步方法
	 *
	 * @param groupId   群组ID
	 * @param members   待删除成员的用户名集合
     * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					败时回调{@link EMCallBack#onError(int, String)}
     *
	 * \~english
	 * Remove members from group
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #removeUsersFromGroup'(String, List)'}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param members   User ids to be removed
     * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
     */
	public void asyncRemoveUsersFromGroup(final String groupId,
										 final List<String> members,
										 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					removeUsersFromGroup(groupId, members);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 当前登录用户退出群组
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Current joined user exit group
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void leaveGroup(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.leaveGroup(groupId, error);
		EMClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 当前登录用户退出群组
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
     * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
     *
	 * \~english
	 * Current joined user exit group
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
     * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
     */
	public void asyncLeaveGroup(final String groupId, final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					leaveGroup(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息
	 * 不获取成员，如需获取成员，使用 {@link #fetchGroupMembers(String, String, int)}
	 * 异步方法见{@link #asyncGetGroupFromServer(String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @return EMGroup				返回群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Get information of group from server
	 * Do not get members, use {@link #fetchGroupMembers(String, String, int)} if want ot get members
	 * Asynchronously method see {@link #asyncGetGroupFromServer(String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @return EMGroup				Return EMGroup instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup getGroupFromServer(String groupId) throws HyphenateException {
		if(groupId == null || groupId.isEmpty())
			throw new HyphenateException(EMError.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息
	 * 参数fetchMembers为true的时候，获取群成员，如需获取更多成员，使用 {@link #fetchGroupMembers(String, String, int)}
	 * 只有群成员才能获取群组的成员列表
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @param fetchMembers 			是否获取成员列表，默认取200人的成员列表
	 * @return EMGroup 				返回更新后的群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Get information of group from server
	 * When parameter of fetchMembers is true, fetch group members from server, maximum count is 200, if want to get more members,
	 * use {@link #fetchGroupMembers(String, String, int)}
	 * Only members in group can get the member list of group
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @param fetchMembers 			Whether need fetch group members, if need fetch members, default action fetch 200 members
	 * @return EMGroup				Return EMGroup instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup getGroupFromServer(String groupId, boolean fetchMembers) throws HyphenateException {
		if(groupId == null || groupId.isEmpty())
			throw new HyphenateException(EMError.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, fetchMembers);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息
	 * 同步方法见{@link #getGroupFromServer(String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
     * @param callback 	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get information of group from server
	 * Synchronization method see {@link #getGroupFromServer(String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
     * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
     */
	public void asyncGetGroupFromServer(final String groupId,
										final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = getGroupFromServer(groupId);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器端获取当前用户的所有群组 （此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 * @return 群组列表
	 * @throws HyphenateException
	 *
	 * \~english
	 * this api will get groups from remote server and update local groups, this method only return the list of group,
	 * not include member list in group, use {@link #getGroupFromServer(String groupId)} if you want it.
	 * @return array of group list 
	 * @throws HyphenateException
	 */
	synchronized List<EMGroup> getGroupsFromServer() throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroups(error);
		handleError(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 分页从服务器端获取当前用户的所有群组 （此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 *
	 * @param pageIndex 页码号
	 * @param pageSize  每页获取群组包含群组数量
	 * @return 群组列表
	 * @throws HyphenateException
	 *
	 * \~english
	 * this api will get groups from remote server and update local groups, this method only return the list of group,
	 * not include member list in group, use {@link #getGroupFromServer(String groupId)} if you want it.
	 *
	 * @param pageIndex  page index
	 * @param pageSize each page contain's group size
	 * @return array of group list
	 * @throws HyphenateException
	 */
	synchronized List<EMGroup> getGroupsFromServer(int pageIndex, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroupsWithPage(pageIndex, pageSize, error);
		handleError(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return result;
	}


	/**
	 * \~chinese
	 * 从服务器端获取当前用户的所有群组 （此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 *
	 * @param callback  包含群组列表
	 *
	 * \~english
	 * this api will get groups from remote server and update local groups, this method only return the list of group,
	 * not include member list in group, use {@link #getGroupFromServer(String groupId)} if you want it.
	 *
	 * @param callback  contains array of group list
	 */
	void asyncGetGroupsFromServer(final EMValueCallBack<List<EMGroup>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<EMGroup> groups = getGroupsFromServer();
					callback.onSuccess(groups);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器端获取当前用户的所有群组
	 * 此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 * 异步方法见{@link #asyncGetJoinedGroupsFromServer(EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @return 						当前用户已加入的群组列表
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Fetch all joined groups from server
	 * Only return the group list, not include member list in group, use
	 * {@link #getGroupFromServer(String groupId)} if you want it
	 * Asynchronously method see {@link #asyncGetJoinedGroupsFromServer(EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @return 						Group list of the current user has joined
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public synchronized List<EMGroup> getJoinedGroupsFromServer() throws HyphenateException{
		return getGroupsFromServer();
	}

	/**
	 * \~chinese
	 * 分页从服务器端获取当前用户已加入的群组
	 * 此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 * 异步方法见{@link #asyncGetJoinedGroupsFromServer(int, int, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param pageIndex 			当前要拉取的页数，从1开始
	 * @param pageSize				每页拉取的数量
	 * @return 						返回分页获取的群组列表
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Fetch joined groups from server by page
	 * Only return the group list, not include member list in group
	 * {@link #getGroupFromServer(String groupId)} if you want it
	 * Asynchronously method see {@link #asyncGetJoinedGroupsFromServer(int, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param pageIndex 			Page number, start from 1
	 * @param pageSize				Size to be fetched per page
	 * @return 						Group list returned by page
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public synchronized List<EMGroup> getJoinedGroupsFromServer(int pageIndex, int pageSize) throws HyphenateException{
		return getGroupsFromServer(pageIndex, pageSize);
	}

	/**
	 * \~chinese
	 * 从服务器端获取当前用户的所有群组
	 * 此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 * 同步方法见{@link #getJoinedGroupsFromServer()}
	 *
	 * 异步方法
	 *
	 * @param callback  结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，返回已加入的群组列表
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch all joined groups from server
	 * Only return the group list, not include member list in group, use
	 * {@link #getGroupFromServer(String groupId)} if you want it
	 * Synchronization method see {@link #getJoinedGroupsFromServer()}
	 *
	 * Asynchronously method
	 *
	 * @param callback 	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the group list has joined;
	 * 					failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncGetJoinedGroupsFromServer(final EMValueCallBack<List<EMGroup>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<EMGroup> groups = getJoinedGroupsFromServer();
					callback.onSuccess(groups);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 分页从服务器端获取当前用户已加入的群组
	 * 此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 * 同步方法见{@link #getJoinedGroupsFromServer(int, int)}
	 *
	 * 异步方法
	 *
	 * @param pageIndex	当前要拉取的页数，从1开始
	 * @param pageSize	每页拉取的数量
	 * @param callback 	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}， 返回分页获取的群组列表
	 *                  失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 *
	 * \~english
	 * Fetch joined groups from server by page
	 * Only return the group list, not include member list in group
	 * {@link #getGroupFromServer(String groupId)} if you want it
	 * Synchronization method see {@link #getJoinedGroupsFromServer(int, int)}
	 *
	 * Asynchronously method
	 *
	 * @param pageIndex	Page number, start from 1
	 * @param pageSize	Size to be fetched per page
     * @param callback 	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the group list by page;
	 * 					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncGetJoinedGroupsFromServer(final int pageIndex,
											   final int pageSize,
											   final EMValueCallBack<List<EMGroup>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<EMGroup> groups = getJoinedGroupsFromServer(pageIndex, pageSize);
					callback.onSuccess(groups);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 分页从服务器获取公开群组
	 * 异步方法见{@link #asyncGetPublicGroupsFromServer(int, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param pageSize				一次取的数量
	 * @param cursor				从这个游标位置开始取数据，首次获取数据时传null即可
	 * @return						分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的cursor以及群组list的对象，
	 * 								如果是最后一页，cursor返回结果为空字符串
	 * @throws HyphenateException	错误码详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Get public groups from server by page
	 * Asynchronously method see {@link #asyncGetPublicGroupsFromServer(int, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param pageSize  			The count fetched a time
	 * @param cursor    			The cursor position to fetch data, use null if first call
	 * @return  		    		Fetch result, see {@link EMCursorResult}, used to get next page and the group list,
	 * 								for the last page, EMCursorResult.getCursor() is an empty string.
	 * @throws HyphenateException	Error code, see {@link com.hyphenate.EMError}
	 */
	public EMCursorResult<EMGroupInfo> getPublicGroupsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMGroupInfo> result = emaObject.fetchPublicGroupsWithCursor(cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 分页从服务器获取公开群组
	 * 同步方法见{@link #getPublicGroupsFromServer(int, String)}
	 *
	 * 异步方法
	 *
	 * @param pageSize		一次取的数量
	 * @param cursor		从这个游标位置开始取数据，首次获取数据时传null即可
     * @param callback      结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      返回分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的cursor以及群组list的对象，
	 * 						如果是最后一页，cursor返回结果为空字符串；
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get public groups from server by page
	 * Synchronization method see {@link #getPublicGroupsFromServer(int, String)}
	 *
	 * Asynchronously method
	 *
	 * @param pageSize  	The count fetched a time
	 * @param cursor    	The cursor position to fetch data, use null if first call
	 * @param callback      Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      return result, see {@link EMCursorResult}, used to get next page and the group list,
	 * 						for the last page, EMCursorResult.getCursor() is an empty string;
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncGetPublicGroupsFromServer(final int pageSize,
											   final String cursor,
											   final EMValueCallBack<EMCursorResult<EMGroupInfo>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMCursorResult<EMGroupInfo> result = getPublicGroupsFromServer(pageSize, cursor);
					callback.onSuccess(result);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 当前登录用户加入公开群
	 * 如果是自由加入的公开群，直接进入群组；需要验证的公开群需要等待群主同意，详见{@link EMGroupStyle}
	 * 异步方法见{@link #asyncJoinGroup(String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Join the group by the group id
	 * If group's style is {@link EMGroupStyle#EMGroupStylePublicOpenJoin}，join it not require owner agree;
	 * If is {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}, need owner's agreement
	 * Asynchronously method see {@link #asyncJoinGroup(String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void joinGroup(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup _group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		if (_group.groupSetting() == null) {
			throw new HyphenateException();
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN) {
			emaObject.joinPublicGroup(groupId, error);
			handleError(error);
			return;
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL) {
			emaObject.applyJoinPublicGroup(groupId, mClient.getCurrentUser(), "hello", error);
			handleError(error);
			return;
		}
	}

	/**
	 * \~chinese
	 * 当前登录用户加入公开群
	 * 如果是自由加入的公开群，直接进入群组；需要验证的公开群需要等待群主同意，详见{@link EMGroupStyle}
	 * 同步方法见{@link #joinGroup(String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Join the group by the group id
	 * If group's style is {@link EMGroupStyle#EMGroupStylePublicOpenJoin}，join it not require owner agree;
	 * If is {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}, need owner's agreement
	 * Synchronization method see {@link #joinGroup(String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncJoinGroup(final String groupId,
							   final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					joinGroup(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 改变群组的名称
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncChangeGroupName(String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId 				需要改变名称的群组的id
	 * @param changedGroupName 		改变后的群组名称
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Change group name
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncChangeGroupName(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId 				Group id to be change name
	 * @param changedGroupName 		New group name
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void changeGroupName(String groupId, String changedGroupName) throws HyphenateException{
		EMAError error = new EMAError();
		emaObject.changeGroupSubject(groupId, changedGroupName, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 改变群组的名称
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #changeGroupName(String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId 			需要改变名称的群组的id
	 * @param changedGroupName 	改变后的群组名称
	 * @param callback			结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 							失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Change group name
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #changeGroupName(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 			Group id to be change name
	 * @param changedGroupName 	New group name
	 * @param callback			Result callback, success call {@link EMCallBack#onSuccess()},
	 * 							failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncChangeGroupName(final String groupId,
									 final String changedGroupName,
									 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					changeGroupName(groupId, changedGroupName);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改群描述
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncChangeGroupDescription(String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId 				群组ID
	 * @param changedDescription 	改变后的群描述
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Change group description
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncChangeGroupDescription(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId 				Group id
	 * @param changedDescription 	Changed group description
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void changeGroupDescription(String groupId, String changedDescription)
			throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.changeGroupDescription(groupId, changedDescription, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 修改群描述
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #changeGroupDescription(String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId 				群组ID
	 * @param changedDescription 	改变后的群描述
     * @param callBack              结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 	                   			失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Change group description. ONLY the group own can change group description
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #changeGroupDescription(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 				Group id
	 * @param changedDescription 	Changed group description
     * @param callBack				Result callback, success call {@link EMCallBack#onSuccess()},
	 *								failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncChangeGroupDescription(final String groupId,
											final String changedDescription,
											final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override public void run() {
				try {
					changeGroupDescription(groupId, changedDescription);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 接受加入群的邀请
	 * 异步方法见{@link #asyncAcceptInvitation(String, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @param inviter				邀请人ID
	 * @return 						用户已经同意邀请的群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Accept group's invitation
	 * Asynchronously method see {@link #asyncAcceptInvitation(String, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param  groupId 				Group ID
	 * @param  inviter 				Inviter
	 * @return 						The group user has accepted.
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup acceptInvitation(String groupId, String inviter) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.acceptInvitationFromGroup(groupId, inviter == null ? "" : inviter, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 接受加入群的邀请
	 * 同步方法见{@link #acceptInvitation(String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param inviter	邀请人ID
	 * @param callback  结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Accept group's invitation
	 * Synchronization method see {@link #acceptInvitation(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group ID
	 * @param inviter 	Inviter
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the updated group instance;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncAcceptInvitation(final String groupId,
									  final String inviter,
									  final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = acceptInvitation(groupId, inviter);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 拒绝加入群的邀请
	 * 异步方法见{@link #asyncDeclineInvitation(String, String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @param inviter				邀请人ID
	 * @param reason				拒绝的原因
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Decline group's invitation
	 * Synchronization method see {@link #asyncDeclineInvitation(String, String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId 				Group ID
	 * @param inviter 				Inviter
	 * @param reason				Declining message
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void declineInvitation(
			String groupId,
			String inviter,
			String reason) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.declineInvitationFromGroup(groupId, inviter == null ? "" : inviter, reason == null ? "" : reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 拒绝加入群的邀请
     * 同步方法见{@link #declineInvitation(String, String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param inviter	邀请人ID
	 * @param reason	拒绝的原因
	 * @param callback  结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Decline group's invitation
	 * Synchronization method see {@link #declineInvitation(String, String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group ID
	 * @param inviter 	Inviter
	 * @param reason	Declining message
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncDeclineInvitation(final String groupId,
									   final String inviter,
									   final String reason,
									   final EMCallBack callback) {

		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					declineInvitation(groupId, inviter, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 同意加群申请
     * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncAcceptApplication(String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param username 				申请人ID
	 * @param groupId  				群组ID
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Accept the application of the user to join this group
     * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncAcceptApplication(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param username 				Application user
	 * @param groupId  				Group id for application
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void acceptApplication(String username, String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.acceptJoinGroupApplication(groupId, username, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 同意加群申请
     * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #acceptApplication(String, String)}
	 *
	 * 异步方法
	 *
	 * @param username 	申请人ID
	 * @param groupId  	群组ID
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Accept the application of the user to join this group
     * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #acceptApplication(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param username 	Application user
	 * @param groupId  	Group id for application
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncAcceptApplication(final String username,
									   final String groupId,
									   final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					acceptApplication(username, groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 拒绝加群申请
     * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncDeclineApplication(String, String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param username 				申请人ID
	 * @param groupId  				群组ID
	 * @param reason   				拒绝的理由
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Decline the application of the user to join this group
     * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncDeclineApplication(String, String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param username 				Application user
	 * @param groupId  				Group id for application
	 * @param reason   				Message for decline
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void declineApplication(String username, String groupId, String reason) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.declineJoinGroupApplication(groupId, username, reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 拒绝加群申请
     * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #declineApplication(String, String, String)}
	 *
	 * 异步方法
	 *
	 * @param username 	申请人ID
	 * @param groupId  	群组ID
	 * @param reason   	拒绝的理由
     * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
     *
	 * \~english
	 * Decline the application of the user to join this group
     * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #declineApplication(String, String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param username 	Application user
	 * @param groupId  	Group id for application
	 * @param reason   	Message for decline
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncDeclineApplication(final String username,
                                        final String groupId,
                                        final String reason,
                                        final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					declineApplication(username, groupId, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 群成员邀请用户加入群组
     * 前提：群组类型是{@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}，即允许群成员邀请用户
	 * 异步方法见{@link #asyncInviteUser(String, String[], String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @param beInvitedUsernames	邀请用户数组
	 * @param reason				邀请原因
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Invite other users to join the group
	 * Note: The group style is {@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}, which allow group
	 * members to invite users
	 * Asynchronously method see {@link #asyncInviteUser(String, String[], String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param beInvitedUsernames   	Array of users to be invited
	 * @param reason               	Message for invitation
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void inviteUser(String groupId, String[] beInvitedUsernames, String reason) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		Collections.addAll(members, beInvitedUsernames);
		emaObject.addGroupMembers(groupId, members, reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 群成员邀请用户加入群组
	 * 前提：群组类型是{@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}，即允许群成员邀请用户
	 * 同步方法见{@link #inviteUser(String, String[], String)}
	 *
	 * 异步方法
     *
	 * @param groupId				群组ID
	 * @param beInvitedUsernames	邀请用户数组
	 * @param reason				邀请原因
	 * @param callback				结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 								失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Invite other users to join the group
	 * Note: The group style is {@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}, which allow group
	 * members to invite users
	 * Synchronization method see {@link #inviteUser(String, String[], String)}
	 *
	 * Asynchronously method
     *
	 * @param groupId				Group id
	 * @param beInvitedUsernames   	Array of users to be invited
	 * @param reason               	Message for invitation
	 * @param callback				Result callback, success call {@link EMCallBack#onSuccess()},
	 *								failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncInviteUser(final String groupId,
                                final String[] beInvitedUsernames,
                                final String reason,
                                final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					inviteUser(groupId, beInvitedUsernames, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 申请加入群组
	 * 前提：群组类型为{@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}，即需要需要验证的公开群
     * 异步方法见{@link #asyncApplyJoinToGroup(String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @param reason				申请加入的原因
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Apply to join the group
     * Note: Group style is {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}, which is the public group
	 * need to verify
	 * Asynchronously method see {@link #asyncApplyJoinToGroup(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @param reason    			Message for join application
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void applyJoinToGroup(String groupId, String reason) throws HyphenateException {
		String userName = mClient.getCurrentUser();
		EMAError error = new EMAError();
		emaObject.applyJoinPublicGroup(groupId, userName, reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 申请加入群组
	 * 前提：群组类型为{@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}，即需要需要验证的公开群
     * 同步方法见{@link #applyJoinToGroup(String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param reason    申请加入的原因
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Apply to join the group
     * Note: Group style is {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}, which is the public group
	 * need to verify
	 * Synchronization method see {@link #applyJoinToGroup(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param reason    Message for join application
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncApplyJoinToGroup(final String groupId,
                                      final String reason,
                                      final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					applyJoinToGroup(groupId, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 屏蔽群消息
	 * 还是群成员，但不再接收群消息
	 * 异步方法见{@link #asyncBlockGroupMessage(String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
     *
	 * @param groupId				群组ID
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 *\~english
	 * Set to disable receiving the group messages
     * Current user is still group member, but no longer receives group message
	 * Asynchronously method see {@link #asyncBlockGroupMessage(String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void blockGroupMessage(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.blockGroupMessage(groupId, error);
		handleError(error);
	}


	/**
	 * \~chinese
	 * 屏蔽群消息
	 * 还是群成员，但不再接收群消息
     * 同步方法见{@link #blockGroupMessage(String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 *\~english
	 * Set to disable receiving the group messages
     * Current user is still group member, but no longer receives group message
	 * Synchronization method see {@link #blockGroupMessage(String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncBlockGroupMessage(final String groupId, final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockGroupMessage(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 取消屏蔽群消息
	 * 异步方法见{@link #asyncUnblockGroupMessage(String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Set to enable to receive the group message again
	 * Asynchronously method see {@link #asyncUnblockGroupMessage(String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void unblockGroupMessage(String groupId) throws HyphenateException {
		EMLog.d(TAG, "try to unblock group msg:" + groupId);
		EMAError error = new EMAError();
		emaObject.unblockGroupMessage(groupId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 取消屏蔽群消息
     * 同步方法见{@link #unblockGroupMessage(String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Set to enable to receive the group message again
     * Synchronization method see {@link #unblockGroupMessage(String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncUnblockGroupMessage(final String groupId,
                                         final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockGroupMessage(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户加入到群组黑名单
	 * 被加入黑名单的用户无法加入群，无法收发此群的消息
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncBlockUser(String, String, EMCallBack)}
	 *
	 * v3.3.0后：被加入到黑名单的用户会被先移出群，再加入到黑名单中，之前版本没有移出群的动作
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId 				群组ID
	 * @param username 				待加入黑名单的用户名
     * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 *\~english
	 * Add user to group blacklist
	 * Set to block member who can not join group again and will be not allowed to receive any group messages
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncBlockUser(String, String, EMCallBack)}
	 *
	 * After v3.3.0: Member who be added to blacklist will be removed to group first, then add to blacklist; The previous version
	 * is not the action to remove from group
	 *
	 * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param username 				User to be blocked
     * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void blockUser(String groupId, String username) throws HyphenateException {
		EMLog.d(TAG, "block user for groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		String reason = "";
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.blockGroupMembers(groupId, members, error, reason);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 将用户加入到群组黑名单
	 * 被加入黑名单的用户无法加入群，无法收发此群的消息
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #blockUser(String, String)}
	 *
	 * 3.3.0后：被加入到黑名单的用户会被先移出群，再加入到黑名单中，之前版本没有移出群的动作
	 *
	 * 异步方法
     *
	 * @param groupId 	群组ID
	 * @param username 	待加入黑名单的用户名
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 *\~english
	 * Add user to group blacklist
	 * Set to block member who can not join group again and will be not allowed to receive any group messages
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #blockUser(String, String)}
     *
	 * After v3.3.0: Member who be added to blacklist will be removed to group first, then add to blacklist; The previous version
	 * is not the action to remove from group
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param username 	User to be blocked
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncBlockUser(final String groupId,
                               final String username,
                               final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockUser(groupId, username);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
     * 将用户加入到群组黑名单
	 * 被加入黑名单的用户无法加入群，无法收发此群的消息
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncBlockUsers(String, List, EMCallBack)}
	 *
	 * v3.3.0后：被加入到黑名单的用户会被先移出群，再加入到黑名单中，之前版本没有移出群的动作
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId 				群组ID
	 * @param members 				待加入黑名单的用户集合
     * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
     * Add users to group blacklist
	 * Set to block members who can not join group again and will be not allowed to receive any group messages
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncBlockUsers(String, List, EMCallBack)}
	 *
	 * After v3.3.0: Members who be added to blacklist will be removed to group first, then add to blacklist; The previous version
	 * is not the action to remove from group
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group Id
	 * @param members 				Users to be blocked
     * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void blockUsers(String groupId, List<String> members) throws HyphenateException {
		EMLog.d(TAG, "block user for groupid:" + groupId + " members:" + members == null ? "":members.toString());
		EMAError error = new EMAError();
		String reason = "";
		emaObject.blockGroupMembers(groupId, members, error, reason);
		handleError(error);
	}

	/**
	 * \~chinese
     * 将用户加入到群组黑名单
	 * 被加入黑名单的用户无法加入群，无法收发此群的消息
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #blockUsers(String, List)}
	 *
	 * 3.3.0后：被加入到黑名单的用户会被先移出群，再加入到黑名单中，之前版本没有移出群的动作
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param members 	待加入黑名单的用户集合
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
     * Add users to group blacklist
	 * Set to block members who can not join group again and will be not allowed to receive any group messages
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #blockUsers(String, List)}
	 *
	 * After v3.3.0: Members who be added to blacklist will be removed to group first, then add to blacklist; The previous version
	 * is not the action to remove from group
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param members 	Users to be blocked
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncBlockUsers(final String groupId,
                               final List<String> members,
                               final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockUsers(groupId, members);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单移除
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncUnblockUser(String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId 				群组ID
	 * @param username 				待从黑名单中移除的用户
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Remove the blocked user from the group blacklist
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncUnblockUser(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @param username				Username to remove from blacklist
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void unblockUser(String groupId, String username) throws HyphenateException{
		EMLog.d(TAG, "unblock user groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.unblockGroupMembers(groupId, members, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单移除
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #unblockUser(String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param username 	待从黑名单中移除的用户
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove the blocked user from the group blacklist
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #unblockUser(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param username	Username to remove from blacklist
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncUnblockUser(final String groupId,
                                 final String username,
                                 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockUser(groupId, username);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单移除
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncBlockUsers(String, List, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId 				群组ID
	 * @param members 				待从黑名单中移除的用户集合
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Remove the blocked users from the group blacklist
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncBlockUsers(String, List, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @param members				The users to remove from blacklist
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void unblockUsers(String groupId, List<String> members) throws HyphenateException{
		EMLog.d(TAG, "unblock user groupid:" + groupId + " members:" + members == null ? "":members.toString());
		EMAError error = new EMAError();
		emaObject.unblockGroupMembers(groupId, members, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单移除
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #unblockUsers(String, List)}
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param members 	待从黑名单中移除的用户集合
	 * @param callback	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove the blocked users from the group blacklist
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #unblockUsers(String, List)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param members	The users to remove from blacklist
	 * @param callback	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncUnblockUsers(final String groupId,
                                 final List<String> members,
                                 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockUsers(groupId, members);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取群组的黑名单列表, 默认最多取200个
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncGetBlockedUsers(String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId				群组ID
	 * @return List					返回黑名单列表，默认最多200条数据
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Fetch black list from server, default maximum is 200
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncGetBlockedUsers(String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @return List					List of users in group blacklist, default maximum is 200
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public List<String> getBlockedUsers(String groupId) throws HyphenateException {
		return getBlockedUsers(groupId, 0, 200);
	}

	/**
	 * \~chinese
	 * 分页获取群组的黑名单列表
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncGetBlockedUsers(String, int, int, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
     *
	 * @param groupId 				群组ID
	 * @param pageIndex 			当前要拉取的页数，从1开始
	 * @param pageSize 				每页拉取的数量
	 * @return List					返回黑名单列表
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Get blacklist of group from server by page
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncGetBlockedUsers(String, int, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
     *
	 * @param groupId               Group id
	 * @param pageIndex            	Page number, start from 1
	 * @param pageSize              Size to be fetched per page
	 * @return List					List of users in group blacklist by page
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public List<String> getBlockedUsers(String groupId, int pageIndex, int pageSize) throws HyphenateException{
		EMLog.d(TAG, "get blocked users of the group: " + groupId);
		EMAError error = new EMAError();
		List<String> members = emaObject.fetchGroupBlackList(groupId, pageIndex, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~chinese
	 * 获取群组的黑名单列表, 默认最多取200个
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #getBlockedUsers(String)}
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回群组的黑名单列表；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get blacklist from server, default maximum is 200
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #getBlockedUsers(String)}
	 *
	 * Asynchronously method
     *
	 * @param groupId	Group id
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the group's blacklist;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncGetBlockedUsers(final String groupId,
                                     final EMValueCallBack<List<String>> callback) {
		asyncGetBlockedUsers(groupId, 0, 200, callback);
	}

	/**
	 * \~chinese
	 * 分页获取群组的黑名单列表
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #getBlockedUsers(String, int, int)}
	 *
	 * 异步方法
	 *
	 * @param groupId     群组ID
	 * @param pageIndex   当前要拉取的页数，从1开始
	 * @param pageSize    每页拉取的数量
	 * @param callback	  结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                    返回黑名单列表；
	 * 					  失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get black list of group by page
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #getBlockedUsers(String, int, int)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId     Group id
	 * @param pageIndex   Page number, start from 1
	 * @param pageSize    Size to be fetched per page
	 * @param callback	  Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                    return the group's blacklist;
	 *					  Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncGetBlockedUsers(final String groupId,
                                     final int pageIndex,
                                     final int pageSize,
                                     final EMValueCallBack<List<String>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<String> users = getBlockedUsers(groupId, pageIndex, pageSize);
					callback.onSuccess(users);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 注册群变动事件监听
	 * 注册后，需要配合{@link #removeGroupChangeListener(EMGroupChangeListener)}使用
	 *
	 * @param listener	注册的群组事件监听
	 *
	 * \~english
	 * Register the group change listener
	 * After registering, need use {@link #removeGroupChangeListener(EMGroupChangeListener)} to unregister
	 *
	 * @param listener	Listener of group event to register
	 */
	public void addGroupChangeListener(EMGroupChangeListener listener) {
		EMLog.d(TAG, "add group change listener:" + listener.getClass().getName());
		if (!groupChangeListeners.contains(listener)) {
			groupChangeListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除群组变化监听器
	 * 在注册{@link #addGroupChangeListener(EMGroupChangeListener)}后调用此方法
     *
	 * @param listener	要解注册的群组监听
	 *
	 * \~english
	 * Remove group change listener
     * Use this method after registering {@link #addGroupChangeListener(EMGroupChangeListener)}
	 *
	 * @param listener 	Listener of group event to unregister
	 */
	public void removeGroupChangeListener(EMGroupChangeListener listener) {
		if (listener != null) {
			EMLog.d(TAG, "remove group change listener:" + listener.getClass().getName());
			groupChangeListeners.remove(listener);
		}
	}


	// ============================= group_reform new add api begin =============================
	/**
	 * \~chinese
	 * 分页获取群组成员列表
	 * 返回的结果中，当EMCursorResult.getCursor()为空字符串("")时，表示没有更多数据
	 * 异步方法见{@link #asyncFetchGroupMembers(String, String, int, EMValueCallBack)}
	 *
     * 同步方法，会阻塞当前线程
	 *
	 * 例如:
	 * <pre>
	 *     EMCursorResult result = fetchGroupMembers(groupId, cursor, pageSize);    // search 1
	 * 	   result = fetchGroupMembers(groupId, result.getCursor(), pageSize);       // search 2
	 * </pre>
     *
	 * @param groupId				群组ID
	 * @param cursor				一次取的数量
	 * @param pageSize				从这个游标位置开始取数据，首次获取数据时传null即可
	 * @return						分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的cursor以及群组list的对象，
	 * 								如果是最后一页，cursor返回结果为空字符串
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
     * Get group's member list by page
	 * When EMCursorResult.getCursor() is an empty string ("") in the result, there is no more data
     * Asynchronously method see {@link #asyncFetchGroupMembers(String, String, int, EMValueCallBack)}
	 *
     * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param cursor				The count fetched a time
	 * @param pageSize				The cursor position to fetch data, use null if first call
	 * @return 						Fetch result, see {@link EMCursorResult}, used to get next page and the group list,
	 * 								for the last page, EMCursorResult.getCursor() is an empty string.
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMCursorResult<String> fetchGroupMembers(String groupId, String cursor, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<String> result = emaObject.fetchGroupMembers(groupId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 分页获取群组成员列表
	 * 同步方法见{@link #fetchGroupMembers(String, String, int)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param cursor	从这个游标位置开始取数据，首次获取数据时传null即可
	 * @param pageSize	一次取的数量
	 * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 * 					返回分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的cursor以及群组成员集合的对象，
	 * 					如果是最后一页，cursor返回结果为空字符串；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get group's member list by page
	 * Synchronization method see {@link #fetchGroupMembers(String, String, int)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId	Group id
	 * @param cursor	The count fetched a time
	 * @param pageSize	The cursor position to fetch data, use null if first call
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  fetch result, see {@link EMCursorResult}, used to get next page and the group list,
	 * 					for the last page, EMCursorResult.getCursor() is an empty string.
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchGroupMembers(final String groupId,
                                       final String cursor,
                                       final int pageSize,
                                       final EMValueCallBack<EMCursorResult<String>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMCursorResult<String> result = fetchGroupMembers(groupId, cursor, pageSize);
					callback.onSuccess(result);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 移交群组所有权
     * 需要群组owner权限
	 * 异步方法见{@link #asyncChangeOwner(String, String, EMValueCallBack)}
	 *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId				群组ID
	 * @param newOwner				新的群主
	 * @return						返回更新后的群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
     * Transfer group ownership
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncChangeOwner(String, String, EMValueCallBack)}
     *
     * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param newOwner				New owner id
	 * @return						Return the updated group instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup changeOwner(String groupId, String newOwner) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.transferGroupOwner(groupId, newOwner, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 移交群组所有权
	 * 需要群组owner权限
     * 同步方法见{@link #changeOwner(String, String)}
	 *
	 * 异步方法
	 *
	 * @param groupId	群组ID
	 * @param newOwner	新的群主
	 * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Transfer group ownership
	 * Owner‘s authority is required
	 * Synchronization method see {@link #changeOwner(String, String)}
	 *
	 * Asynchronously method
     *
	 * @param groupId	Group id
	 * @param newOwner	New Owner id
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the updated group instance;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncChangeOwner(final String groupId,
                                 final String newOwner,
                                 final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = changeOwner(groupId, newOwner);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 增加群组管理员
	 * 需要群组owner权限，admin无权限
	 * 异步方法见{@link #asyncAddGroupAdmin(String, String, EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId				群组ID
	 * @param admin					新增加的管理员ID
	 * @return						返回更新后的群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Add group admin
	 * Owner‘s authority is required, admin has no authority
	 * Asynchronously method see {@link #asyncAddGroupAdmin(String, String, EMValueCallBack)}
     *
     * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param admin					The admin id to add
	 * @return						Return the updated group instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup addGroupAdmin(final String groupId, final String admin) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.addGroupAdmin(groupId, admin, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 增加群组管理员
	 * 需要群组owner权限，admin无权限
	 * 同步方法见{@link #addGroupAdmin(String, String)}
	 *
	 * 异步方法
     *
	 * @param groupId	群组ID
	 * @param admin		新增加的管理员ID
	 * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
     * Add group admin
	 * Owner‘s authority is required, admin has no authority
	 * Synchronization method see {@link #addGroupAdmin(String, String)}
	 *
	 * Asynchronously method
     *
	 * @param groupId	Group id
	 * @param admin		New owner id
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the updated group instance;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncAddGroupAdmin(final String groupId,
                                   final String admin,
                                   final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(addGroupAdmin(groupId, admin));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除群组管理员
	 * 需要群组owner权限
	 * 异步方法见{@link #asyncRemoveGroupAdmin(String, String, EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId				群组ID
	 * @param admin					删除的管理员ID
	 * @return						返回更新后的群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Remove group administrator
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncRemoveGroupAdmin(String, String, EMValueCallBack)}
     *
     * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param admin					Admin id to remove
	 * @return						Return the updated group instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup removeGroupAdmin(String groupId, String admin) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.removeGroupAdmin(groupId, admin, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 删除群组管理员
	 * 需要群组owner权限
	 * 同步方法见{@link #removeGroupAdmin(String, String)}
	 *
	 * 异步方法
     *
	 * @param groupId	群组ID
	 * @param admin		删除的管理员ID
	 * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove group administrator
	 * Owner‘s authority is required
	 * Synchronization method see {@link #removeGroupAdmin(String, String)}
	 *
	 * Asynchronously method
     *
	 * @param groupId	Group id
	 * @param admin		Admin id to remove
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the updated group instance;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncRemoveGroupAdmin(final String groupId,
                                      final String admin,
                                      final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(removeGroupAdmin(groupId, admin));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 禁止指定的群组成员发言
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #aysncMuteGroupMembers(String, List, long, EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId				群组ID
	 * @param muteMembers 			需要禁言的用户列表
	 * @param duration 				禁言的时间，单位是毫秒
	 * @return						返回更新后的群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
     * Mute specified members from posting message for specified duration
     * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #aysncMuteGroupMembers(String, List, long, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param muteMembers 			Mute member list
	 * @param duration 				Mute duration, in milli-seconds
	 * @return						Return the updated group instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup muteGroupMembers(String groupId, List<String> muteMembers, long duration) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.muteGroupMembers(groupId, muteMembers, duration, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 禁止指定的群组成员发言
	 * 需要群组拥有者或者管理员权限
     * 同步方法见{@link #muteGroupMembers(String, List, long)}
	 *
	 * 异步方法
	 *
	 * @param groupId		群组ID
	 * @param muteMembers	需要禁言的用户列表
	 * @param duration 		禁言的时间，单位是毫秒
	 * @param callback		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      返回更新后的群组对象；
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
     * Mute specified members from posting message for specified duration
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #muteGroupMembers(String, List, long)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId		Group id
	 * @param muteMembers 	Mute group members
	 * @param duration 		Mute duration, in milli-seconds
	 * @param callback		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      return the updated group instance;
	 *						Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void aysncMuteGroupMembers(final String groupId,
                                      final List<String> muteMembers,
                                      final long duration, final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(muteGroupMembers(groupId, muteMembers, duration));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除指定用户的禁言
     * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncUnMuteGroupMembers(String, List, EMValueCallBack)}
	 *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId				群组ID
	 * @param members				需要解除禁言的用户列表
	 * @return						返回更新后的群组对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Unban the specified users
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncUnMuteGroupMembers(String, List, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
     *
     * @param groupId				Group id
     * @param members				The users to release mute
     * @return 						Return the updated group instance
     * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup unMuteGroupMembers(String groupId, List<String> members) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.unMuteGroupMembers(groupId, members, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 解除指定用户的禁言
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #unMuteGroupMembers(String, List)}
	 *
	 * 异步方法
     *
	 * @param groupId	群组ID
	 * @param members	需要解除禁言的用户列表
	 * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Unban the specified users
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #unMuteGroupMembers(String, List)}
	 *
	 * Asynchronously method
     *
	 * @param groupId	Group id
	 * @param members	The users to release mute
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the updated group instance;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncUnMuteGroupMembers(final String groupId,
                                        final List<String> members,
                                        final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(unMuteGroupMembers(groupId, members));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取群组的禁言列表
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId				群组ID
	 * @param pageNum				当前要拉取的页数，从1开始
	 * @param pageSize				每页拉取的数量
	 * @return 						返回的包含禁言成员ID及其禁言时长的Map，其中每一项的key是禁言的成员ID，value是禁言动作失效的时间，单位是毫秒
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Fetch mute list
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
     *
	 * @param groupId				Group id
	 * @param pageNum				The count fetched a time
	 * @param pageSize				The cursor position to fetch data
	 * @return 						Return a map which contains the mute member id and mute duration, in which the key of each entry
	 * 								is the mute member id and the value is the expired time of banning post action, in milli-seconds
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public Map<String, Long> fetchGroupMuteList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, Long> muteMembers = emaObject.fetchGroupMutes(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~chinese
	 * 获取群组的禁言列表
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #fetchGroupMuteList(String, int, int)}
	 *
	 * 异步方法
     *
	 * @param groupId	群组ID
	 * @param pageNum	当前要拉取的页数，从1开始
	 * @param pageSize	每页拉取的数量
	 * @param callBack 	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回的包含禁言成员ID及其禁言时长的Map，其中每一项的key是禁言的成员ID，value是禁言动作失效的时间，单位是毫秒;
	 *                  失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch mute list
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #fetchGroupMuteList(String, int, int)}
	 *
	 * Asynchronously method
     *
	 * @param groupId	Group id
	 * @param pageNum	The count fetched a time
	 * @param pageSize	The cursor position to fetch data
	 * @param callBack 	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return a map which contains the mute member id and mute duration, in which the key of each entry
	 * 					is the mute member id and the value is the expired time of banning post action, in milli-seconds;
	 *                  Failure call {@link EMValueCallBack#onError(int, String)}
	 *
	 */
	public void asyncFetchGroupMuteList(final String groupId,
                                        final int pageNum,
                                        final int pageSize,
                                        final EMValueCallBack<Map<String, Long>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupMuteList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 分页从服务器获群组黑名单
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId       		群组ID
	 * @param pageNum       		当前要拉取的页数，从1开始
	 * @param pageSize      		每页拉取的数量
	 * @return						返回分页获取的群组黑名单集合
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Fetch group blacklist from server by page
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
     *
	 * @param groupId       		Group id
	 * @param pageNum       		Page number, start from 1
	 * @param pageSize      		Size to be fetched per page
	 * @return						Return the updated group instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public List<String> fetchGroupBlackList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> muteMembers = emaObject.fetchGroupBlackList(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~chinese
	 * 分页从服务器获群组黑名单
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #fetchGroupBlackList(String, int, int)}
	 *
	 * 异步方法
     *
	 * @param groupId       群组ID
	 * @param pageNum       当前要拉取的页数，从1开始
	 * @param pageSize      每页拉取的数量
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      返回分页获取的黑名单列表；
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch group blacklist from server by page
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #fetchGroupBlackList(String, int, int)}
	 *
	 * Asynchronously method
     *
	 * @param groupId       Group id
	 * @param pageNum       Page number, start from 1
	 * @param pageSize      Size to be fetched per page
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      return the group blacklist by page;
	 *						Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchGroupBlackList(final String groupId,
                                         final int pageNum,
                                         final int pageSize,
                                         final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupBlackList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加用户到群组白名单
	 * 需要群组拥有者或者管理员权限
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param members 	需要加入白名单的成员ID列表
	 * @param callBack  结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Add members to whitelist
	 * Owner or administrator‘s authority is required
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group id
	 * @param members 	Member list to add to whitelist
	 * @param callBack	Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void addToGroupWhiteList(final String groupId, final List<String> members, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					emaObject.addToWhiteList(groupId, members, error);
					handleError(error);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户从群组白名单中移除
	 * 需要群组拥有者或者管理员权限
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param members 	需要从白名单中移除的成员ID列表
	 * @param callBack  结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove members from whitelist
	 * Owner or administrator‘s authority is required
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group id
	 * @param members 	Member list to add to whitelist
	 * @param callBack  Result callback, success call {@link EMCallBack#onSuccess()},
	 * 					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void removeFromGroupWhiteList(final String groupId, final List<String> members, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					emaObject.removeFromWhiteList(groupId, members, error);
					handleError(error);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 检查自己是否在群组白名单中
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param callBack	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回是否在群组白名单的布尔值；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Check if current user in group whitelist
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group id
	 * @param callBack	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the Boolean whether the current user is in group whitelist;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void checkIfInGroupWhiteList(final String groupId, EMValueCallBack<Boolean> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(groupId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~chinese
	 * 从服务器获取白名单成员列表
	 * 需要群组拥有者或者管理员权限
	 *
	 * 异步方法
	 *
	 * @param groupId 	群组ID
	 * @param callBack	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回白名单列表；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch group whitelist from serer
	 * Owner or administrator‘s authority is required
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group id
	 * @param callBack	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the group whitelist;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void fetchGroupWhiteList(final String groupId, final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchGroupWhiteList(groupId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}


    /**
     * \~chinese
     * 禁言所有成员
	 * 需要群组拥有者或者管理员权限
	 *
	 * 异步方法
	 *
     * @param groupId 	群组ID
	 * @param callBack	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
     *
     * \~english
     * Mute all members
	 * Owner or administrator‘s authority is required
	 *
	 * Asynchronously method
	 *
     * @param groupId 	Group id
	 * @param callBack	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the updated group instance;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
     */
    public void muteAllMembers(final String groupId, final EMValueCallBack<EMGroup> callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMAGroup group = emaObject.muteAllMembers(groupId, error);
                    handleError(error);
                    callBack.onSuccess(new EMGroup(group));
                } catch (HyphenateException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 解除所有成员禁言
	 * 需要群组拥有者或者管理员权限
	 *
	 * 异步方法
	 *
     * @param groupId 	群组ID
	 * @param callBack	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
     *
     * \~english
     * Release all members muted
	 * Owner or administrator‘s authority is required
	 *
	 * Asynchronously method
	 *
     * @param groupId 	Group id
	 * @param callBack	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  return the updated group instance;
	 *					Failure call {@link EMValueCallBack#onError(int, String)}
     */
    public void unmuteAllMembers(final String groupId, final EMValueCallBack<EMGroup> callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMAGroup group = emaObject.unmuteAllMembers(groupId, error);
                    handleError(error);
                    callBack.onSuccess(new EMGroup(group));
                } catch (HyphenateException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 更新群公告
	 * 需要群组拥有者或者管理员权限
	 * 异步方法见{@link #asyncUpdateGroupAnnouncement(String, String, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId 				群组ID
	 * @param announcement 			公告内容
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Update group announcement
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncUpdateGroupAnnouncement(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId 				Group id
	 * @param announcement 			Group announcement
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void updateGroupAnnouncement(String groupId, String announcement) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.updateGroupAnnouncement(groupId, announcement, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 更新群公告
	 * 需要群组拥有者或者管理员权限
	 * 同步方法见{@link #updateGroupAnnouncement(String, String)}
	 *
	 * 异步方法
     *
	 * @param groupId 		群组ID
	 * @param announcement 	公告内容
	 * @param callBack		结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 						失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Update group announcement
     * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #updateGroupAnnouncement(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 		Group id
	 * @param announcement 	Group announcement
	 * @param callBack 		Result callback, success call {@link EMCallBack#onSuccess()},
	 * 						failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncUpdateGroupAnnouncement(final String groupId,
                                             final String announcement,
                                             final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateGroupAnnouncement(groupId, announcement);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取群组公告
	 * 需要是群成员权限
     * 异步方法见{@link #asyncFetchGroupAnnouncement(String, EMValueCallBack)}
	 *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId 				群组ID
	 * @return						群组公告
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Get group announcement from server
	 * Group member‘s authority is required
	 * Asynchronously method see {@link #asyncFetchGroupAnnouncement(String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId				Group id
	 * @return						Group announcement
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public String fetchGroupAnnouncement(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.fetchGroupAnnouncement(groupId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取群公告
	 * 需要是群成员权限
	 * 同步方法见{@link #fetchGroupAnnouncement(String)}
	 *
	 * 异步方法
     *
	 * @param groupId 	群组ID
	 * @param callBack	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get group announcement from server
	 * Group member‘s authority is required
	 * Synchronization method see {@link #fetchGroupAnnouncement(String)}
	 *
	 * Asynchronously method
     *
	 * @param groupId	Group id
	 * @param callBack	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 * 					failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchGroupAnnouncement(final String groupId,
                                            final EMValueCallBack<String> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupAnnouncement(groupId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 上传共享文件至群组
	 * 注意：callback只做进度回调用
	 * 异步方法见{@link #asyncUploadGroupSharedFile(String, String, EMCallBack)}
	 *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId 				群组ID
	 * @param filePath 				文件本地路径
	 * @param callBack 				上传文件进度回调
	 * @return EMMucSharedFile		返回共享文件属性对象
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Upload file to group
	 * Note: The callback only callback the progress
	 * Asynchronously method see {@link #asyncUploadGroupSharedFile(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId 				group id
	 * @param filePath 				File local path
	 * @param callBack 				Progress callback of uploading file
	 * @return						The shared file instance
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMMucSharedFile uploadGroupSharedFile(String groupId, String filePath, EMCallBack callBack) throws HyphenateException {
		EMAError error = new EMAError();
		EMACallback aCallback = new EMACallback(callBack);
		EMAMucShareFile aFile = emaObject.uploadGroupShareFile(groupId, filePath, aCallback, error);
		handleError(error);
		callBack.onSuccess();

		return new EMMucSharedFile(aFile);
	}


	/**
	 * \~chinese
	 * 上传共享文件至群组
	 * 同步方法见{@link #uploadGroupSharedFile(String, String, EMCallBack)}
	 *
	 * 异步方法
     *
	 * @param groupId 	群组ID
	 * @param filePath 	文件本地路径
	 * @param callBack 	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 *                  上传文件进度回调{@link EMCallBack#onProgress(int, String)}，
	 * 					结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 上传文件进度回调{@link EMCallBack#onProgress(int, String)}，
	 * 失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 *
	 * \~english
	 * Upload file to group
     * Synchronization method see {@link #uploadGroupSharedFile(String, String, EMCallBack)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group id
	 * @param filePath 	File local path
	 * @param callBack 	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncUploadGroupSharedFile(final String groupId,
                                           final String filePath,
                                           final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMMucSharedFile file = uploadGroupSharedFile(groupId, filePath, callBack);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的共享文件列表
	 * 异步方法见{@link #asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId 				群组ID
	 * @param pageNum 				当前要拉取的页数，从1开始
	 * @param pageSize 				每页拉取的数量
	 * @return						返回共享文件列表
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Get the shared file list from server
     * Asynchronously method see {@link #asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}
	 *
     * Synchronization method will block the current thread
     *
	 * @param groupId 				Group id
	 * @param pageNum 				Page number, start from 1
	 * @param pageSize 				Size to be fetched per page
	 * @return						Return the shared files
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public List<EMMucSharedFile> fetchGroupSharedFileList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAMucShareFile> files = emaObject.fetchGroupShareFiles(groupId, pageNum, pageSize, error);
		handleError(error);
		List<EMMucSharedFile> result = new ArrayList<EMMucSharedFile>();
		for(EMAMucShareFile file : files){
			result.add(new EMMucSharedFile(file));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的共享文件列表
	 * 同步方法见{@link #fetchGroupSharedFileList(String, int, int)}
	 *
	 * 异步方法
     *
	 * @param groupId 	群组ID
	 * @param pageNum 	当前要拉取的页数，从1开始
	 * @param pageSize 	每页拉取的数量
	 * @param callBack 	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回共享文件列表;
	 * 					失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get the shared file list from server
     * Synchronization method see {@link #fetchGroupSharedFileList(String, int, int)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group id
	 * @param pageNum 	Page number, start from 1
	 * @param pageSize 	Size to be fetched per page
	 * @param callBack 	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  Return the shared files;
	 * 					Failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchGroupSharedFileList(final String groupId,
                                              final int pageNum,
                                              final int pageSize,
                                              final EMValueCallBack<List<EMMucSharedFile>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupSharedFileList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除群组指定的共享文件
	 * 成员可以删除自己上传的文件，群组拥有者或者群组管理员可以删除所有的共享文件
	 * 异步方法见{@link #asyncDeleteGroupSharedFile(String, String, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId 				群组ID
	 * @param fileId 				文件ID
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Delete file from group
     * Members can delete their own uploaded files, and the group owner or group administrator can delete all shared files
	 * Asynchronously method see {@link #asyncDeleteGroupSharedFile(String, String, EMCallBack)}
	 *
     * Synchronization method will block the current thread
     *
	 * @param groupId 				Group id
	 * @param fileId 				File id
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void deleteGroupSharedFile(String groupId, String fileId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.deleteGroupShareFile(groupId, fileId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 删除群组指定的共享文件
	 * 成员可以删除自己上传的文件，群组拥有者或者群组管理员可以删除所有的共享文件
	 * 同步方法见{@link #deleteGroupSharedFile(String, String)}
	 *
	 * 异步方法
     *
	 * @param groupId 	群组ID
	 * @param fileId 	文件ID
	 * @param callBack 	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Delete file from group
     * Members can delete their own uploaded files, and the group owner or group administrator can delete all shared files
	 * Synchronization method see {@link #deleteGroupSharedFile(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId 	Group id
	 * @param fileId 	File id
	 * @param callBack 	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncDeleteGroupSharedFile(final String groupId,
                                           final String fileId,
                                           final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					deleteGroupSharedFile(groupId, fileId);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 下载群组中指定的共享文件
	 * 注意：callback只做进度回调用
	 * 异步方法见{@link #asyncDownloadGroupSharedFile(String, String, String, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
	 * @param groupId 				群组ID
	 * @param fileId 				文件ID
	 * @param savePath 				文件保存路径
	 * @param callBack 				下载文件进度回调
	 * @throws HyphenateException	错误信息详见{@link EMError}
	 *
	 * \~english
	 * Download the file from group
	 * Note the callback only callback progress
	 * Asynchronously method see {@link #asyncDownloadGroupSharedFile(String, String, String, EMCallBack)}
     *
     * Synchronization method will block the current thread
     *
	 * @param groupId 				Group id
	 * @param fileId 				File id
	 * @param savePath 				File to saved path
	 * @param callBack 				Progress callback to download file
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public void downloadGroupSharedFile(String groupId,
                                        String fileId,
                                        String savePath,
                                        EMCallBack callBack) throws HyphenateException {
		EMAError error = new EMAError();
		EMACallback aCallback = new EMACallback(callBack);
		emaObject.downloadGroupShareFile(groupId, fileId, savePath, aCallback, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 下载群组中指定的共享文件
	 * 同步方法见{@link #downloadGroupSharedFile(String, String, String, EMCallBack)}
	 *
	 * 异步方法
     *
	 * @param groupId   群组ID
	 * @param fileId    文件ID
	 * @param savePath  文件保存路径
	 * @param callBack 	结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 					下载文件进度回调{@link EMCallBack#onProgress(int, String)}，
	 * 					失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Download the file from group
     * Synchronization method see {@link #downloadGroupSharedFile(String, String, String, EMCallBack)}
	 *
	 * Asynchronously method
	 *
	 * @param groupId   Group id
	 * @param fileId    File id
	 * @param savePath  File to saved path
	 * @param callBack 	Result callback, success call {@link EMCallBack#onSuccess()},
	 *					failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncDownloadGroupSharedFile(final String groupId,
											 final String fileId,
											 final String savePath,
											 final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					downloadGroupSharedFile(groupId, fileId, savePath, callBack);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	// ============================= group_reform new add api end =============================

	/**
	 * \~chinese
	 * 更新群组扩展字段
	 * 需要群组拥有者或者管理员权限
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param groupId       群组ID
	 * @param extension     群组扩展字段
	 * @throws HyphenateException   错误信息详见{@link EMError}
	 *
	 * \~english
	 * update group extension
	 * Owner or administrator‘s authority is required
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param groupId       		Group id
	 * @param extension     		Extension string
	 * @throws HyphenateException	Error info, see {@link EMError}
	 */
	public EMGroup updateGroupExtension(String groupId, String extension) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.updateGroupExtension(groupId, extension, error);
		handleError(error);
		return new EMGroup(group);
	}

	void onLogout() {
	}

	private void handleError(EMAError error)  throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new HyphenateException(error);
		}
	}
}
