/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAGroup;
import com.hyphenate.chat.adapter.EMAGroupSetting;
import com.hyphenate.chat.adapter.EMAMucShareFile;

import java.util.ArrayList;
import java.util.List;

/**
 *  \~chinese
 *  群组
 *
 *  \~english
 *  Group
 */
public class EMGroup extends EMBase<EMAGroup> {

    /**
     * \~chinese
     * 当前用户在群组的角色枚举
     *
     * \~english
     * Enum of the current user's role in group
     *
     */
    public enum EMGroupPermissionType {
        /**
         * \~chinese
         * 普通成员
         *
         * \~english
         * Normal member
         */
        member(0),

        /**
         * \~chinese
         * 群组管理员
         *
         * \~english
         * Group admin
         */
        admin(1),

        /**
         * \~chinese
         * 群主
         *
         * \~english
         * Group owner
         */
        owner(2),

        /**
         * \~chinese
         * 未知
         *
         * \~english
         * Unknown
         */
        none(-1);
        private int permissionType;
        private EMGroupPermissionType(int permissionType) {
            this.permissionType = permissionType;
        }
    }

    public EMGroup(EMAGroup group) {
        emaObject = group;
    }

    /**
     * \~chinese
     * 获取群ID
     * @return 群ID
     *
     * \~english
     * Get group id
     * @return Group id
     */
    public String getGroupId() {
        return emaObject.groupId();
    }

    /**
     * \~chinese
     * 获取群名称
     * @return String
     *
     * \~english
     * Get group name
     * @return String
     */
    public String getGroupName() {
        return emaObject.groupSubject();
    }

    /**
     * \~chinese
     * 获取群简介
     * @return String
     *
     * \~english
     * Get group description
     * @return String
     */
    public String getDescription() {
        return emaObject.getDescription();
    }

    /**
     * \~chinese
     * 获取群组是否为公开群
     * @return boolean  true为公开群，否则为私有群
     *
     * \~english
     * Get if the group is public
     * @return boolean  True is public group, otherwise is private group
     */
    public boolean isPublic() {

        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        switch (setting.style()) {
            case EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE:
            case EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE:
                return false;
            default:
                return true;
        }
    }

    /**
     * \~chinese
     * 获取群组是否允许成员邀请
     * @return boolean
     * @deprecated  使用 {@link #isMemberAllowToInvite()} 替代
     *
     * \~english
     * Get if the group member is allow invite the others
     * @return boolean
     * @deprecated  Use {@link #isMemberAllowToInvite()} instead
     */
    @Deprecated
    public boolean isAllowInvites() {
        return isMemberAllowToInvite();
    }

    /**
     * \~chinese
     * 获取群组是否允许成员邀请
     * @return boolean
     *
     * \~english
     * Get if the group member is allow invite the others
     * @return boolean
     */
    public boolean isMemberAllowToInvite()
    {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        if (setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE) {
            return true;
        }
        return false;
    }

    /**
     * \~chinese
     * 群组属性：成员不能自由加入，需要申请或者被邀请
     * 与之对应的属性是除了{@link EMGroupManager.EMGroupStyle#EMGroupStylePublicOpenJoin}之外的三种属性
     *
     * @return boolean
     * @deprecated  使用 {@link #isMemberOnly()} 替代
     *
     * \~english
     * Group properties:members can not auto join the group, need join group application approved or received invitation
     * Corresponding group style is the three other attributes besides {@link EMGroupManager.EMGroupStyle#EMGroupStylePublicOpenJoin}
     *
     * @return boolean
     * @deprecated  use {@link #isMemberOnly()} instead
     */
    @Deprecated
    public boolean isMembersOnly() {
        return isMemberOnly();
    }

    /**
     * \~chinese
     * 群组属性：成员不能自由加入，需要申请或者被邀请
     * 与之对应的属性是除了{@link EMGroupManager.EMGroupStyle#EMGroupStylePublicOpenJoin}之外的三种属性
     *
     * @return boolean
     *
     * \~english
     * Group properties:members can not auto join the group, need join group application approved or received invitation
     * Corresponding group style is the three other attributes besides {@link EMGroupManager.EMGroupStyle#EMGroupStylePublicOpenJoin}
     *
     * @return boolean
     */
    public boolean isMemberOnly()
    {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return true;
        }
        if (setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE ||
                setting.style() == EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE ||
                setting.style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL) {
            return true;
        }
        return false;
    }

    /**
     * \~chinese
     * 获取群允许加入的最大成员数，在创建群时确定。
     * 需要获取群详情才能拿到正确的结果，如果没有获取则返回0
     * @return
     *
     * \~english
     * Max group members capability
     * It depends on group detail, if not fetch group detail, return 0
     * @return
     */
    public int getMaxUserCount() {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return 0;
        }
        return setting.maxUserCount();
    }

    /**
     * \~chinese
     * 获取是否已屏蔽群消息
     * 设置屏蔽群消息见{@link EMGroupManager#blockGroupMessage(String)},
     * 取消屏蔽群消息见{@link EMGroupManager#unblockGroupMessage(String)}
     * @return boolean
     *
     * \~english
     * Get whether the group message is blocked
     * For blocking group message see {@link EMGroupManager#blockGroupMessage(String)},
     * unblocking see {@link EMGroupManager#unblockGroupMessage(String)}
     * @return boolean
     */
    public boolean isMsgBlocked() {
        return emaObject.isMsgBlocked();
    }

    /**
     * \~chinese
     * 获取群创建者
     * @return String   群创建者id
     *
     * \~english
     * Get the owner
     * @return String   Group owner id
     */
    public String getOwner() {
        return emaObject.getOwner();
    }

    /**
     *  \~chinese
     *  群组名称，需要获取群详情
     * @return String
     *
     *  \~english
     *  Subject of the group, need to get group detail
     * @return String
     */
    public String groupSubject() {
        return emaObject.groupSubject();
    }

    /**
     * \~chinese
     * 获取群成员列表
     * 如果没有获取成员列表，返回的列表可能为空
     * 获取成员有如下方式：
     * （1）成员人数少于200人时，可通过{@link EMGroupManager#getGroupFromServer(String, boolean)},
     * 第二个参数传入true，可获取最多200个成员
     * （2）通过{@link EMGroupManager#fetchGroupMembers(String, String, int)}分页获取，也可通过其异步方法
     * {@link EMGroupManager#asyncFetchGroupMembers(String, String, int, EMValueCallBack)}获取
     * @return  成员id集合
     *
     * \~english
     * Get the member list
     * If not get members from server, the return may be empty
     * You can get members in the flowing ways:
     * (1) When member's number is less than 200, can use {@link EMGroupManager#getGroupFromServer(String, boolean)} to fetch,
     * the second parameter pass in true, can get up to 200 members
     * (2) Use the method of {@link EMGroupManager#fetchGroupMembers(String, String, int)} to get by page,
     * also can use its asynchronous method of {@link EMGroupManager#asyncFetchGroupMembers(String, String, int, EMValueCallBack)}
     * to get
     * @return  The list of member IDs
     */
    public List<String> getMembers() {
        List<String> members = new ArrayList<String>();
        members.addAll(emaObject.getMembers());
        return members;
    }

    /**
     * \~chinese
     * 获取成员数量
     * @return  int
     *
     * \~english
     * Get the member count
     * @return  int
     */
    public int getMemberCount() {
        return emaObject.getMemberCount();
    }

    /**
     * \~chinese
     * 返回群组名称或者群组ID，此方法可用于Adapter排序
     * @return  String
     *
     * \~english
     * Return group name or group id, this method can be used for adapter sorting
     * @return  String
     */
    public String toString() {
        String str = getGroupName();
        return str != null ? str : getGroupId();
    }

    /**
     * \~chinese
     * 获取群组管理员列表
     * 如果没有获取群组详情，返回可能为空
     * 获取群组详情见{@link EMGroupManager#getGroupFromServer(String)}
     * @return  List<String>
     *
     * \~english
     * Get group admin list
     * If not fetch the group's detail specification from server, the result may be empty
     * Get Group detail, see {@link EMGroupManager#getGroupFromServer(String)}
     * @return  List<String>
     */
    public List<String> getAdminList() {
        return emaObject.getAdminList();
    }

    /**
     * \~chinese
     * 获取群组黑名单
     * 如果没有获取群组黑名单列表，返回的列表可能为空
     * 获取黑名单列表可调用{@link EMGroupManager#fetchGroupBlackList(String, int, int)},
     * 调用这个方法需要群组所有者或者管理员身份，也可通过它的异步方法进行调用
     * {@link EMGroupManager#asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}
     * @return  黑名单列表
     *
     * \~english
     * Get blacklist
     * If not get blacklist from server, the return may be empty
     * Fetch blacklist can call {@link EMGroupManager#fetchGroupBlackList(String, int, int)},
     * call this method need owner or admin authority, also can call it's asynchronous method
     * {@link EMGroupManager#asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}
     * @return blacklist
     */
    public List<String> getBlackList() {
        return emaObject.getGroupBans();
    }

    /**
     * \~chinese
     * 获取群组禁言列表
     * 如果没有获取禁言列表，返回的列表可能为空
     * 获取禁言列表可调用{@link EMGroupManager#fetchGroupMuteList(String, int, int)},
     * 调用这个方法需要群组所有者或者管理员身份，也可通过它的异步方法进行调用
     * {@link EMGroupManager#asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}
     *
     * @return 群组禁言列表
     *
     * \~english
     * Get group mute list
     * If not get mute list from server, the return may be empty
     * Fetch mute list can call {@link EMGroupManager#fetchGroupMuteList(String, int, int)},
     * call this method need owner or admin authority, also can call it's asynchronous method
     * {@link EMGroupManager#asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}
     *
     * @return Group mute list
     */
    public List<String> getMuteList() {
        return emaObject.getGroupMuteList();
    }

    /**
     * \~chinese
     * 获取白名单列表
     * 如果没有从服务器获取白名单列表，返回的列表可能为空
     * 获取白名单可调用{@link EMGroupManager#fetchGroupWhiteList(String, EMValueCallBack)},
     * 调用这个方法需要群组所有者或者管理员身份
     * @return  群组白名单列表
     *
     * \~english
     * Get white list
     * If not get whitelist from server, the return may be empty
     * Fetch whitelist can call {@link EMGroupManager#fetchGroupWhiteList(String, EMValueCallBack)},
     * call this method need owner or admin authority
     * @return  Group whitelist
     */
    public List<String> getWhiteList() { return emaObject.getWhiteList(); }

    /**
     * \~chinese
     * 返回是否全员禁言
     * 此方法有使用局限性，建议慎重使用
     * 在收到禁言/解禁的回调时，该状态会更新，但是只是更新内存中对象，
     * 内存中对象被回收后以及再次从数据库或者从服务端拉取后，该状态变的不可信
     * @return boolean
     *
     * \~english
     * Return if all member muted
     * This method has limitations and is recommended to be used with caution
     * The state is updated when a all-muted/all-unmuted callback is received, but only for the in-memory object.
     * After the in-memory object is collected and pulled again from the database or server, the state becomes untrusted
     * @return boolean
     */
    public boolean isAllMemberMuted() { return emaObject.isAllMemberMuted(); }

    /**
     * \~chinese
     * 获取群组订制扩展信息
     * @return String
     *
     * \~english
     * Get group customized extension
     * @return String
     */
    public String getExtension() {
        EMAGroupSetting setting = emaObject.groupSetting();
        if (setting == null) {
            return "";
        }
        return setting.extension();
    }

    /**
     * \~chinese
     * 获取群组公告
     * 如果没有获取群组公告，返回的值为空
     * 获取群组公告，可调用{@link EMGroupManager#fetchGroupAnnouncement(String)},
     * 也可调用其异步方法{@link EMGroupManager#asyncFetchGroupAnnouncement(String, EMValueCallBack)}
     * @return String
     *
     * \~english
     * Get group announcement
     * If not get announcement from server, the return may be empty
     * Get group announcement, can call the method of {@link EMGroupManager#fetchGroupAnnouncement(String)},
     * also can call its asynchronous method of {@link EMGroupManager#asyncFetchGroupAnnouncement(String, EMValueCallBack)}
     * @return String
     */
    public String getAnnouncement() {
        return emaObject.getAnnouncement();
    }

    /**
     *  \~chinese
     *  群共享文件列表
     *
     *  \~english
     *  List of shared file
     */
    private List<EMMucSharedFile> shareFileList;

    /**
     * \~chinese
     * 获取群组共享文件
     * 如果没有获取群组共享文件，返回的列表可能为空
     * 获取群组共享文件，可调用{@link EMGroupManager#fetchGroupSharedFileList(String, int, int)},
     * 也可调用其异步方法{@link EMGroupManager#asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}
     * @return  群组共享文件列表
     *
     * \~english
     * Get group shared file list
     * If not get shared file list from server, the return may be empty
     * Fetch shared file list can call {@link EMGroupManager#fetchGroupSharedFileList(String, int, int)},
     * {@link EMGroupManager#asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}
     * @return Group shared file list
     */
    public List<EMMucSharedFile> getShareFileList()
    {
        if (shareFileList == null) {
            shareFileList = new ArrayList<EMMucSharedFile>();
        }
        else {
            shareFileList.clear();
        }
        List<EMAMucShareFile> afiles = emaObject.getShareFiles();
        for (EMAMucShareFile afile : afiles){
            shareFileList.add(new EMMucSharedFile(afile));
        }
        return shareFileList;
    }

    /**
     * \~chinese
     * 获取当前用户在群中的角色
     * @return  EMGroupPermissionType
     *
     * \~english
     * Get the current user's role in group
     * @return  EMGroupPermissionType
     */
    public EMGroupPermissionType getGroupPermissionType() {
        int permissionType = emaObject.permissionType();
        if(permissionType == EMGroupPermissionType.member.permissionType) {
            return EMGroupPermissionType.member;
        }else if(permissionType == EMGroupPermissionType.admin.permissionType) {
            return EMGroupPermissionType.admin;
        }else if(permissionType == EMGroupPermissionType.owner.permissionType) {
            return EMGroupPermissionType.owner;
        }else {
            return EMGroupPermissionType.none;
        }
    }
}
