/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.net.Uri;

import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.chat.adapter.message.EMAVoiceMessageBody;
import com.hyphenate.util.EMFileHelper;

import java.io.File;

/**
 * \~chinese
 * 文件类消息的基类
 * 
 * \~english
 * The base class of file type message
 */
public abstract class EMFileMessageBody extends EMMessageBody {

	/**
	 * \~chinese
	 * 文件下载状态枚举类
	 *
	 * \~english
	 * Enum of file download status
	 */
	public enum EMDownloadStatus {
        DOWNLOADING,
        SUCCESSED,
        FAILED,
        PENDING
    }

	/**
	 * \~chinese
	 * 文件类型消息体构造方法
	 * @param localPath	文件的绝对路径或者资源标志符的字符串形式
	 *
	 * \~english
	 * Constructor method of file message body
	 * @param localPath	The absolute path to the file or the path to the Uri as a string
	 */
	public EMFileMessageBody(String localPath) {
    	this(EMFileHelper.getInstance().formatInUri(localPath));
	}

	/**
	 * \~chinese
	 * 文件类型消息体构造方法
	 * @param localPath	文件的资源标志符
	 *
	 * \~english
	 * Constructor method of file message body
	 * @param localPath Uri of file
	 */
	public EMFileMessageBody(Uri localPath) {
		this(localPath, EMAMessageBody.EMAMessageBodyType_FILE);
	}

	/**
	 * @param localPath
	 * @param type
	 */
	EMFileMessageBody(String localPath, int type) {
    	this(Uri.fromFile(new File(localPath)), type);
	}

	EMFileMessageBody(Uri localPath, int type) {
		switch (type) {
		case EMAMessageBody.EMAMessageBodyType_FILE:
			emaObject = new EMAFileMessageBody(localPath, type);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_IMAGE:
			emaObject = new EMAImageMessageBody(localPath, null);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_VIDEO:
			emaObject = new EMAVideoMessageBody(localPath, null);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_VOICE:
		    emaObject = new EMAVoiceMessageBody(localPath, 0);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
		    break;
		default:
			break;
		}
	}

	public EMFileMessageBody(EMAFileMessageBody body) {
		emaObject = body;
	}
	

	/**
	 * \~chinese
	 * 获得文件名
	 * 
	 * @return	文件名
	 * 
	 * \~english
	 * Get file name
	 * @return File name
	 */
	public String getFileName() {
		return ((EMAFileMessageBody)emaObject).displayName();
	}

	/**
	 * \~chinese
	 * 设置文件名
	 * @param fileName 文件名
	 *
	 * \~english
	 * Set file name
	 * @param fileName File name
	 */
	public void setFileName(String fileName) {
		((EMAFileMessageBody)emaObject).setDisplayName(fileName);
	}

	/**
	 * \~chinese
	 * 本地图片，语音等文件的路径或者资源标志符的字符串形式
	 * 
	 * @return 本地文件的绝对路径及资源标志符的字符串形式
	 *
	 * \~english
	 * Local file path or the string form of Uri
	 *
	 * @return Local file path or the string form of Uri
	 */
	public String getLocalUrl() {
		return EMFileHelper.getInstance().formatOutLocalUrl(((EMAFileMessageBody)emaObject).getLocalUrl());
	}

	/**
	 * \~chinese
	 * 设置本地文件路径
	 * @param localUrl	文件的绝对路径或者资源标志符的字符串形式
	 *
	 * \~english
	 * Set local file path
	 * @param localUrl	The absolute path to the file or the string form of Uri
	 */
	public void setLocalUrl(String localUrl) {
		((EMAFileMessageBody)emaObject).setLocalPath(EMFileHelper.getInstance().formatInUriToString(localUrl));
	}

	/**
	 * \~chinese
	 * 本地图片，语音等文件的资源标识符
	 *
	 * @return
	 *
	 * \~english
	 * Get the Uri of local image, voice and so on
	 */
	public Uri getLocalUri() {
		return EMFileHelper.getInstance().formatOutUri(((EMAFileMessageBody)emaObject).getLocalUrl());
	}

	/**
	 * \~chinese
	 * 设置本地文件的资源标识符
	 * @param localUrl	文件的资源标志符
	 *
	 * \~english
	 * Set the Uri of local file
	 * @param localUrl	The Uri of file
	 */
	public void setLocalUrl(Uri localUrl) {
		((EMAFileMessageBody)emaObject).setLocalPath(EMFileHelper.getInstance().formatInUriToString(localUrl));
	}

	/**
	 * \~chinese
	 * 远程图片，声音等的服务器路径
	 * @return	文件的服务器路径
	 * 
	 * \~english
	 * Get file's remote url
	 * @return File's remote url
	 */
	public String getRemoteUrl() {
		return ((EMAFileMessageBody)emaObject).getRemoteUrl();
	}

	/**
	 * \~chinese
	 * 设置文件消息体文件的服务器地址
	 * @param remoteUrl 文件的服务器地址
	 *
	 * \~english
	 * Set the remote url for file message body
	 * @param remoteUrl File's remote url
	 */
	public void setRemoteUrl(String remoteUrl) {
		((EMAFileMessageBody)emaObject).setRemotePath(remoteUrl);
	}

	/**
	 * \~chinese
	 * 设置文件的访问令牌，此方法一般由SDK调用
	 * @param secret 文件的访问令牌
	 *
	 * \~english
	 * Set file's token, this method is typically invoked by SDK
	 * @param secret File's token
	 */
	public void setSecret(String secret) {
		((EMAFileMessageBody)emaObject).setSecretKey(secret);
	}

	/**
	 * \~chinese
	 * 获取服务器生成的文件的访问令牌
	 * @return 服务器生成的文件的访问令牌
	 * \~english
	 * 获取服务器生成的文件的token
	 * @return 服务器生成的文件的token
	 */
	public String getSecret() {
		return ((EMAFileMessageBody)emaObject).getSecret();
	}

	/**
	 * \~chinese
	 * 获取文件名
	 * @return 文件名
	 *
	 * \~english
	 * Get the  filename
	 * @return The filename
	 */
	public String displayName() {
		return ((EMAFileMessageBody)emaObject).displayName();
	}

	/**
	 * \~chinese
	 * 设置文件大小
	 * @param length  文件大小
	 *
	 * \~english
	 * Set file's length
	 * @param length  File length
	 */
	public void setFileLength(long length) {
		((EMAFileMessageBody)emaObject).setFileLength(length);
	}

	/**
	 * \~chinese
	 * 获取文件的下载状态
	 * @return 文件的下载状态
	 *
	 * \~english
	 * Get the file download status
	 * @return File download status
	 */
	public EMDownloadStatus downloadStatus() {
	    EMADownloadStatus _status = ((EMAFileMessageBody)emaObject).downloadStatus();
	    switch (_status) {
	    case DOWNLOADING:
	        return EMDownloadStatus.DOWNLOADING;
	    case SUCCESSED:
	        return EMDownloadStatus.SUCCESSED;
	    case FAILED:
	        return EMDownloadStatus.FAILED;
	    case PENDING:
	        return EMDownloadStatus.PENDING;
	    }
	    return EMDownloadStatus.SUCCESSED;
	}

	/**
	 * \~chinese
	 * 设置文件的下载状态
	 * @param status	文件的下载状态
	 *
	 * \~english
	 * Set the file download status
	 * @param status	File download status
	 */
	public void setDownloadStatus(EMDownloadStatus status) {
		((EMAFileMessageBody)emaObject).setDownloadStatus(EMADownloadStatus.valueOf(status.name()));
	}

}
