/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.chat.EMMessage.ChatType;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAConversation.EMASearchDirection;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * \~chinese
 * 会话类，表示和一个用户/群组/聊天室的对话，包含发送和接收的消息
 * 例如，从会话中获取未读消息数：
 * <pre>
 *     // conversationId可以为聊天的另一方id，群组id或者聊天室id
 *     EMConversation conversation = EMClient.getInstance().chatManager().getConversation(conversationId);
 *     int unread = conversation.getUnreadMsgCount();
 * </pre>
 * @version 3.0
 * 
 * \~english
 * Conversation class, represent a conversation with a user/group/chat room and contains messages
 * that are sent and received
 * For example, Gets the number of unread messages from the conversation:
 * <pre>
 *     // ConversationId can be the other party id, the group id, or the chat room id
 *     EMConversation conversation = EMClient.getInstance().chatManager().getConversation(conversationId);
 *     int unread = conversation.getUnreadMsgCount();
 * </pre>
 * @version 3.0
*/

public class EMConversation extends EMBase<EMAConversation> {

    private static final String TAG = "conversation";
    private static final int LIST_SIZE = 512;

    EMConversation(EMAConversation conversation) {
        emaObject = conversation;
    }

    /**
     * \~chinese
     * 会话类型枚举类
     * 包含单聊，群聊，聊天室，讨论组(暂时不可用)，客服
     *
     * \~english
     * Conversation type enum class
     * Including single chat, group chat, chat room, discussion group(temporarily unavailable), help desk
     */
    public enum EMConversationType {
        /**
         *\~chinese 
         * 单聊会话
         *
         *\~english
         * One-to-one chat
         */
        Chat,

        /**
         *\~chinese
         * 群聊会话
         *
         *\~english
         * Group chat
         */
        GroupChat,

        /**
         *\~chinese
         *聊天室会话
         *\~english
         *Chat room
         */
        ChatRoom,

        /**
         *\~chinese
         * 目前没有使用
         *\~english
         * Not current used
         */
        DiscussionGroup,

        /**
         * \~chinese
         * 客服会话
         *
         * \~english
         * Help desk
         */
        HelpDesk
    }

    /**
     *  \~chinese
     *  消息搜索方向枚举类
     *
     *  \~english
     *  Message search direction enum class
     */
    public enum EMSearchDirection {
        /**
         *\~chinese
         *  向上搜索 
         *
         *\~english
         *  Search older messages 
         */
        UP,     

        /**
         *\~chinese
         *  向下搜索
         *
         *\~english
         *  Search newer messages
         */
        DOWN    
    }

    /**
     * \~chinese
     * 会话ID
     * 对于单聊类型，会话ID同时也是对方用户的名称
     * 对于群聊类型，会话ID同时也是对方群组的ID，并不同于群组的名称
     * 对于聊天室类型，会话ID同时也是聊天室的ID，并不同于聊天室的名称
     * 对于HelpDesk类型，会话ID与单聊类型相同，是对方用户的名称
     * @return 会话ID
     *
     * \~english
     * Conversation ID
     * For single chat，conversation ID is to chat user's name
     * For group chat, conversation ID is group ID, different with group name
     * For chat room, conversation ID is chatroom ID, different with chat room name
     * For help desk, it is same with single chat, conversationID is also chat user's name
     * @return Conversation ID
     */
    public String conversationId() {
        return emaObject.conversationId();
    }

    /**
     * \~chinese
     * 获取会话类型
     * @return 会话类型
     *  
     * \~english
     * Get conversation type
     * @return Conversation type
     */
    public EMConversationType getType() {
        EMAConversationType t = emaObject._getType();
        if (t == EMAConversationType.CHAT) {
            return EMConversationType.Chat;
        }
        if (t == EMAConversationType.GROUPCHAT) {
            return EMConversationType.GroupChat;
        }
        if (t == EMAConversationType.CHATROOM) {
            return EMConversationType.ChatRoom;
        }
        if (t == EMAConversationType.DISCUSSIONGROUP) {
            return EMConversationType.DiscussionGroup;
        }
        if (t == EMAConversationType.HELPDESK) {
            return EMConversationType.HelpDesk;
        }
        return EMConversationType.Chat;
    }

    /**
     * \~chinese
     * 获取此对话中未读的消息数量
     * @return 未读的消息数量
     *
     * \~english
     * Get unread message count
     * @return Unread message count
     */
    public int getUnreadMsgCount() {
        return emaObject.unreadMessagesCount();
    }

    /**
     *  \~chinese
     *  将所有未读消息设置为已读
     *
     *  \~english
     *  Mark all messages as read
     */
    public void markAllMessagesAsRead() {
        emaObject.markAllMessagesAsRead(true);
    }

    /**
     * \~chinese
     * 获取本地存储中会话的全部消息数目
     * @return 会话的全部消息数量
     *
     * \~english
     * Get all messages count in this conversation
     * @return ALl messages count in this conversation
     */
    public int getAllMsgCount() {
        return emaObject.messagesCount();
    }

    /**
     * \~chinese
     * 从数据库中加载更多消息
     * 根据传入的参数从db加载startMsgId之前(存储顺序)指定数量的消息，
     * 加载到的messages会加入到当前会话的缓存中，通过{@link #getAllMessages()}将会返回所有加载的消息
     *
     * @param startMsgId    加载这个id之前的message，如果传入""或者null，将从最近的消息开始加载
     * @param pageSize      加载多少条
     * @return              消息列表
     *
     * \~english
     * Load more messages from database
     * Load message from database by passing in parameters, which message id starting from param startMsgId,
     * messages will also store in to current conversation's memory cache,
     * so next time calling {@link #getAllMessages()}, result will contain those messages
     *
     * @param startMsgId    message storage time will before this ID, if startMsgId is "" or null,
     *                      will load last messages in database
     * @param pageSize      message count to be loaded
     * @return              messages
     */
    public List<EMMessage> loadMoreMsgFromDB(String startMsgId, int pageSize) {
        return loadMoreMsgFromDB(startMsgId, pageSize, EMSearchDirection.UP);
    }

    /**
     * \~chinese
     * 根据传入的参数从db加载startMsgId之前(存储顺序)指定数量的message，
     * 加载到的messages会加入到当前conversation的messages里
     *
     * @param startMsgId    加载这个id之前的message
     * @param pageSize      加载多少条
     * @param direction     消息加载的方向
     * @return              消息列表
     *
     * \~english
     * load message from database, message id starting from param startMsgId, messages will also store in to memory cache
     * so next time calling getAllMessages(), result will contain those messages
     *
     * @param startMsgId    message storage time will before this ID, if startMsgId is "" or null, will load last messages in database.
     * @param pageSize      message count to be loaded.
     * @param direction     direction in which the message is loaded
     * @return              messages
     */
    public List<EMMessage> loadMoreMsgFromDB(String startMsgId, int pageSize, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;
        List<EMAMessage> msgs = emaObject.loadMoreMessages(startMsgId, pageSize, d);
        List<EMMessage> result = new ArrayList<EMMessage>();
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        getCache().addMessages(result);
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的消息
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param timeStamp  搜索消息的时间点
     * @param maxCount   搜索结果的最大条数
     * @return           消息列表
     *
     * \~english
     * Search message from database according the parameters
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param timeStamp  The timestamp for search
     * @param maxCount   The max number of message to search
     * @return           The list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(long timeStamp, int maxCount, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(timeStamp, maxCount, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的消息
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param type       消息类型，文本、图片、语音等等
     * @param timeStamp  搜索消息的时间点
     * @param maxCount   搜索结果的最大条数
     * @param from       搜索来自某人的消息，也可用于搜索群组里的消息
     * @return           消息列表
     *
     * \~english
     * Search message from database according the parameters
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param type       Message type, TXT、VOICE、IMAGE and so on
     * @param timeStamp  The timestamp for search
     * @param maxCount   The max number of message to search
     * @param from       Who the message from, also can used to search in group
     * @return           The list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(EMMessage.Type type, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的消息
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param keywords   搜索消息中的关键词
     * @param timeStamp  搜索消息的时间点
     * @param maxCount   搜索结果的最大条数
     * @param from       搜索来自某人的消息，也可用于搜索群组里的消息
     * @return           消息列表
     *
     * \~english
     * Search message from database based the parameters
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param keywords   The keywords in message.
     * @param timeStamp  The timestamp for search
     * @param maxCount   The max number of message to search
     * @param from       Who the message from, also can used to search in group
     * @return           The list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索指定数量的消息
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param startTimeStamp   搜索的起始时间
     * @param endTimeStamp     搜索的结束时间
     * @param maxCount         搜索结果的最大条数
     * @return                 消息列表
     *
     * \~english
     * Search message from database based the parameters
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param startTimeStamp    Start timestamp for search in
     * @param endTimeStamp      End timestamp for search
     * @param maxCount          The max number of message to search
     * @return                  The list of searched messages
     */
    public List<EMMessage> searchMsgFromDB(long startTimeStamp, long endTimeStamp, int maxCount) {
        List<EMAMessage> msgs = emaObject.searchMessages(startTimeStamp, endTimeStamp, maxCount);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据传入的参数从本地存储中搜索自定义消息中指定数量的消息
     * 注意：当maxCount非常大时，需要考虑内存消耗
     *
     * @param keywords   搜索消息中的关键词
     * @param timeStamp  搜索消息的时间点
     * @param maxCount   搜索结果的最大条数
     * @param from       搜索来自某人的消息，也可用于搜索群组里的消息
     * @return           消息列表
     *
     * \~english
     * Search custom messages from database based the parameters
     * Note: Cautious about memory usage when the maxCount is large
     *
     * @param keywords   The keywords in message.
     * @param timeStamp  The timestamp for search
     * @param maxCount   The max number of message to search
     * @param from       Who the message from, also can used to search in group
     * @return           The list of searched messages
     */
    public List<EMMessage> searchCustomMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMSearchDirection direction) {
        EMASearchDirection d = direction == EMSearchDirection.UP ? EMASearchDirection.UP : EMASearchDirection.DOWN;

        List<EMAMessage> msgs = emaObject.searchCustomMessages(keywords, timeStamp, maxCount, from, d);
        // to avoid resizing issue of array list, used linked list when size > 512
        List<EMMessage> result;
        if (msgs.size() > LIST_SIZE) {
            result = new LinkedList<EMMessage>();
        } else {
            result = new ArrayList<EMMessage>();
        }
        for (EMAMessage msg : msgs) {
            if (msg != null) {
                result.add(new EMMessage(msg));
            }
        }
        return result;
    }

    /**
     * \~chinese
     * 根据messageId获取指定的消息
     * 优先从内存中加载，如果内存中没有则从数据库中加载，并将其插入到内存中
     * 
     * @param  messageId    消息id
     * @param  markAsRead   是否获取消息的同时标记消息为已读，如果标记为已读，则会发送已读标记到服务器
     * @return              获取到的消息实例
     *
     * \~english
     * Get message by message ID,
     * If the message already loaded into memory cache, will directly return the message,
     * otherwise load message from database, and put it into cache
     *
     * @param  messageId     Message id
     * @param  markAsRead    Whether to mark the message as read while getting it, if true,
     *                       will mark the message as read and send read ack to server
     * @return message       Return the message instance
     */
    public EMMessage getMessage(String messageId, boolean markAsRead) {
        EMMessage msg = getCache().getMessage(messageId);
        if (msg == null) {
            EMAMessage emaMsg = emaObject.loadMessage(messageId);
            if (emaMsg == null) {
                return null;
            }
            msg = new EMMessage(emaMsg);
        }
        emaObject.markMessageAsRead(messageId, markAsRead);
        return msg;
    }

    /**
     * \~chinese
     * 加载一组消息
     * 如果缓存不存在会去DB查询并加载
     *
     * @param msgIds    一组消息ID
     * @return          返回一组消息,如果找不到返回null
     *
     * \~english
     * Load messages,
     * If those message not exists in memory in cache, will load message from database
     *
     * @param  msgIds   message ids to be loaded
     * @return          Return message list, if not find will return null
     */
    @Deprecated
    public List<EMMessage> loadMessages(List<String> msgIds) {

        List<EMMessage> msgs = new ArrayList<EMMessage>();

        for (String msgId : msgIds) {
            EMAMessage msg = emaObject.loadMessage(msgId);
            if (msg != null) {
                msgs.add(new EMMessage(msg));
            }
        }
        getCache().addMessages(msgs);
        return msgs;
    }

    /**
     * \~chinese
     * 设置指定的消息为已读
     * 将消息置为已读，也可用{@link EMMessage#setUnread(boolean)}
     *
     * @param messageId 消息ID
     *
     * \~english
     * Mark the message as read by specified messageId
     * To make the message read, you can also use {@link EMMessage#setUnread(boolean)}
     * @param messageId Message id
     */
    public void markMessageAsRead(String messageId) {
        emaObject.markMessageAsRead(messageId, true);
    }

    /**
     * \~chinese
     * 获取此conversation当前内存所有的message
     * 如果内存中为空，再从db中加载最近的一条消息
     *
     * @return 消息列表
     *
     * \~english
     * Get all message in local cache
     * If empty, then get latest message from local database
     *
     * @return Message list
     */
    public List<EMMessage> getAllMessages() {
        if (getCache().isEmpty()) {
            EMAMessage lastMsg = emaObject.latestMessage();
            List<EMMessage> msgs = new ArrayList<EMMessage>();
            if (lastMsg != null) {
                msgs.add(new EMMessage(lastMsg));
            }
            getCache().addMessages(msgs);
        }
        return getCache().getAllMessages();
    }

    /**
     * \~chinese
     * 删除一条指定的消息
     * 注：只会对本地数据进行操作
     *
     * @param messageId     待删除消息的ID
     *
     * \~english
     * Delete a message
     * Note: Operates only on local data
     *
     * @param messageId     message to be deleted
     */
    public void removeMessage(String messageId) {
        EMLog.d(TAG, "remove msg from conversation: " + messageId);
        emaObject._removeMessage(messageId);
        getCache().removeMessage(messageId);
    }

    /**
     * \~chinese
     * 获取消息队列中的最后一条消息
     * 此操作不会改变未读消息计数
     * 优先从缓存中获取，如果缓存中没有，则从数据库中加载最近的一条消息
     *
     * @return 消息
     *
     * \~english
     * Get last message from conversation
     * The operation does not change the unread message count
     * Get from cache, if empty, will load from database and put it in the cache
     *
     * @return Message instance
     */
    public EMMessage getLastMessage() {
        if (getCache().isEmpty()) {
            EMAMessage _msg = emaObject.latestMessage();
            EMMessage msg = _msg == null ? null : new EMMessage(_msg);
            getCache().addMessage(msg);
            return msg;
        } else {
            return getCache().getLastMessage();
        }
    }

    /**
     * \~chinese
     * 获取会话接收到的最后一条消息
     * @return 消息
     *
     * \~english
     * Get received latest message from conversation
     * @return Message instance
     */
    public EMMessage getLatestMessageFromOthers() {
        EMAMessage _msg = emaObject.latestMessageFromOthers();
        EMMessage msg = _msg == null ? null : new EMMessage(_msg);
        getCache().addMessage(msg);
        return msg;
    }

    /**
     * \~chinese
     * 清除对话中的所有消息
     * 只清除内存的，不清除db的消息
     * 在退出会话的时候清除内存缓存，减小内存消耗
     *
     * \~english
     * Clear messages in this conversation
     * Which clear only memory messages, but will not clear local database
     * Reduce memory consumption by clearing the memory cache when exiting a conversation
     */
    public void clear() {
        getCache().clear();
    }

    /**
     * \~chinese
     * 删除会话中所有消息，同时清除内存和数据库中的消息
     *
     * \~english
     * Delete all messages of the conversation from memory cache and local database
     */
    public void clearAllMessages() {
        emaObject.clearAllMessages();
        getCache().clear();
    }

    /**
     * \~chinese
     * 设置会话扩展字段
     * 该字段只保存在本地，不进行网络同步
     *
     * @param ext       会话对应扩展字段的内容
     *
     * \~english
     * Set Conversation extension
     * Extend field only stored in local database, not sync to network server
     *
     * @param ext       Extension string
     */
    public void setExtField(String ext) {
        emaObject._setExtField(ext);
    }

    /**
     * \~chinese
     * 获取会话扩展字段
     * 该字段只保存在本地，不进行网络同步
     *
     * @return      会话对应扩展字段的内容
     *
     * \~english
     * Get conversation extension
     * Extend field only stored in local database, not sync to network server
     *
     * @return Extension string
     */
    public String getExtField() {
        return emaObject.extField();
    }

    /**
     * \~chinese
     * 从消息类型到会话类型的转化
     *
     * @param       id   消息Id，用来区分客服和单聊，对于其他类型，这个参数没有影响
     * @param       type 消息类型
     * @return      会话类型
     *
     * \~english
     * Provide transformation from message type to conversation type
     *
     * @param       id   Message Id, used to distinguish single chat and help desk, has no effect on other chat type
     * @param       type Message type
     * @return      Conversation type
     */
    public static EMConversationType msgType2ConversationType(String id, EMMessage.ChatType type) {
        if (type == ChatType.Chat) {
//			if(EMCustomerService.getInstance().isCustomServiceAgent(id)){
//				return EMConversationType.HelpDesk;
//			}
            return EMConversationType.Chat;
        } else if (type == ChatType.GroupChat) {
            return EMConversationType.GroupChat;
        } else if (type == ChatType.ChatRoom) {
            return EMConversationType.ChatRoom;
        }

        return EMConversationType.Chat;
    }

    /**
     * \~chinese
     * 获取是否是群组或者聊天室会话
     *
     * @return 群组和聊天室类型都会返回true, 其他类型返回false
     *
     * \~english
     * Gets whether it is a group or chat room conversation
     *
     * @return Group chat and chatroom chat will both return true, otherwise return false
     */
    public boolean isGroup() {
        EMConversationType type = getType();
        return type == EMConversationType.GroupChat ||
                type == EMConversationType.ChatRoom;
    }

    /**
     *  \~chinese
     *  插入一条消息到本地数据库
     *  消息的conversationId应该和会话的conversationId一致，消息会被插入DB，并且更新会话的latestMessage等属性
     *
     *  @param msg 消息实例
     *
     *  \~english
     *  Insert a message to a conversation in local database
     *  ConversationId of the message should be the same as conversationId of the conversation in order to
     *  insert the message into the conversation correctly. And the latestMessage and other properties of the session should be updated
     *
     *  @param msg Instance of Message
     */
    public boolean insertMessage(EMMessage msg) {
        boolean result = emaObject.insertMessage(msg.emaObject);
        if(result) {
            getCache().addMessage(msg);
        }
        return result;
    }

    /**
     *  \~chinese
     *  插入一条消息到会话尾部
     *  消息的conversationId应该和会话的conversationId一致，消息会被插入DB，并且更新会话的latestMessage等属性
     *
     *  @param msg 消息实例
     *
     *  \~english
     *  Insert a message to the end of a conversation in local database
     *  ConversationId of the message should be the same as conversationId of the conversation in order to
     *  insert the message into the conversation correctly. And the latestMessage and other properties of the session should be updated
     *
     *  @param msg Message
     */
	public boolean appendMessage(EMMessage msg) {
	    boolean result = emaObject.appendMessage(msg.emaObject);
	    if(result) {
            getCache().addMessage(msg);
	    }
	    return result;
	}

    /**
     *  \~chinese
     *  更新本地的消息
     *  不能更新消息ID，消息更新后，会话的latestMessage等属性进行相应更新
     *
     *  @param msg 要更新的消息
     *
     *  \~english
     *  Update a message in local database
     *  Message id of the message cannot be updated. Latest Message of the conversation and other properties will be updated accordingly.
     *
     *  @param msg Message
     */
    public boolean updateMessage(EMMessage msg) {
        boolean updateMessage = emaObject.updateMessage(msg.emaObject);
        if(updateMessage) {
            getCache().addMessage(msg);
        }
        return updateMessage;
    }

    /**
     * \~chinese
     * 返回会话对应的附件存储路径
     * 该方法适用于清理该会话磁盘存储，不确保该路径一定存在，请在删除对应路径前加以判断，并加上异常保护
     * @return 附件存储路径
     *
     * \~english
     * Return conversation associated attachment path
     * Which can be used to erase conversation related downloaded files from local storage.
     * Not ensure the return path exists, please handle IOException when deleting directory
     *
     * @return Attachment path
     */
    public String getMessageAttachmentPath() {
        String downloadPath = EMClient.getInstance().getChatConfigPrivate().getDownloadPath();
        return downloadPath + "/" + EMClient.getInstance().getCurrentUser()
                + "/" + conversationId();
    }


    // ====================================== Message cache ======================================

    MessageCache getCache() {
        MessageCache cache;
        synchronized (EMClient.getInstance().chatManager().caches) {
            cache = EMClient.getInstance().chatManager().caches.get(emaObject.conversationId());
            if (cache == null) {
                cache = new MessageCache();
            }
            EMClient.getInstance().chatManager().caches.put(emaObject.conversationId(), cache);
        }
        return cache;
    }

    static class MessageCache {

        TreeMap<Long, Object> sortedMessages = new TreeMap<Long, Object>(new MessageComparator());
        Map<String, EMMessage> messages = new HashMap<String, EMMessage>();
        Map<String, Long> idTimeMap = new HashMap<String, Long>();
        boolean hasDuplicateTime = false;

        final boolean sortByServerTime = EMClient.getInstance().getChatConfigPrivate().getOptions().isSortMessageByServerTime();

        class MessageComparator implements Comparator<Long> {

            @Override
            public int compare(Long time0, Long time1) {
                long val = time0 - time1;
                if (val > 0) {
                    return 1;
                } else if (val == 0) {
                    return 0;
                } else {
                    return -1;
                }
            }
        }

        public synchronized EMMessage getMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return null;
            }
            return messages.get(msgId);
        }

        public synchronized void addMessages(List<EMMessage> msgs) {
            for (EMMessage msg : msgs) {
                addMessage(msg);
            }
        }

        public synchronized void addMessage(EMMessage msg) {
            if (msg == null || msg.emaObject == null || msg.getMsgTime() == 0 || msg.getMsgTime() == -1 || msg.getMsgId() == null
                    || msg.getMsgId().isEmpty() || msg.getType() == Type.CMD) {
                return;
            }
            String id = msg.getMsgId();
            // override message
            if (messages.containsKey(id)) {
                long time = idTimeMap.get(id);
                sortedMessages.remove(time);
                messages.remove(id);
                idTimeMap.remove(id);
            }
            // messages share same time stamp
            long time = sortByServerTime ? msg.getMsgTime() : msg.localTime();
            if (sortedMessages.containsKey(time)) {
                hasDuplicateTime = true;
                Object v = sortedMessages.get(time);
                if (v != null) {
                    if (v instanceof EMMessage) {
                        List<EMMessage> msgs = new LinkedList<>();
                        msgs.add((EMMessage) v);
                        msgs.add(msg);
                        sortedMessages.put(time, msgs);
                    } else if (v instanceof List) {
                        List<EMMessage> msgs = (List<EMMessage>) v;
                        msgs.add(msg);
                    }
                }
            } else {
                sortedMessages.put(time, msg);
            }
            messages.put(id, msg);
            idTimeMap.put(id, time);
        }

        public synchronized void removeMessage(String msgId) {
            if (msgId == null || msgId.isEmpty()) {
                return;
            }
            EMMessage msg = messages.get(msgId);
            if (msg != null) {
                Long time = idTimeMap.get(msgId);
                if (time != null) {
                    if (hasDuplicateTime && sortedMessages.containsKey(time)) {
                        Object v = sortedMessages.get(time);
                        if (v != null && v instanceof List) {
                            List<EMMessage> msgs = (List)v;
                            for (EMMessage m : msgs) {
                                if (m != null && m.getMsgId() != null && m.getMsgId().equals(msgId)) {
                                    msgs.remove(m);
                                    break;
                                }
                            }
                        } else {
                            sortedMessages.remove(time);
                        }
                    } else {
                        sortedMessages.remove(time);
                    }
                    idTimeMap.remove(msgId);
                }
                messages.remove(msgId);
            }
        }

        public synchronized List<EMMessage> getAllMessages() {
            List<EMMessage> list = new ArrayList<EMMessage>();
            if (hasDuplicateTime == false) {
                for (Object v : sortedMessages.values()) {
                    list.add((EMMessage)v);
                }
            } else {
                for (Object v : sortedMessages.values()) {
                    if (v != null) {
                        if (v instanceof List) {
                            list.addAll((List<EMMessage>)v);
                        } else {
                            list.add((EMMessage)v);
                        }
                    }
                }
            }
            return list;
        }

        public synchronized EMMessage getLastMessage() {
            if (sortedMessages.isEmpty()) {
                return null;
            }

            Object o = sortedMessages.lastEntry().getValue();
            if (o == null) {
                return null;
            }
            if (o instanceof EMMessage) {
                return (EMMessage)o;
            } else if (o instanceof List){
                List<EMMessage> msgs = (List<EMMessage>)o;
                if (msgs.size() > 0) {
                    return msgs.get(msgs.size() - 1);
                }
                return null;
            }
            return null;
        }

        public synchronized void clear() {
            //no need to keep the last message
            sortedMessages.clear();
            messages.clear();
            idTimeMap.clear();
        }

        public synchronized boolean isEmpty() {
            return sortedMessages.isEmpty();
        }
    }
}
