/************************************************************
 *  * hyphenate CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 hyphenate Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of hyphenate Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from hyphenate Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMContactListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAContactListener;
import com.hyphenate.chat.adapter.EMAContactManager;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * \~chinese
 * EMContactManager用来记录，查询，修改用户的联系人/好友列表
 *
 * \~english
 * EMContactManager is used to manage friends, including recording, querying, modifying and so on
 */
public class EMContactManager {

    EMAContactManager emaObject;
    EMClient mClient;
    private List<EMContactListener> contactListeners = Collections.synchronizedList(new ArrayList<EMContactListener>());
    private EMAContactListenerImpl contactImpl = new EMAContactListenerImpl();

    EMContactManager(EMClient client, EMAContactManager contactManager) {
        mClient = client;
        emaObject = new EMAContactManager(contactManager);
        emaObject.registerContactListener(contactImpl);
    }

    class EMAContactListenerImpl extends EMAContactListener {

        @Override
        public void onContactAdded(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactAdded(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactDeleted(String username) {
            EMClient.getInstance().chatManager().caches.remove(username);

            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactDeleted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactInvited(String username, String reason) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactInvited(username, reason);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactAgreed(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onFriendRequestAccepted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactRefused(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onFriendRequestDeclined(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void handleError(EMAError error)  throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    /**
     * \~chinese
     * 添加联系人
     * 异步方法见{@link #aysncAddContact(String, String, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username		用户名
     * @param reason		原因、理由（optional. use "" or null)
     * @throws HyphenateException
     *
     * \~english
     * Add a new user
     * Asynchronously method see {@link #aysncAddContact(String, String, EMCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @param username		The user to add
     * @param reason     	message for adding contact (optional. use "" or null))
     * @throws HyphenateException
     */
    public void addContact(String username, String reason) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.inviteContact(username, reason, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 添加联系人
     * 同步方法见{@link #addContact(String, String)}
     *
     * @param username  用户名
     * @param reason    原因、理由（optional)
     * @param callback
     *
     * \~english
     * Add a new user
     * Synchronization method see {@link #addContact(String, String)}
     *
     * @param username  user id
     * @param reason    message for adding contact (optional)
     * @param callback
     */
    public void aysncAddContact(final String username, final String reason, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addContact(username, reason);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 删除好友及其相关的会话
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username
     * @throws HyphenateException
     *
     * \~english
     * Delete a contact and all the conversations associated
     *
     * Synchronization method will block the current thread
     *
     * @param username
     * @throws HyphenateException
     */
    public void deleteContact(String username) throws HyphenateException {
        deleteContact(username, false);
    }

    /**
     * \~chinese
     * 删除好友
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username
     * @param keepConversation  是否保留会话和消息
     * @throws HyphenateException
     *
     * \~english
     * Delete a contact
     *
     * Synchronization method will block the current thread
     *
     * @param username
     * @param keepConversation  If to keep the associated conversation and messages
     * @throws HyphenateException
     */
    public void deleteContact(String username, boolean keepConversation) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.deleteContact(username, error, keepConversation);
        EMClient.getInstance().chatManager().caches.remove(username);
        handleError(error);
    }

    /**
     * \~chinese
     * 删除好友
     * 同步方法见{@link #deleteContact(String)}
     *
     * @param username  要删除的用户ID
     * @param callback
     *
     * \~english
     * Delete a contact
     * Synchronization method see {@link #deleteContact(String)}
     *
     * @param username   The user to delete
     * @param callback
     */
    public void aysncDeleteContact(final String username,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    deleteContact(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从服务器获取所有的好友
     * 异步方法见{@link #aysncGetAllContactsFromServer(EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @return 联系人列表
     * @throws HyphenateException
     *
     * \~english
     * Get all contacts from the server
     * Asynchronously method see {@link #aysncGetAllContactsFromServer(EMValueCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @return list of contacts
     * @throws HyphenateException
     */
    public List<String> getAllContactsFromServer() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> contacts = emaObject.getContactsFromServer(error);
        handleError(error);
        return contacts;
    }

    /**
     * \~chinese
     * 从服务器获取所有的好友
     * 同步方法见{@link #getAllContactsFromServer()}
     *
     * @param callback 包含联系人列表
     *
     * \~english
     * Get all contacts from the server
     * Synchronization method see {@link #getAllContactsFromServer()}
     *
     * @param callback Contains list of contacts
     */
    public void aysncGetAllContactsFromServer(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> allContacts = getAllContactsFromServer();
                    callback.onSuccess(allContacts);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    //region contactListener
    /**
     * \~chinese
     * 注册联系人监听
     * 调用{@link #removeContactListener(EMContactListener)}移除监听
     *
     * @param contactListener
     *
     * \~english
     * Register new contact listener
     * Call {@link #removeContactListener(EMContactListener)} to remove listener
     *
     * @param contactListener
     */
    public void setContactListener(EMContactListener contactListener) {
        if (!contactListeners.contains(contactListener)) {
            contactListeners.add(contactListener);
        }
    }

    /**
     * \~chinese
     * 移除联系人监听
     * 添加联系人监听调用{@link #setContactListener(EMContactListener)}
     *
     * \~english
     * Remove contact listener
     * Add contact listener calls {@link #setContactListener(EMContactListener)}
     */
    public void removeContactListener(EMContactListener contactListener) {
        contactListeners.remove(contactListener);
    }
    //endregion


    /**
     * \~chinese
     * 把指定用户加入到黑名单中
     * 可以给黑名单的中用户发消息，但是接收不到对方发送的消息
     * 异步方法见{@link #aysncAddUserToBlackList(String, boolean, EMCallBack)}
     * 批量加入黑名单见{@link #saveBlackList(List)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username  此用户ID
     * @param both      此参数已废弃
     * @throws HyphenateException
     *
     * \~english
     * Add a user to blacklist
     * You can send message to the user in blacklist, but you can not receive the message sent by the other
     * Asynchronously method see {@link #aysncAddUserToBlackList(String, boolean, EMCallBack)}
     * Add batch blacklist to see {@link #saveBlackList(List)}
     *
     * Synchronization method will block the current thread
     *
     * @param username  User to be blocked
     * @param both      This parameter is deprecated
     * @throws HyphenateException
     */
    public void addUserToBlackList(String username, boolean both) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.addToBlackList(username, both, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 把指定用户加入到黑名单中
     * 可以给黑名单的中用户发消息，但是接收不到对方发送的消息
     * 同步方法见{@link #addUserToBlackList(String, boolean)}
     * 批量加入黑名单见{@link #asyncSaveBlackList(List, EMCallBack)}
     *
     * @param username  此用户ID
     * @param both      此参数已废弃
     * @param callback
     *
     * \~english
     * Add the user to blacklist
     * You can send message to the user in blacklist, but you can not receive the message sent by the other
     * Synchronization method see {@link #addUserToBlackList(String, boolean)}
     * Add batch blacklist to see {@link #asyncSaveBlackList(List, EMCallBack)}
     *
     * @param username  User to be blocked
     * @param both      This parameter is deprecated
     * @param callback
     */
    public void aysncAddUserToBlackList(final String username,
                                        final boolean both,
                                        final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addUserToBlackList(username, both);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 上传黑名单列表到服务器
     * 异步方法见{@link #asyncSaveBlackList(List, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @param blackList
     *
     * \~english
     * Upload blacklist to server
     * Asynchronously method see {@link #asyncSaveBlackList(List, EMCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @param blackList
     */
    public void saveBlackList(List<String> blackList) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.saveBlackList(blackList, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 上传黑名单列表到服务器
     * 同步方法见{@link #saveBlackList(List)}
     *
     * @param blackList
     * @param callback
     *
     * \~english
     * Upload blacklist to server
     * Synchronization method see {@link #saveBlackList(List)}
     *
     * @param blackList
     * @param callback
     */
    public void asyncSaveBlackList(final List<String> blackList,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    saveBlackList(blackList);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 把用户从黑名单中移除
     * 异步方法见{@link #aysncRemoveUserFromBlackList(String, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username
     * @throws HyphenateException
     *
     * \~english
     * Remove the contact from blacklist
     * Asynchronously method see {@link #aysncRemoveUserFromBlackList(String, EMCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @param username
     * @throws HyphenateException
     */
    public void removeUserFromBlackList(String username) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.removeFromBlackList(username, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 把用户从黑名单中移除
     * 同步方法见{@link #removeUserFromBlackList(String)}
     *
     * @param username
     * @param callback
     *
     * \~english
     * Remove the contact from blacklist
     * Synchronization method see {@link #removeUserFromBlackList(String)}
     *
     * @param username
     * @param callback
     */
    public void aysncRemoveUserFromBlackList(final String username,
                                             final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    removeUserFromBlackList(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从本地获取黑名单列表
     *
     * @return List
     *
     * \~english
     * Get local blacklist
     *
     * @return List
     */
    public List<String> getBlackListUsernames() {
        EMAError error = new EMAError();
        return emaObject.getBlackListFromDB(error);
    }

    /**
     * \~chinese
     * 从服务器获取黑名单列表
     * 异步方法见{@link #aysncGetBlackListFromServer(EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @return
     * @throws HyphenateException
     *
     * \~english
     * Get local blacklist
     * Asynchronously method see {@link #aysncGetBlackListFromServer(EMValueCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @return
     * @throws HyphenateException
     */
    public List<String> getBlackListFromServer() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> blacklist = emaObject.getBlackListFromServer(error);
        handleError(error);
        return blacklist;
    }

    /**
     * \~chinese
     * 从服务器获取黑名单列表
     * 同步方法见{@link #getBlackListFromServer()}
     *
     * @param callback 包含黑名单列表
     *
     * \~english
     * Get blacklist from server
     * Synchronization method see {@link #getBlackListFromServer()}
     *
     * @param callback Contains black list
     */
    public void aysncGetBlackListFromServer(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> blackList = getBlackListFromServer();
                    callback.onSuccess(blackList);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 接受加好友的邀请
     * 异步方法见{@link #asyncAcceptInvitation(String, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username
     *
     * \~english
     *
     * Accept a friend invitation
     * Asynchronously method see {@link #asyncAcceptInvitation(String, EMCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @param username
     */
    public void acceptInvitation(String username) throws HyphenateException
    {
        EMAError error = new EMAError();
        emaObject.acceptInvitation(username, error);
        handleError(error);
    }


    /**
     * \~chinese
     * 接受加好友的邀请
     * 同步方法见{@link #acceptInvitation(String)}
     *
     * 异步方法
     *
     * @param username
     * @param callback
     *
     * \~english
     *
     * Accept a friend invitation
     * Synchronization method see {@link #acceptInvitation(String)}
     *
     * Asynchronously method
     *
     * @param username
     * @param callback
     */
    public void asyncAcceptInvitation(final String username,
                                      final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    acceptInvitation(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 拒绝加好友的邀请
     * 异步方法见{@link #asyncDeclineInvitation(String, EMCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username
     *
     * \~english
     * Decline a friend invitation
     * Asynchronously method see {@link #asyncDeclineInvitation(String, EMCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @param username
     */
    public void declineInvitation(String username) throws HyphenateException  {
        EMAError error = new EMAError();
        emaObject.declineInvitation(username, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 拒绝加好友的邀请
     * 同步方法见{@link #declineInvitation(String)}
     *
     * 异步方法
     *
     * @param username
     * @param callback
     *
     * \~english
     * Decline a friend invitation
     * Synchronization method see {@link #declineInvitation(String)}
     *
     * Asynchronously method
     *
     * @param username
     * @param callback
     */
    public void asyncDeclineInvitation(final String username,
                                       final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    declineInvitation(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从数据库获取好友列表
     *
     * @return 好友列表
     * @throws HyphenateException
     *
     * \~english
     * Get contacts from local database
     *
     * @return contact list
     * @throws HyphenateException
     */
    public List<String> getContactsFromLocal() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> contacts = getContactsFromDB(error);
        handleError(error);
        return contacts;
    }

    /**
     * \~chinese
     * 从数据库获取好友列表
     *
     * @param error
     * @return 好友列表
     *
     * \~english
     * get contacts from local database
     *
     * @param error
     * @return contact list
     */
    List<String> getContactsFromDB(EMAError error) {
        return emaObject.getContactsFromDB(error);
    }

    void onLogout() {
    }

    /**
     * \~chinese
     * 从服务器获取登录用户在其他设备上登录的ID
     * 异步方法见{@link #aysncGetSelfIdsOnOtherPlatform(EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程
     *
     * @return
     * @throws HyphenateException
     *
     * \~english
     * Get IDs of current user on other platform
     * Asynchronously method see {@link #aysncGetSelfIdsOnOtherPlatform(EMValueCallBack)}
     *
     * Synchronization method will block the current thread
     *
     * @return
     * @throws HyphenateException
     */
    public List<String> getSelfIdsOnOtherPlatform() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> selfIds = emaObject.getSelfIdsOnOtherPlatform(error);
        handleError(error);
        return selfIds;
    }

    /**
     * \~chinese
     * 从服务器获取登录用户在其他设备上登录的ID
     * 同步方法见{@link #getSelfIdsOnOtherPlatform()}
     *
     * 异步方法
     *
     * @param callback 包含用户在其他设备上登录的ID
     *
     * \~english
     * Get IDs of current user on other platform
     * Synchronization method see {@link #getSelfIdsOnOtherPlatform()}
     *
     * Asynchronously method
     *
     * @param callback Contains self id list on other platform
     */
    public void aysncGetSelfIdsOnOtherPlatform(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> selfIds = getSelfIdsOnOtherPlatform();
                    callback.onSuccess(selfIds);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
}
