/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.annotation.TargetApi;
import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.provider.Settings.Secure;
import android.text.TextUtils;
import android.util.Log;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMConnectionListener;
import com.hyphenate.EMError;
import com.hyphenate.EMMultiDeviceListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatClient;
import com.hyphenate.chat.adapter.EMAChatClient.EMANetwork;
import com.hyphenate.chat.adapter.EMAConnectionListener;
import com.hyphenate.chat.adapter.EMADeviceInfo;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAMultiDeviceListener;
import com.hyphenate.chat.adapter.EMANetCallback;
import com.hyphenate.chat.core.EMAdvanceDebugManager;
import com.hyphenate.chat.core.EMChatConfigPrivate;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.monitor.EMNetworkMonitor;
import com.hyphenate.notification.core.EMNotificationHelper;
import com.hyphenate.push.EMPushConfig;
import com.hyphenate.push.EMPushHelper;
import com.hyphenate.push.EMPushType;
import com.hyphenate.util.CryptoUtils;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.NetUtils;
import com.hyphenate.util.PathUtil;
import com.hyphenate.util.Utils;

import org.json.JSONException;
import org.json.JSONObject;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;

import javax.net.ssl.X509TrustManager;

import internal.com.getkeepsafe.relinker.ReLinker;

import static com.hyphenate.util.EasyUtils.convertToCerts;
import static com.hyphenate.util.EasyUtils.getSystemDefaultTrustManager;

/**
 * \~chinese
 * Chat SDK的入口，负责登录退出及连接管理等，由此可以获得其他模块的入口
 * <pre>
 *  EMChatManager chatManager = EMClient.getInstance().chatManager();
 * </pre>
 * <p>
 * \~english
 * Chat SDK Client, entrance of SDK, used to login, logout, and get access chat modules, such as
 * <pre>
 *  EMChatManager chatManager = EMClient.getInstance().chatManager();
 * </pre>
 */
public class EMClient {
    public final static String TAG = "EMClient";
    static private EMClient instance = null;
    static boolean libraryLoaded = false;

    private EMGroupManager groupManager;
    private EMChatRoomManager chatroomManager;
    private EMChatManager chatManager;
    private EMContactManager contactManager;
    private EMUserInfoManager userInfoManager;
    private EMPushManager pushManager;

    private EMAChatClient emaObject;
    private Context mContext;
    private ExecutorService executor = null;
    private ExecutorService mainQueue = Executors.newSingleThreadExecutor();
    private ExecutorService sendQueue = Executors.newSingleThreadExecutor();
    private EMEncryptProvider encryptProvider = null;
    private CryptoUtils cryptoUtils = new CryptoUtils();
    private boolean sdkInited = false;
    private EMChatConfigPrivate mChatConfigPrivate;
    private List<EMConnectionListener> connectionListeners = Collections.synchronizedList(new ArrayList<EMConnectionListener>());

	private MyConnectionListener connectionListener;
	private EMSmartHeartBeat smartHeartbeat = null;
    private List<EMMultiDeviceListener> multiDeviceListeners = Collections.synchronizedList(new ArrayList<EMMultiDeviceListener>());
    private MyMultiDeviceListener multiDeviceListenerImpl;
	private WakeLock wakeLock;

	private ConnectivityManager connManager;
	private EMANetwork currentNetworkType = EMANetwork.NETWORK_NONE;

	public final static String VERSION = "3.8.6.1";

	/**
	 * preventing client to instantiate this EMClient
	 *
	 */
	private EMClient() {
	}

	public static EMClient getInstance() {
		if(instance == null){
		    synchronized (EMClient.class) {
		        if(instance == null) {
                    instance = new EMClient();
		        }
            }
		}
		return instance;
	}


    /**
    * \~chinese
    * 初始化SDK。
    * 需要在主进程中进行初始化。
    * @param context 上下文，不可为空
    * @param options 配置项，不可为空，见{@link EMOptions}
    *
    * \~english
    * Initialize the SDK
    * Need initialize The SDK in main process
    * @param context Not be null
    * @param options Configurations, not be null, see {@link EMOptions}
    */
    public void init(Context context, EMOptions options) {
        if(sdkInited){
            return;
        }

        final EMTimeTag tag = new EMTimeTag();
        tag.start();

        mContext = context.getApplicationContext();
        connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        //if android sdk >= 14,register callback to know app enter the background or foreground
        registerActivityLifecycleCallbacks();

        loadLibrary();

        mChatConfigPrivate = new EMChatConfigPrivate();
        mChatConfigPrivate.load(context, options);
        options.setConfig(mChatConfigPrivate);

        // multi-device is enabled by setDeviceUuid
        // config setDeviceUuid before EMAChatClient.create being called
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(context);
        getChatConfigPrivate().setDeviceUuid(deviceFactory.getDeviceUuid().toString());
        getChatConfigPrivate().setDeviceName(Build.MANUFACTURER + Build.MODEL);

        // Init push relevant.
        EMPushConfig pushConfig = options.getPushConfig();
        if (pushConfig == null) {
            pushConfig = new EMPushConfig.Builder(mContext).build();
        }
        // Merge EMOptions to EMPushConfig
        EMPushConfig.Builder builder = new EMPushConfig.Builder(mContext, pushConfig);
        if (TextUtils.isEmpty(pushConfig.getFcmSenderId())) {
            builder.enableFCM(options.getFCMNumber());
        }
        if (TextUtils.isEmpty(pushConfig.getMiAppId()) || TextUtils.isEmpty(pushConfig.getMiAppKey())) {
            EMChatConfigPrivate.EMMipushConfig miPushConfig = options.getMipushConfig();
            if (miPushConfig != null) {
                builder.enableMiPush(miPushConfig.appId, miPushConfig.appKey);
            }
        }
        EMPushHelper.getInstance().init(context, builder.build());
        emaObject = EMAChatClient.create(mChatConfigPrivate.emaObject);
        // sandbox
        // rest 103.241.230.122:31080 im-msync 103.241.230.122:31097
//        mChatConfigPrivate.setChatServer("118.193.28.212");
//        mChatConfigPrivate.setChatPort(31097);
//        mChatConfigPrivate.setRestServer("https://118.193.28.212:31443");
//        mChatConfigPrivate.enableDnsConfig(false);

        connectionListener = new MyConnectionListener();
        emaObject.addConnectionListener(connectionListener);

        multiDeviceListenerImpl = new MyMultiDeviceListener();
        emaObject.addMultiDeviceListener(multiDeviceListenerImpl);

        executor = Executors.newCachedThreadPool();

        cryptoUtils.init(CryptoUtils.ALGORIGHM_AES);

        // init all the managers
        initManagers();

        final String lastLoginUser = EMSessionManager.getInstance().getLastLoginUser();

        EMLog.e(TAG, "is autoLogin : " + options.getAutoLogin());
        EMLog.e(TAG, "lastLoginUser : " + lastLoginUser);
        EMLog.e(TAG, "hyphenate SDK is initialized with version : " + getChatConfigPrivate().getVersion());

        PowerManager pm = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);
        wakeLock = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "emclient");
        EMNotificationHelper.getInstance().onInit(context);
        sdkInited = true;

        if (options.getAutoLogin() && isLoggedInBefore()) {
                final String lastLoginToken = EMSessionManager.getInstance().getLastLoginToken();
                final String lastLoginPwd = EMSessionManager.getInstance().getLastLoginPwd();

                //sessionManager.login(lastLoginUser, lastLoginPwd,false, null);
                EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);

                final EMCallBack callback = new EMCallBack() {

                    @Override
                    public void onSuccess() {
                        EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);
                        Log.d(TAG, "hyphenate login onSuccess");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onError(int code, String error) {
                        Log.d(TAG, "hyphenate login onError");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onProgress(int progress, String status) {
                    }
                };

                this.execute(new Runnable() {

                    @Override
                    public void run() {
                        // to ensure database has been open before login,
                        // login takes long time before user can see the result
                        getChatConfigPrivate().openDatabase(lastLoginUser);
                        groupManager().loadAllGroups();
                        chatManager().loadAllConversationsFromDB();

                        _login(lastLoginUser,
                                EMSessionManager.getInstance().isLastLoginWithToken() ? lastLoginToken : lastLoginPwd,
                                callback,
                                /*autoLogin*/ true,
                                /*loginWithToken*/ EMSessionManager.getInstance().isLastLoginWithToken());
                    }
                });
        } else {
            tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
        }
    }

    /**
     * \~chinese
     * 创建账号
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username 账号，不可为空，可以是字母/数字/下划线/横线/英文句号，正则为"^[a-zA-Z0-9_.-]+$"，
     *                 其他的都不允许。如果是大写字母会自动转成小写。长度不可超过64个字符长度。
     * @param password 密码，不可为空。长度不可超过64个字符长度。
     *
     * @throws HyphenateException {@param username}账号，{@param password}密码为空，或者账号不符合要求会报错。
     *    注册账号时发生的错误也会此抛出。
     *
     * \~english
     * create an chat account on server
     *
     * Synchronization method block the current thread
     *
     * @param username The account you want to register, not be null, can be a letter/number/underline/
     *                 cross/English period, regular as "^[a-zA-Z0-9_.-]+$", other is not allowed. It will
     *                 automatically be converted to lowercase if it is a capital letter.
     *                 The username's length can not be longer than 64 bytes.
     * @param password The password you want to register, not be null.
     *                 The password's length can not be longer than 64 bytes.
     *
     * @throws HyphenateException If user account or password is null, or account is illegal will throw exception.
     *    The exception while registering account will be thrown here.
     */
    public void createAccount(String username, String password) throws HyphenateException {
        username = username.toLowerCase();

        Pattern pattern = Pattern.compile("^[a-zA-Z0-9_.-]+$");
        boolean find = pattern.matcher(username).find();
        if (!find) {
            throw new HyphenateException(EMError.USER_ILLEGAL_ARGUMENT, "illegal user name");
        }
        EMAError error = emaObject.createAccount(username, password);
        handleError(error);
    }

    /**
     * \~chinese
     * 通过用户ID和密码登录chat服务器
     * 通过用户ID和token登录chat 服务器见{@link #loginWithToken(String, String, EMCallBack)}
     *
     * 异步方法
     *
     * @param id          用户id             不能为空
     * @param password    用户密码           不能为空
     * @param callback    EMCallback回调函数 不能为空，登录的结果将通过callback返回。
     *
     * @throws IllegalArgumentException 用户id，密码，callback及appkey为空会抛出异常。
     *
     * \~english
     * Login to chat server by login id and password
     * Login to chat server by login id and token, see {@link #loginWithToken(String, String, EMCallBack)}
     *
     * Asynchronously method
     *
     * @param id 		Unique chat Login ID       Not be null
     * @param password 	Password for this chat ID  Not be null
     * @param callback 	Login callback           Not be null, the result of login is returned via the callback
     *
     * @throws IllegalArgumentException If id, password, callback or appkey is null throws an exception.
     */
    public void login(String id, String password, final EMCallBack callback) throws IllegalArgumentException {

        if (callback == null) {
            throw new IllegalArgumentException("callback is null!");
        }

        if (id == null || password == null || id.equals("") || password.equals("")) {
            throw new IllegalArgumentException("username or password is null or empty!");
        }

        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            throw new IllegalArgumentException("please setup your appkey either in AndroidManifest.xml or through the EMOptions");
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        id = id.toLowerCase();
        _login(id, password, callback, false, false);
    }

    /**
     * \~chinese
     * 通过用户id和token登录chat 服务器
     * 另通过用户id和密码登录chat 服务器，见{@link #login(String, String, EMCallBack)}
     *
     * 异步方法
     *
     * @param username 用户id             不能为空
     * @param token    验证token          不能为空
     * @param callback EMCallback回调函数 不能为空，登录的结果将通过callback返回
     *
     * @throws RuntimeException          appkey为空回报运行异常
     * @throws IllegalArgumentException  用户id，token及callbak为空，会抛出异常
     *
     * \~english
     * Login to chat server by login id and token
     * Other method login to chat server by login id and token, see {@link #login(String, String, EMCallBack)}
     *
     * Asynchronously method
     *
     * @param username Unique chat Login ID        Not be null
     * @param token    Password for this chat ID   Not be null
     * @param callback Login callback            Not be null, the result of login is returned via the callback
     *
     * @throws RuntimeException          If appkey is null throws a RuntimeException
     * @throws IllegalArgumentException  If Id, token or callback is null throws an exception
     */
    public void loginWithToken(String username, String token, final EMCallBack callback) {
        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            throw new RuntimeException("please setup your appkey either in AndroidManifest.xml or through the EMOptions");
        }

        if (callback == null) {
            throw new IllegalArgumentException("callback is null!");
        }

        if (username == null || token == null || username.equals("") || token.equals("")) {
            throw new IllegalArgumentException("username or password is null or empty!");
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        username = username.toLowerCase();
        _login(username, token, callback, false, true);
    }

    /**
     * \~chinese
     * 退出登录并返回退出结果
     *
     * 同步方法，会阻塞当前线程
     *
     * @param unbindToken 是否解绑token true表示需要退出时解绑设备token
     *                                  false表示退出时不解绑设备token
     * @return 返回值为0，即{@link EMError#EM_NO_ERROR}，表示退出登录成功；其他值为退出失败，详见{@link EMError}
     *
     * \~english
     * logout and return the result
     *
     * Synchronization method block the current thread
     *
     * @param unbindToken whether unbind token  true means to unbind the device token when logout
     *                                          false means to not unbind the device token when logout
     * @return The return value is 0 {@link EMError#EM_NO_ERROR}, means successful logout; Other values are
     *         exit failure, see {@link EMError}
     */
    public int logout(boolean unbindToken) {
        if (!emaObject.isLogout()) {
            boolean result = EMPushHelper.getInstance().unregister(unbindToken);
            if (!result) {
                return EMError.USER_UNBIND_DEVICETOKEN_FAILED;
            }
        } else {
            EMPushHelper.getInstance().unregister(false);
            EMLog.e(TAG, "already logout, skip unbind token");
        }

        logout();

        return EMError.EM_NO_ERROR;
    }

    /**
     * \~chinese
     * 同步登出聊天服务器
     * 如果需要异步请参考{@link EMClient#logout(EMCallBack callback)}
     *
     * \~english
     * logout hyphenate chat server synchronously
     * to use asynchronously API, see {@link EMClient#logout(EMCallBack callback)}
     */
    void logout() {
        EMLog.d(TAG, " SDK Logout");

        try {
            if (connectivityBroadcastReceiver != null) {
                mContext.unregisterReceiver(connectivityBroadcastReceiver);
            }
        } catch (Exception e) {
        }

        EMSessionManager.getInstance().clearLastLoginUser();
        EMSessionManager.getInstance().clearLastLoginToken();

        // ============ gaode code start ===============
        // EMGDLocation.getInstance().onDestroy();
        // ============ gaode code end ===============

        if (smartHeartbeat != null) {
            smartHeartbeat.stop();
        }

        if (wakeLock.isHeld()) {
            wakeLock.release();
        }

        // make sure we clear the username and pwd in a sync way
        if (emaObject != null) {
            emaObject.logout();
        }
        if (chatManager != null) {
            chatManager.onLogout();
        }
        if (groupManager != null) {
            groupManager.onLogout();
        }
        if (contactManager != null) {
            contactManager.onLogout();
        }
        if (chatroomManager != null) {
            chatroomManager.onLogout();
        }

        // clear all memory cache
        try {
            EMAdvanceDebugManager.getInstance().onDestroy();
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (EMChatConfigPrivate.isDebugTrafficMode()) {
            EMNetworkMonitor.stopTrafficStat();
        }
    }

    /**
     * \~chinese
     * 登出聊天服务器
     *
     * 异步方法
     *
     * @param unbindToken 是否解绑token   true表示需要退出时解绑设备token
     *                                    false表示退出时不解绑设备token
     * @param callback    EMCallback回调  {@link EMCallBack#onSuccess()}表示退出登录成功的回调
     *                                    {@link EMCallBack#onError(int, String)}表示退出登录失败的回调，
     *                                    第一个参数代表错误码，见{@link EMError}，第二个参数代表错误描述
     *
     * \~english
     * Logout hyphenate chat server
     *
     * Asynchronously method
     *
     * @param unbindToken  whether unbind token  true means to unbind the device token when logout
     *                                           false means to not unbind the device token when logout
     * @param callback     EMCallback            {@link EMCallBack#onSuccess()} indicates a successful logout callback
     *                                           {@link EMCallBack#onError(int, String)} indicates a failure logout callback,
     *                                           the first parameter means the error code, see {@link EMError}, the second parameter
     *                                           means the error description
     */
    public void logout(final boolean unbindToken, final EMCallBack callback) {
        new Thread() {
            @Override
            public void run() {
                int error = logout(unbindToken);

                if (error != EMError.EM_NO_ERROR) {
                    if (callback != null) {
                        callback.onError(error, "faild to unbind device token");
                    }
                } else {
                    if (callback != null) {
                        callback.onSuccess();
                    }
                }
            }
        }.start();
    }

    /**
     * \~chinese
     * 异步登出聊天服务器
     *
     * @param callback    EMCallback回调
     *
     * \~english
     * logout hyphenate chat server synchronously
     *
     * @param callback     EMCallback
     */
    void logout(final EMCallBack callback) {
        Thread logoutThread = new Thread() {
            @Override
            public void run() {
                if (callback != null) {
                    callback.onProgress(0, null);
                }

                logout();

                if (callback != null) {
                    callback.onSuccess();
                }
            }
        };
        logoutThread.setPriority(Thread.MAX_PRIORITY - 1);
        logoutThread.start();
    }

    /**
     * \~chinese
     * 更改appkey
     * 只有在未登录状态才能修改appkey
     * 也可以通过{@link EMOptions#setAppKey(String)}设置appkey，同样只有在未登录状态才能修改
     *
     * @param appkey  不可为空
     *
     * \~english
     * Change appkey
     * Can ONLY be changed if not logged in
     * Also can set appkey by {@link EMOptions#setAppKey(String)}, need be set if not logged in
     *
     * @param appkey  Not be {@link null} or ""
     */
    public void changeAppkey(String appkey) throws HyphenateException {
        EMAError error = emaObject.changeAppkey(appkey);
        if (error.errCode() == EMAError.EM_NO_ERROR) {
            this.getOptions().updatePath(appkey);
        }
        handleError(error);
    }

    /**
     * \~chinese
     * 设置chat服务器连接监听
     *
     * @param listener chat服务器连接监听  {@link EMConnectionListener#onConnected()}表示与chat服务器连接成功
     *                                  {@link EMConnectionListener#onDisconnected(int)}表示与chat服务器断开连接,
     *                                  参数表示错误码，详见{@link EMError}
     *
     * \~english
     * Add chat server connection listener
     *
     * @param listener Chat server connection listener  {@link EMConnectionListener#onConnected()} indicates a successful connection to chat server
     *                                                {@link EMConnectionListener#onDisconnected(int)} indicates a failure connection to chat server,
     *                                                the parameter means the error code, see {@link EMError}
     */
    public void addConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }

        synchronized (connectionListeners) {
            if (!connectionListeners.contains(listener)) {
                connectionListeners.add(listener);
            }
        }

        execute(new Runnable() {

            @Override
            public void run() {
                if (isConnected()) {
                    listener.onConnected();
                } else {
                    listener.onDisconnected(EMError.NETWORK_ERROR);
                }
            }
        });
    }

    /**
     * \~chinese
     * 移除chat服务器连接监听
     *
     * @param listener  Chat服务器连接监听
     *
     * \~english
     * Remove chat server connection listener
     *
     * @param listener  Chat server connection listener
     */
    public void removeConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (connectionListeners) {
            connectionListeners.remove(listener);
        }
    }

    /**
     * \~chinese
     * 获取群组管理类，需要在EMClient初始化完成后调用，详见{@link #init(Context, EMOptions)}
     *
     * @return 群组管理类
     *
     * \~english
     * Gets group manager, calls it after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     *
     * @return The group manager
     */
    public EMGroupManager groupManager() {
        if (groupManager == null) {
            synchronized (EMClient.class) {
                if(groupManager == null) {
                    groupManager = new EMGroupManager(this, emaObject.getGroupManager());
                }
            }
        }
        return groupManager;
    }

    /**
     * \~chinese
     * 获取推送管理类，需要在EMClient初始化完成后调用，详见{@link #init(Context, EMOptions)}
     *
     * @return 推送管理类
     *
     * \~english
     * Gets push manager, calls it after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     *
     * @return The push manager
     */
    public EMPushManager pushManager() {
        if (pushManager == null) {
            synchronized (EMClient.class) {
                if(pushManager == null) {
                    pushManager = new EMPushManager(this, emaObject.getPushMnager());
                }
            }
        }
        return pushManager;
    }

    /**
     * \~chinese
     * 获取聊天室管理类，需要在EMClient初始化完成后调用，详见{@link #init(Context, EMOptions)}
     *
     * @return 聊天室管理类
     *
     * \~english
     * Gets chat room manager, calls it after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     *
     * @return The chat room manager
     */
    public EMChatRoomManager chatroomManager() {
        if (chatroomManager == null) {
            synchronized (EMClient.class) {
                if(chatroomManager == null) {
                    chatroomManager = new EMChatRoomManager(this, emaObject.getChatRoomManager());
                }
            }
        }
        return chatroomManager;
    }

    /**
     * \~chinese
     * 获取聊天管理类，需要在EMClient初始化完成后调用，详见{@link #init(Context, EMOptions)}
     *
     * @return 聊天管理类
     *
     * \~english
     * Gets chat manager, calls it after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     *
     * @return The chat manager
     */
    public EMChatManager chatManager() {
        if (chatManager == null) {
            synchronized (EMClient.class) {
                if(chatManager == null) {
                    chatManager = new EMChatManager(this, emaObject.getChatManager());
                }
            }
        }
        return chatManager;
    }


    /**
     * \~chinese
     * 获取用户信息管理类，需要在EMClient初始化完成后调用，详见{@link #init(Context, EMOptions)}
     *
     * @return 用户信息管理类
     *
     * \~english
     * Gets user info manager, calls it after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     *
     * @return The user info manager
     */
    public EMUserInfoManager userInfoManager() {
        if (userInfoManager == null) {
            synchronized (EMClient.class) {
                if(userInfoManager == null) {
                    userInfoManager = new EMUserInfoManager(emaObject.getUserInfoManager());
                }
            }
        }
        return userInfoManager;
    }



    /**
     * \~chinese
     * 获取联系人管理类，需要在EMClient初始化完成后调用，详见{@link #init(Context, EMOptions)}
     *
     * @return 联系人管理类
     *
     * \~english
     * Gets contact manager, calls it after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     *
     * @return The contact manager
     */
    public EMContactManager contactManager() {
        if (contactManager == null) {
            synchronized (EMClient.class) {
                if(contactManager == null) {
                    contactManager = new EMContactManager(this, emaObject.getContactManager());
                }
            }
        }
        return contactManager;
    }

    public Context getContext() {
        return mContext;
    }

    /**
     * \~chinese
     * 获取当前登录用户的用户名
     *
     * @return 当前登录的用户
     *
     * \~english
     * Gets current login user ID
     *
     * @return current login user
     */
    public synchronized String getCurrentUser() {
        if (EMSessionManager.getInstance().currentUser == null ||
                EMSessionManager.getInstance().currentUser.username == null ||
                EMSessionManager.getInstance().currentUser.username.equals("")) {
            return EMSessionManager.getInstance().getLastLoginUser();
        }
        return EMSessionManager.getInstance().currentUser.username;
    }

    /**
     * \~chinese
     * 根据用户名和密码获取token
     *
     * @param username 已经注册过的用户名
     * @param password 密码
     * @param callBack 结果回调   {@link EMValueCallBack#onSuccess(Object)}表示请求成功的回调，onSuccess中的参数即为返回的token
     *                            {@link EMValueCallBack#onError(int, String)}表示请求失败的回调，第一个参数是错误码，
     *                            第二个参数是错误描述
     *
     * \~english
     * Fetch token by username and password
     *
     * @param username The registered username
     * @param password
     * TODO: check
     * @param callBack Result callback  {@link EMValueCallBack#onSuccess(Object)} indicates a successful callback，the parameter in onSuccess is the result of token
     *                                  {@link EMValueCallBack#onError(int, String)} indicates a failure callback, the first parameter is the error code,
     *                                  the second parameter is the error description
     */
    public void getUserTokenFromServer(final String username, final String password, final EMValueCallBack<String> callBack) {
        execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                final String token = emaObject.getUserTokenFromServer(username, password, error);

                if (callBack == null) {
                    return;
                }

                if (error.errCode() == EMError.EM_NO_ERROR) {
                    callBack.onSuccess(token);
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 返回是否登录过
     * 登录成功过没调logout方法，这个方法的返回值一直是true
     * 如果需要判断当前是否连接到服务器，请使用{@link #isConnected()}方法
     *
     * <pre>
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // enter main activity
     * }else{
     *     // enter login activity
     * }
     * </pre>
     *
     * @return 返回是否登录过的结果  true表示之前登录过
     *                              false表示之前未登录过或者已经调用过{@link #logout()}方法
     *
     * \~english
     * Uses to check if user has been logged in before and did not logout
     * If you need check if connected to server, please use {@link #isConnected()}
     *
     * <pre>
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // enter main activity
     * }else{
     *     // enter login activity
     * }
     * </pre>
     *
     * @return The result if has login before  true means that has login before
     *                                         false means that has not login before or has called {@link #logout()} method
     */
    public boolean isLoggedInBefore() {
        EMSessionManager sessionMgr = EMSessionManager.getInstance();
        String user = sessionMgr.getLastLoginUser();
        String pwd = sessionMgr.getLastLoginPwd();
        String token = sessionMgr.getLastLoginToken();

        if (user != null  && !user.isEmpty() &&
                ((pwd != null   && !pwd.isEmpty()) ||
                 (token != null && !token.isEmpty())) ) {
            return true;
        }

        return false;
    }


    /**
     * \~chinese
     * 检查是否连接到chat服务器
     *
     * @return 返回是否连接到chat服务器的结果  true表示已经连接到chat服务器
     *                                      false表示没有连接到chat服务器
     *
     * \~english
     * Checks if connected to chat server.
     *
     * @return Return the result whether connected to chat server  true means that has connected to chat server
     *                                                           false means that has not connected to chat server
     */
    public boolean isConnected() {
        return emaObject.isConnected();
    }

    /**
     * \~chinese
     * 设置是否输出调试信息，在EMClient初始化完成后调用，详见{@link #init(Context, EMOptions)}
     *
     * @param debugMode 传入true，SDK会在log里输出调试信息；传入false则不会输出调试信息
     *
     * \~english
     * Sets whether output debug info, should calls after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     * SDK will out put debug info if debugMode = true
     *
     * @param debugMode  Enter true, SDK will output debug info, enter false will not output
     */
    public void setDebugMode(boolean debugMode) {
        if (sdkInited) {
            //if had set debugmode, use it
            String mode = EMAdvanceDebugManager.getInstance().getDebugMode();
            if (mode != null)
                debugMode = Boolean.parseBoolean(mode);
        }
        EMLog.debugMode = debugMode;
        getChatConfigPrivate().setDebugMode(debugMode);
    }

    /**
     * \~chinese
     * 更新当前用户的推送昵称，用于推送时在通知栏展示
     * 现推荐使用{@link EMPushManager#updatePushNickname(String)}代替
     * 需要与用户属性中的昵称区分开
     * 建议从用户服务器获取到用户信息后，将用户属性中的昵称更新到chat服务器；
     * 当个人信息中昵称发生变化的时候，也要同步更新到chat服务器，防止出现显示差异
     *
     * @param nickname 昵称 不能为空，长度不超过100个字符
     * @deprecated 从V3.3.3废弃，使用 {@link EMPushManager#updatePushNickname(String)} 代替
     *
     * \~english
     * Updates current user's push nickname, which is used to display in the notification bar when pushing
     * Now use {@link EMPushManager#updatePushNickname(String)} instead
     * Need to distinguish the nickname in user properties
     * Recommended to update the push nickname to chat server after getting the user information from the user server;
     * When the nickname of user information change, we need to update to chat server to prevent display discrepancies
     *
     * @param nickname Not be null, the length is not more than 100 characters
     * @deprecated From V3.3.3, use {@link EMPushManager#updatePushNickname(String)} instead
     */
    @Deprecated
    public boolean updateCurrentUserNick(String nickname) throws IllegalArgumentException, HyphenateException {
        return pushManager().updatePushNickname(nickname);
    }

    /**
     * \~chinese
     * 上传本地的日志
     *
     * 同步方法，会阻塞当前线程
     *
     * @param callback 回调方法 此处没有用到此回调方法
     *
     * \~english
     * upload local log file
     *
     * Synchronization method block the current thread
     *
     * @param callback Not use in here
     */
    public void uploadLog(EMCallBack callback) {
        chatManager().emaObject.uploadLog();
    }


    /**
     * \~chinese
     * 获取EMOptions
     *
     * \~english
     * Gets EMOptions
     */
    public EMOptions getOptions() {
        return mChatConfigPrivate.getOptions();
    }


    /**
     * \~chinese
     * 压缩log文件，并返回压缩后的文件路径
     *
     * @return 压缩后的log文件路径
     * @throws HyphenateException 压缩文件失败报的错
     *
     * \~english
     * Compress log folder into zip file, return zip file path
     *
     * @return The path of compressed gz file
     * @throws HyphenateException  The exception by Compress gz file
     */
    public String compressLogs() throws HyphenateException {
        EMAError error = new EMAError();
        String path = emaObject.compressLogs(error);
        handleError(error);
        return path;
    }

    /**
     * \~chinese
     * 添加多设备监听的接口
     *
     * @param listener 见{@link EMMultiDeviceListener}, {@link EMMultiDeviceListener#onContactEvent(int, String, String)}联系人事件回调，
     *                 {@link EMMultiDeviceListener#onGroupEvent(int, String, List)}群组事件回调
     *
     * \~english
     * add multiple devices listener
     * @param listener See {@link EMMultiDeviceListener}, {@link EMMultiDeviceListener#onContactEvent(int, String, String)} is the contact event callback,
     *                 {@link EMMultiDeviceListener#onGroupEvent(int, String, List)} is the group event callback
     */
    public void addMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.add(listener);
    }

    /**
     * \~chinese
     * 删除多设备监听的接口
     * @param listener 见{@link EMMultiDeviceListener}
     *
     * \~english
     * remove multiple devices listener
     * @param listener See {@link EMMultiDeviceListener}
     */
    public void removeMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.remove(listener);
    }

    /**
     * \~chinese
     * 获取指定账号下登陆的在线设备列表
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username  用户名
     * @param password  密码
     * @return 在线设备列表
     * @throws HyphenateException 具体错误详见{@link EMError}
     *
     * \~english
     * Gets all the information about the logged in devices under the specified account
     *
     * Synchronization method block the current thread
     *
     * @param username The account you want to get device info
     * @param password The account's password
     * @return The list of online devices
     * @throws HyphenateException See {@link EMError}
     */
    public List<EMDeviceInfo> getLoggedInDevicesFromServer(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        List<EMADeviceInfo> devices = emaObject.getLoggedInDevicesFromServer(username, password, error);
        handleError(error);
        List<EMDeviceInfo> result = new ArrayList<>();
        for (EMADeviceInfo info : devices) {
            result.add(new EMDeviceInfo(info));
        }
        return result;
    }

    /**
     * \~chinese
     * 在指定账号下，根据设备ID，将指定设备下线, 设备ID：{@link EMDeviceInfo#getResource()}
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username  账户名称
     * @param password  该账户密码
     * @param resource  设备ID, 见{@link EMDeviceInfo#getResource()}
     * @throws HyphenateException 错误码详见{@link EMError}
     *
     * \~english
     * Forced to logout the specified logged in device under the specified account, resource: {@link EMDeviceInfo#getResource()}
     *
     * Synchronization method block the current thread
     *
     * @param username The account you want to kick a device off
     * @param password The account's password
     * @param resource The device ID, see {@link EMDeviceInfo#getResource()}
     * @throws HyphenateException Error code see {@link EMError}
     */
    public void kickDevice(String username, String password, String resource) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickDevice(username, password, resource, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 将指定账号下的所有设备都踢下线
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username 用户名
     * @param password 密码
     * @throws HyphenateException 错误码详见{@link EMError}
     *
     * \~english
     * Forced to logout all logged in devices under the specified account
     *
     * Synchronization method block the current thread
     *
     * @param username The account you want to kick all the devices
     * @param password The account's password
     * @throws HyphenateException Error code see {@link EMError}
     */
    public void kickAllDevices(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickAllDevices(username, password, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 上传FCM token至服务器
     * <pre>
     * token可以被上传的前提条件有：
     * 1.被上传的token不为空
     * 2.当前已有用户登录
     * 3.当前设备支持google推送
     * 4.google推送类型为FCM {@link com.hyphenate.push.EMPushType#FCM}
     * 设置推送类型为FCM的接口为{@link EMOptions#setFCMNumber(String)},即设置了FCM number则推送类型为FCM
     * </pre>
     *
     * @param fcmToken 要上传的token
     *
     * \~english
     * Upload the FCM token to chat server
     * <pre>
     * The token can be uploaded when all the following conditions are met:
     * 1.The token is not empty
     * 2.User logged in
     * 3.Device support google play service
     * 4.Push type is FCM {@link com.hyphenate.push.EMPushType#FCM},
     * You can set with {@link EMOptions#setFCMNumber(String)}
     * </pre>
     * @param fcmToken The token want to upload
     */
    public void sendFCMTokenToServer(String fcmToken) {
        EMLog.i(TAG, "sendFCMTokenToServer: " + fcmToken);
        if (TextUtils.isEmpty(fcmToken)) {
            return;
        }
        // Save the fcm token native.
        String savedToken = EMPushHelper.getInstance().getFCMPushToken();
        if (!fcmToken.equals(savedToken)) {
            EMPushHelper.getInstance().setFCMPushToken(fcmToken);
        }
        // If no user logged in, stop upload the fcm token.
        String userName = getCurrentUser();
        if (TextUtils.isEmpty(userName)) {
            EMLog.i(TAG, "No user login currently, stop upload the token.");
            return;
        }
        // let's check if the push service available
        EMPushType pushType = EMPushHelper.getInstance().getPushType();
        EMLog.i(TAG, "pushType: " + pushType);
        if (pushType == EMPushType.FCM) {
            EMPushHelper.getInstance().onReceiveToken(pushType, fcmToken);
        }
    }

    /**
     * \~chinese
     * 发送华为推送 token 到服务器
     * @param appId 华为 appId
     * @param token 华为推送 token
     * @deprecated  使用 {@link #sendHMSPushTokenToServer(String)} 代替
     *
     * \~english
     * Send Huawei devices token to server
     * @param appId Huawei appId
     * @param token Huawei device token
     * @Deprecated Use {@link #sendHMSPushTokenToServer(String)} for instead.
     */
    @Deprecated
    public void sendHMSPushTokenToServer(String appId, String token){
        sendHMSPushTokenToServer(token);
    }

    /**
     * \~chinese
     * 发送华为推送 token 到服务器
     * @param token 华为推送token
     *
     * \~english
     * Send Huawei devices token to server
     * @param token Huawei device token
     */
    public void sendHMSPushTokenToServer(String token){
        if (EMPushHelper.getInstance().getPushType() == EMPushType.HMSPUSH) {
            EMPushHelper.getInstance().onReceiveToken(EMPushType.HMSPUSH, token);
        }
    }

	//---------------------------private methods ---------------------------------

	private void initManagers(){
		// invoke constructor before login, to listener PB event
		EMHttpClient.getInstance().onInit(mChatConfigPrivate);
		chatManager();
		contactManager();
		groupManager();
		chatroomManager();

		setNatvieNetworkCallback();
		EMSessionManager.getInstance().init(this, emaObject.getSessionManager());
	}

	void _login(final String username, final String code, final EMCallBack callback, final boolean autoLogin, final boolean loginWithToken) {
        if (getChatConfigPrivate() == null || sdkInited == false) {
            callback.onError(EMError.GENERAL_ERROR, "");
            return;
        }

        EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid());

        // convert to lowercase
        execute(new Runnable() {

            @Override
            public void run() {
                EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid() + " threadName:" + Thread.currentThread().getName() + " ID:" + Thread.currentThread().getId());

                if (username == null) {
                    callback.onError(EMError.INVALID_USER_NAME, "Invalid user name");
                    return;
                }

                EMAError error = new EMAError();
                emaObject.login(username, code, autoLogin, loginWithToken, error);

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    EMSessionManager.getInstance().setLastLoginUser(username);
                    EMAError emaError = new EMAError();
                    if (emaError.errCode() == EMAError.EM_NO_ERROR) {
                        if (/*mChatConfigPrivate.getUsingSQLCipher() || */loginWithToken) {
                            String token = emaObject.getUserToken(false, emaError);
                            EMSessionManager.getInstance().setLastLoginToken(token);
                            EMSessionManager.getInstance().setLastLoginWithToken(true);
                            EMSessionManager.getInstance().clearLastLoginPwd();
                        } else {
                            EMSessionManager.getInstance().setLastLoginPwd(code);
                            EMSessionManager.getInstance().setLastLoginWithToken(false);
                            EMSessionManager.getInstance().clearLastLoginToken();
                        }
                    }

                    onNewLogin();
                    EMPushHelper.getInstance().register();
                    // set EMConferenceManager parameters, lite jar build need to remove this line
//                    conferenceManager().set(getAccessToken(), getOptions().getAppKey(), getCurrentUser());
                    callback.onSuccess();
                } else {
                    callback.onError(error.errCode(), error.errMsg());
                }

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    if (getOptions().isEnableStatistics()) {
                        setPresence(getLocationString(autoLogin));
                    } else {
                        EMLog.d(TAG, "statistics is not enabled");
                    }
                }

                if (error.errCode() == EMAError.USER_AUTHENTICATION_FAILED) {
                    EMSessionManager.getInstance().clearLastLoginPwd();
                    EMSessionManager.getInstance().clearLastLoginToken();
                }
            }
        });
    }

    /**
     * \~chinese
     * 检查FCM 推送是否可用
     *
     * @return 返回 true表示FCM 推送可用；返回 false表示FCM 推送不可用
     *
     * \~english
     * Check whether FCM push is available
     *
     * @return Return true means the FCM push is available; return false means not available
     */
    public boolean isFCMAvailable() {
        return EMPushHelper.getInstance().getPushType() == EMPushType.FCM;
    }

    void onNewLogin() {
        EMLog.d(TAG, "on new login created");

        String username = EMSessionManager.getInstance().getLastLoginUser();

        PathUtil.getInstance().initDirs(getChatConfigPrivate().getAppKey(), username, mContext);

        EMAdvanceDebugManager.getInstance().onInit(mChatConfigPrivate);

        mContext.registerReceiver(connectivityBroadcastReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));

        if (smartHeartbeat == null) {
            smartHeartbeat = EMSmartHeartBeat.create(mContext);
        }

        if (getChatConfigPrivate().emaObject.hasHeartBeatCustomizedParams()) {
            smartHeartbeat.setCustomizedParams(getChatConfigPrivate().emaObject.getWifiHeartBeatCustomizedParams(),
                    getChatConfigPrivate().emaObject.getMobileHeartBeatCustomizedParams());
        }
        smartHeartbeat.onInit();
        if (getOptions().getFixedInterval() != -1) {
            smartHeartbeat.setFixedInterval(getOptions().getFixedInterval());
        }
    }

    /**
     * \~chinese
     * 从内存中获取身份认证token
     * 在上传下载附件（语音，图片，文件等）时必须添加到请求header中，当出现任何异常时将返回null，
     * 可通过判断是否为null来检测是否有问题；如果为null，在打开EMLog日志时，是可以看到异常原因
     *
     * 也可以通过{@link EMOptions#getAccessToken()}获取，要从服务端获取token，可以调用{@link EMOptions#getAccessToken(boolean)},
     * 并传入true
     *
     * @return 身份认证token
     *
     * \~english
     * Gets access token from memory
     * When uploading or downloading attachment (voice, image, file and so on), the token should be added to the request header;
     * It is null when any exceptions occurs
     * You can detect whether it is a problem by determining if the token is null. If the token is null, you can open the EMLog file
     * to see the exception
     *
     * Also can get token by {@link EMOptions#getAccessToken()}. If you want to get token from server,
     * can call {@link EMOptions#getAccessToken(boolean)} and enter true
     *
     * @return Access token
     */
    public String getAccessToken() {
        return getChatConfigPrivate().getAccessToken();
    }

    /**
     * \~chinese
     * 判断SDK是否已经初始化完毕
     * @return  SDK是否已经初始化完毕
     *
     * \~english
     * Determine if the SDK has been initialized
     * @return  Whether the SDK has been initialized
     */
    public boolean isSdkInited() {
        return sdkInited;
    }

    private boolean _loadLibrary(final String library, boolean trace) {
        try {
            ReLinker.loadLibrary(mContext, library);
            return true;
        } catch (Throwable e) {
            if (trace) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private boolean _loadLibrary(final String library) {
        return _loadLibrary(library, true);
    }

    private void loadLibrary() {
        if (libraryLoaded == false) {
            //if (_loadLibrary("sqlcipher", false) || _loadLibrary("sqlite")) {}
            //_loadLibrary("sqlite");
            _loadLibrary("sqlcipher", false);
//            _loadLibrary("hyphenate_av");
//            _loadLibrary("hyphenate_av_recorder");
//            ReLinker.loadLibrary(mContext, "crypto");
//            ReLinker.loadLibrary(mContext, "ssl");
            ReLinker.loadLibrary(mContext, "hyphenate");
            libraryLoaded = true;
        }
    }

    class MyConnectionListener extends EMAConnectionListener {

        @Override
        public void onConnected() {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onConnected();
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onDisconnected(final int errCode) {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        if (errCode == EMError.USER_REMOVED) {
                            EMSessionManager.getInstance().clearLastLoginUser();
                            EMSessionManager.getInstance().clearLastLoginToken();
                        } else if (errCode == EMError.USER_LOGIN_ANOTHER_DEVICE ||
                                errCode == EMError.SERVER_SERVICE_RESTRICTED ||
                                errCode == EMError.USER_KICKED_BY_CHANGE_PASSWORD ||
                                errCode == EMError.USER_KICKED_BY_OTHER_DEVICE) {
                            EMSessionManager.getInstance().clearLastLoginToken();
                        }

                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onDisconnected(errCode);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
				}
			});
		}

        @Override
        public boolean verifyServerCert(List<String> certschain) {
            if(certschain==null) {
                EMLog.d(TAG,"List<String> certschain : null ");
                return false;
            }
            X509Certificate[] certsArray=convertToCerts(certschain);
            try {
                X509TrustManager x509TrustManager = getSystemDefaultTrustManager();
                x509TrustManager.checkServerTrusted(certsArray,certsArray[0].getType());
            }catch (Exception e){
                e.printStackTrace();
                EMLog.e(TAG,e.getMessage());
                EMLog.d(TAG,"List<String> certschain :"+certschain.toString());
                EMLog.d(TAG,"certsArray :"+certsArray.toString());
                return false;
            }
            return true;
        }
    }

    class MyMultiDeviceListener extends EMAMultiDeviceListener {
        @Override
        public void onContactEvent(final int event, final String target, final String ext) {
            EMLog.d(TAG, "onContactEvent:" + event + " target:" + target + " ext:" + ext);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onContactEvent(event, target, ext);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onGroupEvent(final int event, final String target, final List<String> usernames) {
            EMLog.d(TAG, "onGroupEvent:" + event + " target:" + target + " usernames:" + usernames);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onGroupEvent(event, target, usernames);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }
    }

	CryptoUtils getCryptoUtils() {
		return cryptoUtils;
	}

    void execute(Runnable runnable) {
        executor.execute(runnable);
    }

    void executeOnMainQueue(Runnable runnable) {
        mainQueue.submit(runnable);
    }

    void executeOnSendQueue(Runnable runnable) {
        sendQueue.submit(runnable);
    }

    // TODO: temp
    public EMChatConfigPrivate getChatConfigPrivate() {
        return mChatConfigPrivate;
    }

    void setNatvieNetworkCallback() {
        EMANetCallback callback = new EMANetCallback() {

            @Override
            public int getNetState() {
                if (!NetUtils.hasDataConnection(mContext)) {
                    return EMANetwork.NETWORK_NONE.ordinal();
                } else {
                    if (NetUtils.isWifiConnected(mContext)
                            || NetUtils.isOthersConnected(mContext)) {
                        return EMANetwork.NETWORK_WIFI.ordinal();
                    } else if (NetUtils.isMobileConnected(mContext)) {
                        return EMANetwork.NETWORK_MOBILE.ordinal();
                    } else if (NetUtils.isEthernetConnected(mContext)) {
                        return EMANetwork.NETWORK_CABLE.ordinal();
                    } else {
                        return EMANetwork.NETWORK_NONE.ordinal();
                    }
                }
            }
        };

        mChatConfigPrivate.emaObject.setNetCallback(callback);
    }

    /**
     * 设置应用的加密实现 如果未设置，SDK 会使用内置的加密算法
     * <p>
     * set the encrypt provider.
     *
     * @param provider
     */
    void setEncryptProvider(EMEncryptProvider provider) {
        this.encryptProvider = provider;
    }

    /**
     * 获取EncryptProvider 如果未设置, 将返回sdk 内置的encrypt provider
     *
     * @return
     */
    EMEncryptProvider getEncryptProvider() {
        if (encryptProvider == null) {
            EMLog.d(TAG, "encrypt provider is not set, create default");
            encryptProvider = new EMEncryptProvider() {

                public byte[] encrypt(byte[] input, String username) {
                    try {
                        return cryptoUtils.encrypt(input);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

                public byte[] decrypt(byte[] input, String username) {
                    try {
                        return cryptoUtils.decrypt(input);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

            };
        }
        return encryptProvider;
    }

    boolean sendPing(boolean waitPong, long timeout) {
        return emaObject.sendPing(waitPong, timeout);
    }

    void forceReconnect() {
        EMLog.d(TAG, "forceReconnect");
        disconnect();
        reconnect();
    }

    void reconnect() {
        EMLog.d(TAG, "reconnect");
        wakeLock.acquire();
        emaObject.reconnect();
        if (wakeLock.isHeld()) {
            wakeLock.release();
        }
    }

    void disconnect() {
        EMLog.d(TAG, "disconnect");
        emaObject.disconnect();
    }

    private void handleError(EMAError error) throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    /**
     * the connectivity change listener
     * it will only interrupt the reconnection thread and reset the attempts times
     */
    private BroadcastReceiver connectivityBroadcastReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(final Context context, Intent intent) {
            String action = intent.getAction();
            if (!action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
                EMLog.d(TAG, "skip no connectivity action");
                return;
            }

            EMLog.d(TAG, "connectivity receiver onReceiver");

            EMANetwork networkType;
            NetUtils.Types types = NetUtils.getNetworkTypes(getContext());
            switch (types) {
                case WIFI:
                case OTHERS:
                    networkType = EMANetwork.NETWORK_WIFI;
                    break;
                case MOBILE:
                    networkType = EMANetwork.NETWORK_MOBILE;
                    break;
                case ETHERNET:
                    networkType = EMANetwork.NETWORK_CABLE;
                    break;
                case NONE:
                default:
                    networkType = EMANetwork.NETWORK_NONE;
                    break;
            }

            boolean prevNetworkAvailable = currentNetworkType != EMANetwork.NETWORK_NONE;
            boolean currentNetworkAvailable = networkType != EMANetwork.NETWORK_NONE;
            currentNetworkType = networkType;
            if (prevNetworkAvailable == currentNetworkAvailable) {
                // Network availability no change, return.
                EMLog.i(TAG, "Network availability no change, just return. " + currentNetworkType + ", but check ping");
                execute(new Runnable() {
                    @Override
                    public void run() {
                        if (smartHeartbeat != null) {
                            smartHeartbeat.sendPingCheckConnection();
                        }
                    }
                });
                return;
            }

            EMLog.i(TAG, "Network availability changed, notify... " + currentNetworkType);

            execute(new Runnable() {

                @Override
                public void run() {
                    emaObject.onNetworkChanged(currentNetworkType);
                }


            });
        }
    };

    void onNetworkChanged() {
        try {
            if (NetUtils.isWifiConnected(mContext)
                    || NetUtils.isOthersConnected(mContext)) {
                EMLog.d(TAG, "has wifi connection");
                currentNetworkType = EMANetwork.NETWORK_WIFI;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_WIFI);
                return;
            }

            if (NetUtils.isMobileConnected(mContext)) {
                EMLog.d(TAG, "has mobile connection");
                currentNetworkType = EMANetwork.NETWORK_MOBILE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_MOBILE);
                return;
            }

            if (NetUtils.isEthernetConnected(mContext)) {
                EMLog.d(TAG, "has ethernet connection");
                currentNetworkType = EMANetwork.NETWORK_CABLE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_CABLE);
                return;
            }
            currentNetworkType = EMANetwork.NETWORK_NONE;
            EMLog.d(TAG, "no data connection");
            emaObject.onNetworkChanged(EMANetwork.NETWORK_NONE);
            return;
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }
    }

    void onNetworkChanged(EMANetwork network) {
        emaObject.onNetworkChanged(network);
    }

    private AppStateListener appStateListener;
    private List<Activity> resumeActivityList = new ArrayList<>();

    void setAppStateListener(AppStateListener appStateListener) {
        this.appStateListener = appStateListener;
    }

    @TargetApi(14)
    private void registerActivityLifecycleCallbacks() {
        if (Utils.isSdk14()) {
            Object lifecycleCallbacks = new ActivityLifecycleCallbacks() {

                @Override
                public void onActivityStopped(Activity activity) {
                    resumeActivityList.remove(activity);
                    if (resumeActivityList.isEmpty()) {
                        if (appStateListener != null) appStateListener.onBackground();
                    }
                }

                @Override
                public void onActivityResumed(Activity activity) {
                }

                @Override
                public void onActivityCreated(Activity activity, Bundle savedInstanceState) {

                }

                @Override
                public void onActivityStarted(Activity activity) {
                    if(!resumeActivityList.contains(activity)) {
                        resumeActivityList.add(activity);
                        if(resumeActivityList.size() == 1) {
                            if (appStateListener != null) appStateListener.onForeground();
                        }
                    }
                }

                @Override
                public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

                }

                @Override
                public void onActivityPaused(Activity activity) {

                }

                @Override
                public void onActivityDestroyed(Activity activity) {

                }

            };
            ((Application) mContext).registerActivityLifecycleCallbacks((ActivityLifecycleCallbacks) lifecycleCallbacks);
        }
    }

    interface AppStateListener {
        void onForeground();

        void onBackground();
    }

    void setPresence(final String location) {
        if(TextUtils.isEmpty(location)) {
            return;
        }
        execute(new Runnable() {
            @Override
            public void run() {
                emaObject.setPresence(location);
            }
        });
    }

    private String getLocationString(boolean autoLogin) {
        JSONObject jObj = null;
        if (!autoLogin) {
            jObj = getDeviceInfo();
        }
        return jObj == null ? "": jObj.toString();
    }

    public JSONObject getDeviceInfo() {
        JSONObject jobj = new JSONObject();
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(mContext);
        String deviceId = deviceFactory.getDeviceUuid().toString();
        try {
            jobj.put("deviceid", deviceId);
            jobj.put("app-id", mContext.getPackageName());
            jobj.put("hid", EMClient.getInstance().getCurrentUser());
            jobj.put("os", "android");
            jobj.put("os-version", Build.VERSION.RELEASE);
            jobj.put("manufacturer", Build.MANUFACTURER);
            jobj.put("model", Build.MODEL);
        } catch (JSONException e) {
            EMLog.d(TAG, e.getMessage());
        }

        return jobj;
    }

    // For service check.
    private boolean duringChecking = false;

    /**
     * \~chinese
     * 服务诊断接口,流程如下:
     * <pre>
     * 1.校验用户名和密码
     * 2.从服务端获取DNS列表
     * 3.从服务端获取Token
     * 4.连接chat服务器
     * 5.断开连接(如果检查前已经有账户登录,则不执行该步骤)
     * </pre>
     * 如果在诊断过程中产生了错误,该流程将会被打断.
     *
     * @param username 用于服务诊断的用户名,如果已有账户登录,该用户名会被替换为已登录账户的用户名,以防止更改当前已登录账户
     *                 的信息,比如Token等...
     * @param password 密码,如果已有账户登录,该密码会被替换为已登录账户的密码.
     * @param listener 诊断结果回调
     *
     * \~english
     * Service check interface, here is the schedule:
     * <pre>
     * 1.Validate the username and password user input.
     * 2.Get dns list from server.
     * 3.Get token from server.
     * 4.Connect to the im server.
     * 5.logout.(If call this method after user logged in, this step will not in schedule.)
     * </pre>
     * If there is a error occurred during checking, this schedule will be broken.
     *
     * @param username User inputted for service check. If account has logged in before, this username
     *                 would be changed to the logged in username to avoid update the current logged in
     *                 account data.
     * @param password The password for username. if this is a logged in service check, the password would
     *                 changed to the logged in password.
     * @param listener A listener for service check results callback.
     */
    public void check(String username, String password, final CheckResultListener listener) {
        // Already during checking, just return.
        if (duringChecking) {
            EMLog.i("EMServiceChecker", "During service checking, please hold on...");
            return;
        }

        duringChecking = true;

        // If is logged in before, the username must be current user.
        if (isLoggedInBefore()) {
            username = getCurrentUser();
            EMSessionManager sessionMgr = EMSessionManager.getInstance();
            password = sessionMgr.getLastLoginPwd();
        }

        final String finalUser = username;
        final String finalPwd = password;

        new Thread(new Runnable() {
            @Override
            public void run() {
                /**
                 * Contains account-validation check, get-dns check, get-token check, login check.
                 * So the {@link EMAChatClient.CheckResultListener#onResult(int, int, String)} callback
                 * will be called four times.
                 */
                emaObject.check(finalUser, finalPwd, new EMAChatClient.CheckResultListener() {
                    @Override
                    public void onResult(final int type, final int result, final String desc) {
                        EMLog.i("EMServiceChecker", "type: " + type + ", result: " + result + ", desc: " + desc);
                        // Notify user once per callback.
                        notifyCheckResult(listener, type, result, desc);

                        // If occur a error during four checks, return.
                        if (result != EMError.EM_NO_ERROR) {
                            duringChecking = false;
                            return;
                        }

                        // If login successfully, logout
                        if (type == EMCheckType.DO_LOGIN) {
                            checkLogout(listener);
                        }
                    }
                });
            }
        }).start();
    }

    private void checkLogout(CheckResultListener listener) {
        // If is not a logged in service check, logout and notify the user.
        if (!isLoggedInBefore()) {
            logout();
            notifyCheckResult(listener, EMCheckType.DO_LOGOUT, EMError.EM_NO_ERROR, "");
        }

        duringChecking = false;
    }

    private void notifyCheckResult(CheckResultListener listener, @EMCheckType.CheckType int type, int result, String desc) {
        if (listener == null) return;
        listener.onResult(type, result, desc);
    }

    public interface CheckResultListener {
        void onResult(@EMCheckType.CheckType int type, int result, String desc);
    }

}
