/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMChatRoomChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatRoom;
import com.hyphenate.chat.adapter.EMAChatRoomManager;
import com.hyphenate.chat.adapter.EMAChatRoomManagerListener;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * \~chinese
 * 聊天室管理类，负责聊天室加入，退出，聊天室列表获取以及对成员的权限管理等
 * 比如，加入聊天室：
 * <pre>
 *  EMClient.getInstance().chatroomManager().joinChatRoom(conversationId, new EMValueCallBack<EMChatRoom>() {
 *
 *     	public void onSuccess(EMChatRoom value) {
 *     	    //成功加入聊天室后的处理逻辑
 *     	}
 *
 *     	public void onError(int error, String errorMsg) {
 *     	    //加入聊天室失败后的处理逻辑
 *     	}
 * 	});
 * </pre>
 *
 * \~english
 * Chat room manager, responsible for chat room join, exit, list access and member rights management and so on
 * Such as, join a chat room:
 * <pre>
 *   EMClient.getInstance().chatroomManager().joinChatRoom(conversationId, new EMValueCallBack<EMChatRoom>() {
 *    	public void onSuccess(EMChatRoom value) {
 *    	    //Do something after successfully joined the chat room
 *        }
 *    	public void onError(int error, String errorMsg) {
 *    	    //Do something after fail to join the chat room
 *        }
 * 	});
 * </pre>
 */
public class EMChatRoomManager {
    EMAChatRoomManager emaObject;
	private List<EMChatRoomChangeListener> chatRoomListeners = Collections.synchronizedList(new ArrayList<EMChatRoomChangeListener>());
	private ExecutorService threadPool = null;
	private List<EMChatRoom> chatRooms = Collections.synchronizedList(new ArrayList<EMChatRoom>());
	
	EMClient mClient;
	public EMChatRoomManager(EMClient client, EMAChatRoomManager manager) {
	    emaObject = manager;
	    emaObject.addListener(chatRoomListenerImpl);
		mClient = client;
		threadPool = Executors.newCachedThreadPool();
	}

	/**
	 * \~chinese
	 * 注册聊天室事件监听
	 * 聊天室被销毁，成员的加入和退出，禁言，加入白名单等操作均可通过设置{@link EMChatRoomChangeListener}进行监听
	 *
	 * 调用本方法后，需要在适当的时机调用{@link #removeChatRoomListener(EMChatRoomChangeListener)}移除监听
	 *
	 * @param listener 聊天室事件监听，详见{@link EMChatRoomChangeListener}
	 *
	 * \~english
	 * Register chat room change listener
	 * Chat room destruction, member entry and exit, silence, whitelisting and so on can be listened by setting
	 * {@link EMChatRoomChangeListener}
	 *
	 * After calling this method need to call {@link #removeChatRoomListener(EMChatRoomChangeListener)} at the
	 * appropriate time to remove the listener
	 *
	 * @param listener Chat room listener, see {@link EMChatRoomChangeListener}
	 */
	public void addChatRoomChangeListener(EMChatRoomChangeListener listener) {
		chatRoomListeners.add(listener);
	}

    /**
     * \~chinese
     * 移除聊天室事件监听
	 * 在调用{@link #addChatRoomChangeListener(EMChatRoomChangeListener)}后，可以调用本方法移除监听，现推荐使用
	 * {@link #removeChatRoomListener(EMChatRoomChangeListener)}去移除监听
	 *
	 * @param listener 	要移除的聊天室监听对象
     * @deprecated  	使用 {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} 替代
	 *
	 * \~english
	 * Remove chat room change listener
	 * After calling {@link #addChatRoomChangeListener(EMChatRoomChangeListener)} can call this method to remove listener,
	 * now it is recommended to use {@link #removeChatRoomListener(EMChatRoomChangeListener)} to remove
	 *
	 * @param listener	The chat room listener to be removed
     * @deprecated  	Use {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} instead
	 */
    @Deprecated
	public void removeChatRoomChangeListener(EMChatRoomChangeListener listener) {
	    removeChatRoomListener(listener);
    }

	/**
	 * \~chinese
	 * 移除聊天室事件监听
	 * 在调用{@link #addChatRoomChangeListener(EMChatRoomChangeListener)}后，需要调用本方法移除监听
	 *
	 * @param listener	要移除的聊天室监听对象
	 *
	 * \~english
	 * Remove chat room change listener
	 * After calling {@link #addChatRoomChangeListener(EMChatRoomChangeListener)} need to call this method to remove listener
	 *
	 * @param listener	The chat room listener to be removed
	 */
	public void removeChatRoomListener(EMChatRoomChangeListener listener)
	{
		chatRoomListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 加入聊天室
	 * 退出聊天室调用{@link #leaveChatRoom(String)}
	 *
	 * 异步方法
	 *
	 * @param roomId 	聊天室ID
     * @param callback	加入聊天室的结果回调。加入聊天室成功，回调{@link EMValueCallBack#onSuccess(Object)}并返回聊天室对象；
	 *                  加入聊天室失败，回调{@link EMValueCallBack#onError(int, String)}，其中第一个参数为错误参数，第二各参数为错误信息
	 * 
	 * \~english
     * Join the chat room
	 * Leave chat room call {@link #leaveChatRoom(String)}
	 *
	 * Asynchronously method
	 *
	 * @param roomId 	Chat room id to join
	 * @param callback 	Callback for success or fail. If join chat room success calls {@link EMValueCallBack#onSuccess(Object)}, return the EMChatRoom instance;
	 *                  Failed to  join chat room calls {@link EMValueCallBack#onError(int, String)}, where the first parameter is the error code,
	 *                  the second is the error message
	 */
	public void joinChatRoom(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMAChatRoom r = EMChatRoomManager.this.emaObject.joinChatRoom(roomId, error);
				EMChatRoom room = new EMChatRoom(r);
				if (error.errCode() == EMAError.EM_NO_ERROR || error.errCode() == EMAError.CHATROOM_ALREADY_JOINED) {
					callback.onSuccess(room);
				} else {
					callback.onError(error.errCode(), error.errMsg());
				}
			}
		});

	}
	
	/**
	 * \~chinese
	 * 离开聊天室
	 * 加入聊天室{@link #joinChatRoom(String, EMValueCallBack)}后，在离开聊天室时，调用此方法
	 *
	 * 异步方法
	 *
	 * @param roomId 	聊天室ID
	 *
	 * \~english
	 * Leave the chat room
	 * After join chat room {@link #joinChatRoom(String, EMValueCallBack)}, you need to call this method if want to leave chat room
	 *
	 * Asynchronously method
	 *
	 * @param roomId 	Chat room id to leave
	 */
	public void leaveChatRoom(final String roomId) {
		EMChatRoom room = getChatRoom(roomId);
		if (room == null) {
		    return;
		}

        EMOptions options = EMClient.getInstance().getOptions();

        boolean allowed = options.isChatroomOwnerLeaveAllowed();
        String owner = room.getOwner();
        if(!allowed && owner.equals(EMSessionManager.getInstance().getLastLoginUser())){
            return;
        }

        // temp workaround to delete conversation in case of leaving the room
        // but deleteConversation could be a time consuming operation?
		boolean delete = options.isDeleteMessagesAsExitChatRoom();
        if(delete) {
			EMClient.getInstance().chatManager().deleteConversation(roomId,true);
        }
		threadPool.submit(new Runnable() {

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMChatRoomManager.this.emaObject.leaveChatRoom(roomId, error);
			}
		});
	}

	/**
	 * \~chinese
	 * 分页从服务器获取聊天室
	 * 此方法为同步方法，可以使用异步方法{@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param pageNum 				当前要拉取的页数，从1开始
	 * @param pageSize 				每页拉取的数量
	 * @return  					分页获取结果 {@link EMPageResult}
	 * @throws HyphenateException	错误码详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Fetch the chat room data in page
	 * As the synchronization method, can use asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param pageNum 				Page number, start from 1
	 * @param pageSize 				Size to be fetched per page
	 * @return      				Fetch result, {@link EMPageResult}
	 * @throws HyphenateException	Error code see {@link com.hyphenate.EMError}
	 */
	public EMPageResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMPageResult<EMAChatRoom> result = emaObject.fetchChatroomsWithPage(pageNum, pageSize, error);
		handleError(error);

		List<EMAChatRoom> rooms = result.getData();
		int count = result.getPageCount();

		EMPageResult<EMChatRoom> ret = new EMPageResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : rooms) {
			data.add(new EMChatRoom(room));
		}
		ret.setPageCount(count);
		ret.setData(data);

		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}

	/**
	 * \~chinese
	 * 分页从服务器获取聊天室
	 * 返回的结果中，当EMCursorResult.getCursor()为空字符串("")时，表示没有更多数据
	 * 此方法为耗时操作，可以调用异步方法{@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param pageSize 				一次取的数量
     * @param cursor 				从这个游标位置开始取数据
     * @return   					分页获取结果 {@link EMCursorResult}
     * @throws HyphenateException	错误码详见{@link com.hyphenate.EMError}
     * @deprecated 					使用 {@link #fetchPublicChatRoomsFromServer(int, int)} 代替
	 *
	 * \~english
	 * Fetch the chat rooms in page
	 * When EMCursorResult.getCursor() is an empty string ("") in the result, there is no more data
	 * As the synchronization method, can use asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param pageSize 				The count fetched a time
	 * @param cursor 				The cursor position to fetch data
	 * @return 						Fetch result, see {@link EMCursorResult}
	 * @throws HyphenateException	Error code, see {@link com.hyphenate.EMError}
	 * @deprecated 					Use {@link #fetchPublicChatRoomsFromServer(int, int)} instead
	 */
	@Deprecated
	public EMCursorResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAChatRoom> cursorResult = emaObject.fetchChatroomsWithCursor(cursor, pageSize, error);
		handleError(error);
		
		EMCursorResult<EMChatRoom> ret = new EMCursorResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : cursorResult.getData()) {
			data.add(new EMChatRoom(room));
		}
		ret.setCursor(cursorResult.getCursor());
		ret.setData(data);
		
		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}
	
	/**
     * \~chinese
     * 分页从服务器获取聊天室
	 * 返回的结果中，当EMCursorResult.getCursor()为空字符串("")时，表示没有更多数据
	 *
     * 异步方法
	 *
     * @param pageSize 	一次取的数量
     * @param cursor 	从这个游标位置开始取
     * @param callback	结果回调，成功回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  失败回调{@link EMValueCallBack#onError(int, String)}
	 *
     * @deprecated 		使用 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} 代替
     *
     * \~english
     * Fetch the chat rooms in page
     * When EMCursorResult.getCursor() is an empty string ("") in the result, there is no more data
	 *
	 * Asynchronously method
     *
     * @param pageSize 	The count fetched a time
     * @param cursor 	The cursor position to fetch data
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  failure call {@link EMValueCallBack#onError(int, String)}
	 *
     * @deprecated 		Use {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} instead
     */
	@Deprecated
    public void asyncFetchPublicChatRoomsFromServer(final int pageSize, final String cursor, final EMValueCallBack<EMCursorResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMCursorResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageSize, cursor);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
    
    /**
     * \~chinese
     * 分页从服务器获取聊天室
	 *
     * 异步方法
	 *
     * @param pageNum 	当前要拉取的页数，从1开始
     * @param pageSize 	每页拉取的数量
     * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  失败时回调{@link EMValueCallBack#onError(int, String)}
     *
     * \~english
     * Fetch the chat rooms in page
	 *
     * Asynchronously method
     *
     * @param pageNum 	Page number, start from 1
     * @param pageSize 	Size to be fetched per page
     * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  failure call {@link EMValueCallBack#onError(int, String)}
     */
	public void asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack<EMPageResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMPageResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageNum, pageSize);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 从服务器获取聊天室详情，默认不取成员列表
	 * 异步方法见{@link #asyncFetchChatRoomFromServer(String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param roomId				聊天室ID
	 * @return EMChatRoom			返回聊天室对象
	 * @throws HyphenateException	错误码详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Fetch the chat room from server by room id, default not to fetch members
	 * Asynchronous method see {@link #asyncFetchChatRoomFromServer(String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param roomId				Chat room id
	 * @return EMChatRoom			Return the EMChatRoom instance
	 * @throws HyphenateException	Error code, see {@link com.hyphenate.EMError}
	 */
	public EMChatRoom fetchChatRoomFromServer(String roomId) throws HyphenateException {
		return fetchChatRoomFromServer(roomId, false);
	}
	
	/**
	 * \~chinese
	 * 从服务器获取聊天室详情
	 * 如果需要取成员列表，默认最多取200个成员，超出部分，请调用{@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param roomId 				聊天室ID
	 * @param fetchMembers 			是否需要获取聊天室成员
	 * @return EMChatRoom			返回的聊天室对象
	 * @throws HyphenateException	错误码详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Fetch the chat room from server by room id.
	 * If fetchMembers is true, default maximum of fetched member count is 200,
	 * to continue fetch members, call {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param roomId 				chat room id
	 * @param fetchMembers 			Whether you need to get chat room members
	 * @return EMChatRoom			Return the EMChatRoom instance
	 * @throws HyphenateException	Error code, see {@link com.hyphenate.EMError}
	 */
 	public EMChatRoom fetchChatRoomFromServer(String roomId, boolean fetchMembers)throws HyphenateException {
 	   EMAError error = new EMAError();
       EMAChatRoom room = emaObject.fetchChatroomSpecification(roomId, error, fetchMembers);
       handleError(error);
       return new EMChatRoom(room);
 	}
	
	/**
     * \~chinese
     * 从服务器获取聊天室详情，默认不取成员列表
	 * 同步方法见{@link #fetchChatRoomFromServer(String)}
	 *
     * 异步方法
	 *
     * @param roomId	聊天室ID
	 * @param callback	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                  失败时回调{@link EMValueCallBack#onError(int, String)}
     *
     * \~english
     * Fetch the chat room from server by room id, default not to fetch members
     * Synchronization method see {@link #fetchChatRoomFromServer(String)}
	 *
	 * Asynchronously method
	 *
     * @param roomId	Chat room id
	 * @param callback	Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                  failure call {@link EMValueCallBack#onError(int, String)}
     */
    public void asyncFetchChatRoomFromServer(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMChatRoom chatRoom = fetchChatRoomFromServer(roomId);
                    callback.onSuccess(chatRoom);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });   
    }
	
	/**
	 * \~chinese
	 * 从内存中获取聊天室
	 *
	 * @param roomId		聊天室ID
     * @return				返回聊天室对象，如果内存中没有会返回null
	 *
	 * \~english
	 * Get the chat room in the cache
	 *
	 * @param roomId		Chat room id
	 * @return 				Return the EMChatRoom instance and return null if not find in the cache
	 */
	public EMChatRoom getChatRoom(String roomId) {
		EMAChatRoom room = emaObject.getChatroom(roomId);
		if(room == null){
			return null;
		}
		return new EMChatRoom(room);
	}
	
	/**
	 * \~chinese
	 * 获取当前内存的聊天室列表
	 * 注：此方法需要调用以下方法：
	 * （1）{@link #fetchPublicChatRoomsFromServer(int, int)}及其异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 * （2）{@link #fetchPublicChatRoomsFromServer(int, String)}及其异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 * 后，才会有数据，且是以上方法最后一次分页获取的数据。
	 *
	 * @return	返回EMChatRoomManager维护的聊天室列表
	 * @deprecated
	 * 
	 * \~english
	 * Get all chat rooms in the cache in EMChatRoomManager
	 * Note: The method needs to call the following methods:
	 * (1) {@link #fetchPublicChatRoomsFromServer(int, int)} and it's asynchronous method of {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 * (2) {@link #fetchPublicChatRoomsFromServer(int, String)} and it's asynchronous method of {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 *
	 * @return	Return all the chat rooms maintained by EMChatRoomManager
	 * @deprecated
	 */
	@Deprecated
	public List<EMChatRoom> getAllChatRooms() {
		return Collections.unmodifiableList(chatRooms);
	}

	EMAChatRoomManagerListener chatRoomListenerImpl = new EMAChatRoomManagerListener() {

		@Override
		public void onLeaveChatRoom(EMAChatRoom chatroom, int leaveReason) {
            EMClient.getInstance().chatManager().caches.remove(chatroom.getId());
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            if (leaveReason == DESTROYED) {
			                listener.onChatRoomDestroyed(chatroom.getId(), chatroom.getName());
			            } else {
							listener.onRemovedFromChatRoom(leaveReason, chatroom.getId(), chatroom.getName(), EMClient.getInstance().getCurrentUser());
						}
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberJoinedChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
				    for (EMChatRoomChangeListener listener : chatRoomListeners) {
					    listener.onMemberJoined(chatroom.getId(), member);
				    }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberLeftChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            listener.onMemberExited(chatroom.getId(), chatroom.getName(), member);
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		public void onAddMuteList(EMAChatRoom chatRoom, List<String> mutes, long expireTime) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListAdded(chatRoom.getId(), mutes, expireTime);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveMutes(EMAChatRoom chatRoom, List<String> mutes) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListRemoved(chatRoom.getId(), mutes);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListAdded(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListAdded(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListRemoved(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListRemoved(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAllMemberMuteStateChanged(EMAChatRoom chatRoom, final boolean isMuted) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAllMemberMuteStateChanged(chatRoom.getId(), isMuted);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAddAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminAdded(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminRemoved(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onOwnerChanged(EMAChatRoom chatRoom, String newOwner, String oldOwner) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onOwnerChanged(chatRoom.getId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAChatRoom chatRoom, String announcement) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAnnouncementChanged(chatRoom.getId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	};


	/**
	 * \~chinese
	 * 创建聊天室
	 * 异步方法见{@link #asyncCreateChatRoom(String, String, String, int, List, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param subject           	聊天室名称
	 * @param description       	聊天室描述
	 * @param welcomeMessage    	邀请成员加入聊天室的消息
	 * @param maxUserCount      	允许加入聊天室的最大成员数
	 * @param members           	邀请加入聊天室的成员列表
	 * @return EMChatRoom 			创建成功的聊天室对象
     * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Create chat room
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param subject           	Name of chat room
	 * @param description       	Description of chat room
	 * @param welcomeMessage    	Welcome message inviting members to join the chat room
	 * @param maxUserCount      	Maximum number of members allowed to join
	 * @param members           	Members inviting to join
	 * @return EMChatRoom			EMChatRoom instance created successful
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom createChatRoom(String subject, String description, String welcomeMessage,
	                                 int maxUserCount, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.createChatRoom(subject, description, welcomeMessage, EMChatRoom.EMChatRoomStyle.EMChatRoomStylePublicOpenJoin.ordinal(), maxUserCount, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 创建聊天室
	 * 同步方法见{@link #createChatRoom(String, String, String, int, List)}
	 *
	 * 异步方法
	 *
	 * @param subject           聊天室名称
	 * @param description       聊天室描述
	 * @param welcomeMessage    邀请成员加入聊天室的消息
	 * @param maxUserCount      允许加入聊天室的最大成员数
	 * @param members           邀请加入聊天室的成员列表
	 * @param callBack          结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)},
	 *                          失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Create chat room
	 * Synchronization method see {@link #createChatRoom(String, String, String, int, List)}
	 *
	 * Asynchronously method
	 *
	 * @param subject           Name of chat room
	 * @param description       Description of chat room
	 * @param welcomeMessage    Welcome message inviting members to join the chat room
	 * @param maxUserCount      Maximum number of members allowed to join
	 * @param members           Members inviting to join
	 * @param callBack			Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                          failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncCreateChatRoom(final String subject, final String description, final String welcomeMessage,
	                                final int maxUserCount, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(createChatRoom(subject, description, welcomeMessage, maxUserCount, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 销毁聊天室
	 * 需要拥有owner权限
	 * 异步方法见{@link #asyncDestroyChatRoom(String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Destroy chat room
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncDestroyChatRoom(String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public void destroyChatRoom(String chatRoomId)
			throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyChatroom(chatRoomId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 销毁聊天室
	 * 需要拥有owner权限
	 * 同步方法见{@link #destroyChatRoom(String)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param callBack		结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 *                      失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Destroy chat room
	 * Owner‘s authority is required
	 * Synchronization method see {@link #destroyChatRoom(String)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param callBack		Result callback, success call {@link EMCallBack#onSuccess()},
	 *                      failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncDestroyChatRoom(final String chatRoomId, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyChatRoom(chatRoomId);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改聊天室标题
	 * 需要拥有owner权限
	 * 异步方法见{@link #asyncChangeChatRoomSubject(String, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param newSubject			新的聊天室名字
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Change chat room subject
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncChangeChatRoomSubject(String, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param newSubject			New name of chat room
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom changeChatRoomSubject(String chatRoomId, String newSubject)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomSubject(chatRoomId, newSubject, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改聊天室标题
	 * 需要拥有owner权限
	 * 同步方法见{@link #changeChatRoomSubject(String, String)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param newSubject	新的聊天室名字
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Change chat room subject
	 * Owner‘s authority is required
	 * Synchronization method see {@link #changeChatRoomSubject(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param newSubject	New name of chat room
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncChangeChatRoomSubject(final String chatRoomId, final String newSubject, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatRoomSubject(chatRoomId, newSubject));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改群描述信息
	 * 需要拥有owner权限
	 * 异步方法见{@link #asyncChangeChatroomDescription(String, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param newDescription		新的聊天室描述
	 * @return						修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * change chat room description
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncChangeChatroomDescription(String, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param newDescription		New description of chat room
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom changeChatroomDescription(String chatRoomId, String newDescription)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomDescription(chatRoomId, newDescription, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改群描述信息
	 * 需要拥有owner权限
	 * 同步方法见{@link #changeChatroomDescription(String, String)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId		聊天室ID
	 * @param newDescription	新的聊天室描述
	 * @param callBack   		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                          失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Change chat room description
	 * Owner‘s authority is required
	 * Synchronization method see {@link #changeChatroomDescription(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId		Chat room id
	 * @param newDescription	New description of chat room
	 * @param callBack			Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                          failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncChangeChatroomDescription(final String chatRoomId, final String newDescription, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatroomDescription(chatRoomId, newDescription));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表
	 * 返回的结果中，当EMCursorResult.getCursor()为空字符串("")时，表示没有更多数据
	 * 异步方法见{@link #asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param cursor				从这个游标位置开始取数据
	 * @param pageSize				一次取的数量
	 * @return						分页获取结果 {@link EMCursorResult}
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Fetch chat room member list
	 * When EMCursorResult.getCursor() is an empty string ("") in the result, there is no more data
	 * As the synchronization method, can use asynchronous method {@link #asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param cursor				The cursor position to fetch data
	 * @param pageSize				The count fetched a time
	 * @return						Fetch result, see {@link EMCursorResult}
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMCursorResult<String> fetchChatRoomMembers(String chatRoomId, String cursor, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult result = emaObject.fetchChatroomMembers(chatRoomId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表
	 * 返回的结果中，当EMCursorResult.getCursor()为空字符串("")时，表示没有更多数据
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param cursor		从这个游标位置开始取数据
	 * @param pageSize		一次取的数量
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch chat room member list
	 * When EMCursorResult.getCursor() is an empty string ("") in the result, there is no more data
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param cursor		The cursor position to fetch data
	 * @param pageSize		The count fetched a time
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchChatRoomMembers(final String chatRoomId, final String cursor, final int pageSize, final EMValueCallBack<EMCursorResult<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMembers(chatRoomId, cursor, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 禁止聊天室成员发言
	 * 需要聊天室拥有者或者管理员权限
	 * 异步方法见{@link #asyncMuteChatRoomMembers(String, List, long, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param muteMembers 			禁言的用户列表
	 * @param duration 				禁言的时间，单位是毫秒
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Stop members post messages for some time
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncMuteChatRoomMembers(String, List, long, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param muteMembers 			Mute member list
	 * @param duration 				Mute duration, in milli-seconds
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom muteChatRoomMembers(String chatRoomId, List<String> muteMembers, long duration)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.muteChatroomMembers(
				chatRoomId, muteMembers, duration, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 禁止聊天室成员发言
	 * 需要聊天室拥有者或者管理员权限
	 * 同步方法见{@link #muteChatRoomMembers(String, List, long)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param muteMembers   禁言的用户列表
	 * @param duration 		禁言的时间，单位是毫秒
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Stop members post message for some time
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #muteChatRoomMembers(String, List, long)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param muteMembers 	Mute members list
	 * @param duration 		Mute duration, in milli-seconds
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncMuteChatRoomMembers(final String chatRoomId, final List<String> muteMembers, final long duration, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(muteChatRoomMembers(chatRoomId, muteMembers, duration));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除禁言
	 * 需要聊天室拥有者或者管理员权限
	 * 异步方法见{@link #asyncUnMuteChatRoomMembers(String, List, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param members				解除禁言的用户列表
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Release mute
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncUnMuteChatRoomMembers(String, List, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param members				Release mute members list
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom unMuteChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unmuteChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 解除禁言
	 * 需要聊天室拥有者或者管理员权限
	 * 同步方法见{@link #unMuteChatRoomMembers(String, List)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param members		解除禁言的用户列表
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Release mute
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #unMuteChatRoomMembers(String, List)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param members		Release mute members list
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncUnMuteChatRoomMembers(final String chatRoomId, final List<String> members,
	                                             final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unMuteChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权
	 * 需要拥有owner权限
	 * 异步方法见{@link #asyncChangeOwner(String, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param newOwner				新的聊天室拥有者ID
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Transfer chat room ownership to others
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncChangeOwner(String, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param newOwner				New owner id
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
	 */
	public EMChatRoom changeOwner(String chatRoomId, String newOwner)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.transferChatroomOwner(chatRoomId, newOwner, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权
	 * 需要拥有owner权限
	 * 同步方法见{@link #changeOwner(String, String)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param newOwner		新的聊天室拥有者ID
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Transfer chat room ownership to others
	 * Owner‘s authority is required
	 * Synchronization method see {@link #changeOwner(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param newOwner		New owner id
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncChangeOwner(final String chatRoomId, final String newOwner, final EMValueCallBack<EMChatRoom> callBack)
			throws HyphenateException {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeOwner(chatRoomId, newOwner));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 设置聊天室管理员
	 * 需要聊天室owner权限
	 * 异步方法见{@link #asyncAddChatRoomAdmin(String, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param admin					要设置的管理员ID
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Add chat room administrator
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncAddChatRoomAdmin(String, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param admin					Admin id to set
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom addChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.addChatroomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 设置聊天室管理员
	 * 需要聊天室owner权限
	 * 同步方法见{@link #addChatRoomAdmin(String, String)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param admin			要设置的管理员ID
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Add chat room administrator
	 * Owner‘s authority is required
	 * Synchronization method see {@link #addChatRoomAdmin(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param admin			Admin id to set
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncAddChatRoomAdmin(final String chatRoomId, final String admin, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(addChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 移除聊天室管理员权限
	 * 需要聊天室owner权限
	 * 异步方法见{@link #asyncRemoveChatRoomAdmin(String, String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param admin					要移除管理员权限的ID
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Remove administrator's authority in chat room
	 * Owner‘s authority is required
	 * Asynchronously method see {@link #asyncRemoveChatRoomAdmin(String, String, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param admin					Admin id to remove
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom removeChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 移除聊天室管理员权限
	 * 需要聊天室owner权限
	 * 同步方法见{@link #removeChatRoomAdmin(String, String)}}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param admin			要移除管理员权限的ID
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove administrator's authority in chat room
	 * Owner‘s authority is required
	 * Synchronization method see {@link #removeChatRoomAdmin(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param admin			Admin id to remove
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncRemoveChatRoomAdmin(final String chatRoomId, final String admin,
	                                final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室的禁言列表
	 * 需要聊天室拥有者或者管理员权限
	 * 异步方法见{@link #asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param pageNum				当前要拉取的页数，从1开始
	 * @param pageSize				每页拉取的数量
	 * @return  					返回的包含禁言成员ID及其禁言时长的Map，其中每一项的key是禁言的成员ID，value是禁言动作失效的时间，单位是毫秒
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Fetch mute list from server, which contains mute members and mute time
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param pageNum				The count fetched a time
	 * @param pageSize				The cursor position to fetch data
	 * @return 						Return a map which contains the mute member id and mute duration, in which the key of each entry
	 * 								is the mute member id and the value is the expired time of banning post action, in milli-seconds
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public Map<String, Long> fetchChatRoomMuteList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, Long> members = emaObject.fetchChatRoomMuteList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室的禁言列表
	 * 需要聊天室拥有者或者管理员权限
	 * 同步方法见{@link #fetchChatRoomMuteList(String, int, int)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param pageNum		当前要拉取的页数，从1开始
	 * @param pageSize		每页拉取的数量
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch mute list from server, which contains mute members and mute time
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #fetchChatRoomMuteList(String, int, int)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param pageNum		The count fetched a time
	 * @param pageSize		The cursor position to fetch data
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchChatRoomMuteList(final String chatRoomId, final int pageNum, final int pageSize,
	                                       final EMValueCallBack<Map<String, Long>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMuteList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 移除聊天室成员
	 * 需要聊天室拥有者或者管理员权限
	 * 异步方法见{@link #asyncRemoveChatRoomMembers(String, List, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param members				要移除的成员ID集合
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Remove chat room members
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncRemoveChatRoomMembers(String, List, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param members				The members list to be removed
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom removeChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 移除聊天室成员
	 * 需要聊天室拥有者或者管理员权限
	 * 同步方法见{@link #removeChatRoomMembers(String, List)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param members		要移除的成员ID集合
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove chat room members
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #removeChatRoomMembers(String, List)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param members		The members list to be removed
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncRemoveChatRoomMembers(final String chatRoomId, final List<String> members,
	                                       final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加成员到黑名单
	 * 需要聊天室拥有者或者管理员权限
	 * 成员被加入聊天室黑名单后，有如下行为：
	 * <pre>
	 * 1、成员被加入黑名单的同时，将被服务器移出聊天室
	 * 2、可通过{@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)}回调通知，
	 * 第一个参数是具体的原因，被加入黑名单返回的原因是{@link EMAChatRoomManagerListener#BE_KICKED}
	 * 3、加入黑名单的成员禁止再次加入到聊天室
	 * </pre>
	 * 异步方法见{@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param members				要加入黑名单的成员列表
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Add members to chat room's black list
	 * Owner or administrator‘s authority is required
	 * As member is added to blacklist, have the following behaviors:
	 * <pre>
	 * 1.As member is added to blacklist, will be removed from chat room
	 * 2.Call the method {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} to notice,
	 *   the first parameter is the reason, which is {@link EMAChatRoomManagerListener#BE_KICKED}
	 * 3.The members have been added to blacklist are barred from rejoining
	 * </pre>
	 * Asynchronously method see {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param members				The member list to be added to blacklist
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom blockChatroomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.blockChatroomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 添加成员到黑名单
	 * 需要聊天室拥有者或者管理员权限
	 * 成员被加入聊天室黑名单后，有如下行为：
	 * <pre>
	 * 1、成员被加入黑名单的同时，将被服务器移出聊天室
	 * 2、可通过{@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)}回调通知，
	 * 第一个参数是具体的原因，被加入黑名单返回的原因是{@link EMAChatRoomManagerListener#BE_KICKED}
	 * 3、加入黑名单的成员禁止再次加入到聊天室
	 * </pre>
	 * 同步方法见{@link #blockChatroomMembers(String, List)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param members		要加入黑名单的成员列表
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Add members to chat room's black list
	 * Owner or administrator‘s authority is required
	 * As member is added to blacklist, have the following behaviors:
	 * <pre>
	 * 1.As member is added to blacklist, will be removed from chat room
	 * 2.Call the method {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} to notice,
	 *   the first parameter is the reason, which is {@link EMAChatRoomManagerListener#BE_KICKED}
	 * 3.The members have been added to blacklist are barred from rejoining
	 * </pre>
	 * Synchronization method see {@link #blockChatroomMembers(String, List)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param members		The member list to be added to blacklist
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncBlockChatroomMembers(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(blockChatroomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员从黑名单中移除
	 * 需要聊天室拥有者或者管理员权限
	 * 异步方法见{@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param members				要移除黑名单的成员列表
	 * @return						返回修改后的聊天室对象
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Remove members from chat room's black list
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param members				The members list will be removed from blacklist
	 * @return						Return the EMChatRoom instance modified
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public EMChatRoom unblockChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unblockChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 将成员从黑名单中移除
	 * 需要聊天室拥有者或者管理员权限
	 * 同步方法见{@link #unblockChatRoomMembers(String, List)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param members		要移除黑名单的成员列表
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove members from chat room's black list
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #unblockChatRoomMembers(String, List)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param members		The members list will be removed from blacklist
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 *                      failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncUnBlockChatRoomMembers(final String chatRoomId, final List<String> members,
	                                        final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unblockChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室黑名单列表，分页显示
	 * 需要聊天室拥有者或者管理员权限
	 * 异步方法见{@link #asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId			聊天室ID
	 * @param pageNum				当前要拉取的页数，从1开始
	 * @param pageSize				每页拉取的数量
	 * @return						返回聊天室黑名单列表
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Fetch chat room black list members in pages
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId			Chat room id
	 * @param pageNum				Page number, start from 1
	 * @param pageSize				Size to be fetched per page
	 * @return						Return the blacklist in chat room
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
     */
	public List<String> fetchChatRoomBlackList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		List<String> blockList = emaObject.fetchChatRoomBlackList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return blockList;
	}

	/**
	 * \~chinese
	 * 获取聊天室黑名单列表，分页显示
	 * 需要聊天室拥有者或者管理员权限
	 * 同步方法见{@link #fetchChatRoomBlackList(String, int, int)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId	聊天室ID
	 * @param pageNum		当前要拉取的页数，从1开始
	 * @param pageSize		每页拉取的数量
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，返回聊天室黑名单列表；
	 *                      失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch chat room black list members in pages
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #fetchChatRoomBlackList(String, int, int)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId	Chat room id
	 * @param pageNum		Page number, start from 1
	 * @param pageSize		Size to be fetched per page
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)}, return the blacklist in chat room;
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchChatRoomBlackList(final String chatRoomId, final int pageNum, final int pageSize,
	                                        final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomBlackList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加成员到白名单
	 * 需要聊天室拥有者或者管理员权限
	 * 添加到白名单后，聊天室拥有者或者管理员执行{@link #muteAllMembers(String, EMValueCallBack)} 时，加入白名单的用户不受影响
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 			聊天室ID
	 * @param members 				加入白名单的成员列表
	 * @param callBack				结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 * 								失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Add members to whitelist
	 * Owner or administrator‘s authority is required
	 * The whitelisted user is not affected when {@link #muteAllMembers(String, EMValueCallBack)} is executed by owner or admins
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId 			Chat room id
	 * @param members 				The member list to add
	 * @param callBack				Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 * 								failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void addToChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.addToWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员从白名单移除
	 * 需要聊天室拥有者或者管理员权限
	 * 成员从白名单移除后，将受到{@link #muteAllMembers(String, EMValueCallBack)}功能的影响
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 			聊天室ID
	 * @param members 				移除白名单的用户列表
	 * @param callBack 				结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 * 								失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Remove members from whitelist
	 * Owner or administrator‘s authority is required
	 * When members are removed from whitelist, they are affected by {@link #muteAllMembers(String, EMValueCallBack)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId 			Chat room id
	 * @param members 				The member list to remove from whitelist
	 * @param callBack
	 */
	public void removeFromChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.removeFromWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 检查自己是否在白名单中
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 	聊天室ID
	 * @param callBack 		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，返回是否在白名单中；
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Check if in whitelist
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId 	chat room id
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)}, return the result if in whitelist
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void checkIfInChatRoomWhiteList(final String chatRoomId, EMValueCallBack<Boolean> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~chinese
	 * 从服务器获取白名单列表
	 * 需要聊天室拥有者或者管理员权限
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 	聊天室ID
	 * @param callBack		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Fetch whitelist from server
	 * Owner or administrator‘s authority is required
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId 	Chat room id
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void fetchChatRoomWhiteList(final String chatRoomId, final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchChatRoomWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 禁言所有成员
	 * 需要聊天室拥有者或者管理员权限
	 * 聊天室拥有者、管理员及加入白名单的用户不受影响
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 	聊天室ID
	 * @param callBack 		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Mute all members
	 * Owner or administrator‘s authority is required
	 * Chat room owners, administrators and whitelisted users will not be affected
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId  	Chat room id
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void muteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.muteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除所有成员的禁言状态
	 * 需要聊天室拥有者或者管理员权限
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 	聊天室ID
	 * @param callBack 		结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Release the mute status of all members
	 * Owner or administrator‘s authority is required
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId 	Chat room id
	 * @param callBack		Result callback, success call {@link EMValueCallBack#onSuccess(Object)},
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void unmuteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.unmuteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 更新聊天室公告
	 * 需要聊天室拥有者或者管理员权限
	 * 异步方法见{@link #asyncUpdateChatRoomAnnouncement(String, String, EMCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId 			聊天室ID
	 * @param announcement 			公告内容
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Update chat room announcement
	 * Owner or administrator‘s authority is required
	 * Asynchronously method see {@link #asyncUpdateChatRoomAnnouncement(String, String, EMCallBack)}
	 *
	 * Synchronization method will block the current thread
	 *
	 * @param chatRoomId 			Chat room id
	 * @param announcement 			Announcement content
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
	 */
	public void updateChatRoomAnnouncement(String chatRoomId, String announcement) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.updateChatRoomAnnouncement(chatRoomId, announcement, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 更新聊天室公告
	 * 需要聊天室拥有者或者管理员权限
	 * 同步方法见{@link #updateChatRoomAnnouncement(String, String)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 	聊天室ID
	 * @param announcement 	公告内容
	 * @param callBack 		结果回调，成功时回调{@link EMCallBack#onSuccess()}，
	 * 						失败时回调{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Update chat room announcement
	 * Owner or administrator‘s authority is required
	 * Synchronization method see {@link #updateChatRoomAnnouncement(String, String)}
	 *
	 * Asynchronously method
	 *
	 * @param chatRoomId 	Chat room id
	 * @param announcement 	Announcement content
	 * @param callBack 		Result callback, success call {@link EMCallBack#onSuccess()},
	 * 						failure call {@link EMCallBack#onError(int, String)}
	 */
	public void asyncUpdateChatRoomAnnouncement(final String chatRoomId, final String announcement, final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateChatRoomAnnouncement(chatRoomId, announcement);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容
	 * 异步方法见{@link #asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}
	 *
	 * 同步方法，会阻塞当前线程
	 *
	 * @param chatRoomId 			聊天室ID
	 * @return						聊天室公告
	 * @throws HyphenateException	错误信息详见{@link com.hyphenate.EMError}
	 *
	 * \~english
	 * Get the chat room announcement from server
	 * @param chatRoomId 			Chat room id
	 * @return						Chat room announcement
	 * @throws HyphenateException	Error info, see {@link com.hyphenate.EMError}
	 */
	public String fetchChatRoomAnnouncement(String chatRoomId) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.fetchChatRoomAnnouncement(chatRoomId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容
	 * 同步方法见{@link #fetchChatRoomAnnouncement(String)}
	 *
	 * 异步方法
	 *
	 * @param chatRoomId 	聊天室ID
	 * @param callBack  	结果回调，成功时回调{@link EMValueCallBack#onSuccess(Object)}，返回聊天室公告；
	 * 						失败时回调{@link EMValueCallBack#onError(int, String)}
	 *
	 * \~english
	 * Get the chat room announcement from server
	 * @param chatRoomId 	Chat room id
	 * @param callBack  	Result callback, success call {@link EMValueCallBack#onSuccess(Object)}, return the chat room announcement;
	 * 						failure call {@link EMValueCallBack#onError(int, String)}
	 */
	public void asyncFetchChatRoomAnnouncement(final String chatRoomId, final EMValueCallBack<String> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomAnnouncement(chatRoomId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	// ============================= chatRoom_reform new add api end
	private void handleError(EMAError error)  throws HyphenateException {
	    if (error.errCode() != EMAError.EM_NO_ERROR) {
	        throw new HyphenateException(error);
	    }
	}
	
	void onLogout() {
	    chatRoomListeners.clear();
	}
}
