/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatRoom;

import java.util.List;
import java.util.Map;

/**
 * \~chinese
 * 聊天室对象类
 *
 * \~english
 * Chatroom object
 */
public class EMChatRoom extends EMBase<EMAChatRoom> {

    enum EMChatRoomStyle{
        /**
         *\~chinese
         * 私有群组，创建完成后，只允许Owner邀请用户加入
         *
         * \~english
         * Private groups, after created, only allowing owner invite users to join
         */
        EMChatRoomStylePrivateOnlyOwnerInvite,

        /**
         * \~chinese
         * 私有群组，创建完成后，只允许Owner和群成员邀请用户加入
         *
         * \~english
         * Private groups, after created, only allowing owner and members invite users to join
         */
        EMChatRoomStylePrivateMemberCanInvite,

        /**
         * \~chinese
         * 公开群组，创建完成后，只允许Owner邀请用户加入; 非群成员用户需发送入群申请，Owner同意后才能入组
         *
         * *\~english
         * Public groups, after created, only allowing owner invite users to join; Non-group members into the group must send an application to the owner after the Owner agree
         */
        EMChatRoomStylePublicJoinNeedApproval,

        /**
         * \~chinese
         * 公开群组，创建完成后，允许非群组成员加入，不需要Owner同意
         *
         * \~english Public groups, after created, to allow non-members to join the group, does not require owner agrees
         */
        EMChatRoomStylePublicOpenJoin
    }

    /**
     * \~chinese
     *  当前用户在聊天室的角色枚举
     *
     *  \~english
     *  Enum of the current user's role in chat room
     */
    public enum EMChatRoomPermissionType {
        /**
         * \~chinese
         * 普通成员
         *
         * \~english
         * Normal member
         */
        member(0),

        /**
         * \~chinese
         * 群组管理员
         *
         * \~english
         * Group admin
         */
        admin(1),

        /**
         * \~chinese
         * 群主
         *
         * \~english
         * Group owner
         */
        owner(2),

        /**
         * \~chinese
         * 未知
         *
         * \~english
         * Unknown
         */
        none(-1);
        private int permissionType;
        private EMChatRoomPermissionType(int permissionType) {
            this.permissionType = permissionType;
        }
    }

	public EMChatRoom(){
	    emaObject = new EMAChatRoom();
	}
	
	public EMChatRoom(EMAChatRoom room) {
	    emaObject = new EMAChatRoom(room);
	}
	
    public EMChatRoom(String roomId){
        emaObject = new EMAChatRoom(roomId);
    }
    
    public EMChatRoom(String roomId, String roomName){
        emaObject = new EMAChatRoom(roomId);
//        this.nick = roomName;
    }

    /**
     * \~chinese
     * 获取聊天室id
     *
     * @return 聊天室id
     *
     * \~english
     * Get chat room id
     *
     * @return Chat room id
     */
    public String getId() {
        return emaObject.getId();
     }

    /**
     * \~chinese
     * 获取聊天室名称
     * @return  聊天室名称
     *
     * \~english
     * Get chat room name
     * @return Chat room name
     */
     public String getName() {
        return emaObject.getName();
     }

    /**
     * \~chinese
     * 获取聊天室描述
     * @return 聊天室描述
     *
     * \~english
     * Get chat room description
     * @return Chat room description
     */
     public String getDescription() {
        return emaObject.getDescription();
    }

    /**
     * \~chinese
     * 获取聊天室的所有者
     * 如果没有获取聊天室详情，返回可能为空
     * 获取聊天室详情见{@link EMChatRoomManager#fetchChatRoomFromServer(String)}
     * @return 聊天室所有者id
     *
     * \~english
     * Get chat room owner
     * If not fetch the chat room's detail specification, the return may be empty
     * Get chat room detail, see {@link EMChatRoomManager#fetchChatRoomFromServer(String)}
     * @return
     */
    public String getOwner() {
        return emaObject.getOwner();
    }

    /**
     * \~chinese
     * 获取聊天室的管理员列表
     * 如果没有获取聊天室详情，返回可能为空
     * 获取聊天室详情见{@link EMChatRoomManager#fetchChatRoomFromServer(String)}
     * @return 管理员id集合
     *
     * \~english
     * Get chat room administrator list
     * If not fetch the chat room's detail specification, the result may be empty
     * Get chat room detail, see {@link EMChatRoomManager#fetchChatRoomFromServer(String)}
     * @return List of administrator IDs
     */
    public List<String> getAdminList() { return emaObject.getAdministratorList();}

    /**
     * \~chinese
     * 返回在线成员人数
     * @return 在线成员数
     *
     * \~english
     * Return the number of online members
     */
    public int getMemberCount()
    {
        return emaObject.getAffiliationsCount();
    }

    /**
     * \~chinese
     * 获取聊天室最大成员数，在创建时确定
     * 需要获取聊天室详情才能拿到正确的结果，如果没有获取则返回0
     * @return  聊天室最大成员数
     *
     * \~english
     * Max chat room members capability
     * It depends on chat room detail, if not fetch chat room detail, return 0
     * @return  The maximum number of members in chat room
     */
    public int getMaxUsers() {
        return emaObject.getMaxUserCount();
    }

    /**
     * \~chinese
     * 返回成员列表
     * 如果没有获取成员列表，返回的列表可能为空
     * 获取成员有如下方式：
     * （1）成员人数少于200人时，可通过{@link EMChatRoomManager#fetchChatRoomFromServer(String, boolean)},
     * 第二个参数传入true，可获取最多200个成员
     * （2）通过{@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}分页获取，也可通过其异步方法
     * {@link EMChatRoomManager#asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}获取
     * @return  成员id集合
     *
     * \~english
     * Return member list
     * If not get members from server, the return may be empty
     * You can get members in the flowing ways:
     * (1) When member's number is less than 200, can use {@link EMChatRoomManager#fetchChatRoomFromServer(String, boolean)} to fetch,
     * the second parameter pass in true, can get up to 200 members
     * (2) Use the method of {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)} to get by page,
     * also can use its asynchronous method of {@link EMChatRoomManager#asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}
     * to get
     * @return The list of member IDs
     */
    public List<String> getMemberList(){
        return emaObject.getMemberList();
    }

    /**
     * \~chinese
     * 返回聊天室黑名单
     * 如果没有获取黑名单列表，返回的列表可能为空
     * 获取黑名单可调用{@link EMChatRoomManager#fetchChatRoomBlackList(String, int, int)},
     * 调用这个方法需要聊天室所有者或者管理员身份，也可通过它的异步方法进行调用
     * {@link EMChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}
     * @return 返回黑名单列表
     *
     * \~english
     * Return black list
     * If not get blacklist from server, the return may be empty
     * Fetch blacklist can call {@link EMChatRoomManager#fetchChatRoomBlackList(String, int, int)},
     * call this method need owner or admin authority, also can call it's asynchronous method
     * {@link EMChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}
     * @return Return the blacklist
     */
    public List<String> getBlackList() { return emaObject.getBlockList(); }

    /**
     * \~chinese
     * 返回禁言列表
     * 如果没有获取禁言列表，返回的列表可能为空
     * 获取禁言列表可调用{@link EMChatRoomManager#fetchChatRoomMuteList(String, int, int)},
     * 调用这个方法需要聊天室所有者或者管理员身份，也可通过它的异步方法进行调用
     * {@link EMChatRoomManager#asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}
     *
     * @return  返回禁言列表，Map.entry.key 是禁言的成员id，Map.entry.value是禁言动作存在的时间，单位是毫秒
     *
     * \~english
     * Return mute list
     * If not get mute list from server, the return may be empty
     * Fetch mute list can call {@link EMChatRoomManager#fetchChatRoomMuteList(String, int, int)},
     * call this method need owner or admin authority, also can call it's asynchronous method
     * {@link EMChatRoomManager#asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}
     *
     * @return  Return the mute list, Map.entry.key is username of mute action,
     * Map.entry.value is expired time of banning post action, in milli-seconds
     */
    public Map<String, Long> getMuteList() { return emaObject.getMuteList(); }

    /**
     * \~chinese
     * 返回白名单列表
     * 如果没有获取白名单列表，返回的列表可能为空
     * 获取白名单可调用{@link EMChatRoomManager#fetchChatRoomWhiteList(String, EMValueCallBack)},
     * 调用这个方法需要聊天室所有者或者管理员身份
     *
     * @return  白名单列表
     *
     * \~english
     * Return white list
     * If not get whitelist from server, the return may be empty
     * Fetch whitelist can call {@link EMChatRoomManager#fetchChatRoomMuteList(String, int, int)},
     * call this method need owner or admin authority
     * @return  whitelist
     */
    public List<String> getWhiteList() { return emaObject.getWhiteList(); }

    /**
     * \~chinese
     * 返回是否全员禁言
     * 此方法有使用局限性，建议慎重使用
     * 使用此方法需要注意使用的时机：
     * （1）加入聊天室后，接收到一键禁言/取消禁言的回调后，该状态会被更新，此时使用该方法获取的状态是正确的
     * （2）从聊天室退出后，再进入聊天室，从该方法获取到的状态是不可信的
     * @return boolean
     *
     * \~english
     * Return if all member muted
     * This method has limitations and is recommended to be used with caution
     * You need to be careful when you use this method：
     * (1) After joining the chat room, the status will be updated after receiving the all-mute/all-unmute callback,
     * and the status obtained using this method is correct
     * (2) After exiting from the chat room and reentering the chat room, the status obtained from this method is not trusted
     * @return boolean
     */
    public boolean isAllMemberMuted() { return emaObject.isAllMemberMuted(); }


    /**
     * \~chinese
     * 获取聊天室公告
     * 如果没有获取聊天室公告，返回的值为空
     * 获取聊天室公告，可调用{@link EMChatRoomManager#fetchChatRoomAnnouncement(String)},
     * 也可调用其异步方法{@link EMChatRoomManager#asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}
     * @return 公告
     *
     * \~english
     * Get chatroom announcement
     * If not get announcement from server, the return may be empty
     * Get chat room announcement, can call the method of {@link EMChatRoomManager#fetchChatRoomAnnouncement(String)},
     * also can call its asynchronous method of {@link EMChatRoomManager#asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}
     * @return  announcement
     */
    public String getAnnouncement() {
        return emaObject.getAnnouncement();
    }

    /**
     * \~chinese
     * 获取当前用户在群中的角色
     * @return EMChatRoomPermissionType
     *
     * \~english
     * Get the current user's role in chat room
     * @return EMChatRoomPermissionType
     */
    public EMChatRoomPermissionType getChatRoomPermissionType() {
        int permissionType = emaObject.permissionType();
        if(permissionType == EMChatRoomPermissionType.member.permissionType) {
            return EMChatRoomPermissionType.member;
        }else if(permissionType == EMChatRoomPermissionType.admin.permissionType) {
            return EMChatRoomPermissionType.admin;
        }else if(permissionType == EMChatRoomPermissionType.owner.permissionType) {
            return EMChatRoomPermissionType.owner;
        }else {
            return EMChatRoomPermissionType.none;
        }
    }

}
