/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

import com.hyphenate.chat.EMGroupReadAck;
import com.hyphenate.chat.EMMessage;

import java.util.List;

/**
 * \~chinese
 * 消息监听接口
 * 可以用来监听消息接收情况，成功发送到对方手机后会有回执（需要开启了送达回执，见{@link com.hyphenate.chat.EMOptions#setRequireDeliveryAck(boolean)}），
 * 对方阅读了这条消息也会收到回执（需要开启了允许已读回执，见{@link com.hyphenate.chat.EMOptions#setRequireAck(boolean)}）
 * 发送消息过程中，消息的ID会发生改变，由最初本地生成的一个UUID，变更为服务器端生成的全局唯一的ID，这个ID在所有使用SDK的设备上都是唯一的
 * 应用需要实现此接口来监听消息变更状态
 * <pre>
 *     添加消息监听：
 *     EMClient.getInstance().chatManager().addMessageListener(mMessageListener);
 *
 *     移除消息监听：
 *     EMClient.getInstance().chatManager().removeMessageListener(mMessageListener);
 * </pre>
 *
 * \~english
 * Message event listener.
 * Listening receive message, delivery acked (need enable delivery ack, see {@link com.hyphenate.chat.EMOptions#setRequireDeliveryAck(boolean)}),
 * read acked (need enable read ack, see {@link com.hyphenate.chat.EMOptions#setRequireAck(boolean)}
 * If send a message, message ID will change during delivery process, first SDK local device will generate a UUID, and send to server,
 * server will create a global unique ID, and notify local device to update local database's message related stuff.
 * App should implement this interface for monitoring any event related to the message
 * <pre>
 *     Add message listener:
 *     EMClient.getInstance().chatManager().addMessageListener(mMessageListener);
 *
 *     Remove message listener:
 *     EMClient.getInstance().chatManager().removeMessageListener(mMessageListener);
 * </pre>
 */
public interface EMMessageListener
{
	/**
     * \~chinese
	 * 接收消息
	 * 在接收到文本，图片，视频，语音，地理位置，文件这些消息的时候，会通过此回调通知用户
     *
     * \~english
	 * Messages received
	 * When receive the message of text, image, video, voice, location, file, call the callback to notify user
	 */
	void onMessageReceived(List<EMMessage> messages);

	/**
     * \~chinese
	 * 接收CMD消息
	 * 区别于{@link #onMessageReceived(List)}, 这个回调只包含命令的消息体，包含命令的消息体通常不对用户展示
	 *
     * \~english
	 * Command messages received.
	 * Difference from {@link EMMessageListener#onMessageReceived(List)}, this callback only contains command message,
	 * which normally not be displayed to user
	 */
	void onCmdMessageReceived(List<EMMessage> messages);

    /**
     * \~chinese
     * 接受到消息体的已读回执，消息的接收方已经阅读此消息
     *
     * \~english
     * Received message read ack by recipient as message had been read
     */
	void onMessageRead(List<EMMessage> messages);

	/**
	 * \~chinese
	 * 接受到群组消息体的已读回执, 消息的接收方已经阅读此消息。
	 *
	 * \~english
	 * Call back when received group message read ack by recipient as message had been read
	 */
	default void onGroupMessageRead(List<EMGroupReadAck> groupReadAcks) {}

	/**
	 * \~chinese
	 * 接受到群组消息体的已读数据更新
	 *
	 * \~english
	 * Call back when the data of group message read ack has updated
	 */
	default void onReadAckForGroupMessageUpdated() {}

    /**
     * \~chinese
     * 收到消息体的发送回执，消息体已经成功发送到对方设备
     *
     * \~english
     * Received message delivered ack as message delivered to recipient successfully
     */
	void onMessageDelivered(List<EMMessage> messages);

	/**
 	 * \~chinese
 	 * 收到消息体的撤回回调，消息体已经成功撤回
 	 *
 	 * \~english
 	 * Sender has recalled the messages
 	 */
    void onMessageRecalled(List<EMMessage> messages);
    
	/**
     * \~chinese
     * 接受消息发生改变的通知，包括消息ID的改变。消息是改变后的消息
     * @param message	发生改变的消息
     * @param change
     *
     * \~english
     * Received message change event, including message ID change
     *
     * @param message   Message value updated
     * @param change   
     */
	void onMessageChanged(EMMessage message, Object change);
}
