package tv.athena.live.player.vodplayer.utils;

import android.app.ActivityManager;
import android.content.Context;
import android.os.Build;
import android.os.Process;
import android.text.TextUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;

import tv.athena.live.utils.ALog;

/**
 * create by dengqu 2020/6/28
 */
public class SystemUtil {
    private final static String TAG = SystemUtil.class.getSimpleName();
    private static RandomAccessFile mProcStatFile;
    private static RandomAccessFile mAppStatFile;
    private static Long mLastCpuTime;
    private static Long mLastAppCpuTime;

    {
        getCpuRate();
    }

    /**
     * 获取当前手机系统版本号
     *
     * @return 系统版本号
     */
    public static String getSystemVersion() {
        return android.os.Build.VERSION.RELEASE;
    }

    public static String getOs() {
        try {
            StringBuilder stringBuilder = new StringBuilder();
            String version = getSystemVersion();
            int pos = version.indexOf(".") + 1;
            String subVersion = pos < version.length() ? version.substring(pos) : "0";
            stringBuilder.append("ANDROID");
            stringBuilder.append(" ");
            stringBuilder.append(version);
            stringBuilder.append(".");
            stringBuilder.append(subVersion);
            return stringBuilder.toString();
        } catch (Exception e) {
            ALog.e(TAG, "getOs", e);
        }
        return "-1";
    }

    /**
     * 获取cpu核数
     *
     * @return
     */
    public static int getNumberOfCPUCores() {
        if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.GINGERBREAD_MR1) {
            // Gingerbread doesn't support giving a single application access to both cores, but a
            // handful of devices (Atrix 4G and Droid X2 for example) were released with a dual-core
            // chipset and Gingerbread; that can let an app in the background run without impacting
            // the foreground application. But for our purposes, it makes them single core.
            return 1;  //上面的意思就是2.3以前不支持多核,有些特殊的设备有双核...不考虑,就当单核!!
        }
        int cores;
        try {
            cores = new File("/sys/devices/system/cpu/").listFiles(CPU_FILTER).length;
        } catch (Exception e) {
            cores = -1;   //这个常量得自己约定
            ALog.e(TAG, "getNumberOfCPUCores", e);
        }
        return cores;
    }

    private static final FileFilter CPU_FILTER = new FileFilter() {
        @Override
        public boolean accept(File pathname) {
            String path = pathname.getName();
            //regex is slow, so checking char by char.
            if (path.startsWith("cpu")) {
                for (int i = 3; i < path.length(); i++) {
                    if (path.charAt(i) < '0' || path.charAt(i) > '9') {
                        return false;
                    }
                }
                return true;
            }
            return false;
        }
    };

    /**
     * 获取当前app进程cpu利用率
     *
     * @return
     */
    public static int[] getCpuRate() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            return getCurrentCpuDataForO();
        } else {
            return getCPUDataBelowO();
        }
    }

    /**
     * 8.0一下获取cpu的方式
     *
     * @return
     */
    private static int[] getCPUDataBelowO() {
        long cpuTime;
        long appTime;
        int value = 0;
        int[] result = new int[2];
        result[0] = 0;
        result[1] = 0;
        int totalValue = 0;
        try {
            if (mProcStatFile == null || mAppStatFile == null) {
                mProcStatFile = new RandomAccessFile("/proc/stat", "r");
                mAppStatFile = new RandomAccessFile("/proc/" + android.os.Process.myPid() + "/stat", "r");
            } else {
                mProcStatFile.seek(0L);
                mAppStatFile.seek(0L);
            }
            String procStatString = mProcStatFile.readLine();
            String appStatString = mAppStatFile.readLine();
            String[] procStats = procStatString.split(" ");
            String[] appStats = appStatString.split(" ");
            cpuTime = Long.parseLong(procStats[2]) + Long.parseLong(procStats[3])
                + Long.parseLong(procStats[4]) + Long.parseLong(procStats[5])
                + Long.parseLong(procStats[6]) + Long.parseLong(procStats[7])
                + Long.parseLong(procStats[8]);
            appTime = Long.parseLong(appStats[13]) + Long.parseLong(appStats[14]);
            if (mLastCpuTime == null && mLastAppCpuTime == null) {
                mLastCpuTime = cpuTime;
                mLastAppCpuTime = appTime;
                return result;
            }
            totalValue = (int) ((cpuTime - mLastCpuTime) * 100);
            value = (int) (((float) (appTime - mLastAppCpuTime) / (float) (cpuTime - mLastCpuTime)) * 100);
            mLastCpuTime = cpuTime;
            mLastAppCpuTime = appTime;
            ALog.i("getCPUData", "mLastCpuTime =" + mLastCpuTime + ",mLastAppCpuTime =" +
                mLastAppCpuTime + ",value =" + value);
        } catch (Exception e) {
            ALog.e(TAG, "getCurrentCPUData", e);
        }
        result[0] = value;
        result[1] = totalValue;
        return result;
    }

    /**
     * 8.0以上获取cpu的方式
     *
     * @return
     */
    private static int[] getCurrentCpuDataForO() {
        java.lang.Process process = null;
        int[] result = new int[2];
        result[0] = 0;
        result[1] = 0;
        int totalValue = 0;
        try {
            process = Runtime.getRuntime().exec("top -n 1");
            BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String line;
            int cpuIndex = -1;
            while ((line = reader.readLine()) != null) {
                line = line.trim();
                if (TextUtils.isEmpty(line)) {
                    continue;
                }
                int tempIndex = getCPUIndex(line);
                if (tempIndex != -1) {
                    cpuIndex = tempIndex;
                    continue;
                }
                if (line.startsWith(String.valueOf(Process.myPid()))) {
                    if (cpuIndex == -1) {
                        continue;
                    }
                    String[] param = line.split("\\s+");
                    if (param.length <= cpuIndex) {
                        continue;
                    }
                    String cpu = param[cpuIndex];
                    if (cpu.endsWith("%")) {
                        cpu = cpu.substring(0, cpu.lastIndexOf("%"));
                    }
                    float rate = Float.parseFloat(cpu) / Runtime.getRuntime().availableProcessors();
                    int value = (int) rate;
                    ALog.i(TAG,
                        "getCurrentCpuDataForO line.startsWith(String.valueOf(Process.myPid())) value =" + value);
                    result[0] = value;
                    totalValue = totalValue + value;
                } else {
                    if (cpuIndex == -1) {
                        continue;
                    }
                    String[] param = line.split("\\s+");
                    if (param.length <= cpuIndex) {
                        continue;
                    }
                    String cpu = param[cpuIndex];
                    if (cpu.endsWith("%")) {
                        cpu = cpu.substring(0, cpu.lastIndexOf("%"));
                    }
                    float rate = Float.parseFloat(cpu) / Runtime.getRuntime().availableProcessors();
                    int value = (int) rate;
                    ALog.i(TAG, "getCurrentCpuDataForO value =" + value);
                    totalValue = totalValue + value;
                }
            }
        } catch (Exception e) {
            ALog.e(TAG, "getCurrentCpuDataForO", e);
        } finally {
            if (process != null) {
                process.destroy();
            }
        }
        result[1] = totalValue;
        ALog.i(TAG, "result result[0] =" + result[0] + ",result[1] =" + result[1]);
        return result;
    }

    private static int getCPUIndex(String line) {
        if (line.contains("CPU")) {
            String[] titles = line.split("\\s+");
            for (int i = 0; i < titles.length; i++) {
                if (titles[i].contains("CPU")) {
                    return i;
                }
            }
        }
        return -1;
    }

    //获取某进程CPU占用率的方法
    public static int getProcessCpuUsage(String pid) {
        try {
            RandomAccessFile reader = new RandomAccessFile("/proc/stat", "r");
            String load = reader.readLine();
            String[] toks = load.split(" ");

            double totalCpuTime1 = 0.0;
            int len = toks.length;
            for (int i = 2; i < len; i++) {
                totalCpuTime1 += Double.parseDouble(toks[i]);
            }


            BufferedReader readerBB = new BufferedReader(new FileReader("/proc/"));
            String processNameB = readerBB.readLine();
            RandomAccessFile reader2 = new RandomAccessFile("/proc/" + pid + "/stat", "r");
            String load2 = reader2.readLine();
            String[] toks2 = load2.split(" ");

            double processCpuTime1 = 0.0;
            double utime = Double.parseDouble(toks2[13]);
            double stime = Double.parseDouble(toks2[14]);
            double cutime = Double.parseDouble(toks2[15]);
            double cstime = Double.parseDouble(toks2[16]);

            processCpuTime1 = utime + stime + cutime + cstime;

            try {
                Thread.sleep(360);
            } catch (Exception e) {
                e.printStackTrace();
            }
            reader.seek(0);
            load = reader.readLine();
            reader.close();
            toks = load.split(" ");
            double totalCpuTime2 = 0.0;
            len = toks.length;
            for (int i = 2; i < len; i++) {
                totalCpuTime2 += Double.parseDouble(toks[i]);
            }
            reader2.seek(0);
            load2 = reader2.readLine();
            String[] toks3 = load2.split(" ");

            double processCpuTime2 = 0.0;
            utime = Double.parseDouble(toks3[13]);
            stime = Double.parseDouble(toks3[14]);
            cutime = Double.parseDouble(toks3[15]);
            cstime = Double.parseDouble(toks3[16]);

            processCpuTime2 = utime + stime + cutime + cstime;
            int usage = (int) ((processCpuTime2 - processCpuTime1) * 100.00
                / (totalCpuTime2 - totalCpuTime1));
            return usage;
        } catch (Exception e) {
            ALog.e(TAG, "getProcessCpuUsage", e);
        }
        return -1;
    }

    /**
     * 获取手机内存大小
     */
    public static long getTotalMemorySize() {
        String dir = "/proc/meminfo";
        try {
            FileReader fr = new FileReader(dir);
            BufferedReader br = new BufferedReader(fr, 2048);
            String memoryLine = br.readLine();
            String subMemoryLine = memoryLine.substring(memoryLine.indexOf("MemTotal:"));
            br.close();
            long totalMemorySize = Integer.parseInt(subMemoryLine.replaceAll("\\D+", ""));
            return totalMemorySize;
        } catch (Exception e) {
            ALog.e(TAG, "getTotalMemorySize", e);
        }
        return -1;
    }

    /**
     * 计算已使用内存的百分比，并返回。
     */
    public static int getUsedPercentValue(Context context) {
        String dir = "/proc/meminfo";
        try {
            FileReader fr = new FileReader(dir);
            BufferedReader br = new BufferedReader(fr, 2048);
            String memoryLine = br.readLine();
            String subMemoryLine = memoryLine.substring(memoryLine.indexOf("MemTotal:"));
            br.close();
            long totalMemorySize = Integer.parseInt(subMemoryLine.replaceAll("\\D+", ""));
            long availableSize = getAvailableMemory(context) / 1024;
            float percent = ((totalMemorySize - availableSize) / (totalMemorySize * 1.0f));
            return (int) (percent * 100);
        } catch (Exception e) {
            ALog.e(TAG, "getUsedPercentValue", e);
        }
        return -1;
    }

    /**
     * 获取当前可用内存，返回数据以字节为单位。
     */
    private static long getAvailableMemory(Context context) {
        ActivityManager.MemoryInfo mi = new ActivityManager.MemoryInfo();
        ActivityManager activityManager = getActivityManager(context);
        if (activityManager == null) {
            return 0;
        }
        activityManager.getMemoryInfo(mi);
        if (mi == null) {
            return 0;
        }
        return mi.availMem;
    }

    private static ActivityManager mActivityManager;

    public synchronized static ActivityManager getActivityManager(Context context) {
        if (mActivityManager == null) {
            if (context == null) {
                return null;
            }
            mActivityManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
        }
        return mActivityManager;
    }

    /**
     * 当前app内存使用率
     *
     * @param context
     * @return
     */
    public static int getAppMemoryUser(Context context) {
        ActivityManager activityManager = getActivityManager(context);
        if (activityManager == null) {
            return 0;
        }
        //最大分配内存
        int memory = activityManager.getMemoryClass();
        System.out.println("memory: " + memory);
        //当前分配的总内存
        float totalMemory = (float) (Runtime.getRuntime().totalMemory() * 1.0 / (1024 * 1024));
        //剩余内存
        float freeMemory = (float) (Runtime.getRuntime().freeMemory() * 1.0 / (1024 * 1024));
        float percent = ((totalMemory - freeMemory) / (totalMemory * 1.0f));
        return (int) (percent * 100);
    }
}
