package tv.athena.live.player.vodplayer

import android.content.Context
import android.os.Looper
import android.text.TextUtils
import com.yy.transvod.downloader.MediaDownloader
import com.yy.transvod.downloader.MediaDownloaderOptions
import com.yy.transvod.player.DataSource
import com.yy.transvod.preference.Preference
import tv.athena.annotation.ServiceRegister
import tv.athena.core.axis.Axis
import tv.athena.live.basesdk.liveroom.LivePlatformSdk
import tv.athena.live.player.IAthLiveMediaPlayer
import tv.athena.live.player.IAthLivePlayerEngine
import tv.athena.live.player.IAthLivePlayerStatisticsService
import tv.athena.live.player.statistics.common.CommonStatisticsConfig
import tv.athena.live.utils.ALog
import java.util.concurrent.ConcurrentHashMap

/**
 *create by dengqu 2020/6/1
 */
@ServiceRegister(serviceInterface = IAthLivePlayerEngine::class)
class VodPlayerEngineImpl : IAthLivePlayerEngine {
    companion object {
        private const val TAG = "VodPlayerEngineImpl"
    }

    private var mContext: Context? = null
    private var mCacheDirectory: String? = null
    private var mMediaDownloader: MediaDownloader? = null
    private var mDataSources = ConcurrentHashMap<String, AthDownloadDataSource>()

    /**
     * 下载数据实体类
     */
    data class AthDownloadDataSource(
        val uuid: String, val url: String, val isSupportQuic: Boolean, val dataSource: DataSource
    )

    override fun initialize(
        context: Context, appId: String, sceneId: Long, logLevel: Int, cacheDirectory:
        String?, looper: Looper?
    ) {
        LivePlatformSdk.getInstance().tryLoadTransvodSo()
        this.mContext = context?.applicationContext
        mCacheDirectory = cacheDirectory
        CommonStatisticsConfig.mAppid = appId
        CommonStatisticsConfig.packageName = context.packageName
        CommonStatisticsConfig.mContext = context.applicationContext
        VodPlayerPreference.initVodPlayerDns()
        Axis.getService(IAthLivePlayerStatisticsService::class.java)?.init(context?.packageName, context)
        initMediaDownloader()
    }

    private fun initMediaDownloader() {
        val downOptions = MediaDownloaderOptions()
        downOptions.mApplicationContext = getApplicationContext()
        downOptions.mCacheDir = if (TextUtils.isEmpty(mCacheDirectory)) {
            AthLiveMediaPlayerFactory.DEFAULT_CACHE_DIRECTORY
        } else {
            mCacheDirectory
        }
        mMediaDownloader = MediaDownloader(downOptions)
    }

    override fun deInitialize() {
    }

    override fun startPlayerEngine(uid: String) {
    }

    override fun stopPlayerEngine() {
    }

    override fun createPlayer(
        uuid: String, userUUID: String,
        samplerFilter: Int, forceUse601ColorStandard: Boolean, directRenderToSurface: Boolean, isMultiInstance:
        Boolean, userTextureView: Boolean
    ): IAthLiveMediaPlayer {
        ALog.i(TAG, "createPlayer ---------- $mContext")
        return AthLiveMediaPlayerFactory.instance.getAthLiveMediaPlayer(
            mContext, mCacheDirectory,
            uuid, userUUID, samplerFilter, forceUse601ColorStandard, directRenderToSurface, isMultiInstance,
            userTextureView
        )
    }

    override fun destroyPlayer(player: IAthLiveMediaPlayer, uuid: String, userUUID: String, isMultiInstance: Boolean) {
        AthLiveMediaPlayerFactory.instance.releasePlayer(uuid, userUUID, isMultiInstance)
    }

    override fun setVodConfigs(map: HashMap<String, String>?) {
        ALog.i(TAG, "setVodConfigs map = $map")
        VodPlayerPreference.setVodConfigs(map)
    }

    override fun getApplicationContext(): Context? {
        return mContext
    }

    override fun startDownloadMedia(uuid: String, url: String, isSupportQuic: Boolean, isDash: Boolean) {
        ALog.i(
            TAG, "startDownloadMedia before uuid = $uuid,url = $url ,isSupportQuic = $isSupportQuic,mDataSources " +
                "size = " +
                "${
                    mDataSources
                        .size
                }"
        )
        if (isHasDownload(uuid, url)) {
            ALog.i(TAG, "uuid =$uuid has startDownloadMedia-------- ")
            return
        }
        ALog.i(TAG, "uuid =$uuid not startDownloadMedia-------- ")

        val srcFmt = if (isDash) {
            DataSource.SOURCE_FMT_DASH
        } else {
            DataSource.SOURCE_FMT_FLV
        }
        val cachePolicy = if (isDash) {
            DataSource.CACHE_POLICY_NOCACHE
        } else {
            DataSource.CACHE_POLICY_STREAM_LATEST_GOP
        }
        val proto = if (isSupportQuic || isDash) {
            DataSource.URL_PROTO_USER
        } else {
            DataSource.URL_PROTO_HTTP
        }
        val mDataSource = DataSource(url, proto, srcFmt, cachePolicy, true)
        mMediaDownloader?.startDownloadMedia(mDataSource)
        mDataSources[uuid] = AthDownloadDataSource(uuid, url, isSupportQuic, mDataSource)
        ALog.i(
            TAG,
            "startDownloadMedia after uuid = $uuid,url = $url ," +
                "isSupportQuic = $isSupportQuic," +
                "mDataSources " +
                "size = " +
                "${
                    mDataSources
                        .size
                }"
        )
    }

    override fun stopDownloadMedia(uuid: String, url: String) {
        ALog.i(
            TAG,
            "stopDownloadMedia before uuid = $uuid,url = $url," +
                "size = ${mDataSources?.size}"
        )
        mDataSources?.let { it ->
            if (it.isNotEmpty()) {
                var tempDataSources = ConcurrentHashMap<String, AthDownloadDataSource>()
                for ((key, value) in it) {
                    if ((TextUtils.equals(key, uuid) || TextUtils.equals(value.url, url))) {
                        ALog.i(TAG, "stopDownloadMedia find --------------")
                        mMediaDownloader?.stopDownloadMedia(value.dataSource)
                    } else {
                        tempDataSources[key] = value
                    }
                }
                mDataSources.clear()
                mDataSources = tempDataSources
            }
        }
        ALog.i(
            TAG,
            "stopDownloadMedia after uuid = $uuid,url = $url," +
                "size = ${mDataSources?.size}"
        )
    }

    override fun onAabReady() {
        Preference.loadSoLibForAabMode()
    }

    private fun isHasDownload(uuid: String, url: String): Boolean {
        if (mDataSources.isNotEmpty()) {
            for ((key, value) in mDataSources) {
                if ((TextUtils.equals(key, uuid) || TextUtils.equals(value.url, url))) {
                    ALog.i(TAG, "isHasDownload find-------------------")
                    return true
                }
            }
        }
        return false
    }
}