package tv.athena.amp.core.log;

import java.util.UUID;

import tv.athena.amp.core.log.trace.Trace;

public final class LogId {

    private final static long MAX_10BIT_NUM = 1024;
    private final static long MAX_18BIT_NUM = 262144;
    private static final LogId instance = new LogId();
    private long mBeginTime = System.currentTimeMillis() / 1000;
    private long mDeviceHashID = 0;
    private long mCurrLoopCnt;
    private LogId() {
        mCurrLoopCnt = 0;

        try {
            String deviceId = UUID.randomUUID().toString();
            byte[] deviceIdBytes = deviceId.getBytes();
            mDeviceHashID = genHashNum(deviceIdBytes);
        } catch (Throwable t) {
            AmpLog.e("LogId", Trace.once("Failed constructing IDGenerator")
                    .info("exception", t.getMessage()));
        }
    }

    public static long generate() {
        return instance.makeId();
    }

    private long makeId() {
        if (mCurrLoopCnt >= (MAX_10BIT_NUM - 1)) {
            mBeginTime = System.currentTimeMillis() / 1000; // 每1024条消息重新计算新的时间
            mCurrLoopCnt = 0;
        }

        long resID = ((mBeginTime << 32) & 0x7fffffff00000000L);

        long mTermType = Terminal.ANDROID;
        resID |= ((mTermType << 28) & 0xf0000000);
        resID |= ((mDeviceHashID << 10) & 0x0ffffc00);
        resID |= (mCurrLoopCnt & 0x000003ff);

        mCurrLoopCnt++;

        return resID;
    }

    private long genHashNum(byte[] data) {
        if (data == null || data[0] == 0) {
            return 0;
        }

        long resHash = 5381;
        int curr = 0;
        int end = data.length;

        while (curr < end) {
            resHash *= 33;
            resHash ^= (int) (data[curr]);
            curr++;
            if (0 == curr) {
                break;
            }
        }

        resHash = resHash % LogId.MAX_18BIT_NUM;
        return resHash;
    }

    private final class Terminal {
        static final int ANDROID = 1;
    }
}
