package tv.athena.amp.core

import com.google.protobuf.nano.MessageNano
import tv.athena.amp.api.IAmpChannel
import tv.athena.amp.api.IAmpChannelCallback
import tv.athena.amp.api.IAmpGroup
import tv.athena.amp.api.IAmpSdk
import tv.athena.amp.api.IAmpView
import tv.athena.amp.core.log.AmpLog
import tv.athena.amp.proto.nano.Amp
import tv.athena.core.axis.Axis

internal class AmpDataCenter(channel: IAmpChannel) : IAmpChannelCallback {


    companion object {
        const val TAG = "AmpDataCenter"
        const val KEY_MOBILE_URL = "mobUrl"
    }

    private var mChannel: IAmpChannel? = channel
    private var mGroupMap: MutableMap<String, IAmpGroup> = mutableMapOf()
    private var mTsMap: MutableMap<String, Long?> = mutableMapOf()

    fun init() {
        AmpLog.d(TAG, "init")
        mChannel?.apply {
            addCallback(this@AmpDataCenter)
            AmpLog.d(TAG, "init channel, channel=${mChannel?.getName()}")
            init()
        } ?: kotlin.run {
            AmpLog.e(TAG, "init failed, channel is null")
        }
    }

    fun load(req: Amp.PRMPReq) {
        mChannel?.apply {
            AmpLog.d(TAG, "start request, req=>\n$req")
            request(req)
        } ?: kotlin.run {
            AmpLog.e(TAG, "request failed, channel isn't init")
        }
    }

    override fun onMgrData(data: Amp.PAMPMgr) {

        // 匹配活动组
        val groupMatch = getAmpGroup(gid = data.gid, insId = data.insid, bcId = data.bcid)
        if (groupMatch == null || groupMatch.isEmpty()) {
            AmpLog.d(
                TAG, "discard mgr data, act group not found, " +
                    "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}"
            )
            return
        }

        groupMatch.filterNotNull().forEach { group ->

            AmpLog.d(TAG, "dispatch mgr data, group[gid=${group.getGid()}, insId=${group.getInsId()}]")

            // 校验ts
            val tsKey = "${group.getGid()}_${group.getInsId()}_${data.actid}"
            val lastTs: Long? = mTsMap[tsKey]
            if (lastTs != null && data.ts <= lastTs) {
                AmpLog.w(TAG, "discard mgr data, tsKey=$tsKey, lastTs=$lastTs, ts=${data.ts}")
                return@forEach
            }
            mTsMap[tsKey] = lastTs

            // 校验是否存在这个活动
            val views = group.getAmpViews() ?: emptyList()
            var view: IAmpView? = views.find { it.getActId() == data.actid }
            if (view == null) {
                if (data.isShow) {
                    // 新增活动
                    AmpLog.d(
                        TAG, "dispatch mgr data, create a new act, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${data.actid}"
                    )
                    view = group.getAmpViewFactory().createView()
                    view.init(data.actid, data.isShow, data.prio, group)
                    val newViews = views.toMutableList().apply {
                        add(view)
                        // 根据优先级重新排序
                        sortByDescending  { it.getPriority() }
                    }.toList()

                    // 刷新活动组
                    AmpLog.d(
                        TAG, "dispatch mgr data, refresh act group, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                            "views=${views.count()}, newViews=${newViews.count()}"
                    )
                    group.setAmpViews(newViews)
                    group.getCallback()?.onRefresh(group, newViews)

                    // 刷新活动
                    AmpLog.d(
                        TAG, "dispatch mgr data, refresh act view, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                            "actId=${view.getActId()}, views=${views.count()}, newViews=${newViews.count()}"
                    )
                    view.getCallback()?.onRefresh(group, data.urlMap?.get(KEY_MOBILE_URL), data.bd, data.extend)

                    //新增广播订阅[new]
                    AmpLog.d(
                        TAG, "dispatch mgr data, process subscribe[new] start, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${view.getActId()}, " +
                            "bc=${data.bc?.contentToString()}, group.bcs=${group.getBcs()}, " +
                            "group.fullBcs=${group.getFullBcs()}"
                    )
                    val newBcs = (group.getBcs() ?: emptySet()).toMutableSet()
                    data.bc?.filterNot {
                        group.getBcs()?.contains(it) ?: false
                    }?.toTypedArray()?.takeIf { it.isNotEmpty() }?.apply {
                        AmpLog.d(
                            TAG, "dispatch mgr data, process subscribe[new], channel subscribe, " +
                                "bcs=${contentToString()}"
                        )
                        IAmpChannel.parseBcId(this)?.also {
                            AmpLog.d(
                                TAG, "dispatch mgr data, process subscribe[new], channel subscribe, " +
                                    "bcItems=${it.contentToString()}"
                            )
                            mChannel?.subscribe(it)
                        }
                        newBcs.addAll(this)
                    }
                    group.setBcs(bcs = newBcs.toSet())
                    group.addFullBcs(view.getActId(), data.bc?.toSet())

                    AmpLog.d(
                        TAG, "dispatch mgr data, process subscribe[new] end, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${view.getActId()}, " +
                            "bc=${data.bc?.contentToString()}, group.bcs=${group.getBcs()}," +
                            " group.fullBcs=${group.getFullBcs()}"
                    )
                } else {
                    AmpLog.d(
                        TAG, "dispatch mgr data, discard create a new act, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                            "actId=${data.actid}, isShow=${data.isShow}"
                    )
                }
            } else {
                if (data.isShow) {
                    // 更新活动（这种情况基本应该不会存在？）
                    AmpLog.d(
                        TAG, "dispatch mgr data, update a act, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${data.actid}"
                    )

                    // 刷新活动组
                    if (data.prio != view.getPriority()) {
                        view.setPriority(data.prio)
                        val newViews = views.toMutableList().apply {
                            sortByDescending { it.getPriority() }
                        }.toList()
                        AmpLog.d(
                            TAG, "dispatch mgr data, refresh act group, " +
                                "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                                "views=${newViews.count()}"
                        )
                        group.setAmpViews(newViews)
                        group.getCallback()?.onRefresh(group, newViews)
                    }

                    // 刷新活动
                    AmpLog.d(
                        TAG, "dispatch mgr data, refresh act view, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                            "actId=${view.getActId()}, views=${views.count()}"
                    )
                    view.getCallback()?.onRefresh(group, data.urlMap?.get(KEY_MOBILE_URL), data.bd, data.extend)

                    //更新广播订阅[update]
                    AmpLog.d(
                        TAG, "dispatch mgr data, process subscribe[update] start, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${view.getActId()}, " +
                            "bc=${data.bc?.contentToString()}, group.bcs=${group.getBcs()}, " +
                            "group.fullBcs=${group.getFullBcs()}"
                    )
                    val newBcs = (group.getBcs() ?: emptySet()).toMutableSet()
                    data.bc?.filterNot {
                        group.getBcs()?.contains(it) ?: false
                    }?.toTypedArray()?.takeIf { it.isNotEmpty() }?.apply {
                        AmpLog.d(
                            TAG, "dispatch mgr data, process subscribe[update], channel subscribe, " +
                                "bcs=${contentToString()}"
                        )
                        IAmpChannel.parseBcId(this)?.also {
                            AmpLog.d(
                                TAG, "dispatch mgr data, process subscribe[update], channel subscribe, " +
                                    "bcItems=${it.contentToString()}"
                            )
                            mChannel?.subscribe(it)
                        }
                        newBcs.addAll(this)
                    }
                    val otherFullBcs: Map<Int, Set<String>?>? = group.getFullBcs()
                        ?.filterNot { it.key == view.getActId() }
                    group.getFullBcs()[view.getActId()]?.filterNot {
                        data.bc?.contains(it) ?: false
                                || otherFullBcs?.any { entry -> entry.value?.contains(it) ?: false } ?: false
                    }?.toTypedArray()?.takeIf { it.isNotEmpty() }?.apply {
                        AmpLog.d(
                            TAG, "dispatch mgr data, process subscribe[update], channel unSubscribe, " +
                                "bcs=${contentToString()}"
                        )
                        IAmpChannel.parseBcId(this)?.also {
                            AmpLog.d(
                                TAG, "dispatch mgr data, process subscribe[update], channel unSubscribe, " +
                                    "bcItems=${it.contentToString()}"
                            )
                            mChannel?.unSubscribe(it)
                        }
                        newBcs.removeAll(this)
                    }
                    group.setBcs(newBcs)
                    group.addFullBcs(view.getActId(), data.bc?.toSet())

                    AmpLog.d(
                        TAG, "dispatch mgr data, process subscribe[update] end, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${view.getActId()}, " +
                            "bc=${data.bc?.contentToString()}, group.bcs=${group.getBcs()}," +
                            " group.fullBcs=${group.getFullBcs()}"
                    )
                } else {
                    // 移除活动
                    AmpLog.d(
                        TAG, "dispatch mgr data, delete a act, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${data.actid}"
                    )
                    val newViews = views.toMutableList().apply {
                        remove(view)
                    }.toList()

                    // 刷新活动组
                    AmpLog.d(
                        TAG, "process mgr data, refresh act group, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                            "views=${views.count()}, newViews=${newViews.count()}"
                    )
                    group.setAmpViews(newViews)
                    group.getCallback()?.onRefresh(group, newViews)

                    // 刷新活动
                    AmpLog.d(
                        TAG, "dispatch mgr data, destroy act view, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                            "actId=${view.getActId()}, views=${views.count()}, newViews=${newViews.count()}"
                    )
                    view.destroy()

                    //取消广播订阅(remove)
                    AmpLog.d(
                        TAG, "dispatch mgr data, process subscribe[remove] start, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${view.getActId()}, " +
                            "bc=${data.bc?.contentToString()}, group.bcs=${group.getBcs()}," +
                            " group.fullBcs=${group.getFullBcs()}"
                    )
                    val newBcs = (group.getBcs() ?: emptySet()).toMutableSet()
                    val otherFullBcs: Map<Int, Set<String>?>? = group.getFullBcs()
                        ?.filterNot { it.key == view.getActId() }
                    group.getFullBcs()[view.getActId()]?.filterNot {
                        data.bc?.contains(it) ?: false
                                || otherFullBcs?.any { entry -> entry.value?.contains(it) ?: false } ?: false
                    }?.toTypedArray()?.takeIf { it.isNotEmpty() }?.apply {
                        AmpLog.d(
                            TAG, "dispatch mgr data, process subscribe[remove], channel unSubscribe," +
                                "bcs=${contentToString()}"
                        )
                        IAmpChannel.parseBcId(this)?.also {
                            AmpLog.d(
                                TAG, "dispatch mgr data, process subscribe[remove], channel unSubscribe," +
                                    "bcItems=${it.contentToString()}"
                            )
                            mChannel?.unSubscribe(it)
                        }
                        newBcs.removeAll(this)
                    }
                    group.setBcs(newBcs)
                    group.removeFullBcs(view.getActId())

                    AmpLog.d(
                        TAG, "dispatch mgr data, process subscribe[remove] end, " +
                            "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${view.getActId()}, " +
                            "bc=${data.bc?.contentToString()}, group.bcs=${group.getBcs()}, " +
                            "group.fullBcs=${group.getFullBcs()}"
                    )
                }
            }
        }
    }

    override fun onUpdData(data: Amp.PAMPUpdate) {

        // 匹配活动组
        val groupMatch = getAmpGroup(gid = data.gid, insId = data.insid, bcId = data.bcid)
        if (groupMatch == null || groupMatch.isEmpty()) {
            AmpLog.d(
                TAG, "discard upd data, act group not found, " +
                    "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}"
            )
            return
        }

        groupMatch.filterNotNull().forEach { group ->

            AmpLog.d(TAG, "dispatch upd data, group[gid=${group.getGid()}, insId=${group.getInsId()}]")

            // 校验ts
            val tsKey = "${group.getGid()}_${group.getInsId()}_${data.actid}"
            val lastTs: Long? = mTsMap[tsKey]
            if (lastTs != null && data.ts <= lastTs) {
                AmpLog.w(TAG, "discard upd data, tsKey=$tsKey, lastTs=$lastTs, ts=${data.ts}")
                return@forEach
            }
            mTsMap[tsKey] = lastTs

            // 校验是否存在这个活动
            val views = group.getAmpViews() ?: emptyList()
            val view: IAmpView? = views.find { it.getActId() == data.actid }
            if (view != null) {
                // 刷新活动
                AmpLog.d(
                    TAG, "dispatch upd data, refresh act view, " +
                        "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, " +
                        "actId=${view.getActId()}, views=${views.count()}"
                )
                view.getCallback()?.onRefresh(group, null, data.bd, data.extend)
            } else {
                AmpLog.d(
                    TAG, "discard upd data, act not found, " +
                        "gid=${data.gid}, insId=${data.insid}, bcId=${data.bcid}, actId=${data.actid}"
                )
                return
            }
        }
    }

    override fun onLoadRsp(data: Amp.PAMPLoadRsp) {
        AmpLog.i(TAG, "onLoadRsp:msg=$data")
        IAmpChannel.parseBcId(data.baseBc)?.also {
            AmpLog.d(
                    TAG, "dispatch mgr data, process subscribe[update], channel subscribe, " +
                    "bcItems=${it.contentToString()}"
            )
            mChannel?.subscribe(it)
        }
    }

    override fun onReLoad(data: Amp.PAMPReLoad) {
        getAmpGroup(data.gid, data.insid, data.bcid)?.forEach { group ->
            group?.let {
                val req = Amp.PRMPReq().apply {
                    this.serviceName = IAmpChannel.AMP_SERVICE
                    this.functionName = IAmpChannel.AMP_FUN_LOAD
                    this.appid = Axis.getService(IAmpSdk::class.java)?.getConfig()?.bizId ?: 0
                    this.ticket = Axis.getService(IAmpSdk::class.java)?.getConfig()?.ticket?.getToken()
                    this.data = MessageNano.toByteArray(group.getLoadParam().apply {
                        // 自动补全部分参数
                        this.gid = group.getGid()
                        this.insid = group.getInsId()
                        this.extend = data.extend
                    })
                }
                load(req)
            }
        }
    }

    fun addAmpGroup(group: IAmpGroup) {
        AmpLog.d(TAG, "addAmpGroup, group=${group.getKey()}")
        mGroupMap[group.getKey()] = group
    }

    fun getAmpGroup(key: String): IAmpGroup? {
        return mGroupMap[key]
    }

    private fun getAmpGroup(gid: String, insId: String?, bcId: String?): List<IAmpGroup?>? {
        (if (insId == null || insId.trim().isEmpty()) null else insId)?.apply {
            // 指定insId, 根据gid + insId 筛选
            val key = "${gid}_$insId"
            return listOfNotNull(getAmpGroup(key))
        } ?: kotlin.run {
            (if (bcId == null || bcId.trim().isEmpty()) null else bcId)?.apply {
                // 指定bcId, 根据gid + bc[bcid] 筛选
                return mGroupMap.filter { (key, value) ->
                    key.startsWith(gid) && (value.getBcs()?.contains(bcId) ?: false)
                }.values.toList()
            } ?: kotlin.run {
                // insId、bcId 均未指定，根据gid筛选
                return mGroupMap.filterKeys { key -> key.startsWith(gid) }.values.toList()
            }
        }
        return null
    }

    fun removeAmpGroup(group: IAmpGroup) {
        removeAmpGroupByKey(group.getKey())
    }

    private fun removeAmpGroupByKey(key: String) {
        mGroupMap[key]?.apply {
            AmpLog.d(TAG, "removeAmpGroupByKey, remove group, group=${getKey()}")

            // 取消活动组订阅的广播(要考虑不同活动组bcid相同的情况)
            val allBcs = getAllBcs(this)
            IAmpChannel.parseBcId(getBcs()?.filterNot { allBcs.contains(it) }?.toTypedArray())
                    ?.takeIf { it.isNotEmpty() }?.let {
                        AmpLog.d(TAG, "removeAmpGroupByKey, channel unSubscribe, bcItems=${it.contentToString()}")
                        mChannel?.unSubscribe(it)
                    }

            // 移除活动组对应的ts缓存
            mTsMap.keys.filter { it.startsWith(getKey()) }.forEach {
                AmpLog.d(TAG, "removeAmpGroupByKey, remove ts, tsKey=$it")
                mTsMap.remove(it)
            }

            // 销毁活动组
            destroy()

            // 移除活动组缓存
            mGroupMap.remove(key)
        }
    }

    private fun removeAllAmpGroup() {
        AmpLog.d(TAG, "removeAllAmpGroup")
        mGroupMap.apply {
            forEach { it.value.apply { removeAmpGroup(this) } }
            clear()
        }
    }

    private fun getAllBcs(excludeGroup: IAmpGroup?): Set<String> {
        val bcs = mutableSetOf<String>()
        mGroupMap.filterNot { it.key == excludeGroup?.getKey() }.values.forEach {
            it.getBcs()?.toTypedArray()?.takeIf { array -> array.isNotEmpty() }?.apply {
                bcs.addAll(this)
            }
        }
        return bcs
    }

    fun destroy() {
        AmpLog.d(TAG, "destroy")
        mChannel?.apply {
            removeCallback(this@AmpDataCenter)
            destroy()
        }
        mChannel = null
        removeAllAmpGroup()
    }
}