package com.yy.mobile.ipc

import android.content.SharedPreferences
import android.text.TextUtils
import android.util.Log
import com.google.gson.Gson
import java.util.*

/**
 * Rule : input key cannot be null.
 */
abstract class AbstractSharedPref(protected val mPref: SharedPreferences) {
    fun putString(key: String, value: String) {
        put(key, value)
    }

    fun getString(key: String): String {
        return get(key) ?: ""
    }

    fun getString(key: String, defaultValue: String?): String {
        return mPref.getString(key, defaultValue)?:""
    }

    fun putInt(key: String, value: Int) {
        put(key, value.toString())
    }

    fun putBoolean(key: String, value: Boolean) {
        put(key, value.toString())
    }

    fun getBoolean(key: String, defaultValue: Boolean): Boolean {
        val rawValue = get(key)
        return if (TextUtils.isEmpty(rawValue)) {
            defaultValue
        } else try {
            java.lang.Boolean.parseBoolean(rawValue)
        } catch (e: Exception) {
            Log.e(
                TAG,
                "failed to parse boolean value for key $key",
                e
            )
            defaultValue
        }
    }

    fun getInt(key: String, defaultValue: Int): Int {
        val rawValue = get(key) ?: ""
        return if (TextUtils.isEmpty(rawValue)) {
            defaultValue
        } else parseInt(rawValue, defaultValue)
    }

    private fun parseInt(value: String, defaultValue: Int): Int {
        return try {
            value.toInt()
        } catch (e: NumberFormatException) {
            Log.e(
                TAG,
                "lcy failed to parse value for key $value",
                e
            )
            defaultValue
        }
    }

    fun getInt(key: String): Int {
        return getInt(key, -1)
    }

    fun putLong(key: String, value: Long) {
        put(key, value.toString())
    }

    fun getLong(key: String, defaultValue: Long): Long {
        val rawValue = get(key)
        return if (TextUtils.isEmpty(rawValue)) {
            defaultValue
        } else try {
            rawValue?.toLong() ?: 0L
        } catch (e: NumberFormatException) {
            Log.e(
                TAG,
                "lcy failed to parse $rawValue as long, for key $key, ex : %s", e
            )
            defaultValue
        }
    }

    fun getLong(key: String): Long {
        return getLong(key, -1L)
    }

    fun putIntArray(key: String, values: Array<Int?>) {
        putIntList(key, listOf(*values))
    }

    fun getIntArray(key: String): IntArray? {
        return getIntArray(key, intArrayOf())
    }

    /**
     * @param key
     * @param outValues For memory reuse, if the result is no greater than this space,
     * will fill into this, the redundant elements won't be touched.
     * If it is null, a new int array will be created if result is
     * not empty.
     * @return The result list, null if no correlated.
     */
    fun getIntArray(key: String, outValues: IntArray): IntArray? {
        val list = getIntList(key)
        if (list == null || list.isEmpty()) {
            return null
        }
        val ret =
            if (list.size <= outValues.size) outValues else IntArray(list.size)
        var i = 0
        for (e in list) {
            ret[i++] = e
        }
        return ret
    }

    fun putIntList(key: String, values: List<Int?>?) {
        if (values == null || values.size == 0) {
            return
        }
        val value =
            TextUtils.join(DELIMITER, values)
        put(key, value)
    }

    fun getIntList(key: String): List<Int>? {
        val `val` = get(key)
        if (TextUtils.isEmpty(`val`)) {
            return null
        }
        val values =
            TextUtils.split(`val`, DELIMITER)
        if (values == null || values.size == 0) {
            return null
        }
        val list = ArrayList<Int>()
        for (e in values) {
            try {
                list.add(e.toInt())
            } catch (ex: NumberFormatException) {
                Log.e(
                    TAG,
                    "lcy failed to parse value for key: $key, value: $e, exception: %s",
                    ex
                )
                continue
            }
        }
        return list
    }

    fun put(key: String, value: String) {
        mPref.edit().putString(key, value).apply()
    }

    operator fun get(key: String): String? {
        return mPref.getString(key, null)
    }

    fun remove(key: String) {
        mPref.edit().remove(key).apply()
    }

    fun clear() {
        mPref.edit().clear().apply()
    }

    val all: Map<String, *>
        get() = mPref.all

    fun contain(key: String): Boolean {
        return if (key == null || key.isEmpty()) {
            false
        } else mPref.contains(key)
    }

    fun putObject(key: String, obj: Any) {
        val gson = Gson()
        val json = gson.toJson(obj)
        put(key, json)
    }

    fun getObj(key: String, className: Class<*>): Any {
        val gson = Gson()
        val json = getString(key, "")
        return gson.fromJson(json, className)
    }

    companion object {
        private const val TAG = "YSharedPref"
        private const val DELIMITER = ","
    }

}