package com.yy.mobile.replugin

import android.util.Log
import com.yy.mobile.ipc.IPCManager
import com.yy.mobile.ipc.IpcApiSetting
import xiaofei.library.hermes.Hermes
import xiaofei.library.hermes.util.TypeUtils
import java.util.concurrent.ConcurrentHashMap

/**
 * Created by wangfeihang on 2019/5/14.
 *
 * replugin插件与宿主接口注册类
 */
object ApiBridge {

    private val TAG = "RepluginBridge"

    /**
     * [T]接口被注册后，其他插件或者宿主都可以通过[getApi]方法来调用接口[T]
     * 前提条件是本插件加载成功
     * @param clazz [T]的class
     * @param lazyInitialize 懒加载[T]的实例的方法
     */
    fun <T> register(clazz: Class<T>, lazyInitialize: () -> T?) {
        Log.i(TAG, "register,clazz:$clazz")
        ImplUtil.addImpl(clazz, lazyInitialize)
        ApiIpcBridge.registerIpc(clazz)

    }


    /**
     * 获取[T]的实例，调用实例后会运行别的插件的代码逻辑
     * @param classLoader
     *  [T]接口的实现如果在插件A中被[register]，则[classLoader]的值为插件A的classloader，可通过[RePlugin.fetchClassLoader(pluginName)]获得
     *  [T]接口的实现如果在宿主中被[register]，则[classLoader]的值为宿主的classloader，可在插件中通过[RePlugin.getHostCLassLoader()]获得
     * @param clazz [T]的class
     * @return [T]的实例
     */
    fun <T> getApi(clazz: Class<T>): T? {
        Log.i(TAG, "getApi,clazz:$clazz")
//        return ApiInvoker(classLoader).getInstance(clazz) as T
        ApiIpcBridge.getApi(clazz)?.let {
            return it
        }
        return ImplUtil.getImpl(clazz)
    }
}

object ImplUtil {

    private val TAG = "ImplUtil"

    private var implFactoryMap = ConcurrentHashMap<Class<*>, ImplFactory<*>>()

    fun <T> addImpl(clazz: Class<T>, lazyInitialize: () -> T?) {
        implFactoryMap[clazz] = ImplFactory(lazyInitialize)
    }

    @Suppress("UNCHECKED_CAST")
    @JvmStatic
    fun <T> getImpl(clazz: Class<T>): T? {
        return implFactoryMap[clazz]?.getImplInstance() as T?
    }
}

class ImplFactory<T>(
    private val lazyInitialize: () -> T?
) {

    private var implInstance: T? = null

    fun getImplInstance(): T? {
        if (implInstance == null) {
            implInstance = lazyInitialize()
        }
        return implInstance
    }
}