package com.yy.mobile.framework.revenuesdk.payapi;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.Keep;
import android.support.annotation.NonNull;

import com.yy.mobile.framework.revenuesdk.baseapi.IResult;
import com.yy.mobile.framework.revenuesdk.baseapi.IToken;
import com.yy.mobile.framework.revenuesdk.payapi.bean.ProductInfo;
import com.yy.mobile.framework.revenuesdk.payapi.bean.PurchaseInfo;
import com.yy.mobile.framework.revenuesdk.payapi.bean.PurchaseStatusInfo;
import com.yy.mobile.framework.revenuesdk.payapi.bean.SkuDetailInfo;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.CouponDiscountResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.ExchangeResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.HasChargeInActivityResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.MyBalanceResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.PayOrderResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.ProductListResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.RechargeHistoryResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.RevenueRecordResult;
import com.yy.mobile.framework.revenuesdk.payapi.callbackresult.UserCouponStoreResult;
import com.yy.mobile.framework.revenuesdk.payapi.reporter.IPayReporter;
import com.yy.mobile.framework.revenuesdk.payapi.request.ChargeCurrencyReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.DoHangPayJobReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.ExchangeCurrencyReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.GetChargeCouponDiscountReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.GetHasChargeInActivityReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.GetUserCouponStoreParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.QueryCurrencyChannelsReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.QueryCurrencyReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.QueryRechargeHistoryReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.QueryUserAccountHistoryReqParams;
import com.yy.mobile.framework.revenuesdk.payapi.request.ReportPurchaseReqParams;

import org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * Created by lhf on 2018/7/5.
 * <p>
 * App 支付功能service
 * YY.Inc
 */
@Keep
@SuppressWarnings("javadoc")
public interface IAppPayService {
    /**
     * 获取商品列表
     *
     * @param params   请求参数
     * @param callback 获取商品列表回调
     */
    void queryProductList(@NonNull QueryCurrencyReqParams params,
                          @NonNull String pageId,
                          IResult<ProductListResult> callback);

    /**
     * 获取商品列表从本地目录
     *
     * @param fileName 文件路径
     * @param callback 获取商品列表回调
     */
    void queryLocalProductList(@NonNull String fileName, Context context,
                               IResult<ProductListResult> callback);

    /**
     * 获取商品列表
     *
     * @param params   请求参数
     * @param callback 获取商品列表回调
     */
    void queryProductListChannels(@NonNull QueryCurrencyChannelsReqParams params,
                                  IResult<ProductListResult> callback);

    /**
     * 查询用户有效优惠券
     *
     * @param params   请求参数
     * @param callback 获取用户有效优惠券回调
     */
    void GetUserCouponStore(@NonNull GetUserCouponStoreParams params,
                            IResult<UserCouponStoreResult> callback);

    /**
     * 查询充值优惠券折扣信息（海外服务）
     *
     * @param params   请求参数
     * @param callback 获取用户有效优惠券回调
     */
    void GetChargeCouponDiscount(@NonNull GetChargeCouponDiscountReqParams params,
                                 IResult<CouponDiscountResult> callback);

    /**
     * 查询活动期间是否充值过
     *
     * @param params   请求参数
     * @param callback 查询活动期间是否充值过回调
     */
    public void GetHasChargeInActivity(@NonNull GetHasChargeInActivityReqParams params,
                                       IResult<HasChargeInActivityResult> callback);

    /**
     * 支付接口
     *
     * @param act      请求上下文
     * @param params   请求参数
     * @param product  商品信息
     * @param payType  支付类型
     * @param callback 支付结果listener
     */
    void payWithProductInfo(@NonNull Activity act,
                            @NonNull ChargeCurrencyReqParams params,
                            @NonNull ProductInfo product,
                            @NonNull PayType payType,
                            @NonNull String pageId,
                            IPayCallback<String> callback);

    /**
     * 支持重试的支付接口
     *
     * @param act        请求上下文
     * @param params     请求参数
     * @param product    商品信息
     * @param payType    支付类型
     * @param retryCount 重试次数
     * @param intervalMs 重试间隔 单位毫秒
     * @param timeOutMs  超时时间 单位毫秒
     * @param callback   支付结果listener
     */
    void payWithProductInfo(@NonNull Activity act,
                            @NonNull ChargeCurrencyReqParams params,
                            @NonNull ProductInfo product,
                            @NonNull PayType payType,
                            int retryCount, int intervalMs, int timeOutMs,
                            String pageId,
                            IPayCallback<String> callback);

    /**
     * 支付接口 payload 可自定义.
     *
     * @param act
     * @param info
     * @param params
     * @param payload
     * @param payType
     * @param callback
     */
    void payWithProductId(@NonNull Activity act,
                          @NonNull ProductInfo info,
                          @NonNull ChargeCurrencyReqParams params,
                          String payload,
                          @NonNull PayType payType,
                          IPayCallback<String> callback);

    /**
     * 支持重试的支付接口 payload 可自定义.
     *
     * @param act
     * @param info
     * @param params
     * @param payload
     * @param payType
     * @param retryCount
     * @param intervalMs
     * @param timeOutMs
     * @param callback
     */
    void payWithProductId(@NonNull Activity act,
                          @NonNull ProductInfo info,
                          @NonNull ChargeCurrencyReqParams params,
                          String payload,
                          @NonNull PayType payType,
                          int retryCount, int intervalMs, int timeOutMs,
                          IPayCallback<String> callback);

    /**
     * 下单接口（PayTM 的H5类型，只需要下单，然后交给h5处理支付，不需要调用第三方SDK支付）
     *
     * @param act      请求上下文
     * @param params   请求参数
     * @param product  商品信息
     * @param payType  支付类型
     * @param callback 支付结果listener
     */
    void orderWithProductInfo(@NonNull Activity act,
                              @NonNull ChargeCurrencyReqParams params,
                              @NonNull ProductInfo product,
                              @NonNull PayType payType,
                              @NotNull String pageId,
                              IPayCallback<PayOrderResult> callback);

    /**
     * 支持重试的下单接口（PayTM 的H5类型，只需要下单，然后交给h5处理支付，不需要调用第三方SDK支付）
     *
     * @param act        请求上下文
     * @param params     请求参数
     * @param product    商品信息
     * @param payType    支付类型
     * @param retryCount 重试次数
     * @param intervalMs 重试间隔 单位毫秒
     * @param timeOutMs  超时时间 单位毫秒
     * @param callback   支付结果listener
     */
    void orderWithProductInfo(@NonNull Activity act,
                              @NonNull ChargeCurrencyReqParams params,
                              @NonNull ProductInfo product,
                              @NonNull PayType payType,
                              int retryCount, int intervalMs, int timeOutMs,
                              String pageId,
                              IPayCallback<PayOrderResult> callback);

    /**
     * 请求支付
     *
     * @param act
     * @param payType
     * @param productId
     * @param payload
     * @param callback
     */
    void requestPay(@NonNull Activity act, @NonNull PayType payType, String productId,
                    String payload,
                    boolean isSetAccountId, IPayCallback<PurchaseInfo> callback);

    /**
     * 请求订阅
     *
     * @param act
     * @param payType
     * @param productId
     * @param payload
     * @param callback
     */
    void requestSubscription(@NonNull Activity act, @NonNull PayType payType, String productId,
                             String payload,
                             boolean isSetAccountId, IPayCallback<PurchaseInfo> callback);

    /**
     * 消耗商品
     *
     * @param payType
     * @param info
     */
    void consumeProduct(@NonNull PayType payType, PurchaseInfo info, IResult<String> result);

    /**
     * 获取指定用户的账号余额
     *
     * @param params
     * @param callback //获取我的余额信息回调
     */
    void queryMyBalance(@NonNull QueryCurrencyReqParams params, IResult<MyBalanceResult> callback);

    /**
     * 查询流水记录
     *
     * @param params   请求参数
     * @param callback 获取流水记录回调
     */
    void queryRevenueRecordByUid(@NonNull QueryUserAccountHistoryReqParams params,
                                 IResult<RevenueRecordResult> callback);

    /**
     * 货币兑换.
     *
     * @param params
     * @param iResult
     */
    void exchangeCurrency(@NonNull ExchangeCurrencyReqParams params,
                          IResult<ExchangeResult> iResult);

    /**
     * 查询用户充值历史（最多能查最近1个月数据）
     *
     * @param params
     * @param iResult
     */
    void queryRechargeHistory(@NonNull QueryRechargeHistoryReqParams params,
                              IResult<RechargeHistoryResult> iResult);

    /**
     * 判断是否支持
     *
     * @param act
     * @param payType
     * @return true 支持，false 不支持.
     */
    boolean isSupported(@NonNull Activity act, @NonNull PayType payType);

    /**
     * 查询是否有未完成的第一个支付订单.
     * 查询结果会通过iResult回调,有未完成的支付订单的话会回调iResult的onSuccess.并且返回未完成订单对应的信息.
     * 如果没有未完成的支付订单会回调iResult的onFail.并且回调对应的code, code == 0
     * 如果调用失败也会回调iResult的onFail.并且回调对应的code
     * 使用{@link #hasHangPayJobs(Activity, PayType, IResult<List<PurchaseInfo>>)}替代
     *
     * @param act
     * @param payType
     * @param iResult
     * @return
     */
    @Deprecated
    boolean hasHangPayJob(@NonNull Activity act, @NonNull PayType payType,
                          IResult<PurchaseInfo> iResult);

    /**
     * 是否有取消订阅但未失效的第一个订单
     * 使用{@link #hasHangSubscribeJobs(Activity, PayType, IResult<List<PurchaseInfo>>)}替代
     *
     * @param act
     * @param iResult
     * @return
     */
    @Deprecated
    boolean hasHangSubscribeJob(@NonNull Activity act, @NonNull PayType payType,
                                IResult<PurchaseInfo> iResult);

    /**
     * 查询所有未完成的支付订单.
     *
     * @param act
     * @param payType
     * @param iResult
     * @return
     */
    boolean hasHangPayJobs(@NonNull Activity act, @NonNull PayType payType,
                           IResult<List<PurchaseInfo>> iResult);

    /**
     * 查询所有取消订阅但未失效的订单
     *
     * @param act
     * @param payType
     * @param iResult
     * @return
     */
    boolean hasHangSubscribeJobs(@NonNull Activity act, @NonNull PayType payType,
                                 IResult<List<PurchaseInfo>> iResult);

    /**
     * 根据本地缓存文件查询未完成订单信息
     *
     * @param act
     * @param iResult
     * @return
     */
    boolean queryHangJobsByLocalCache(@NonNull Activity act, PayType payType,
                                      IResult<List<PurchaseStatusInfo>> iResult);

    /**
     * 根据本地缓存文件处理未完成订单信息
     *
     * @param act
     * @return
     */
    boolean doHangJobByLocalCache(@NonNull Activity act, ReportPurchaseReqParams params,
                                  IResult<String> iResult);

    /**
     * 根据本地缓存文件处理未完成订单信息
     *
     * @param act
     * @return
     */
    boolean doHangJobsByLocalCache(@NonNull Activity act, PayType payType,
                                   IResult<List<PurchaseStatusInfo>> iResult);

    /**
     * 查询所有gp购买订单
     *
     * @param payType
     * @param iResult
     * @return
     */
    boolean queryHistoryPurchaseBySkuType(@NonNull Activity act, @NonNull PayType payType,
                                          String skuType, IResult<List<PurchaseInfo>> iResult);

    /**
     * 继续完成未完成的订单.
     *
     * @param act
     * @param payType
     * @param params
     * @param callback
     */
    boolean doHangJob(@NonNull Activity act,
                      @NonNull ReportPurchaseReqParams params,
                      @NonNull PayType payType,
                      IResult<String> callback);

    /**
     * 继续完成未完成的订单（支持重试）.
     *
     * @param act
     * @param payType
     * @param params
     * @param retryCount
     * @param intervalMs
     * @param timeOutMs
     * @param callback
     */
    boolean doHangJob(@NonNull Activity act,
                      @NonNull ReportPurchaseReqParams params,
                      @NonNull PayType payType,
                      int retryCount, int intervalMs, int timeOutMs,
                      IResult<String> callback);

    /**
     * 这个接口主要用来给测试使用的,方便测试随时可以清理掉手机的卡单订单.
     *
     * @param act
     * @param payType
     * @param type     该参数没用，因为只有消耗型商品才能消耗
     * @param callback
     */
    void clearHangJob(@NonNull Activity act, @NonNull PayType payType, int type,
                      IResult<PurchaseInfo> callback);

    /**
     * 根据订单号去完成卡单订单.
     *
     * @param act
     * @param params
     * @param payType
     * @param retryCount
     * @param intervalMs
     * @param timeOutMs
     * @param callback
     * @return
     */
    boolean doHangJobByProductId(@NotNull Activity act,
                                 @NotNull DoHangPayJobReqParams params,
                                 @NonNull PayType payType,
                                 int retryCount, int intervalMs, int timeOutMs,
                                 IResult<String> callback);

    /**
     * 是否已经跳转到第三方的支付页面.
     *
     * @return
     */
    boolean isPayingStatus(@NonNull PayType payType);

    /**
     * 微信支付回调.
     * 微信支付比较特别，因为WXPayEntryActivity放的路径
     * 需要跟APP的包名一样，所以WXPayEntryActivity不能放在
     * 中间件里面，只能放在了业务外面,所以业务在WXPayEntryActivity的onResp
     * 回调的时候需要回调下结果给中间件.
     *
     * @param code
     * @param msg
     */
    void onWxPayResult(int code, String msg);

    /**
     * 注册广播监听
     *
     * @param listener
     */
    void addPayListener(IAppPayServiceListener listener);

    /**
     * 移除一个监听.
     *
     * @param listener
     */
    void removePayListener(IAppPayServiceListener listener);

    /**
     * 注册上报.
     *
     * @param iPayReporter
     */
    void registerPayReporter(IPayReporter iPayReporter);

    /**
     * 更新国家码（业务端不要直接调用，请调用updateConfig方法）
     *
     * @param countryCode
     */
    void setCountryCode(@NonNull String countryCode);

    /**
     * 查新GP商品信息
     *
     * @param skusList
     * @param {SkuType.INAPP, SkuType.SUBS}
     * @param iResult
     * @return
     */
    void querySkuDetails(@NotNull Activity act, PayType payType, List<String> skusList,
                         String skuType, IResult<List<SkuDetailInfo>> iResult);

    /**
     * 上报支付入口曝光页面
     *
     * @param from
     */
    void reportPayEntrancePage(int from);

    /**
     * 查询卡单并且处理卡单的方法
     * @param act
     * @param payType
     * @param productId
     * @param uid
     * @param token
     * @param useChannel
     * @param currencyType
     * @param subscriptionType
     * @param tokenCallback
     * @param callback
     */
    void hasHangPayJobAndDoHangPayJob(
            Activity act,
            PayType payType,
            String productId,
            Long uid,
            String token,
            int useChannel,
            int currencyType,
            int subscriptionType,
            IToken tokenCallback,
            IResult<String> callback
    );

}
