﻿#ifndef _LZ4_WRAPPER_H
#define _LZ4_WRAPPER_H

#include "lz4.h"

namespace lz4
{

enum {
  MESSAGE_MAX_BYTES = 64 * 1024,
  ENCODE_RING_BUFFER = 256 * 1024 + MESSAGE_MAX_BYTES,
  DECODE_RING_BUFFER = ENCODE_RING_BUFFER
};

class Lz4Encoder
{
public:
  Lz4Encoder()
    : m_lz4EncStream(NULL)
    , m_pInpBuf(NULL)
    , m_inpOffset(0)
    , m_isInited(false)
  {
  }

  ~Lz4Encoder()
  {
    destory();
  }

  // must init before use
  bool init()
  {
    bool ret = false;
    if (m_lz4EncStream != NULL)
    {
      LZ4_resetStream(m_lz4EncStream);
      ret = true;
    }
    else
    {
      m_lz4EncStream = LZ4_createStream();
      if (m_lz4EncStream != NULL)
      {
        ret = true;
      }
      else
      {
        ret = false;
      }
    }

    if ((ret == true) && (m_pInpBuf == NULL))
    {
      m_pInpBuf = new char[ENCODE_RING_BUFFER];
      if (m_pInpBuf == NULL)
      {
        ret = false;
      }
    }
    m_inpOffset = 0;

    if (ret == true)
    {
      m_isInited = true;
    }
    else
    {
      destory();
    }

    return ret;
  }

  void destory()
  {
    LZ4_freeStream(m_lz4EncStream);
    m_lz4EncStream = NULL;
    delete m_pInpBuf;
    m_pInpBuf = NULL;
    m_inpOffset = 0;
    m_isInited = false;
  }

  bool isInited()
  {
    return m_isInited;
  }

  int compress(char* dest, const char* src, int srcLen)
  {
    if (m_isInited == false)
    {
      return -1;
    }

//    char* inpPtr = &m_inpBuf[m_inpOffset];
    char* inpPtr = m_pInpBuf + m_inpOffset;
    memcpy(inpPtr, src, srcLen);
    int cmpLen = LZ4_compress_continue(m_lz4EncStream, inpPtr, dest, srcLen);
    if (cmpLen > 0)
    {
      m_inpOffset += srcLen;
      // Wraparound the ringbuffer offset
      if (m_inpOffset >= ENCODE_RING_BUFFER - MESSAGE_MAX_BYTES)
      {
        m_inpOffset = 0;
      }
    }
    return cmpLen;
  }

private:
  LZ4_stream_t* m_lz4EncStream;
//  char m_inpBuf[ENCODE_RING_BUFFER];
  char* m_pInpBuf;
  int m_inpOffset;
  bool m_isInited;
};

class Lz4Decoder
{
public:
  Lz4Decoder()
    : m_lz4DecStream(NULL)
    , m_pDecBuf(NULL)
    , m_decOffset(0)
    , m_isInited(false)
  {
  }

  ~Lz4Decoder()
  {
    destroy();
  }

  // must init before use
  bool init()
  {
    bool ret = false;
    if (m_lz4DecStream != NULL)
    {
      LZ4_freeStreamDecode(m_lz4DecStream);
    }

    m_lz4DecStream = LZ4_createStreamDecode();
    if (m_lz4DecStream != NULL)
    {
      ret = true;
    }
    else
    {
      ret = false;
    }

    if ((ret == true) && (m_pDecBuf == NULL))
    {
      m_pDecBuf = new char[DECODE_RING_BUFFER];
      if (m_pDecBuf == NULL)
      {
        ret = false;
      }
    }
    m_decOffset = 0;

    if (ret == true)
    {
      m_isInited = true;
    }
    else
    {
      destroy();
    }

    return ret;
  }

  void destroy()
  {
    LZ4_freeStreamDecode(m_lz4DecStream);
    m_lz4DecStream = NULL;
    delete m_pDecBuf;
    m_pDecBuf = NULL;
    m_isInited = false;
  }

  bool isInited()
  {
    return m_isInited;
  }

  int decompress(char*& dest, const char* src, int srcLen)
  {
    if (m_isInited == false)
    {
      return -1;
    }

//    dest = &m_decBuf[m_decOffset];
    dest = m_pDecBuf + m_decOffset;

    int decLen = LZ4_decompress_safe_continue(m_lz4DecStream, src, dest, srcLen, MESSAGE_MAX_BYTES);
    if (decLen > 0)
    {
      m_decOffset += decLen;
      if (m_decOffset >= DECODE_RING_BUFFER - MESSAGE_MAX_BYTES)
      {
        m_decOffset = 0;
      }
    }
    else
    {
      dest = NULL;
    }
    return decLen;
  }

private:
  LZ4_streamDecode_t* m_lz4DecStream;
//  char m_decBuf[DECODE_RING_BUFFER];
  char* m_pDecBuf;
  int m_decOffset;
  bool m_isInited;
};

}

#endif