package com.yy.platform.baseservice.task;

import android.os.Bundle;

import com.yy.platform.baseservice.marshal.ProtoPacket;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

public class TaskOptions {

    /**
     * 指定该请求永远不需要bind(不需登录Service进行绑定)既可以发送请求到服务端
     * 将该值作为Bundle option的一个key，value 给true 或false
     */
    public final static String OPT_NEVERBIND = "neverbind";

    /**
     * 指定该请求是否需要绑定UID(登录Service进行绑定)
     * 但如果调用了登录bind 接口，后续请求如无指定OPT_NEVERBIND选项，则此OPTION 会被置为true
     * 将该值作为Bundle option的一个key，value 给true 或false
     */
    public final static String OPT_NEEDBIND = "needbind";

    /**
     * 定制该请求的超时时间，单位ms，最小值1000即1秒
     * 约定：[1-5)秒内，由SDK决定超时返回，在[5, 60]秒范围时由服务器决定返回超时
     * 将该值作为Bundle option的一个key，value 给指定的超时时间
     */
    public final static String OPT_TIMOUTTS = "timeout";
    /**
     * 定制该请求的重试策略，即每隔多久重试一次，
     * 此时忽略定制的超时时间，在所有策略执行完成后仍为收到服务器响应则返回超时
     * 将该值作为Bundle option的一个key，value 给指定一个重试间隔的ArrayList<Integer>，单位均为ms
     */
    public final static String OPT_RETRYSTRATEGY = "retrystrategy";

    public final static String RET_COSTTS = "costts";

    protected static class TaskOption extends ProtoPacket {
        protected Map<String, Byte> mExtraBool;
        protected Map<String, Long> mExtraLong;
        protected Map<String, String> mExtraString;
        protected Map<String, OptionObject> mExtraObject;

        public TaskOption(Bundle options) {
            mExtraBool = new HashMap<>();
            mExtraLong = new HashMap<>();
            mExtraString = new HashMap<>();
            mExtraObject = new HashMap<>();

            if (options != null) {
                if (options.containsKey(TaskOptions.OPT_NEVERBIND)) {
                    boolean neverNeedBind = options.getBoolean(TaskOptions.OPT_NEVERBIND, false);
                    mExtraBool.put(TaskOptions.OPT_NEVERBIND, (byte) (neverNeedBind ? 1 : 0));
                }
                if (options.containsKey(TaskOptions.OPT_TIMOUTTS)) {
                    //超时时间，如果未给则默认10s，在超时时间内发送一次，无结果则返回超时
                    long timeout = options.getLong(TaskOptions.OPT_TIMOUTTS, 0);
                    if (timeout >= 1000) {
                        mExtraLong.put(TaskOptions.OPT_TIMOUTTS, timeout);
                    } else {
                        mExtraLong.put(TaskOptions.OPT_TIMOUTTS, 1000L);
                    }
                }

                mExtraObject.put(TaskOptions.OPT_RETRYSTRATEGY, new OptionObject(options));
            }
        }

        @Override
        public void marshall(ByteBuffer byteBuffer) {
            super.marshall(byteBuffer);
            pushMap(mExtraBool, Byte.class);
            pushMap(mExtraLong, Long.class);
            pushMap(mExtraString, String.class);
            pushMap(mExtraObject, OptionObject.class);
        }
    }

    protected static class OptionObject extends ProtoPacket {
        protected ArrayList<Integer> mExtraIntegerList;

        OptionObject(Bundle options) {
            mExtraIntegerList = options.getIntegerArrayList(TaskOptions.OPT_RETRYSTRATEGY);
            if (mExtraIntegerList == null || (mExtraIntegerList.size() == 0)) {
                mExtraIntegerList = new ArrayList<Integer>();
            } else {
                ArrayList<Integer> mIntervals = new ArrayList<>();
                for (Integer i : mExtraIntegerList) {
                    if (i < 1000) {
                        i = 1000;
                    }
                    mIntervals.add(i);
                }
                mExtraIntegerList = mIntervals;
            }
        }

        @Override
        public void marshall(ByteBuffer byteBuffer) {
            super.marshall(byteBuffer);
            this.pushCollection(mExtraIntegerList, Integer.class);
        }
    }
}
