package com.yy.yylivesdk4cloud.helper;

/**
 * Created by xiaojun on 2017/11/22.
 * Copyright (c) 2017 YY Inc. All rights reserved.
 */

import android.util.Log;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

public class Marshallable implements IPtotoPacket {

	public static final int kProtoPacketSize = 4 * 1024;
	protected ByteBuffer mBuffer = null;

	public Marshallable() {
		mBuffer = ByteBuffer.allocate(kProtoPacketSize);
		mBuffer.order(ByteOrder.LITTLE_ENDIAN);
	}

	public Marshallable(boolean bAllocBuff) {
		if (bAllocBuff) {
			mBuffer = ByteBuffer.allocate(kProtoPacketSize);
			mBuffer.order(ByteOrder.LITTLE_ENDIAN);
		}
	}

	public Marshallable(int buffSize) {
		mBuffer = ByteBuffer.allocate(buffSize);
		mBuffer.order(ByteOrder.LITTLE_ENDIAN);
	}

	@Override
	public byte[] marshall() {
		int len = mBuffer.position();

		byte[] data = new byte[len];
		mBuffer.position(0);
		mBuffer.get(data);
		return data;
	}

	@Override
	public void marshall(ByteBuffer buf) {
		mBuffer = buf;
	}

	@Override
	public void unmarshall(byte[] buf) {
		mBuffer = ByteBuffer.wrap(buf);
		mBuffer.order(ByteOrder.LITTLE_ENDIAN);
	}

	@Override
	public void unmarshall(ByteBuffer buf) {
		this.mBuffer = buf;
	}

	public ByteBuffer getBuffer() {
		return mBuffer;
	}

	public void pushBool(Boolean val) {
		byte b = 0;
		if (val)
			b = 1;

		check_capactiy(1);
		mBuffer.put(b);
	}

	public Boolean popBool() {
		byte b = mBuffer.get();
		return (b == 1);
	}

	public void pushByte(byte val) {
		check_capactiy(1);
		mBuffer.put(val);
	}

	public byte popByte() {
		return mBuffer.get();
	}

	public void pushBytes(byte[] buf) {
		if (buf == null) {
			check_capactiy(2);
			mBuffer.putShort((short) 0);
		} else if (buf.length > 0xffff) {
			YYLiveLog.info(YYLiveLog.kLogTagSdk, "pushBytes, buf overflow, size=" + buf.length);
			check_capactiy(2);
			mBuffer.putShort((short) 0);
		} else {

			check_capactiy(2 + buf.length);
			mBuffer.putShort((short) buf.length);
			mBuffer.put(buf);
		}
	}

	public byte[] popBytes() {
		int len = mBuffer.getShort();
		byte[] buf = null;
		if (len >= 0) {
			buf = new byte[len];
			mBuffer.get(buf);
		}
		return buf;
	}

	public void pushBytes32(byte[] buf) {
		if (buf == null) {
			check_capactiy(4);
			mBuffer.putInt(0);
		} else {
			check_capactiy(4 + buf.length);
			mBuffer.putInt(buf.length);
			mBuffer.put(buf);
		}
	}

	public byte[] popBytes32() {
		int len = mBuffer.getInt();
		byte[] buf = null;
		if (len >= 0) {
			buf = new byte[len];
			mBuffer.get(buf);
		}

		return buf;
	}

	public byte[] popAll() {
		int len = mBuffer.remaining();
		byte[] buf = new byte[len];
		mBuffer.get(buf);
		return buf;
	}

	public void pushShort(short val) {
		check_capactiy(2);
		mBuffer.putShort(val);
	}

	public short popShort() {
		return mBuffer.getShort();
	}

	public void pushInt(int val) {
		check_capactiy(4);
		mBuffer.putInt(val);
	}

	public void pushInt(long val) {
		check_capactiy(4);
		mBuffer.putInt((int) val);
	}

	public void pushFloat(float val) {
		check_capactiy(4);
		mBuffer.putFloat(val);
	}

	public float popFloat() {
		return mBuffer.getFloat();
	}

	public int popInt() {
		return mBuffer.getInt();
	}

	public long popInt2Long() {
		return mBuffer.getInt() & 0XFFFFFFFFL;
	}

	public void pushInt64(long val) {
		check_capactiy(8);
		mBuffer.putLong(val);
	}

	public long popInt64() {
		return mBuffer.getLong();
	}

	public void pushString16(String val) {
		if (val == null) {
			check_capactiy(2);
			mBuffer.putShort((short) 0);
			return;
		}

		check_capactiy(2 + val.getBytes().length);
		mBuffer.putShort((short) val.getBytes().length);
		if (val.getBytes().length > 0) {
			mBuffer.put(val.getBytes());
		}
	}

	public String popString16() {
		short len = mBuffer.getShort();
		byte[] buf = null;
		if (len >= 0) {
			buf = new byte[len];
			mBuffer.get(buf);
			try {
				return new String(buf, "ISO-8859-1"); // ansi
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}

		return "";
	}

	public String popString16UTF8() {
		short len = mBuffer.getShort();
		byte[] buf = null;
		if (len >= 0) {
			buf = new byte[len];
			mBuffer.get(buf);
			try {
				return new String(buf, "utf-8");// utf-8
			} catch (UnsupportedEncodingException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		return "";
	}

	public void pushIntArray(int[] vals) {
		if (vals == null) {
			pushInt(0);
			return;
		}

		int len = vals.length;
		pushInt(len);
		for (int i = 0; i < len; i++) {
			pushInt(vals[i]);
		}
	}

	public void pushIntArray(Integer[] vals) {
		if (vals == null) {
			pushInt(0);
			return;
		}

		int len = vals.length;
		pushInt(len);
		for (int i = 0; i < len; i++) {
			pushInt(vals[i]);
		}
	}

	public void pushIntArray(long[] vals) {
		if (vals == null) {
			pushInt(0);
			return;
		}

		int len = vals.length;
		pushInt(len);
		for (int i = 0; i < len; i++) {
			pushInt(vals[i]);
		}
	}

	public int[] popIntArray() {
		int len = popInt();
		int[] vals = new int[len];
		for (int i = 0; i < len; i++) {
			vals[i] = popInt();
		}

		return vals;
	}

	public long[] popIntArray2Long() {
		int len = popInt();
		long[] vals = new long[len];
		for (int i = 0; i < len; i++) {
			vals[i] = popInt2Long();
		}

		return vals;
	}

	public void pushShortArray(short[] vals) {
		if (vals == null) {
			pushInt(0);
			return;
		}

		int len = vals.length;
		pushInt(len);
		for (int i = 0; i < len; i++) {
			pushShort(vals[i]);
		}
	}

	public short[] popShortArray() {
		int len = popInt();
		short[] vals = new short[len];
		for (int i = 0; i < len; i++) {
			vals[i] = popShort();
		}

		return vals;
	}

	public String popString32() {
		int len = mBuffer.getInt();
		byte[] buf = null;
		if (len >= 0) {
			buf = new byte[len];
			mBuffer.get(buf);
			try {
				return new String(buf, "ISO-8859-1"); // ansi
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}

		return "";
	}

	public String popString16(String charsetName) {
		short len = mBuffer.getShort();
		byte[] buf = null;
		if (len >= 0) {
			buf = new byte[len];
			mBuffer.get(buf);
			try {
				return new String(buf, charsetName);
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}

		return "";
	}

	public void pushString32(String val) {
		if (val == null) {
			check_capactiy(4);
			mBuffer.putInt(0);
			return;
		}

		check_capactiy(4 + val.getBytes().length);
		mBuffer.putInt(val.getBytes().length);
		if (val.getBytes().length > 0) {
			mBuffer.put(val.getBytes());
		}
	}

	public void pushString32(String val, String charsetName) {
		if (val == null) {
			check_capactiy(4);
			mBuffer.putInt(0);
			return;
		}
		try {
			check_capactiy(4 + val.getBytes().length);
			mBuffer.putInt(val.getBytes(charsetName).length);
			if (val.getBytes().length > 0) {
				mBuffer.put(val.getBytes(charsetName));
			}
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
	}

	public String popString32(String charsetName) {
		int len = mBuffer.getInt();
		byte[] buf = null;
		if (len >= 0) {
			buf = new byte[len];
			mBuffer.get(buf);
			try {
				return new String(buf, charsetName);
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}

		return "";
	}

	protected void check_capactiy(int size) {
		if (mBuffer.capacity() - mBuffer.position() < size) {
			increase_capacity(size - (mBuffer.capacity() - mBuffer.position()));
		}
	}

	protected void increase_capacity(int minIncrement) {
		int capacity = mBuffer.capacity();

		if (capacity == 0) {
			return;
		}

		int size = 2 * capacity;
		if (minIncrement > capacity) {
			size = capacity + minIncrement;
		}

		ByteBuffer tempBuf = ByteBuffer.allocate(size);
		tempBuf.order(ByteOrder.LITTLE_ENDIAN);
		mBuffer.limit(mBuffer.position());
		mBuffer.position(0);
		tempBuf.put(mBuffer);
		mBuffer = tempBuf;
		YYLiveLog.info(YYLiveLog.kLogTagSdk, "increase_capacity, size=" + size);
	}

	public void pushMarshallable(Marshallable val) {
		if (val != null) {
			val.marshall(mBuffer);
		}
	}

	public Marshallable popMarshallable(Class<? extends Marshallable> MarshalClass) {
		Marshallable val = null;
		try {
			val = MarshalClass.newInstance();
		} catch (InstantiationException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		}
		val.unmarshall(mBuffer);
		return val;
	}

	/**
	 * @param key     待压入的数据，不支持压入Marshallable及其子类
	 * @param lenType 如果待压入的数据是String和byte[]等需要先压入数据长度，再压入数据的，通过这个选项选择长度存放的类型${@link com.yy.yylivesdk4cloud.helper.Marshallable.ELenType}
	 *                如果压入的数据是Integer/Long等直接压入的数据，lenType填入ELenType.E_NONE
	 * @Title: pushKey
	 * @Description: 向Marshallable缓冲区压入一个数据，pushKey与pushElem的区别在于，pushKey不支持压入Marshallable及其子类
	 * @return: void
	 */
	private <K> void pushKey(K key, ELenType lenType) {
		if (key instanceof java.lang.Byte) {
			pushByte((Byte) key);
		} else if (key instanceof java.lang.Short) {
			pushShort((Short) key);
		} else if (key instanceof java.lang.Integer) {
			pushInt((Integer) key);
		} else if (key instanceof java.lang.Long) {
			pushInt64((Long) key);
		} else if (key instanceof String) {
			pushString16((String) key);
		} else if (key instanceof byte[]) {
			pushBytes((byte[]) key);
		} else {
			throw new IllegalStateException("marshall Map but unknown key type: " + key.getClass().getName());
		}
	}

	/**
	 * @param keyClass    弹出值的类型，popKey与popElem的区别在于，popKey不支持弹出Marshallable及其子类
	 * @param lenType     如果待弹出的数据是String和byte[]等需要先弹出数据长度，再弹出数据的，通过这个选项选择长度存放的类型${@link com.yy.yylivesdk4cloud.helper.Marshallable.ELenType}
	 *                    如果弹出的数据是Integer/Long等直接弹出的数据，lenType填入ELenType.E_NONE
	 * @param charsetName 如果待弹出的数据是一个String，在此处设置该String的编码格式；如果不是String，则填""即可。
	 * @Title: popKey
	 * @Description: 弹出一个值
	 * @return: T
	 */
	@SuppressWarnings("unchecked")
	private <K> K popKey(Class<K> keyClass, ELenType lenType, String charsetName) {
		K key = null;
		if (keyClass == java.lang.Byte.class) {
			key = (K) Short.valueOf(popByte());
		} else if (keyClass == java.lang.Short.class) {
			key = (K) Short.valueOf(popShort());
		} else if (keyClass == java.lang.Integer.class) {
			key = (K) Integer.valueOf(popInt());
		} else if (keyClass == java.lang.Long.class) {
			key = (K) Long.valueOf(popInt64());
		} else if (keyClass == byte[].class) {
			if (lenType == ELenType.E_SHORT) {
				key = (K) popBytes();
			} else if (lenType == ELenType.E_INT) {
				key = (K) popBytes32();
			} else {
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, YYLiveLog.kLogTagSdk, "invalid lenType=%d for popBytes", lenType);
			}
		} else if (keyClass == String.class) {
			if (lenType == ELenType.E_SHORT) {
				key = (K) popString16(charsetName);
			} else if (lenType == ELenType.E_INT) {
				key = (K) popString32(charsetName);
			} else {
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, "invalid lenType=%d for popString", lenType);
			}
		} else {
			throw new IllegalStateException("unMarshall Map but unknown key type: " + keyClass.getName());
		}
		return key;
	}

	/**
	 * @param elem      待压入的数据
	 * @param elemClass 待压入的数据的类型，允许压入Marshallable的子类
	 * @param lenType   如果待压入的数据是String和byte[]等需要先压入数据长度，再压入数据的，通过这个选项选择长度存放的类型${@link com.yy.yylivesdk4cloud.helper.Marshallable.ELenType}
	 *                  如果压入的数据是Integer/Long等直接压入的数据，lenType填入ELenType.E_NONE
	 * @Title: pushElem
	 * @Description: 向Marshallable缓冲区压入一个数据
	 * @return: void
	 */
	private <T> void pushElem(T elem, Class<T> elemClass, ELenType lenType) {
		if (elemClass == Integer.class) {
			pushInt((Integer) elem);
		} else if (elemClass == Short.class) {
			pushShort((Short) elem);
		} else if (elemClass == Long.class) {
			pushInt64((Long) elem);
		} else if (elemClass == Byte.class) {
			pushByte((Byte) elem);
		} else if (elemClass == String.class) {
			if (lenType == ELenType.E_SHORT) {
				pushString16((String) elem);
			} else if (lenType == ELenType.E_INT) {
				pushString32((String) elem);
			} else {
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, "invalid lenType=%d for pushString", lenType);
			}
		} else if (elemClass == byte[].class) {
			if (lenType == ELenType.E_SHORT) {
				pushBytes((byte[]) elem);
			} else if (lenType == ELenType.E_INT) {
				pushBytes32((byte[]) elem);
			} else {
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, "invalid lenType=%d for pushBytes", lenType);
			}
		} else if (elem instanceof Marshallable) {
			((Marshallable) elem).marshall(mBuffer);
			//	byte[] tmp = ((Marshallable)elem).marshall();
			//	mBuffer.put(tmp);	//TODO check if right
		} else {
			throw new RuntimeException("unable to marshal element of class " + elemClass.getName());
		}
	}

	/**
	 * @param elemClass   弹出值的类型
	 * @param lenType     如果待弹出的数据是String和byte[]等需要先弹出数据长度，再弹出数据的，通过这个选项选择长度存放的类型${@link com.yy.yylivesdk4cloud.helper.Marshallable.ELenType}
	 *                    如果弹出的数据是Integer/Long等直接弹出的数据，lenType填入ELenType.E_NONE
	 * @param charsetName 如果待弹出的数据是一个String，在此处设置该String的编码格式；如果不是String，则填""即可。
	 * @Title: popElem
	 * @Description: 弹出一个值
	 * @return: T
	 */
	@SuppressWarnings("unchecked")
	private <T> T popElem(Class<T> elemClass, ELenType lenType, String charsetName) {
		T elem = null;
		if (elemClass == java.lang.Integer.class) {
			elem = (T) Integer.valueOf(popInt());
		} else if (elemClass == java.lang.Short.class) {
			elem = (T) Short.valueOf(popShort());
		} else if (elemClass == java.lang.Long.class) {
			elem = (T) Long.valueOf(popInt64());
		} else if (elemClass == java.lang.Byte.class) {
			elem = (T) Byte.valueOf(popByte());
		} else if (elemClass == java.lang.String.class) {
			if (lenType == ELenType.E_SHORT) {
				elem = (T) popString16(charsetName);
			} else if (lenType == ELenType.E_INT) {
				elem = (T) popString32(charsetName);
			} else {
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, "invalid lenType=%d for popString", lenType);
			}
		} else if (elemClass == byte[].class) {
			if (lenType == ELenType.E_SHORT) {
				elem = (T) popBytes();
			} else if (lenType == ELenType.E_INT) {
				elem = (T) popBytes32();
			} else {
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, "invalid lenType=%d for popBytes", lenType);
			}
		} else {
			try {
				elem = elemClass.newInstance();
			} catch (InstantiationException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			}
			if (elem instanceof Marshallable) {
				((Marshallable) elem).unmarshall(mBuffer);
			} else {
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, "unmarshall invalid elemClass type=%s ", elemClass.getName());
			}
		}
		return elem;
	}

	/**
	 * @param data      待压入的数据
	 * @param elemClass Collection中存放的数据类型
	 * @param lenType   长度存放的类型${@link com.yy.yylivesdk4cloud.helper.Marshallable.ELenType}
	 * @Title: pushCollection
	 * @Description: 压入Collection的子类
	 * @return: void
	 */
	public <T> void pushCollection(Collection<T> data, Class<T> elemClass, ELenType lenType) {
		if (data == null || data.size() == 0) {
			pushInt(0);
		} else {
			pushInt(data.size());
			for (T elem : data) {
				pushElem(elem, elemClass, lenType);
			}
		}
	}

	/**
	 * @param data      待压入的数据
	 * @param elemClass Collection中存放的数据类型；如果类型是String或byte[]，则默认以short压入String或byte[]的长度。
	 * @Title: pushCollection
	 * @Description: 压入Collection的子类
	 * @return: void
	 */
	public <T> void pushCollection(Collection<T> data, Class<T> elemClass) {
		pushCollection(data, elemClass, ELenType.E_NONE);
	}

	/**
	 * @param container 容器类型，如Vector.class, ArrayList.class, ......
	 * @param elemClass xxx.class：如果这个类是Marshallable的子类，那么它必须重写 ${@link com.yy.yylivesdk4cloud.helper.Marshallable#unmarshall(ByteBuffer)}方法；
	 *                  如果这个类型是String/byte[]，则默认其长度保存在short中。
	 * @Title: popCollection
	 * @Description: 弹出Collection的子类
	 * @return: Collection<T>
	 * <p>
	 * for instance:
	 * <p>
	 * Vector<Integer> vec = (Vector<Integer>)popCollection(Vector.class, Integer.class);
	 */
	public <T> Collection<T> popCollection(Class<? extends Collection> container, Class<T> elemClass) {
		return popCollection(container, elemClass, ELenType.E_SHORT, "utf-8");
	}

	/**
	 * @param container   容器类型，如Vector.class, ArrayList.class, ......
	 * @param elemClass   xxx.class 如果这个类是Marshallable的子类，那么它必须重写 ${@link com.yy.yylivesdk4cloud.helper.Marshallable#unmarshall(ByteBuffer)}方法。
	 * @param lenType     长度存放的类型${@link com.yy.yylivesdk4cloud.helper.Marshallable.ELenType}
	 * @param charsetName
	 * @return
	 * @Title: popCollection
	 * @Description: TODO
	 * @return: Collection<T>
	 * @throws:
	 */
	@SuppressWarnings("unchecked")
	public <T> Collection<T> popCollection(Class<? extends Collection> container, Class<T> elemClass, ELenType lenType, String charsetName) {
		int size = popInt();
//		if (size > InvalidProtocolData.MAX_PROTO_ELEMENT_COUNT) {
//			throw new InvalidProtocolData(InvalidProtocolData.ERROR_LEN);
//		}
		Collection<T> c = null;
		try {
			c = container.newInstance();
		} catch (InstantiationException e1) {
			e1.printStackTrace();
		} catch (IllegalAccessException e1) {
			e1.printStackTrace();
		}
		if (c == null) {
			return null;
		}
		for (int i = 0; i < size; i++) {
			T elem = null;
			elem = popElem(elemClass, lenType, charsetName);
			c.add(elem);
		}
		return c;
	}

	public <K, T> void pushMap(Map<K, T> data, Class<T> elemClass) {
		pushMap(data, elemClass, ELenType.E_SHORT, ELenType.E_SHORT);
	}

	public <K, T> void pushMap(Map<K, T> data, Class<T> elemClass, ELenType keyLenType, ELenType valueLenType) {
		if (data == null || data.size() == 0) {
			pushInt(0);
		} else {
			pushInt(data.size());
			for (Entry<K, T> entry : data.entrySet()) {
				K key = entry.getKey();
				pushKey(key, keyLenType);
				T elem = entry.getValue();
				pushElem(elem, elemClass, valueLenType);
			}
		}
	}

	/**
	 * @param keyClass:  Map的索引类型
	 * @param elemClass: Map的值的类型，如果这个类是Marshallable的子类，那么它必须重写public void unmarshall(ByteBuffer buf)方法。
	 * @Title: popMap
	 * @Description: 从mBuffer中弹出Map（方法实现为TreeMap）
	 * @return: Map<K,T>
	 */
	@SuppressWarnings("unchecked")
	public <K, T> Map<K, T> popMap(Class<K> keyClass, Class<T> elemClass) {
		return popMap(keyClass, ELenType.E_SHORT, "utf-8", elemClass, ELenType.E_SHORT, "utf-8");
	}

	@SuppressWarnings("unchecked")
	public <K, T> Map<K, T> popMap(Class<K> keyClass, ELenType keyLenType, String keyCharsetName,
								   Class<T> elemClass, ELenType elemLenType, String elemCharsetName) {
		int size = popInt();
//		if (size > InvalidProtocolData.MAX_PROTO_ELEMENT_COUNT) {
//			throw new InvalidProtocolData(InvalidProtocolData.ERROR_LEN);
//		}
		Map<K, T> out = new TreeMap<K, T>();
		for (int i = 0; i < size; i++) {
			K key = null;
			key = popKey(keyClass, keyLenType, keyCharsetName);
			T elem = null;
			elem = popElem(elemClass, elemLenType, elemCharsetName);
			out.put(key, elem);
		}
		return out;
	}

	/**
	 * @ClassName: ELenType
	 * @Description: 长度类型，用于表示String/byte[]等（需要先压入/弹出长度short/int，再压入/弹出实际数据）的长度类型。
	 * E_SHORT代表short，E_INT代表int，E_NONE表示不需要使用到长度。
	 * <p>
	 * for example
	 * if (lenType == ELenType.E_SHORT) {
	 * pushString16((String)elem);
	 * } else if (lenType == ELenType.E_INT) {
	 * pushString32((String)elem);
	 * } else {
	 * YLog.warn(YYLiveLog.kLogTagSdk, "invalid lenType=%d for pushString", lenType);
	 * }
	 */
	public enum ELenType {
		E_SHORT,
		E_INT,
		E_NONE
	}
}
