#ifndef _AUDIO_TOOLBOX_IMPL_H_
#define _AUDIO_TOOLBOX_IMPL_H_

#include <string>
#include <vector>
#include "audioengine/int_type.h"
#include "audiodsp/audio_dsp_impl.h"

typedef std::vector<std::string*> AudioBlockList;

//only for the same samplerate and channels
class IAudioBlockMixer
{
public:
	virtual ~IAudioBlockMixer(){}
	virtual void Destroy() = 0;
	virtual bool Process(const AudioBlockList& inBlocks, std::string& outBlock) = 0;
};

class IAudioResamplerEx
{
public:
	virtual ~IAudioResamplerEx() {}
	// src_frames, dst_frames: per channel
	virtual bool Convert(const int16_t* pSrc, uint32_t srcSamples, int16_t* dst, uint32_t dstSamples) = 0;
	virtual bool IsFormatChange(uint32_t src_fs, uint32_t src_ch, uint32_t dst_fs, uint32_t dst_ch) = 0;
	virtual uint32_t GetDestSamples() = 0;
	virtual uint32_t GetSrcSamples() = 0;

	// don't support 22050, 10ms
	static IAudioResamplerEx* Create(uint32_t src_frames, uint32_t src_fs, uint32_t src_channels, uint32_t dst_frames, uint32_t dst_fs, uint32_t dst_channels, const char* create_place = NULL);
    static void Destroy(IAudioResamplerEx*& resampler);  // will Set NULL
};

enum DenoiseLevel
{
	DenoiseLevelNo    = 0,
	DenoiseLevelLight = 1,
	DenoiseLevelHeavy = 2, 
};

class IAudioDenoise
{
public:
	virtual ~IAudioDenoise(){}
	virtual void Destroy() = 0;
	virtual bool Process(int16_t* pData) = 0;//10ms
	virtual void EnableAGC(bool enable) = 0;
	virtual void EnableLightDenoise(bool enable) = 0;
	virtual void SetDenoiseLevel(DenoiseLevel level) = 0;
};

class IAudioLevel
{
public:
	virtual ~IAudioLevel() {}
	virtual void Reset() = 0;
	virtual bool ComputeLevel(const void* audioSamples, const uint32_t dataSize, const uint8_t nBytesPerSample) = 0;
	virtual int  GetLevel() = 0;
};

class IBeatTrackNotify
{
public:
	virtual ~IBeatTrackNotify(){}
	virtual void NotifyBeat(int bpm, float power) = 0;
	virtual void NotifyStartFive() = 0;
};

class IBeatTrack
{
public:
	virtual ~IBeatTrack(){}
	virtual void Destroy() = 0;
	virtual int Start(int mode, IBeatTrackNotify* pNotify, int delaydecision) = 0;
	virtual int Detectbeat(int16_t* sample, int size) = 0;
	virtual void Stop() = 0;
};

/*
	Speech Message Player 
*/
enum SPEECHMSG_PLAYER_STATUS {
	SPEECHMSG_PLAYER_ERROR_NONE = 0,
	SPEECHMSG_PLAYER_ERROR_FILE_OPEN = -1,
	SPEECHMSG_PLAYER_ERROR_HEADER_LENGTH = -2,
	SPEECHMSG_PLAYER_ERROR_UNKNOWN_CODEC = -3,
	SPEECHMSG_PLAYER_ERROR_DECODER_INIT = -4,
};

class ISpeechMsgPlayerNotify
{
public:
	virtual ~ISpeechMsgPlayerNotify(){}
	virtual void OnAudioPlayStatus(uint32_t nowPlayTime, uint32_t filePlayTime, uint32_t volume) = 0;
	virtual void OnAudioPlayError() = 0;
	virtual void OnFirstStartPlayData() = 0;
	virtual void OnStopPlayData(uint32_t nowPlayTime, uint32_t filePlayTime) = 0;
	virtual void OnReachMaxPlayTime(uint32_t nowPlayTime, uint32_t filePlayTime) = 0;
};

class ISpeechMsgPlayer
{
public:
	static ISpeechMsgPlayer* CreateAudioPlayer(const char* pFileName);

	virtual ~ISpeechMsgPlayer(){}
	virtual SPEECHMSG_PLAYER_STATUS Init() = 0;
	virtual void Start(ISpeechMsgPlayerNotify *pNotify) = 0;
	virtual void Stop() = 0;
	virtual void Destroy() = 0;
	virtual int  GetRecordFileTime() = 0;
	virtual void SetAudioSemitone(float val) = 0;
	virtual void SetReverbParam(ReverbParam reverbParam) = 0;
	virtual void SaveToFile(const char* pFileName) = 0;
};

/*
	Speech Message Recorder
*/
enum SPEECHMSG_RECORD_STATUS {
	SPEECHMSG_RECORD_ERROR_NONE          = 0,
	SPEECHMSG_RECORD_ERROR_UNKNOWN_CODEC = -1,
	SPEECHMSG_RECORD_ERROR_ENCODER_INIT  = -2,
	SPEECHMSG_RECORD_ERROR_FILE_OPEN_FAIL= -3,
};

enum SpeechMsgCodecType {
	SpeechMsgCodecUnknown  = -1,
	SpeechMsgCodecSilk     = 0,
	SpeechMsgCodecSpeex    = 1,
	SpeechMsgCodecEaacPlus = 2,
	SpeechMsgCodecLcAAC    = 3,
	SpeechMsgCodecWav      = 4,
};

class ISpeechMsgRecorderNotify
{
public:
	virtual ~ISpeechMsgRecorderNotify(){}
	virtual void OnAudioVolumeVisual(uint32_t recordTime, uint32_t volume) = 0;//0~100
	virtual void OnAudioRecordError() = 0;
	virtual void OnGetFirstRecordData() = 0;
	virtual void OnStopRecordData(uint32_t recordTime, uint32_t maxDuration) = 0;
	virtual void OnReachMaxDuration(uint32_t recordTime, uint32_t maxDuration) = 0;
};

class ISpeechMsgRecorder
{
public:
	static ISpeechMsgRecorder* CreateAudioRecorder(const char* pFileName, int uid, SpeechMsgCodecType codecType, const uint32_t maxDuration/*ms*/);

	virtual ~ISpeechMsgRecorder(){}
	virtual SPEECHMSG_RECORD_STATUS Init() = 0;
	virtual void SetVolume(uint32_t volume) = 0;
	virtual void Start(ISpeechMsgRecorderNotify* pNotify) = 0;
	virtual void Stop() = 0;
	virtual void Pause() = 0;
	virtual void Resume() = 0;
	virtual bool Seek(uint32_t seekTime) = 0;
	virtual bool EnablePlayBack(bool enable) = 0;
	virtual void Destroy() = 0;
};

/*  multi-channel pcm stream mixer, each stream length fixed to 20ms, reduce delay
 *  before call Process, each stream is put together, and add stream header
 *  each stream will be converted to dstFs,dstChannel, and be mixed to one stream
 *
 *	One Path Header Structure:
 *	|<-              Fs                   ->|<- channel->|<- precision ->|<-    Data length    ->|
 *	|<- 4th ->|<- 3th ->|<- 2th ->|<- 1th ->|<-  1byte ->|<-   1byte   ->|<- 1byte ->|<- 1byte ->|
 */
class IPushPcmModule
{
public:
	static IPushPcmModule* CreatePuchPcmModule(uint32_t dstFs, uint32_t dstChannel, uint32_t streamCount);
	static std::string MakeStreamInfo(uint32_t* fs_array, uint32_t* channel_array, uint32_t* precision_array, uint32_t* length_array, uint32_t streamCount);

	virtual ~IPushPcmModule() {}
	virtual bool Process(const uint8_t* pushData, const uint32_t dataSize, const uint32_t streamCount, const bool enDenoise, uint8_t* outData, uint32_t* outLen) = 0;
	virtual void Destroy() = 0;
};

IAudioBlockMixer* CreateAudioBlockMixer(int sampleCount, int channels);

IAudioDenoise* CreateDenoise(int sampleRate, int channels);

IAudioLevel* CreateAudioLevel(bool newModule = false);

//input and output are in wav format
//sample_rate_hz must be one of 8000 || 16000 || 32000 || 48000;
//return 0:success, 1:no accelerator, -1, error occur
//NOTICE: input_length MUST BE 30ms data
int AccelerateSoundProcess(uint32_t sample_rate_hz, uint32_t num_channels, const int16_t* input, uint32_t input_length, int16_t* output, uint32_t* output_length, bool fastMode = false);
int StretchSoundProcess(uint32_t sample_rate_hz, uint32_t num_channels, const int16_t* input, uint32_t input_length, int16_t* output, uint32_t* output_length);

IBeatTrack* CreateBeatTrack(int samplerate, int channles);

#endif
