#ifndef _AUDIO_KARAOKE_IMPL_H_
#define _AUDIO_KARAOKE_IMPL_H_

#include "int_type.h"
#include "audio_engine_impl.h"
#include "audio_engine_define.h"

/*
 Karaoke Recorder
*/
enum KARAOKE_RECORD_STATUS {
	KARAOKE_RECORD_ERROR_NONE = 0,
	KARAOKE_RECORD_ERROR_UNKNOWN_CODEC = -1,
	KARAOKE_RECORD_ERROR_ENCODER_INIT = -2,
	KARAOKE_RECORD_ERROR_FILE_OPEN_FAIL = -3,
};

enum KaraokeRecordCodecType {
	KaraokeRecordCodecUnknown = -1,
	KaraokeRecordCodecSilk = 0,
	KaraokeRecordCodecSpeex = 1,
	KaraokeRecordCodecEaacPlus = 2,
	KaraokeRecordCodecLcAAC = 3,
	KaraokeRecordCodecWav = 4,
};

class IKaraokeVoiceRecorderNotify
{
public:
	virtual ~IKaraokeVoiceRecorderNotify(){}
	virtual void OnAudioVolumeVisual(uint32_t recordTime, uint32_t volume) = 0;//0~100
	virtual void OnAudioRecordError() = 0;
	virtual void OnGetFirstRecordData() = 0;
	virtual void OnStopRecordData(uint32_t recordTime, uint32_t maxDuration) = 0;
	virtual void OnReachMaxDuration(uint32_t recordTime, uint32_t maxDuration) = 0;
};

class IKaraokeVoiceRecorder
{
public:
	static AUDIO_ENGINE_EXPORT IKaraokeVoiceRecorder* CreateVoiceRecorder(const char* pFileName, int uid, KaraokeRecordCodecType codecType, const uint32_t maxDuration/*ms*/);

	virtual ~IKaraokeVoiceRecorder(){}
	virtual KARAOKE_RECORD_STATUS Init() = 0;
	virtual void Start(IKaraokeVoiceRecorderNotify* pNotify) = 0;
	virtual void SetVolume(uint32_t volume) = 0;
	virtual void Stop() = 0;
	virtual void Pause() = 0;
	virtual void Resume() = 0;
	virtual bool Seek(uint32_t seekTime) = 0;
	virtual bool SeekToEnd(uint32_t seekTime) = 0;
	virtual void SetLyricPauseTime(uint32_t pauseTime) = 0;
	virtual bool EnablePlayBack(bool enable) = 0;
	virtual void Destroy() = 0;
	virtual void EnableStartEncode(bool enable) = 0;
	virtual bool IsRecorded() = 0;

	virtual void EnableReverbEx(bool enable) = 0;
	virtual void SetReverbParamEx(YYReverbExParam reverbParam) = 0;

	virtual void SetEqGains(YYEqGains gains) = 0;
	virtual void SetEqualizerEnable(bool enable) = 0;

	virtual void SetCompressorEnable(bool enable) = 0;
	virtual void EnableReverbFv3(bool enable) = 0;

	virtual void SetCompressorParam(YYCprParam parameter) = 0;
	virtual void SetReverbFv3Param(YYNreverbFv3Parameter parameter) = 0;

	virtual void EnableLimiter(bool enable) = 0;
	virtual void SetLimiterParam(YYLimiterParameter parameter) = 0;

	virtual void EnableDenoiser(bool enable) = 0;
	virtual void SetDenoiseType(DenoiseModuleType noiseType) = 0;
	virtual void StartAudioPreview() = 0;
	virtual void StopAudioPreview() = 0;

	virtual int GetDebugEncodedTime() = 0;
};

class IKaraokePlayerNotify
{
public:
	virtual ~IKaraokePlayerNotify() {}
	virtual void OnGetFirstPlayData() {}
	virtual void OnAudioPlayEnd() = 0;
	virtual void OnAudioPlayVolume(uint32_t volume, uint32_t current_ms, uint32_t total_ms) = 0;
};

enum KaraokePlayerOpenStatus
{
	KARAOKE_PLAYER_OPEN_SUCCESS = 0,
	KARAOKE_PLAYER_CREATE_FILE_DECODER_FAILED = -1,
	KARAOKE_PLAYER_OPEN_FILE_DECODER_FAILED = -2,
	KARAOKE_PLAYER_OPEN_FILE_CHANNEL_ERROR = -3,
	KARAOKE_PLAYER_OPEN_FILE_SAMPLERATE_ERROR = -4,
};

class IKaraokeFilePlayer
{
public:
	static AUDIO_ENGINE_EXPORT IKaraokeFilePlayer* Create();

	virtual ~IKaraokeFilePlayer(){}
	virtual void Destroy() = 0;
	virtual KaraokePlayerOpenStatus Open(const char* pFileName) = 0;
	virtual uint32_t Seek(uint32_t timeMS) = 0;
	virtual uint32_t GetTotalPlayLengthMS() = 0;
	virtual void Play() = 0;
	virtual void Stop() = 0;
	virtual void Pause() = 0;
	virtual void Resume() = 0;
	virtual uint32_t GetCurrentPlayTimeMS() = 0;
	virtual bool SetFeedBackToMicMode(AudioSourceType type) = 0;
	virtual void SetPlayerNotify(IKaraokePlayerNotify *nofity) = 0;
	virtual void SetPlayerVolume(uint32_t volume) = 0;
	virtual void EnableEqualizer(bool enable) = 0;
	virtual void SetEqualizerGain(int nBandIndex, float fBandGain) = 0;
	virtual void SetSemitone(float val) = 0;
	virtual void EnableVolumeNotify(bool enable) = 0;
	virtual void EnableLoopPlay(bool enable, int cycle = -1) = 0;
	virtual void EnableMute(bool enable) = 0;  // decoder data mute...
	virtual void EnableSoundKitEq(bool enable) = 0;

	virtual bool SetPlayOffsetTime(int32_t offsetTime/*ms*/, int32_t accompanyCurTime) = 0;
	virtual int32_t GetPlayOffsetTime() = 0;

	virtual bool StartSaver(const char* path) = 0;
	virtual bool SeekSaver(uint32_t seekTime) = 0;
	virtual bool StopSaver() = 0;

	virtual void EnableEffect(bool enable){}
	virtual void EnableReverbEx(bool enable){}
	virtual void EnableEqualizerEx(bool enable){}
	virtual void EnableCompressor(bool enable){}
	virtual void EnableReverbFv3(bool enable) {}
	virtual void EnableLimiter(bool enable) {};
	virtual void EnableDrc(bool enable) {}

	virtual void SetReverbParamEx(YYReverbExParam reverbParam) {}
	virtual void SetEqGains(YYEqGains gains, bool includePreamp = false){}
	virtual void SetCompressorParam(YYCprParam parameter){}
	virtual void SetReverbFv3Param(YYNreverbFv3Parameter parameter) {}
	virtual void SetLimiterParam(YYLimiterParameter parameter) {};

	virtual void EnableDenoiser(bool enable) {}
	virtual void SetDenoiseType(DenoiseModuleType noiseType) {}

	virtual void EnableEtb(bool enable) {}
	virtual void SetTrebleBassVal(int value) {}
};

class IKaraokeScore
{
public:
	static AUDIO_ENGINE_EXPORT IKaraokeScore* Create(int sampleRate, int channel);

	virtual void Destroy() = 0;
	virtual void SetNotify(IKaraokeScoreNotify* pNotify) = 0;
	//virtual bool OpenScoreRefFile(char* pFileName) = 0;
	virtual void ResetKaraokeScore(uint32_t seekTime) = 0;
	virtual void SetScoreTimeRange(const ScoreTimeRange* pTimeRanges, int count, int8_t* pOutPitchVisualVal) = 0;
	virtual void SetLineInfo(const ScoreLineInfo* pLineInfo, int lineNum) = 0;
	virtual void Process(void* pData, int dataLen, bool isVoiceActive) = 0;
	virtual void SetKaraokeTone(int32_t value) = 0;
	virtual void SetDisplayTolerate(float lowValue, float highValue) = 0;
};

class IKaraokeRecorderNotify
{
public:
	virtual ~IKaraokeRecorderNotify() {}

	virtual void OnKaraokePlayEnd() = 0;
	virtual void OnKaraokeCurrentPlayVolume(uint32_t volume, uint32_t cur_time, uint32_t total_time) = 0;
	virtual void OnKaraokeCurrentRecordVolume(uint32_t volume, uint32_t cur_time) = 0;
	virtual void OnKaraokeScore(uint32_t score, uint32_t lineId) = 0;
	virtual void OnKaraokePitch(int8_t pitch, int32_t lineId, int32_t wordId, int32_t wordNum) = 0;
	virtual void OnKaraokeInitError() = 0;
	virtual void OnKaraokeFirstGetPlayData() = 0;
};

class IKaraokeRecorder
{
public:
	static AUDIO_ENGINE_EXPORT IKaraokeRecorder* Create();

	virtual ~IKaraokeRecorder() {}

	virtual void SetKaraokeRecorderNotify(IKaraokeRecorderNotify* nofity) = 0;

	virtual bool OpenInputFile(const char* voicFile, const char* accompanyFile) = 0;
	virtual bool OpenOutputFile(const char* voiceRecordFile, const char* accompanyRecordFile) = 0;

	virtual bool OpenMixFileEx(const char* mixFile) = 0;
	virtual bool OpenAccompanyFileEx(const char* accompanyFile) = 0;

	virtual bool SetScoreTimeRange(ScoreTimeRange* timeRange, uint32_t count, int8_t* pitchVisual) = 0;
	virtual void SetLineInfo(ScoreLineInfo* lineInfo, int lineNum) = 0;
	virtual uint32_t GetPlayerTime() = 0;
	virtual uint32_t GetCurrentPlayerTime() = 0;
	virtual void SetVoiceVolume(uint32_t volume) = 0;
	virtual void SetAccompanyVolume(uint32_t volume) = 0;
	virtual void SetTone(int tone) = 0;
	virtual void SetDisplayTolerate(float lowValue, float highValule) = 0;

	virtual bool SetChooseMixFilePlay(bool mixPlay) = 0;
	virtual bool EnablePlayBack(bool enable) = 0;
	virtual bool IsRecordPlayBackEnable() = 0;

	virtual void EnableReverbNew(bool enable) = 0;
	virtual void EnableEqualizer(bool enable) = 0;
	virtual void EnableCompressor(bool enable) = 0;
	virtual void EnableLimiter(bool enable) = 0;

	virtual void SetEqGains(YYEqGains gains) = 0;
	virtual void SetCompressorParam(YYCprParam parameter) = 0;
	virtual void SetReverbFv3Param(YYNreverbFv3Parameter parameter) = 0;
	virtual void SetLimiterParam(YYLimiterParameter parameter) = 0;

	virtual bool Start() = 0;
	virtual void Stop() = 0;
	virtual void Pause() = 0;
	virtual void Resume() = 0;
	virtual void Seek(uint32_t seekTime) = 0;
	virtual void SeekToEnd(uint32_t seekTime) = 0;
	virtual void SetLyricPauseTime(uint32_t pauseTime) = 0;
	virtual void Destroy() = 0;
};

class IKaraokePlayBackNotify
{
public:
	virtual ~IKaraokePlayBackNotify() {}

	virtual void OnAudioPlayEnd() = 0;
	virtual void OnAudioPlayVolume(uint32_t volume, uint32_t current_ms, uint32_t total_ms) = 0;
};

class IKaraokePlayBack
{
public:
	static AUDIO_ENGINE_EXPORT IKaraokePlayBack* Create();

	virtual ~IKaraokePlayBack() {}

	virtual void SetKaraokePlayBackNotify(IKaraokePlayBackNotify* nofity) = 0;

	virtual bool Open(const char* voiceFile, const char* accompanyFile) = 0;

	virtual uint32_t GetTotalPlayBackTime() = 0;
	virtual uint32_t GetCurrentPlayerTime() = 0;
	virtual void SetVoiceVolume(uint32_t volume) = 0;
	virtual void SetAccompanyVolume(uint32_t volume) = 0;
	virtual void SetTone(int tone) = 0;
	virtual void SetEtbValue(int value) = 0;
	virtual void SetVoiceOffset(int offsetTime) = 0;
	virtual void EnableDenoise(bool enable) = 0;
	virtual void EnableDrc(bool enable) = 0;

	virtual bool OpenAutoToneFilePlayer(const char* autoToneFile) = 0;
	virtual bool StopAutoToneFilePlayer() = 0;
	virtual bool ChooseAutoToneFilePlayer(bool isAutoTonePlay) = 0;
	virtual bool IsAutoToneFilePlaying() = 0;

	virtual void EnableReverbNew(bool enable) = 0;
	virtual void EnableEqualizer(bool enable) = 0;
	virtual void EnableCompressor(bool enable) = 0;
	virtual void EnableLimiter(bool enable) = 0;

	virtual void SetEqGains(YYEqGains gains) = 0;
	virtual void SetCompressorParam(YYCprParam parameter) = 0;
	virtual void SetReverbFv3Param(YYNreverbFv3Parameter parameter) = 0;
	virtual void SetLimiterParam(YYLimiterParameter parameter) = 0;

	virtual void Play() = 0;
	virtual void Stop() = 0;
	virtual void Pause() = 0;
	virtual void Resume() = 0;
	virtual void Seek(uint32_t seekTime) = 0;
	virtual void Destroy() = 0;
};

class IKaraokeFileMixerNotify
{
public:
	virtual ~IKaraokeFileMixerNotify() {}

	virtual void OnFileMixerState(uint32_t current_time, uint32_t total_time) = 0;
	virtual void OnFinishMixer() = 0;
};

class IKaraokeFileMixer
{
public:
	static AUDIO_ENGINE_EXPORT IKaraokeFileMixer* Create();

	virtual ~IKaraokeFileMixer() {}

	virtual void SetKaraokeFileMixerNotify(IKaraokeFileMixerNotify* nofity) = 0;

	virtual bool Open(const char* voiceFile, const char* accompanyFile) = 0;

	virtual void SetVoiceVolume(uint32_t volume) = 0;
	virtual void SetAccompanyVolume(uint32_t volume) = 0;
	virtual void SetTone(int tone) = 0;
	virtual void SetEtbValue(int value) = 0;
	virtual void SetVoiceOffset(int offsetTime) = 0;
	virtual void EnableDenoise(bool enable) = 0;

	virtual void EnableReverbNew(bool enable) = 0;
	virtual void EnableEqualizer(bool enable) = 0;
	virtual void EnableCompressor(bool enable) = 0;
	virtual void EnableLimiter(bool enable) = 0;

	virtual void SetEqGains(YYEqGains gains) = 0;
	virtual void SetCompressorParam(YYCprParam parameter) = 0;
	virtual void SetReverbFv3Param(YYNreverbFv3Parameter parameter) = 0;
	virtual void SetLimiterParam(YYLimiterParameter parameter) = 0;

	virtual bool Start(const char* filePath) = 0;
	virtual void Stop() = 0;
	virtual void Destroy() = 0;
};

#endif 
