#ifndef _AUDIO_ENGINE_IMPL_H_
#define _AUDIO_ENGINE_IMPL_H_

#include "audioengine/int_type.h"
#include "audioengine/codec_type.h"
#include "audioengine/audio_jitter_buffer_impl.h"
#include "audioengine/audio_toolbox_impl.h"
#include "audioengine/audio_engine_define.h"

typedef void (*LogCallback)(const char* pLogStr);

typedef struct
{
	uint8_t frameData[1024];
	uint16_t frameLength;
	uint32_t timestamp;
	uint8_t ssrc;
	uint32_t frameIndex;
	uint8_t codecType;
	uint8_t keyFrame;
	uint8_t discard;
} AudioFrameData;

typedef struct 
{
	uint8_t volumeEnhence;
	uint8_t aecMode; //nlpmode aecm also in it
	uint8_t nsMode;
	uint8_t agcMode;
} AudioProperConfig;

enum VADResultKind 
{
	vadResultIsActive = 0,
	vadResultIsPassive = 1,
	vadResultIsUnknown = 2,
	vadResultIsActiveWithoutVoice = 3
};

enum AudioSourceType
{
	AudioSourceTypeMic = 0,
	AudioSourceTypeOuterPCM,
	AudioSourceTypeMix, // mic and outer-pcm
};

enum AudioEngineFeature_t
{
	AudioEngineFeatureVOIP           = 0,
	AudioEngineFeatureCommon         = 1,
	AudioEngineFeatureLiveBroadcast  = 3,
	AudioEngineKaraokeOffLine        = 5,
	TotalFeatureMode            
};

enum VoiceDetectionMode
{
	voiceDetectionUnknown = -1,
	voiceDetectionVeryLowLikelihood = 0,
	voiceDetectionLowLikelihood,
	voiceDetectionModerateLikelihood,
	voiceDetectionHighLikelihood
};

enum AudioSaverMode
{
	AudioSaverOnlyCapture    = 0,
	AudioSaverOnlyRender     = 1,
	AudioSaverBoth           = 2,
    AusioSaverOnlyCaptureOri = 3,
};

enum AudioSaverWfMode
{
	AudioSaverFileAppend   = 0,
	AudioSaverFileOverride = 1,
};

enum AppCommonEvent
{
    AppForegroundEvent,
    AppBackgroundEvent,
    AppScreenLockedEvent,
    AppScreenUnlockedEvent
};

enum DenoiseModuleType
{
	AecImproveNsType = 0,
	SpeexNsType = 1,
	WebrtcNsType = 2, // 10ms, only support 32k,16k now
};


enum AudioOutputRouting
{
    AudioOutputRoutingDefault = -1,
    AudioOutputRoutingHeadset = 0,
    AudioOutputRoutingEarpiece = 1,
    AudioOutputRoutingHeadsetNoMic = 2,
    AudioOutputRoutingSpeakerphone = 3,
    AudioOutputRoutingLoudspeaker = 4,
    AudioOutputRoutingHeadsetBluetooth = 5,
};

class IAudioCaptureNotify
{
public:
	virtual ~IAudioCaptureNotify() {}
	virtual void OnCaptureAudioData(const void* audioSamples, const uint32_t dataSize, const uint8_t nBytesPerSample, 
		const VADResultKind vadResultKind, uint32_t timeStamp, bool firstRecFrame) = 0;
	virtual bool OnCaptureVolumeVisual(const uint32_t& volume) = 0;
};

class IAudioRenderNotify
{
public:
	virtual ~IAudioRenderNotify() {}
	virtual bool OnRenderAudioData(void* audioSamples, const uint32_t dataSize, const uint8_t nBytesPerSample, const uint16_t playoutDelayMS = 0) = 0;
	virtual void OnRenderAudioFormatChange(uint32_t& newSampleRate, uint32_t& newChannels) = 0;
	virtual bool OnAudioVolumeData(const uint32_t& volume) = 0;
	virtual bool OnRenderPostProcess(void* audioSamples, const uint32_t dataSize, const uint32_t channels, const uint32_t sampleRate, bool needNofity) { return true; }
};

class IAudioCapture
{
public:
	virtual ~IAudioCapture(){}
	virtual void Destroy() = 0;
	virtual bool Open(uint32_t sampleRate, uint32_t channels, uint32_t bps) = 0;
	virtual void Start(IAudioCaptureNotify* pNotify) = 0;
	virtual void Stop() = 0;
};

class IAudioRender
{
public:
	virtual ~IAudioRender(){}
	virtual void Destroy() = 0;
	virtual bool Open(uint32_t sampleRate, uint32_t channels, uint32_t bps, uint32_t fadeInMS = 0) = 0;
	virtual void Start(IAudioRenderNotify* pNotify) = 0;
	virtual void Stop() = 0;
    virtual void Pause(bool pause) = 0;
	virtual void SetVolume(uint32_t volume) = 0;
	virtual void SetMute(bool mute) = 0;   // only mute render speaker, have no effect on inner volume display
};

enum AudioPackWarning
{
    kAudioPackWarning_Silence = 0
};

class IAudioFramePackerNotify 
{
public:
	virtual ~IAudioFramePackerNotify() {}
	virtual void OnPackAudioFrame(const uint8_t* pEncodeFrame, const uint32_t encodeFrameSize,
	        const VADResultKind vadResultKind, uint32_t timestamp, uint8_t ssrc, bool firstRecFrame) = 0;
	virtual void OnPackAudioFirstFrameEncoded() = 0;
	virtual void OnPackAudioEncodeFailed() = 0;
    virtual void OnAudioCaptureVolume(const uint32_t volume) = 0;
    virtual void OnHowlingDetectResult(bool bHowling) {}
    virtual void OnPackAudioWarning(int warning) {}
    virtual void OnEchoDetectResult(bool bHowling) {}
};

struct FramePlayerFeedbackInfo
{
	FramePlayerFeedbackInfo()
	{
		acceleratePlay = false;
		fasterPlayOn = false;
		stretchPlay = false;
		isFakeAudioFrame = false;
		accelerateMS = 0;
		stretchMS = 0;
		playDelayMS = 0;
	}

	// read & write
	bool acceleratePlay;
	bool fasterPlayOn; // only effect when acceleratePlay is true 
	bool stretchPlay;
	bool isFakeAudioFrame;

	// only read 
	double accelerateMS;
	double stretchMS;
	double playDelayMS;
};

//reason for AudioFramePlayerStatus
enum AudioFramePlayerErrorCode
{
    AUDIO_FRAME_PLAYER_STATUS_SUCCESS = 0,
    AUDIO_FRAME_PLAYER_STATUS_FAILED = -1,
    AUDIO_FRAME_PLAYER_DECODED_FORMAT_NOT_SUPPORT = -2,
    AUDIO_FRAME_PLAYER_DECODED_REASON_INTERNAL = -3,
    AUDIO_FRAME_PLAYER_DEVICE_START_FAILED = -4,
};

enum AudioFramePlayerStatus
{
    AUDIO_FRAME_PLAYER_START = 0,   //start successfully，errorCode should  be AUDIO_FRAME_PLAYER_STATUS_SUCCESS
    AUDIO_FRAME_PLAYER_DECODE = 1,  //first audio frame decoded successfully, errorCode should be AUDIO_FRAME_PLAYER_STATUS_SUCCESS
    AUDIO_FRAME_PLAYER_STOP = 2,    //stop successfully，errorCode should be AUDIO_FRAME_PLAYER_STATUS_SUCCESS
    AUDIO_FRAME_PLAYER_FAILED = 100,
};

class IAudioFramePlayerNotify 
{
public:
	virtual ~IAudioFramePlayerNotify() {}
	virtual bool OnPullAudioFrame(uint32_t id, AudioFrameData& frame, bool forcePull, FramePlayerFeedbackInfo& feedbackInfo) = 0;
	virtual bool OnAudioVolume(uint32_t id, const uint32_t volume) = 0;  
	virtual bool OnAudioRenderPTS(const uint32_t uid, const uint32_t pts) = 0;
	virtual int  OnAudioAncData(uint32_t id, char** ancOffset, int* ancSize, int* eleTag, int number) = 0;
	virtual int  OnAudioFirstDecodeFrame(uint32_t id) = 0;
	virtual void OnAudioPlayerStatus(uint32_t uid, AudioFramePlayerStatus state, AudioFramePlayerErrorCode errorCode) = 0;
    virtual bool OnPlaybackAudioFrameBeforeMixing(uint32_t uid, void* samples, uint32_t numOfSamples, uint32_t bytesPerSample, uint32_t channels, uint32_t samplesPerSec) = 0;
};

class IAudioChannel
{
public:
	virtual ~IAudioChannel(){}
	virtual void Destroy() = 0;

	virtual bool StartPacker(IAudioFramePackerNotify* pNotify, int codecId, int bitrate = 48000, int samplesPerSec = 48000, int channels = 1, int profile = 2) = 0;
    virtual bool OpenCapture() = 0;
	virtual bool CloseCapture() = 0;
	virtual void StopPacker() = 0;
    virtual void EnablePackerVad(bool enable) = 0;
	virtual bool SetVolume(uint32_t volume) = 0;
    virtual void PushPcmData(const char* audioSamples, const uint32_t dataSize, const uint32_t timeStamp, const VADResultKind vadResultKind) = 0;
	virtual void PushPcmData(uint32_t id, const char* ptr, const uint32_t length, uint32_t fs, uint32_t ch, const uint32_t timeStamp) = 0;

	virtual bool StartPlay(IAudioFramePlayerNotify* pNotify, uint32_t id, uint32_t fs, uint32_t ch) = 0;
	virtual void StopPlay(uint32_t id) = 0;
    virtual bool SetPlayVolume(uint32_t id, uint32_t volume) = 0;
    virtual bool IsPlayStarted(uint32_t id) = 0;

    virtual bool SetRemoteVoicePosition(uint32_t id, int azimuth, int elevation) = 0;  // azimuth: [-180, 180], elevation:[-40, 90]

    static AUDIO_ENGINE_EXPORT IAudioChannel* Create();
};

class IAudioFramePacker
{
public:
	virtual ~IAudioFramePacker() {}
	virtual void Destroy() = 0;

	virtual bool Start(IAudioFramePackerNotify* pNotify, int codecId, int bitrate = 48000, int samplesPerSec = 48000, int channels = 1, int profile = 2) = 0;
	virtual bool SetVolume(uint32_t volume) = 0;
	virtual void OpenCapture() = 0; // packer source from capture
	virtual void CloseCapture() = 0; 
	virtual void Stop() = 0;
    virtual void EnableVad(bool enable) = 0;
	virtual void PushPcmData(const char* ptr, const uint32_t length, const uint32_t timeStamp, const VADResultKind vadResultKind) = 0;
	virtual void PushPcmData(uint32_t id, const char* ptr, const uint32_t length, uint32_t fs, uint32_t ch, const uint32_t timeStamp) = 0;

    static AUDIO_ENGINE_EXPORT IAudioFramePacker* Create();
};

class IAudioFramePlayer
{
public:
	virtual ~IAudioFramePlayer(){}
	virtual void Destroy() = 0;

	virtual bool Start(IAudioFramePlayerNotify* pNotify, uint32_t id, uint32_t fs, uint32_t ch) = 0;
	virtual void Stop() = 0;
    virtual void SetPlayVolume(uint32_t volume) = 0;
    virtual bool IsPlayStarted() = 0;

	virtual bool SetRemoteVoicePosition(int azimuth, int elevation) = 0;
    
    static AUDIO_ENGINE_EXPORT IAudioFramePlayer* Create();
};

class IAudioWizard
{
public:
	virtual ~IAudioWizard(){}
	virtual bool Init() = 0;
	virtual void Uninit() = 0;
	virtual void Destroy() = 0;
	virtual bool StartSpeakerWizard(const char* pFullPathFileName) = 0;
	virtual void StopSpeakerWizard() = 0;
	virtual bool StartMicrophoneWizard() = 0;
	virtual void StopMicrophoneWizard() = 0;
};

class IAudioConvert
{
public:
	virtual ~IAudioConvert(){}
	virtual void Destroy() = 0;
	virtual int16_t Process(const int16_t* pSrc, int16_t* pDst) = 0;//use GetFixProcessSize, result is real dst len
	virtual uint32_t GetSrcProcessSize() = 0;//10ms
	virtual uint32_t GetDstProcessSize() = 0;//10ms
};

class ISpeexConvert   // TODO: cpu high, remove later
{
public:
	virtual ~ISpeexConvert() {}
	virtual void Process(void* pData, int dataLen, std::string& outData) = 0;
	virtual void Process(void* pData, int dataLen, char* pOut, int* outLen) = 0;
	virtual bool IsFormatChange(uint32_t fs, uint32_t ch) = 0;
	virtual void Destroy() = 0;

	static AUDIO_ENGINE_EXPORT ISpeexConvert* Create(int srcSampleRate, int outSampleRate, int srcChannels, int outChannels);
};

class IFileMixerPoint
{
public:
	virtual ~IFileMixerPoint() {}
	virtual bool Open(const char* fileName, bool enableThreadRead = false) = 0;
	virtual void Close() = 0;  // TODO: remove this interface, or use flag to mark open instead of destroy. if not, will cause crash with DoEffectProcess()

	virtual void SetOffsetMs(int32_t offsetMs) = 0;

	virtual void EnableReverbEx(bool enable) = 0;
	virtual void EnableReverbFv3(bool enable) = 0;
	virtual void EnableEqualizerEx(bool enable) = 0;
	virtual void EnableCompressor(bool enable) = 0;
	virtual void EnableLimiter(bool enable) = 0;

	virtual void SetReverbParamEx(YYReverbExParam reverbParam) = 0;
	virtual void SetReverbFv3Param(YYNreverbFv3Parameter reverbParam) = 0;
	virtual void SetEqGains(YYEqGains gains) = 0;
	virtual void SetCompressorParam(YYCprParam parameter) = 0;
	virtual void SetLimiterParameter(YYLimiterParameter parameter) = 0;
	virtual void SetVolume(uint32_t volume) = 0;
	virtual void SetTrebleBassVal(uint32_t value) = 0;
	virtual void SetSemitone(float value) = 0;
	virtual void SetDenoiseType(DenoiseModuleType type) = 0;
	virtual void EnableEtb(bool enable) = 0;
	virtual void EnableDrc(bool enable) = 0;
	virtual void EnableDenoiser(bool enable) = 0;
	virtual void EnableEqReverb(bool enable) = 0;
	virtual void EnableEffect(bool enable) = 0;
};

class IFileMixerNotify
{
public:
	virtual ~IFileMixerNotify() {}
	virtual void OnFileMixerState(uint32_t current_time, uint32_t total_time) = 0;
	virtual void OnFinishMixer() = 0;
};

class IFileMixer
{
public:
	static AUDIO_ENGINE_EXPORT IFileMixer* Create();

	virtual ~IFileMixer() {}
	virtual IFileMixerPoint* CreateMixerPoint() = 0;
	virtual void Destroy() = 0;
	virtual void SetNotify(IFileMixerNotify* notify) = 0;
	virtual uint32_t GetMixerTime() = 0;
	virtual bool Start(const char* outputFileName) = 0;
	virtual void Stop() = 0;
	virtual void SetMusicPoint(IFileMixerPoint* point) = 0;
	virtual void SetVoicePoint(IFileMixerPoint* point) = 0;
};

class IAudioPreview
{
public:
	virtual ~IAudioPreview(){}
	virtual bool Init() = 0;
	virtual void Uninit() = 0;
	virtual void Destroy() = 0;
	virtual bool StartPreview() = 0;
	virtual void StopPreview() = 0;
	virtual void SetVoiceEffectOption(YYVoiceEffectOption mode) = 0;
	virtual void SetVoiceBeauify(YYVoiceBeautifyOption mode, float pitch, float reverb) = 0;
	virtual void SetVoiceBeauifyEnable(bool enable) = 0;
};

enum AudioDeviceErrorType
{
    AudioDeviceInitCaptureSuccess,
    AudioDeviceInitCaptureErrorOrNoPermission,
    AudioDeviceReleaseCaptureSuccess,
    AudioDeviceInitCaptureErrorMayOnBackGround,
    AudioDeviceCaptureRuntimeError,
    AudioDeviceInitRenderSuccess,
    AudioDeviceInitRenderError,
    AudioDeviceRenderRuntimeError,
    AudioDeviceInitRenderErrorMayOnBackGround,
    AudioDeviceStartRenderSuccess,
    AudioDeviceStopRenderSuccess,
    AudioDeviceStartRenderError,
    AudioDeviceInitRenderErrorOrNoPermission,
};

class IAudioEngineNotify
{
public:
	virtual ~IAudioEngineNotify() {}
	virtual void OnAudioCaptureError(AudioDeviceErrorType errorType) {}
	virtual void OnAudioRenderError(AudioDeviceErrorType errorType){}
	virtual void OnReceivePhoneCall(bool isInCall) {}
	virtual void OnAudioModeChange() {}
	virtual void OnHeadsetPlug(bool enable) {}
	virtual void OnInterruptStatus(bool status) {}  // status: 1 begin  0 end
	virtual void OnAudioRenderSpectrumData(float* data, int len) {}
	virtual void OnAudioCapturePcmData(char* data, uint32_t dataSize, uint32_t sampleRate, uint32_t channel) {}
	virtual void OnAudioRenderPcmData(char* data, uint32_t dataSize, uint32_t sampleRate,  uint32_t channel) {}
	virtual void OnAudioCapturePcmDataVolume(uint32_t volume) {}
	virtual void OnHowlingDetector(bool isHowling, bool isForceNotify) {}
    virtual void OnEchoDetector(bool isHowling, bool isForceNotify) {}
    //input: true for input device, false for output device
    //state: 0 for plugin, 1 for plugout
    virtual void OnAudioDeviceStateChanged(char* deviceID, char* deviceName, bool input, int state) {}
    virtual void OnAudioRouteChanged(AudioOutputRouting router) {};
    virtual void OnAudioCaptureWarning(int warning) {}

};

class IKaraokeScoreNotify
{
public:
	virtual void OnCalcKaraokeScore(int8_t score, uint32_t lineId) = 0; //0 ~ 100
	virtual void OnSingerPitchVisual(int8_t val, int32_t lineId, int32_t wordId, int32_t wordNum) = 0; //0~8
};

class IAudioFilePlayerNotify
{
public:
	virtual ~IAudioFilePlayerNotify() {}
	virtual void OnAudioPlayVolume(uint32_t volume, uint32_t current_ms, uint32_t total_ms) = 0;
    virtual void OnAudioPlayStateChange(int state, int errorCode) = 0;
};

enum FilePlayerErrorCode
{
	PLAYER_STATUS_SUCCESS = 0,
	PLAYER_CREATE_FILE_DECODER_FAILED = -1,
	PLAYER_OPEN_FILE_DECODER_FAILED = -2,
    PLAYER_OPEN_FILE_FORMAT_NOT_SUPPORT = -3,
	PLAYER_OPEN_FILE_PATH_ERROR = -4,
};

enum FilePlayerStatus
{
	PLAYER_OPEN = 100,
	PLAYER_PLAY = 101,
	PLAYER_PAUSE = 102,
	PLAYER_STOP = 103,
    PLAYER_RESUME = 104,
    PLAYER_SEEK_COMPLETE = 105,
    PLAYER_END = 106,
	PLAYER_ERROR = 107,
};

class IAudioFilePlayer
{
public:
    static AUDIO_ENGINE_EXPORT IAudioFilePlayer* Create();
    
	virtual ~IAudioFilePlayer(){}
	virtual void Destroy() = 0;
    virtual uint32_t GetPlayerId() = 0;
	virtual FilePlayerErrorCode Open(const char* pFileName) = 0;
	virtual uint32_t Seek(uint32_t timeMS) = 0;
	virtual uint32_t GetTotalPlayLengthMS() = 0;
	virtual void Play() = 0;
	virtual void Stop() = 0;
//    virtual void EnableFakePause(bool enable) = 0;
//    virtual void FakePause() = 0;
	virtual void Pause() = 0;
	virtual void Resume() = 0;
	virtual uint32_t GetCurrentPlayTimeMS() = 0;
	virtual bool SetFeedBackToMicMode(AudioSourceType type) = 0;
	virtual void SetPlayerNotify(IAudioFilePlayerNotify *nofity) = 0;
	virtual void SetPlayerVolume(uint32_t volume) = 0;
    virtual void SetPlayerLocalVolume(uint32_t volume) = 0;
    virtual void SetPlayerPublishVolume(uint32_t volume) = 0;
    virtual int GetPlayerLocalVolume() = 0;
    virtual int GetPlayerPublishVolume() = 0;
	virtual void EnableEqualizer(bool enable) = 0;
	virtual void SetEqualizerGain(int nBandIndex, float fBandGain) = 0;	
	virtual void SetSemitone(float val) = 0;
    virtual void SetTempo(float val) = 0;
	virtual void EnableVolumeNotify(bool enable) = 0;
	virtual void EnableLoopPlay(bool enable, int cycle = -1) = 0;
	virtual void EnableMute(bool enable) = 0;  // decoder data mute...

//	virtual bool StartSaver(const char* path) = 0;
//	virtual bool SeekSaver(uint32_t seekTime) = 0;
//	virtual bool StopSaver() = 0;

	virtual bool SelectAudioTrack(int audioTrack) = 0;
	virtual int GetAudioTrackCount() = 0;

	virtual void EnableEffect(bool enable){}
	virtual void EnableReverbEx(bool enable){}
	virtual void EnableReverbFv3(bool enable){}
	virtual void EnableEqualizerEx(bool enable){}
	virtual void EnableCompressor(bool enable){}
	virtual void EnableLimiter(bool enable){}

	virtual void SetReverbParamEx(YYReverbExParam reverbParam) {}
	virtual void SetEqGains(YYEqGains gains, bool includePreamp = false){}
	virtual void SetCompressorParam(YYCprParam parameter){}
	virtual void SetLimiterParam(YYLimiterParameter parameter){}

	virtual void EnableSpectrum(bool enable) = 0;
	virtual int GetCurrentSpectrum(float* buffer, int len) = 0;

//	virtual void SetDspExEnable(bool enable) = 0;
//	virtual void SetDspExReverbEnable(bool enable) = 0;
//	virtual void SetDspExReverbFv3Param(YYNreverbFv3Parameter reverbParam) = 0;
//	virtual void SetDspExLimiterEnable(bool enable) = 0;
//	virtual void SetDspExLimiterParameter(YYLimiterParameter parameter) = 0;
//	virtual void SetDspExEqEnable(bool enable) = 0;
//	virtual void SetDspExEqGains(YYEqGains gains) = 0;
    virtual bool SetPosition(int azimuth, int elevation) = 0;
};

enum AUDIO_EXTERNAL_PROCESS_MODE
{
	AUDIO_EXTERNAL_PROCESS_MODE_READ = 1,
	AUDIO_EXTERNAL_PROCESS_MODE_WRITE = 2, 
	AUDIO_EXTERNAL_PROCESS_MODE_READ_AND_WRITE = 3,
};

// TODO: remove later
class IExternalAudioProcessor {
public:
    virtual ~IExternalAudioProcessor(){}
    virtual void OnCaptureStart() = 0;
    virtual void OnCaptureData(void* pData, uint32_t len, uint32_t sampleRate, uint32_t channelCount, uint32_t bitPerSample) = 0;
    virtual void OnCaptureStop() = 0;
    virtual void OnRenderStart() = 0;
    virtual void OnRenderData(void* pData, uint32_t len, uint32_t sampleRate, uint32_t channelCount, uint32_t bitPerSample) = 0;
    virtual void OnRenderStop() = 0;
};

class IExternProcessor 
{
public:
	virtual ~IExternProcessor() {}
	virtual bool OnCaptureProcessData(void* samples, uint32_t numOfSamples, uint32_t bytesPerSample, uint32_t channels, uint32_t samplesPerSec) = 0;
	virtual bool OnRenderProcessData(void* samples, uint32_t numOfSamples, uint32_t bytesPerSample, uint32_t channels, uint32_t samplesPerSec) = 0;
    virtual bool OnMixedProcessData(void* samples, uint32_t numOfSamples, uint32_t bytesPerSample, uint32_t channels, uint32_t samplesPerSec, uint32_t timestamp) { return true; }
};

struct ScoreTimeRange
{
	float beginTime;
	float endTime;
	uint32_t lineID;
	uint32_t lineEnd;
	int midiValue;
};

struct ScoreLineInfo
{
	int lineIdx;
	float lineStartTime;
	float lineEndTime;
};


enum AudioEventMessageType
{
    AudioEvent_PhoneCall = 0,
    AudioEvent_HeadsetIn = 1,
    AudioEvent_AppBackground = 2,
    AudioEvent_BluetoothOn = 3,
    AudioEvent_SystemVolume = 4,
    AudioEvent_CaptureStatus = 5,
    AudioEvent_RenderStatus = 6,
    AudioEvent_InterruptStatus = 7,

};

class IAudioBehaviorEventNotify
{
public:
    virtual ~IAudioBehaviorEventNotify() {}
    virtual int OnBehaviorEventNotify(AudioEventMessageType msg, int value) = 0;
};

#define kMAX_YY_AUDIO_DEVICE_COUNT  64
#define kMAX_YY_AUDIO_DEVICE_NAME_LEN 128
#define kMAX_YY_AUDIO_DEVICE_ID_LEN 128

class IAudioEngine
{
public:
	virtual ~IAudioEngine(){}

	virtual void Destroy() = 0;

	virtual int EnableEngine() = 0;
	virtual int DisableEngine() = 0;
	virtual int SetAudioProfile(AudioEngineFeature_t profile) = 0;
	virtual AudioEngineFeature_t CurAudioProfile() = 0;
	virtual int IsEngineEnabled() = 0;
	virtual void ResetAudioEngine() = 0;

	virtual void SetAudioProperConfig(const AudioProperConfig& config) = 0;

	virtual void AddAudioEngineNotify(IAudioEngineNotify* pNotify) = 0;
	virtual void RemoveAudioEngineNotify(IAudioEngineNotify* pNotify) = 0;

	virtual IAudioCapture* CreateAudioCapture() = 0;
	virtual IAudioRender* CreateAudioRender() = 0;
	virtual IAudioWizard* CreateAudioWizard() = 0;
	virtual IAudioPreview* CreateAudioPreview() = 0;

	virtual void StartAudioPreview() = 0;	
	virtual void StopAudioPreview() = 0;
	virtual bool SetSpeakerVolume(uint32_t volume) = 0; 
	virtual uint32_t GetSpeakerVolumeRange() = 0;
	virtual uint32_t GetSpeakerVolume() = 0;
	
	virtual bool SetMicrophoneVolume(uint32_t volume) = 0; //0~100
	virtual bool SetLoudspeakerStatus(bool enable) = 0; //for mobile
	virtual bool GetLoudspeakerStatus() = 0;
	virtual bool SetVirtualSpeakerVolume(uint32_t volume) = 0; //0~100
	virtual bool SetVirtualMicVolume(uint32_t volume) = 0; //0~100
	virtual void SetMicMute(bool mute) = 0;
    virtual bool IsMicMute() = 0;

	virtual void SetAudioSourceType(AudioSourceType sourceType) = 0;
	virtual void PushOuterAudioData(const char* dataBlock, int dataSize, int sampleRate, int channel) = 0; // 10ms 

	virtual void EnableAutoGainControl(bool enable) = 0; 
	virtual void ResetPreProc() = 0;
	virtual void SetVoiceDetectionMode(VoiceDetectionMode mode) = 0;

	virtual void SetVeoMode(YYVoiceEffectOption val) = 0;

	virtual void SetNewNsOn(bool enable) = 0;
    virtual void SetNewAGCOn(bool enable) = 0;

	virtual void SetVoiceBeauify(YYVoiceBeautifyOption val, float pitch, float reverb) = 0;
	virtual void SetVoiceBeauifyEnable(bool enable) = 0;

	virtual void EnableReverb(bool enable) = 0;
	virtual void SetReverbParam(YYReverbParam reverbParam) = 0;
	virtual void SetReverbPreset(int mode) = 0;
	virtual void EnableReverbEx(bool enable) = 0;
	virtual void SetReverbParamEx(YYReverbExParam reverbParam) = 0;

	virtual void SetPlayerMixerCompValue(int value) = 0;

    virtual void EnableEqualizer(bool enable) = 0;
    virtual void EnableCompressor(bool enable) = 0;
    virtual void SetEqGains(YYEqGains gains, bool includePreamp = false) = 0;
    virtual void SetCompressorParam(YYCprParam parameter) = 0;
	virtual void SetLimiterEnable(bool enable) = 0;
	virtual void SetLimiterParameter(YYLimiterParameter parameter) = 0;

    virtual int SetLocalVoicePitch(float value) = 0;//limits [-12, 12], 0:disable, non 0:enable

	virtual void StartLogger(const char* path) = 0;
	virtual void EventLog(const char* log) = 0;
	virtual void StopLogger() = 0;
	virtual bool SetPlaybackModeOn(bool enable) = 0;
	virtual bool SetPlayBackVolume(uint32_t volume) = 0;
	virtual bool SetBuildInMicLocation(int location) = 0;
	virtual bool RecoverAudioMode() =  0;
//	virtual void SetKaraokeTone(int32_t toneValue) = 0;
//	virtual void SetDisplayTolerate(float lowValue, float highValue) = 0;

	virtual void EnablePhoneAdaptation(bool enable) = 0;
	virtual void EnableMicrophone(bool enable) = 0;
	virtual void EnableRenderVolumeVisual(bool enable) = 0;
	virtual void EnableCaptureVolumeVisual(bool enable) = 0;

	virtual void EnableDenoise(bool enable) = 0;
    virtual void EnableAutoStatisLog(bool enable) = 0;

	virtual void SetAudioSaverEncodeType(AudioLocalCodec::AudioCodecType encType) = 0;
	virtual bool StartAudioSaver(const char* fileName, AudioSaverMode saverMode = AudioSaverOnlyCapture, AudioSaverWfMode fileMode = AudioSaverFileAppend) = 0;
	virtual bool StopAudioSaver() = 0;
	virtual bool CheckPhoneCallState() = 0;
	virtual bool GetHeadSetMode() = 0;
    // Only For Ios
    virtual void NotifyAppCommonEvent(AppCommonEvent appEvent) = 0;

    // @deprecated
	virtual bool GetAudioCaptureAndEncodeHiidoStatInfo(char*& pCaptureAndEncodeInfo, int& stringLen) = 0;
	virtual bool GetAudioRenderHiidoStatInfo(char*& pRenderInfo, int& stringLen) = 0;
	virtual bool GetAudioStreamDecodeHiidoStatInfo(uint32_t uid, char*& pDecoderInfo, int& stringLen) = 0;

	virtual void EnableReleaseWhenCloseMic(bool enable) = 0;
    virtual void EnableStereoPlayWhenHeadsetIn(bool enable) = 0;
	virtual bool IsDenoiseEnabled() = 0;
    virtual void SetExternalAudioProcessor(IExternalAudioProcessor* eap) = 0;
	virtual void SetExternalCaptureParameter(uint32_t sampleRate, uint32_t channels) = 0;
	virtual void SetExternalRenderParameter(uint32_t sampleRate, uint32_t channels) = 0;

	virtual bool SetExternalProcessor(IExternProcessor* processor) = 0;
	virtual bool SetExternalCaptureProcessorParameter(uint32_t sampleRate, uint32_t channels, AUDIO_EXTERNAL_PROCESS_MODE mode, uint32_t samplesPerCall) = 0;
	virtual bool SetExternalRenderProcessorParameter(uint32_t sampleRate, uint32_t channels, AUDIO_EXTERNAL_PROCESS_MODE mode, uint32_t samplesPerCall) = 0;
    virtual bool SetExternalMixedProcessorParameter(uint32_t sampleRate, uint32_t channels, uint32_t samplesPerCall) = 0;

//	virtual void EnableKaraokeScore(bool enable) = 0;
//	virtual bool SetScoreTimeRange(const ScoreTimeRange* pTimeRanges, int count, int8_t* pOutPitchVisualVal) = 0;
//	virtual void SetLineInfo(ScoreLineInfo* lineInfo, int lineNum) = 0;
//	virtual bool StartKaraokeScore(IKaraokeScoreNotify* notify) = 0;
//	virtual bool PauseKaraokeScore(bool enable) = 0;
//	virtual bool StopKaraokeScore() = 0;
//	virtual void ResetKaraokeScore(uint32_t seekTime) = 0;
//	virtual void EnableKaraokeScoreEx(bool enable) = 0;
//	virtual void OnKaraokePause(bool isPause) = 0;

	virtual void EnableRenderSpectrum(bool enable) = 0;
	virtual void SetRenderSpectrumUpdateIntervalMS(uint32_t ms) = 0 ;// ms:must N * 10
	virtual void SetRenderSpectrumLen(uint32_t len) = 0 ;// // len :[12-256]

	virtual void EnableCapturePcmDataCallBack(bool enable, int sampleRate, int channel) = 0 ;
	virtual void EnableRenderPcmDataCallBack(bool enable, int sampleRate, int channel) = 0 ;
	virtual void EnableCaptureVolumeCallBack(bool enable) = 0 ;

//	virtual void SetDspExEnable(bool enable) = 0;
//
//	virtual void SetDspExReverbEnable(bool enable) = 0;
//	virtual void SetDspExReverbParam(YYNreverbFv3Parameter reverbParam) = 0;
//	virtual void SetDspExLimiterEnable(bool enable) = 0;
//	virtual void SetDspExLimiterParameter(YYLimiterParameter parameter) = 0;
//	virtual void SetDspExEqEnable(bool enable) = 0;
//	virtual void SetDspExEqGains(YYEqGains gains) = 0;

	virtual void EnableSoundPositionIndication(bool enable) = 0;
	virtual void EnableAudioProcessingModule(bool enable) = 0;
	virtual void EnableDebugLoopDelay(bool enable) = 0;
    virtual void SetBehaviorEventNotify(IAudioBehaviorEventNotify *notify) = 0;

	virtual int EnableHowlingDetector(bool enable) = 0;
    virtual int EnableEchoDetector(bool enable) = 0;

    // deviceIDS: deviceIDs[len] = new char[kMAX_YY_AUDIO_DEVICE_ID_LEN]
    // deviceNames: deviceNames[len] = new char[kMAX_YY_AUDIO_DEVICE_NAME_LEN]
    // len: len <= kMAX_YY_AUDIO_DEVICE_COUNT
    virtual uint32_t EnumerateAudioDevice(char** deviceIDs, char** deviceNames, uint32_t len, bool input) = 0;
    //deviceIDBuffer = new char[kMAX_YY_AUDIO_DEVICE_ID_LEN]
    virtual bool GetCurrentAudioDevice(char* deviceIDBuffer, bool input) = 0;
    virtual bool SetCurrentAudioDevice(char* deviceID, bool input) = 0;
    virtual bool GetDefaultAudioDevice(char* deviceIDBuffer, bool input) = 0;

    // volume [0-100]
    //return : succeed :0 failed: -1
    virtual int32_t SetCurrentMicDeviceVolume(uint32_t volume) = 0;
    virtual int32_t GetCurrentMicDeviceVolume(uint32_t &volume) = 0;
    virtual int32_t SetCurrentMicDeviceMute(bool mute) = 0;
    virtual int32_t GetCurrentMicDeviceMute(bool &mute) = 0;
    virtual int32_t SetCurrentSpeakerDeviceVolume(uint32_t volume) = 0;
    virtual int32_t GetCurrentSpeakerDeviceVolume(uint32_t &volume) = 0;
    virtual int32_t SetCurrentSpeakerDeviceMute(bool mute) = 0;
    virtual int32_t GetCurrentSpeakerDeviceMute(bool &mute) = 0;

    //Default use 'ThunderboltAudioDevice' to record
    virtual bool SetBackgroundSoundRecordDevice(char* deviceID) = 0;
    virtual void StartRecordBackgroundSound() = 0;
    virtual void StopRecordBackgroundSound() = 0;

    virtual bool GetAudioBaseStatics(bool bUpload, uint32_t sendSeq, char* data, uint32_t* dataLength) = 0;
    virtual bool GetAudioModuleStatics(bool bUpload, bool bKeyStat, uint32_t uid, char* data, uint32_t*
    dataLength) = 0;
};

// TODO: remove arg, arglen 
AUDIO_ENGINE_EXPORT IAudioEngine* CreateAudioEngine(const char* arg, int arglen, const char* logPath = NULL, LogCallback logFunc = NULL);

AUDIO_ENGINE_EXPORT void SetAndroidAudioDeviceObjects(void* javaVM, void* env, void* context);//must be call before CreateAudioEngine
AUDIO_ENGINE_EXPORT void UseAndroidOpenSlEs(bool enable);
AUDIO_ENGINE_EXPORT void EnableOpenSlEsInputJavaOutput(bool enable);
AUDIO_ENGINE_EXPORT void EnableOpenSlEsOutputNewLogic(bool enable);
AUDIO_ENGINE_EXPORT void SetAndroidOpenSlEsParam(int inputSLNumBuffers, int inputSL10msBuffers, int outputSLNumBuffers, int outputSL10msBuffers, int openslesPlayerBuffers);
AUDIO_ENGINE_EXPORT void SetAudioAdaptationConfig(int* array, uint32_t length);
AUDIO_ENGINE_EXPORT void SetAudioKaraokeCompValue(int value);

AUDIO_ENGINE_EXPORT AudioEngineFeature_t GetAudioEngineFeature();

AUDIO_ENGINE_EXPORT const char* GetSdkVersion();

AUDIO_ENGINE_EXPORT void RunTestCase();

AUDIO_ENGINE_EXPORT void OutputDebugInfo(const char* format, ...);

AUDIO_ENGINE_EXPORT uint32_t GetExactTickCount();

AUDIO_ENGINE_EXPORT int ConvertFlvAudioObjectType(uint8_t audioObjectType, uint8_t sampleRate, uint8_t channels);

//AUDIO_ENGINE_EXPORT void setAudioConfig(const char* config, int len);
//AUDIO_ENGINE_EXPORT const char* getAudioConfig();

AUDIO_ENGINE_EXPORT void WavShowInfo(const char* wavFileName, uint32_t width, uint32_t height, uint16_t* wavInfo);  // only support mono;
AUDIO_ENGINE_EXPORT bool TransAudioFileToWav(const char* inPath, const char* outPath, int32_t duration);
AUDIO_ENGINE_EXPORT int TransPCM2AAC(char* pcmPtr, int dataSize, int sampleRate, int channel, std::string& aacStr);
AUDIO_ENGINE_EXPORT long GetAudioFileTime(const char* inPath);
AUDIO_ENGINE_EXPORT int TransYYStreamToAAC(const char*srcBuffer, int len, unsigned char**dstBuffer);
namespace YYAudio
{
	AUDIO_ENGINE_EXPORT AudioStreamFormat GetAudioFormat(int netCodecId);
    AUDIO_ENGINE_EXPORT bool IsSupportCodec(int codecId);
    AUDIO_ENGINE_EXPORT int GetDataFrameLength(AudioStreamFormat& infmt, void *data, uint32_t *outLength);
}

#endif
