/*
 * Copyright (C) 2013 Andreas Stuetz <andreas.stuetz@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.yy.mobile.ui.widget.tabstrip;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Typeface;
import android.os.Build;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.v4.view.ViewPager;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.View;
import android.view.ViewTreeObserver;
import android.widget.FrameLayout;
import android.widget.HorizontalScrollView;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.yy.mobile.logs.MLog;
import com.yy.mobile.plugin.pluginunionhomepage.R;
import com.yy.mobile.utils.DensityUtil;
import com.yy.mobile.utils.ScreenUtil;
import com.yy.mobile.utils.StringUtil;
import com.yy.mobile.utils.TextViewUtil;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

public class PagerSlidingTabStrip extends HorizontalScrollView {
    private static final String TAG = "PagerSlidingTabStrip";


    public interface IconTabProvider {
        public int getPageIconResId(int position);
    }

    public interface CustomTabProvider {
        public View getPageViewLayout(int position);
    }

    public interface IndicatorSizeFetcher {
        public float getLength(int position);
    }

    public interface CustomTabDecorator {
        public void onSelected(int position, boolean selected, View tab);
    }

    public interface CustomTabUpdateTextDecorator {
        public void updateViewText(int position, String text, int resourcesId, View tab);
    }

    OnClickCallBack onClickCallBack;

    public interface OnClickCallBack {
        void notifyScrollToHead(int pos);
    }

    public void setOnClickCallBack(OnClickCallBack callBack) {
        onClickCallBack = callBack;
    }

    // @formatter:off
    private static final int[] ATTRS = new int[]{
            android.R.attr.textSize,
            android.R.attr.textColor
    };
    // @formatter:on
    private LinearLayout.LayoutParams matchparentTabLayoutParams;

    private LinearLayout.LayoutParams defaultTabLayoutParams;
    private LinearLayout.LayoutParams expandedTabLayoutParams;

    private final PageListener pageListener = new PageListener();
    public SlidingTabListener delegatePageListener;

    private LinearLayout tabsContainer;
    private ViewPager pager;

    private final String FADING_NORMAL = "normal";
    private final String FADING_SELECTED = "selected";

    private int tabCount;
    private int currentPosition = 0;
    private float currentPositionOffset = 0f;
    private int selectedPosition = 0;

    private Paint rectPaint;
    private Paint dividerPaint;
    private RectF rect;

    private int indicatorColor = 0xFFFF8900;
    private int underlineColor = 0xFFFFDD00;
    private int dividerColor = 0xFFfffefd;
    private boolean smoothScroll = false;

    private boolean shouldExpand = false;
    private boolean expandWrap = false;
    private boolean textAllCaps = false;

    private int scrollOffset;
    private int indicatorHeight = 3;
    private int underlineHeight = 0;
    private int dividerPadding = 12;
    private int tabPadding = 16;
    private int indicatorPadding = 5;
    private int dividerWidth = 1;
    private int lineBottomPadding = 4;

    private float rectOvalRx = 2;
    private float rectOvalRy = 2;

    private int oldPage;
    private int tabTextSize = 16;
    private int tabPressTextSize = 16;

//    private int tabTextColor = 0xFF666666;
//    private int tabPressTextColor = 0xFF000000;

    private int tabTextColor = 0xFF999999;
    private int tabPressTextColor = 0xFF1D1D1D;

    private Typeface tabTypeface = null;
    private int tabTypefaceStyle = Typeface.NORMAL;
    private int lastScrollX = 0;

    private boolean centerMode = false;

    private int tabBackgroundResId = R.drawable.hp_pst_background_tab;

    private CustomTabDecorator tabDecorator;

    private CustomTabUpdateTextDecorator tabUpdateTextDecorator;

    private Locale locale;

    private Context context;

    List<Map<String, TextView>> tabViews = new ArrayList<>();
    private boolean mFadeEnabled = false;
    private boolean useFadeEffect = false;
    private boolean hasOffset = false;
    private int paddingLeft;
    private boolean mFirstLayout;

    private MyViewTreeGlobalLayoutObserver mMyGlobalLayoutObserver;

    public PagerSlidingTabStrip(Context context) {
        this(context, null);
    }

    public PagerSlidingTabStrip(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public PagerSlidingTabStrip(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        this.context = context;
        setFillViewport(true);
        setWillNotDraw(false);

        tabsContainer = new LinearLayout(context);
        tabsContainer.setOrientation(LinearLayout.HORIZONTAL);
        tabsContainer.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));

        DisplayMetrics dm = getResources().getDisplayMetrics();
        scrollOffset = ScreenUtil.INSTANCE.getScreenWidth(context);


//        scrollOffset = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, scrollOffset, dm);
        indicatorHeight = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, indicatorHeight, dm);
        underlineHeight = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, underlineHeight, dm);
        dividerPadding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dividerPadding, dm);
        tabPadding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, tabPadding, dm);
        indicatorPadding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, indicatorPadding, dm);
        dividerWidth = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dividerWidth, dm);
        tabTextSize = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, tabTextSize, dm);
        tabPressTextSize = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, tabPressTextSize, dm);

        lineBottomPadding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, lineBottomPadding, dm);
        // get system attrs (android:textSize and android:textColor)
        rectOvalRx = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, rectOvalRx, dm);
        rectOvalRy = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, rectOvalRy, dm);

        TypedArray a = context.obtainStyledAttributes(attrs, ATTRS);
        tabTextSize = a.getDimensionPixelSize(0, tabTextSize);
        tabPressTextSize = a.getDimensionPixelSize(0, tabPressTextSize);
        tabTextColor = a.getColor(1, tabTextColor);//NOSONAR

        a.recycle();

        // get custom attrs

        a = context.obtainStyledAttributes(attrs, R.styleable.PagerSlidingTabStrip);

        indicatorColor = a.getColor(R.styleable.PagerSlidingTabStrip_pstsIndicatorColor, indicatorColor);
//        underlineColor = a.getColor(R.styleable.PagerSlidingTabStrip_pstsUnderlineColor, underlineColor);
        dividerColor = a.getColor(R.styleable.PagerSlidingTabStrip_pstsDividerColor, dividerColor);
        indicatorHeight = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsIndicatorHeight, indicatorHeight);
        underlineHeight = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsUnderlineHeight, underlineHeight);
        dividerPadding = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsDividerPadding, dividerPadding);
        tabPadding = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsTabPaddingLeftRight, tabPadding);
        indicatorPadding = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsIndicatorPadding, indicatorPadding);
        tabBackgroundResId = a.getResourceId(R.styleable.PagerSlidingTabStrip_pstsTabBackground, tabBackgroundResId);
        shouldExpand = a.getBoolean(R.styleable.PagerSlidingTabStrip_pstsShouldExpand, shouldExpand);
        expandWrap = a.getBoolean(R.styleable.PagerSlidingTabStrip_pstsExpandWrap, expandWrap);
//        scrollOffset = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsScrollOffset, scrollOffset);
        textAllCaps = a.getBoolean(R.styleable.PagerSlidingTabStrip_pstsTextAllCaps, textAllCaps);
        smoothScroll = a.getBoolean(R.styleable.PagerSlidingTabStrip_pstsSmoothScroll, smoothScroll);
        tabTextSize = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsTextSize, tabTextSize);
        tabPressTextSize = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsPressTextSize, tabPressTextSize);
        lineBottomPadding = a.getDimensionPixelSize(R.styleable.PagerSlidingTabStrip_pstsLineBottomPadding, lineBottomPadding);
        centerMode = a.getBoolean(R.styleable.PagerSlidingTabStrip_pstsCenterMode, false);
        tabTextColor = a.getColor(R.styleable.PagerSlidingTabStrip_pstsTextColor, tabTextColor);
        tabPressTextColor = a.getColor(R.styleable.PagerSlidingTabStrip_pstsTextPressColor, tabPressTextColor);

        a.recycle();

        rect = new RectF();
        rectPaint = new Paint();
        rectPaint.setAntiAlias(true);
        rectPaint.setStyle(Paint.Style.FILL);

        dividerPaint = new Paint();
        dividerPaint.setAntiAlias(true);
        dividerPaint.setStrokeWidth(dividerWidth);

        paddingLeft = getPaddingLeft();
        defaultTabLayoutParams = new LinearLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.MATCH_PARENT);
        expandedTabLayoutParams = new LinearLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.MATCH_PARENT, 1.0f);

        matchparentTabLayoutParams = new LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);

        if (locale == null) {
            locale = getResources().getConfiguration().locale;
        }

        if (centerMode) {
            tabsContainer.setGravity(Gravity.CENTER);
        }
        addView(tabsContainer);
    }

    public void setTabTextColor(int tabTextColor) {
        this.tabTextColor = tabTextColor;
    }

    public void setViewPager(ViewPager pager) {
        this.pager = pager;
        if (pager.getAdapter() == null) {
            throw new IllegalStateException("ViewPager does not have adapter instance.");
        }
        mFirstLayout = true;
        pager.setOnPageChangeListener(pageListener);
        notifyDataSetChanged();
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        mFirstLayout = true;
    }

    public void setOnPageChangeListener(SlidingTabListener listener) {
        this.delegatePageListener = listener;
    }

    public void notifyDataSetChanged() {
        tabsContainer.removeAllViews();
        tabCount = pager.getAdapter().getCount();
        for (int i = 0; i < tabCount; i++) {
            if (pager.getAdapter() instanceof IconTabProvider) {
                addIconTab(i, ((IconTabProvider) pager.getAdapter()).getPageIconResId(i));
            } else if (pager.getAdapter() instanceof CustomTabProvider) {
                addTab(i, ((CustomTabProvider) pager.getAdapter()).getPageViewLayout(i));
            } else {
                if (mFadeEnabled) {
                    addFadingTextTab(i, pager.getAdapter().getPageTitle(i).toString());
                } else {
                    addTextTab(i, pager.getAdapter().getPageTitle(i).toString());
                }
            }
        }

        if (mMyGlobalLayoutObserver == null) {
            mMyGlobalLayoutObserver = new MyViewTreeGlobalLayoutObserver(this);
            getViewTreeObserver().addOnGlobalLayoutListener(mMyGlobalLayoutObserver);
        }
    }

    private void handleGlobalLayout() {
        currentPosition = pager.getCurrentItem();
        scrollToChild(currentPosition, 0);
        updateTab();
        mMyGlobalLayoutObserver = null;
    }

    private static class MyViewTreeGlobalLayoutObserver implements ViewTreeObserver.OnGlobalLayoutListener {
        private WeakReference<PagerSlidingTabStrip> mPagerSlidingTabStrip;
        public MyViewTreeGlobalLayoutObserver(PagerSlidingTabStrip pagerSlidingTabStrip) {
            mPagerSlidingTabStrip = new WeakReference<PagerSlidingTabStrip>(pagerSlidingTabStrip);
        }

        @SuppressWarnings("deprecation")
        @SuppressLint("NewApi")
        @Override
        public void onGlobalLayout() {
            if (mPagerSlidingTabStrip != null) {
                PagerSlidingTabStrip tabStrip = mPagerSlidingTabStrip.get();
                if (tabStrip != null) {
                    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN) {
                        tabStrip.getViewTreeObserver().removeGlobalOnLayoutListener(this);
                    } else {
                        tabStrip.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                    }

                    tabStrip.handleGlobalLayout();
                }
            }
        }
    }

    private void addTextTab(final int position, String title) {
        TextView tab = new TextView(getContext());
        tab.setText(title);
        tab.setGravity(Gravity.CENTER);
        tab.setSingleLine();

        addTab(position, tab);
    }

    private void addFadingTextTab(final int position, String title) {
        TextView tab = new TextView(getContext());
        tab.setText(title);
        tab.setGravity(Gravity.CENTER);
        tab.setSingleLine();
        tab.setTextColor(tabTextColor);
        tab.setTextSize(TypedValue.COMPLEX_UNIT_PX, tabTextSize);

        TextView tab2 = new TextView(getContext());
        tab2.setText(title);
        tab2.setGravity(Gravity.CENTER);
        tab2.setSingleLine();
        tab2.setTextColor(tabPressTextColor);
        tab2.setTextSize(TypedValue.COMPLEX_UNIT_PX, tabTextSize);
        tab2.setTypeface(null, Typeface.BOLD);

        FrameLayout framelayout = new FrameLayout(context);
        framelayout.addView(tab, 0, matchparentTabLayoutParams);
        framelayout.addView(tab2, 1, matchparentTabLayoutParams);

        Map<String, TextView> map = new HashMap<>();
        tab.setAlpha(1);
        map.put(FADING_NORMAL, tab);

        tab2.setAlpha(0);
        map.put(FADING_SELECTED, tab2);
        tabViews.add(position, map);
        addTab(position, framelayout);
    }

    private Rect getTextRect(String text) {
        if (TextUtils.isEmpty(text))
            return new Rect();
        try {
            text = StringUtil.INSTANCE.filterSpecificChar(text);
        } catch (Exception e) {
            MLog.INSTANCE.error(TAG, "getTextWidth text filter err = " + e.getMessage());
        }
        return TextViewUtil.INSTANCE.getTextRect(text, tabTextSize);
    }

    private int getTextHeight(String text) {
        return TextViewUtil.INSTANCE.getTextRect(text, tabTextSize).height();
    }

    private void addIconTab(final int position, int resId) {
        ImageButton tab = new ImageButton(getContext());
        tab.setImageResource(resId);
        addTab(position, tab);
    }

    public void setReddotMargin(int left, int top, int right, int bottom) {
        int count = tabsContainer.getChildCount();
        for (int i = 0; i < count; i++) {
            View view = tabsContainer.getChildAt(i).findViewById(R.id.hp_red_dot);
            if (view instanceof BadgeView) {
                BadgeView badgeView = (BadgeView) view;
                badgeView.setBadgeMargin(left, top, right, bottom);
            }
        }
    }

    private void addTab(final int position, View tab) {
        tab.setFocusable(true);
        tab.setId(R.id.hp_tab_content);
        tab.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (pager.getCurrentItem() == position && onClickCallBack != null) {
                    onClickCallBack.notifyScrollToHead(position);
                }
                if (mOnTabClickListener != null) {
                    mOnTabClickListener.onClick(position, pager.getCurrentItem());
                }
                if (pager != null) {
                    pager.setCurrentItem(position, smoothScroll);
                }
                currentPosition = position;
                scrollToChild(position, 0);
            }
        });
        tab.setPadding(tabPadding, 0, tabPadding, 0);
        tab.setTag(position);
        tabsContainer.addView(tab, position, shouldExpand ? expandedTabLayoutParams : defaultTabLayoutParams);
        BadgeView badgeView = new BadgeView(getContext());
        badgeView.setBadgeGravity(Gravity.CENTER);
        badgeView.setTargetView(tab);
        Rect rect = getContentTextRect(position);
        // 红点一定在文字的右上角
        int paddingleft = rect.width() / 2 + DensityUtil.dp2px(6.5f);
        int paddingbottom = rect.height() / 2 + DensityUtil.dp2px(5f);
        badgeView.setBadgeMarginPx(paddingleft, 0, 0, paddingbottom);
    }

    //旧效果使用
    private void updateTabStyles() {
        for (int i = 0; i < tabCount; i++) {
            View v = tabsContainer.getChildAt(i).findViewById(R.id.hp_tab_content);
            v.setBackgroundResource(tabBackgroundResId);
            boolean selected;
            if (i == currentPosition) {
                selected = true;
            } else {
                selected = false;
            }

            if (tabDecorator != null) {
                tabDecorator.onSelected(i, selected, v);
            }

            if (v instanceof TextView) {
                TextView tab = (TextView) v;
                tab.setTypeface(tabTypeface, tabTypefaceStyle);

                if (i == currentPosition) {
                    tab.setTextColor(tabPressTextColor);
                    tab.setTextSize(TypedValue.COMPLEX_UNIT_PX, tabPressTextSize);
                } else {
                    tab.setTextColor(tabTextColor);
                    tab.setTextSize(TypedValue.COMPLEX_UNIT_PX, tabTextSize);
                }

                // setAllCaps() is only available from API 14, so the upper case is made manually if we are on a
                // pre-ICS-build
                if (textAllCaps) {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH) {
                        tab.setAllCaps(true);
                    } else {
                        tab.setText(tab.getText().toString().toUpperCase(locale));
                    }
                }
            }
        }
    }

    public void setPressTextSize(int textSize) {
        tabPressTextSize = textSize;
        updateTabStyles();
    }

    private void updateTabTextStyles() {
        for (int i = 0; i < tabCount; i++) {
            if (tabsContainer.getChildAt(i) != null && !tabViews.isEmpty()) {
                if (i == selectedPosition) {
                    tabViews.get(i).get(FADING_NORMAL).setAlpha(0);
                    tabViews.get(i).get(FADING_SELECTED).setAlpha(1);
                } else {
                    tabViews.get(i).get(FADING_NORMAL).setAlpha(1);
                    tabViews.get(i).get(FADING_SELECTED).setAlpha(0);
                }
            }
        }
    }

    private void updateTab() {
        if (mFadeEnabled) {
            updateTabTextStyles();
        } else {
            updateTabStyles();
        }
    }

    private void scrollToChild(int position, int offset) {
        if (tabCount == 0) {
            return;
        }
        int newScrollX = 0;
        View childAt = tabsContainer.getChildAt(position);
        int textLength = 0;
        if (childAt != null) {
            textLength = childAt.getWidth() / 2;
        }
        //position == 0 不滚动
        if (position > 0) {
            int left = 0;
            if (childAt != null) {
                left = childAt.getLeft();
            }
            newScrollX = left + offset + textLength;
        } else {
            if (childAt != null) {
                newScrollX = childAt.getLeft();
            }
        }

        if (position > 0 || offset > 0) {
            if (hasOffset) {
                newScrollX = newScrollX - scrollOffset;
            } else {
                newScrollX = newScrollX - scrollOffset / 2;
            }

        }
        if (newScrollX != lastScrollX) {
            lastScrollX = newScrollX;
            smoothScrollTo(newScrollX, 0);
        }

    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        super.onLayout(changed, l, t, r, b);
        if (mFirstLayout && selectedPosition == 0 && delegatePageListener != null) {
            delegatePageListener.onPageSelected(0, 0);
        }
        mFirstLayout = false;
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        if (isInEditMode() || tabCount == 0) {
            return;
        }

        final int height = getHeight();

        // draw indicator line

        rectPaint.setColor(underlineColor);

        // default: line below current tab
        if (tabsContainer.getChildAt(currentPosition) == null) {
            currentPosition = 0;
        }
        View currentTab = tabsContainer.getChildAt(currentPosition);
        float lineLeft = currentTab.getLeft();
        float lineRight = currentTab.getRight();

        float currentTextLong = mFetcher.getLength(currentPosition);
        //底部线条长度固定
//        float currentTextLong = mFetcherDefault.getDefaultLength();

        float paddingLong = (((lineRight - lineLeft) - currentTextLong) / 2) - indicatorPadding;

        lineLeft = lineLeft + paddingLong;
        lineRight = lineRight - paddingLong;
        // if there is an offset, start interpolating left and right coordinates between current and next tab
        if (currentPositionOffset > 0f && currentPosition < tabCount - 1) {

            View nextTab = tabsContainer.getChildAt(currentPosition + 1);
            final float nextTabLeft = nextTab.getLeft();
            final float nextTabRight = nextTab.getRight();
            float nextTextLong = mFetcher.getLength(currentPosition + 1);
            //底部线条长度固定
//            float nextTextLong = mFetcherDefault.getDefaultLength();
            float nextpaddingLong = (((nextTabRight - nextTabLeft) - nextTextLong) / 2) - indicatorPadding;

            lineLeft = (currentPositionOffset * (nextTabLeft + nextpaddingLong) + (1f - currentPositionOffset) * lineLeft);
            lineRight = (currentPositionOffset * (nextTabRight - nextpaddingLong) + (1f - currentPositionOffset) * lineRight);

        }
//        if (!useFadeEffect) {
        //去掉底部线
//            canvas.drawRect(lineLeft, height - indicatorHeight, lineRight, height, rectPaint);
        // draw underline
           /* rectPaint.setColor(underlineColor);
            canvas.drawRect(0, height - underlineHeight, tabsContainer.getWidth(), height, rectPaint);*/
        rect.left = lineLeft + paddingLeft;
        rect.top = height - lineBottomPadding - indicatorHeight;
        rect.right = lineRight + paddingLeft;
        rect.bottom = height - lineBottomPadding;
        canvas.drawRoundRect(rect, rectOvalRx, rectOvalRy, rectPaint);
    }

    private boolean needAnimate = true;

    private class PageListener implements ViewPager.OnPageChangeListener {

        @Override
        public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
            currentPosition = position;
            currentPositionOffset = positionOffset;
            if (mFadeEnabled) {
                if (mState == State.IDLE && positionOffset > 0) {
                    oldPage = pager.getCurrentItem();
                    mState = position == oldPage ? State.GOING_RIGHT : State.GOING_LEFT;
                }
                boolean goingRight = position == oldPage;
                if (mState == State.GOING_RIGHT && !goingRight)
                    mState = State.GOING_LEFT;
                else if (mState == State.GOING_LEFT && goingRight)
                    mState = State.GOING_RIGHT;
                float effectOffset = isSmall(positionOffset) ? 0 : positionOffset;

                View mLeft = tabsContainer.getChildAt(position);
                View mRight = tabsContainer.getChildAt(position + 1);
                if (effectOffset == 0) {
                    mState = State.IDLE;
                }
                if (needAnimate) {
                    animateFadeScale(mLeft, mRight, effectOffset, position);
                }
                View v = tabsContainer.getChildAt(position);
                int width = 0;
                if (v != null) {
                    width = v.getWidth();
                }
                scrollToChild(position, (int) (positionOffset * width));
                invalidate();
            } else {
                View v = tabsContainer.getChildAt(position);
                int width = 0;
                if (v != null) {
                    width = v.getWidth();
                }
                scrollToChild(position, (int) (positionOffset * width));
                invalidate();

            }
            if (delegatePageListener != null) {
                delegatePageListener.onPageScrolled(position, positionOffset, positionOffsetPixels);
            }
        }

        @Override
        public void onPageScrollStateChanged(int state) {
            if (useFadeEffect) {
                if (state == ViewPager.SCROLL_STATE_IDLE) {
                    needAnimate = true;
                    scrollToChild(pager.getCurrentItem(), 0);
                    mFadeEnabled = true;
//                    MLog.error(TAG, " onPageScrollStateChanged = SCROLL_STATE_IDLE");
                } else if (state == ViewPager.SCROLL_STATE_DRAGGING) {
//                    MLog.error(TAG, " onPageScrollStateChanged = SCROLL_STATE_DRAGGING");
                } else if (state == ViewPager.SCROLL_STATE_SETTLING) {
//                    MLog.error(TAG, " onPageScrollStateChanged = SCROLL_STATE_SETTLING");
                }
            }
            if (delegatePageListener != null) {
                delegatePageListener.onPageScrollStateChanged(state);
            }
        }

        @Override
        public void onPageSelected(int position) {
            int oldPosition = selectedPosition;
            selectedPosition = position;
            if (useFadeEffect) {
                updateTabTextStyles();
                needAnimate = false;
            } else {
                for (int i = 0; i <= tabCount - 1; i++) {
                    View tab = tabsContainer.getChildAt(i).findViewById(R.id.hp_tab_content);
                    if (tab instanceof TextView) {
                        if (i == position) {
                            ((TextView) tab).setTextColor(tabPressTextColor);
                            ((TextView) tab).setTextSize(TypedValue.COMPLEX_UNIT_PX, tabPressTextSize);
                        } else {
                            ((TextView) tab).setTextColor(tabTextColor);
                            ((TextView) tab).setTextSize(TypedValue.COMPLEX_UNIT_PX, tabTextSize);
                        }
                    }
                    if (tabDecorator != null) {
                        tabDecorator.onSelected(i, i == position, tab);
                    }
                }
            }
            if (delegatePageListener != null) {
                delegatePageListener.onPageSelected(oldPosition, position);
            }
        }

    }

    /**
     * 显示or隐藏红点
     *
     * @param index
     * @param isShow
     */
    public void showOrHideRed(int index, boolean isShow) {
        if (index < tabCount && index >= 0) {
            View v = tabsContainer.getChildAt(index).findViewById(R.id.hp_red_dot);
            if (v instanceof BadgeView) {
                BadgeView badge = (BadgeView) v;
                badge.showBadgeRedDot(isShow);
            }
        }
    }

    public void setLineBottomPadding(int lineBottomPadding) {
        this.lineBottomPadding = lineBottomPadding;
        invalidate();
    }

    public int getLineBottomPadding() {
        return lineBottomPadding;
    }

    public void setIndicatorColor(int indicatorColor) {
        this.indicatorColor = indicatorColor;
        invalidate();
    }

    public void setIndicatorColorResource(int resId) {
        this.indicatorColor = getResources().getColor(resId);
        invalidate();
    }

    public int getIndicatorColor() {
        return this.indicatorColor;
    }

    public void setIndicatorHeight(int indicatorLineHeightPx) {
        this.indicatorHeight = indicatorLineHeightPx;
        invalidate();
    }

    public int getIndicatorHeight() {
        return indicatorHeight;
    }

    public void setUnderlineColor(int underlineColor) {
        this.underlineColor = underlineColor;
        invalidate();
    }

    public void setUnderlineColorResource(int resId) {
        this.underlineColor = getResources().getColor(resId);
        invalidate();
    }

    public int getUnderlineColor() {
        return underlineColor;
    }

    public void setDividerColor(int dividerColor) {
        this.dividerColor = dividerColor;
        invalidate();
    }

    public void setDividerColorResource(int resId) {
        this.dividerColor = getResources().getColor(resId);
        invalidate();
    }

    public int getDividerColor() {
        return dividerColor;
    }

    public void setUnderlineHeight(int underlineHeightPx) {
        this.underlineHeight = underlineHeightPx;
        invalidate();
    }

    public int getUnderlineHeight() {
        return underlineHeight;
    }

    public void setDividerPadding(int dividerPaddingPx) {
        this.dividerPadding = dividerPaddingPx;
        invalidate();
    }

    public int getDividerPadding() {
        return dividerPadding;
    }

    public void setScrollOffset(int scrollOffsetPx) {
        this.scrollOffset = scrollOffsetPx;
        invalidate();
    }

    public int getScrollOffset() {
        return scrollOffset;
    }

    public void minusOffset(int minusOffset) {
        hasOffset = true;
        scrollOffset = (scrollOffset - minusOffset) / 2;
    }

    public void setShouldExpand(boolean shouldExpand) {
        this.shouldExpand = shouldExpand;
        requestLayout();
    }

    public boolean getShouldExpand() {
        return shouldExpand;
    }

    public boolean isTextAllCaps() {
        return textAllCaps;
    }

    public void setAllCaps(boolean textAllCaps) {
        this.textAllCaps = textAllCaps;
    }

    public void setTextSize(int textSizePx) {
        this.tabTextSize = textSizePx;
        updateTab();
    }

    public int getTextSize() {
        return tabTextSize;
    }

    public int getPressTextSize() {
        return tabPressTextSize;
    }

    public void setTextColor(int textColor) {
        this.tabTextColor = textColor;
        if (mFadeEnabled) {
            for (int i = 0; i < tabCount; i++) {
                if (tabsContainer.getChildAt(i) != null && !tabViews.isEmpty()) {
                    if (i == selectedPosition) {
                        tabViews.get(i).get(FADING_NORMAL).setTextColor(tabTextColor);
                    }
                }
            }
        }
        updateTab();
    }

    public void setPressTextColor(int textColor) {
        this.tabPressTextColor = textColor;
        if (mFadeEnabled) {
            for (int i = 0; i < tabCount; i++) {
                if (tabsContainer.getChildAt(i) != null && !tabViews.isEmpty()) {
                    tabViews.get(i).get(FADING_SELECTED).setTextColor(tabPressTextColor);
                }
            }
        }
        updateTab();
    }

    public void setTextColorResource(int resId) {
        this.tabTextColor = getResources().getColor(resId);
        updateTabStyles();
    }

    public int getTextColor() {
        return tabTextColor;
    }

    public void setTypeface(Typeface typeface, int style) {
        this.tabTypeface = typeface;
        this.tabTypefaceStyle = style;
        updateTabStyles();
    }

    public void setTabBackground(int resId) {
        this.tabBackgroundResId = resId;
    }

    public int getTabBackground() {
        return tabBackgroundResId;
    }

    /**
     * update text in tab
     *
     * @param position
     * @param text
     */
    public void updateTextStyleTitle(int position, String text) {
        if (pager.getAdapter() instanceof IconTabProvider || pager.getAdapter() instanceof CustomTabProvider) {
            //Do nothing
        } else {
            View selectTab = tabsContainer.getChildAt(position).findViewById(R.id.hp_tab_content);
            if (selectTab instanceof TextView) {
                ((TextView) selectTab).setText(text);
                selectTab.setPadding(0, 0, 0, 0);
            }
        }
    }

    /**
     * update text in tab
     *
     * @param position
     * @param text
     */
    public void updateTextStyleTitle(int position, String text, int resourcesId) {
        if (pager.getAdapter() instanceof IconTabProvider || pager.getAdapter() instanceof CustomTabProvider) {
            if (tabUpdateTextDecorator != null) {
                View selectTab = tabsContainer.getChildAt(position);
                tabUpdateTextDecorator.updateViewText(position, text, resourcesId, selectTab);
            }
        } else {
            View selectTab = tabsContainer.getChildAt(position);
            if (selectTab instanceof TextView) {
                ((TextView) selectTab).setText(text);
                ((TextView) selectTab).setPadding(0, 0, 0, 0);
            }
        }
    }


    public View getPositionView(int position) {
        return tabsContainer.getChildAt(position);
    }


    public void setTabPaddingLeftRight(int paddingPx) {
        this.tabPadding = paddingPx;
        updateTabStyles();
    }

    public int getTabPaddingLeftRight() {
        return tabPadding;
    }

    private IndicatorSizeFetcher mFetcher = new IndicatorSizeFetcher() {
        @Override
        public float getLength(int position) {
            return TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, 14, getResources().getDisplayMetrics());
        }
    };

    // 自适应长度的实现样板
    private IndicatorSizeFetcher mAutoAdjustFetcher = new IndicatorSizeFetcher() {
        @Override
        public float getLength(int position) {
            float width = getContentTextRect(position).width();
            if (width != -1) {
                return width;
            }
            return TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, 14, getResources().getDisplayMetrics());
        }
    };

    private Rect getContentTextRect(int position) {
        View tab = tabsContainer.getChildAt(position).findViewById(R.id.hp_tab_content);
        if (tab != null && tab instanceof TextView) {
            return getTextRect(((TextView) tab).getText().toString());
        } else if (tab != null && tab instanceof FrameLayout) {
            FrameLayout tabFl = (FrameLayout) tab;
            if (tabFl.getChildAt(0) != null && tabFl.getChildAt(0) instanceof TextView) {
                return getTextRect(((TextView) tabFl.getChildAt(0)).getText().toString());
            }
        }
        return new Rect();
    }

    public void setIndicatotLengthFetcher(IndicatorSizeFetcher fetcher) {
        this.mFetcher = fetcher;
    }

    public void setTabDecorator(CustomTabDecorator decorator) {
        this.tabDecorator = decorator;
    }

    public void setTabUpdateTextDecorator(CustomTabUpdateTextDecorator decorator) {
        this.tabUpdateTextDecorator = decorator;
    }

    public void setFadeEnabled(boolean enabled) {
        mFadeEnabled = enabled;
    }

    public void setUseFadeEffect(boolean mUseFadeEffect) {
        this.useFadeEffect = mUseFadeEffect;
    }

    public void setZoomMax(float zoomMax) {
        this.zoomMax = zoomMax;
    }

    private boolean isSmall(float positionOffset) {
        return Math.abs(positionOffset) < 0.0001;
    }

    private float zoomMax = 0.2f;
    private State mState;

    private enum State {
        IDLE, GOING_LEFT, GOING_RIGHT
    }

    protected void animateFadeScale(View left, View right, float positionOffset, int position) {
        if (mState != State.IDLE) {
            int tabSize = tabViews.size();
            if (left != null && tabSize != 0 && tabSize >= position) {
                tabViews.get(position).get(FADING_NORMAL).setAlpha(positionOffset);
                tabViews.get(position).get(FADING_SELECTED).setAlpha(1 - positionOffset);
            }
            if (right != null && tabSize != 0 && tabSize >= (position + 1)) {
                tabViews.get(position + 1).get(FADING_NORMAL).setAlpha(1 - positionOffset);
                tabViews.get(position + 1).get(FADING_SELECTED).setAlpha(positionOffset);
            }
        }
    }

    @Override
    public void onRestoreInstanceState(Parcelable state) {
        SavedState savedState = (SavedState) state;
        super.onRestoreInstanceState(savedState.getSuperState());
        currentPosition = savedState.currentPosition;
        requestLayout();
    }

    @Override
    public Parcelable onSaveInstanceState() {
        Parcelable superState = super.onSaveInstanceState();
        SavedState savedState = new SavedState(superState);
        savedState.currentPosition = currentPosition;
        return savedState;
    }

    static class SavedState extends BaseSavedState {
        int currentPosition;

        public SavedState(Parcelable superState) {
            super(superState);
        }

        private SavedState(Parcel in) {
            super(in);
            currentPosition = in.readInt();
        }

        @Override
        public void writeToParcel(Parcel dest, int flags) {
            super.writeToParcel(dest, flags);
            dest.writeInt(currentPosition);
        }

        public static final Creator<SavedState> CREATOR = new Creator<SavedState>() {
            @Override
            public SavedState createFromParcel(Parcel in) {
                return new SavedState(in);
            }

            @Override
            public SavedState[] newArray(int size) {
                return new SavedState[size];
            }
        };
    }

    public int getCurrentPosition() {
        if (pager == null) {
            return -1;
        } else {
            return pager.getCurrentItem();
        }
    }

    public interface SlidingTabListener {

        /**
         * This method will be invoked when the current page is scrolled, either as part
         * of a programmatically initiated smooth scroll or a user initiated touch scroll.
         *
         * @param position             Position index of the first page currently being displayed.
         *                             Page position+1 will be visible if positionOffset is nonzero.
         * @param positionOffset       Value from [0, 1) indicating the offset from the page at position.
         * @param positionOffsetPixels Value in pixels indicating the offset from position.
         */
        void onPageScrolled(int position, float positionOffset, int positionOffsetPixels);

        /**
         * This method will be invoked when a new page becomes selected. Animation is not
         * necessarily complete.
         */
        void onPageSelected(int oldPosition, int newPosition);

        /**
         * Called when the scroll state changes. Useful for discovering when the user
         * begins dragging, when the pager is automatically settling to the current page,
         * or when it is fully stopped/idle.
         *
         * @param state The new scroll state.
         * @see ViewPager#SCROLL_STATE_IDLE
         * @see ViewPager#SCROLL_STATE_DRAGGING
         * @see ViewPager#SCROLL_STATE_SETTLING
         */
        void onPageScrollStateChanged(int state);
    }

    public void setOnTabClickListener(OnTabClickListener listener) {
        mOnTabClickListener = listener;
    }

    private OnTabClickListener mOnTabClickListener;

    public interface OnTabClickListener {
        void onClick(int selectedPos, int previouPos);
    }
}
