package com.yy.mobile.presenter.homepage

import com.yy.mobile.constant.LivingCoreConstant
import com.yy.mobile.data.base.HomepageData
import com.yy.mobile.data.follow.FollowTitle
import com.yy.mobile.data.mainbean.LineDataMultiType
import com.yy.mobile.data.meipai.*
import com.yy.mobile.data.nav.LiveNavInfo
import com.yy.mobile.data.nav.SubLiveNavItem
import com.yy.mobile.logs.MLog
import com.yy.mobile.model.HomepageDataModule
import com.yy.mobile.ui.homepage.IHomeContentViewExtend
import com.yy.mobile.utils.RxUtils
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.functions.Function
import io.reactivex.schedulers.Schedulers
import java.util.concurrent.TimeUnit

/**
 * Created by PYF on 2019/3/4
 */
@Suppress("UNCHECKED_CAST")
class HomeContentPresenter(var view: IHomeContentViewExtend?) {

    var hasClick: Boolean = false// 直播列表是否发生点击事件
    private var needLoadMore = false
    private var currentLine: MPLineData<HomepageData>? = null
    private var cacheLine = mutableListOf<MPLineData<HomepageData>>()
    private val disposableList = mutableListOf<Disposable>()

    private val module: HomepageDataModule = HomepageDataModule()
    private val dataMapper = Function<MPBaseNetData, MutableList<LineDataMultiType>> { t ->
        MLog.info(TAG, "dataMapper: $t")

        // 究极转型
        t.data.apply {
            removeAll {
                // 过滤空模块
                it.data.isEmpty()
            }
            forEach {
                if (needLoadMore) {// 该模块需要分页，缓存后续模块
                    remove(it)
                    cacheLine.add(it.apply {
                        // 只缓存模块基本信息，数据清零，加载时从第一页开始加载
                        data.clear()
                        page = 0
                    })
                }
                if (it.needLoadMore()) {
                    currentLine = it
                    needLoadMore = true
                }
            }
        }
        needLoadMore = true
        currentLine = t.data.last()
        if (!needLoadMore) {
            t.data.add(MPLineData<HomepageData>().apply {
                type = LivingCoreConstant.Live_MODULE_NO_MORE_CODE
            })
        }
        // 数据加工，包装成通用数据类型
        LineDataMultiType.mpConvert(t.data)
    }

    private val moreDataMapper = Function<MPBaseNetData, MutableList<LineDataMultiType>> { t ->
        MLog.info(TAG, "moreDataMapper: $t")

        currentLine = t.data.getOrNull(0)
        if (currentLine == null || !currentLine!!.needLoadMore()) {// 当前模块加载完毕，换下一个模块加载
            if (cacheLine.isEmpty()) {// 没有缓存模块需要加载
                needLoadMore = false
            } else {
                currentLine = cacheLine[0]
                cacheLine.removeAt(0)
            }
        }
        if (!needLoadMore) {
            t.data.add(MPLineData<HomepageData>().apply {
                type = LivingCoreConstant.Live_MODULE_NO_MORE_CODE
            })
        }
        LineDataMultiType.mpConvert(t.data)
    }

    private val recommendAuthorMapper = Function<MPBaseNetData, MPRecommendDialogData> { mpBaseNetData ->
        MLog.info(TAG, "recommendAuthorMapper: $mpBaseNetData")

        val dialogData = MPRecommendDialogData()
        mpBaseNetData.data.forEach {
            when (it.type) {
                LivingCoreConstant.MP_RECOMMEND_AUTHOR_LIST_CODE ->// 推荐主播列表
                    dialogData.liveList = it.data as List<MPLiveInfoData>
                LivingCoreConstant.MP_RECOMMEND_AUTHOR_DATA_CODE ->// 弹窗相关配置
                    dialogData.apply {
                        val data = (it.data as List<MPRecommendData>).getOrNull(0)
                        noData = data == null
                        data ?: return@forEach
                        title = data.title
                        noClickTime = data.noClickTime
                        popupTimes = data.times
                        countdown = data.countdown
                    }
                LivingCoreConstant.MP_RECOMMEND_AUTHOR_DESC_CODE ->// 推荐语配置(目前用不上)
                    dialogData.descList = (it.data as List<MPRecommendDesc>)
            }
        }
        dialogData
    }

    //first
    fun requestHomePageData(loadType: String, navInfo: LiveNavInfo, subNav: SubLiveNavItem) {
        val disposable = module.requestMPHomepageData(loadType, navInfo, subNav)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .map(dataMapper)
            .subscribe({
                if ("subscribe" == navInfo.biz) {
                    MLog.info(TAG, "requestHomePageData: current is in subscribe")
                    it.add(
                        0,
                        LineDataMultiType(
                            LivingCoreConstant.Live_MODULE_COMMON_TITLE_CODE,
                            FollowTitle(6)
                        )
                    )
                }
                if (it.isEmpty()) {
                    view?.switchNoDataStatus(true)
                    return@subscribe
                }
                MLog.info(TAG, "requestHomePageData: data = $it")
                view?.switchLoadingStatus(false)
                view?.switchNoDataStatus(false)
                view?.refreshOrAddRecyclerView(true, it)
            }, {
                MLog.error(TAG, "requestHomePageData: t = $it")
                view?.switchNoDataStatus(true)
            })
        disposableList.add(disposable)
    }

    /**
     * 请求推荐主播弹窗数据
     */
    fun requestRecommendAuthor() {
        disposableList.add(
            module.requestRecommendAuthor()
                .map(recommendAuthorMapper)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe {
                    MLog.info(TAG, "requestRecommendAuthor: $it")

                    if (it.shouldShow) {
                        startCountdown(it)
                    }
                }
        )
    }

    /**
     * 开始倒计时，倒计时结束时未发生点击事件，则显示弹窗
     */
    private fun startCountdown(dialogData: MPRecommendDialogData) {
        disposableList.add(
            Observable.timer(dialogData.noClickTime.toLong(), TimeUnit.SECONDS)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(
                    {
                        if (!hasClick) {
                            view?.showRecommendAuthorDialog(dialogData)
                        }
                    },
                    {
                        MLog.error(TAG, "startCountdown: t = $it")
                    }
                )
        )
    }

    fun requestMoreHomePageData() {
        if (!needLoadMore || currentLine == null) {
            view?.finishRefreshAnimation(true, 100)
            return
        }
        disposableList.add(
            module.requestMPMoreHomepageData(currentLine!!)
                .map(moreDataMapper)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe({
                    MLog.info(TAG, "requestMoreHomePageData: $it")

                    if (it.isNotEmpty()) {
                        view?.refreshOrAddRecyclerView(false, it)
                    }
                    view?.finishRefreshAnimation(true, 100)
                }, {
                    MLog.error(TAG, "requestMoreHomePageData: error = $it")
                    view?.finishRefreshAnimation(true, 100)
                })
        )
    }

    fun destroy() {
        view = null
        disposableList.forEach {
            RxUtils.dispose(it)
        }
    }

    companion object {
        const val TAG = "HomeContentPresenter"
    }
}