package com.tencent.imsdk.community;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.group.GroupManager;
import com.tencent.imsdk.manager.BaseManager;
import java.util.HashMap;
import java.util.List;

public class CommunityManager {
    private CommunityListener mInternalCommunityListener;
    private CommunityListener mCommunityListener;

    private static class CommunityManagerHolder {
        private static final CommunityManager communityManager = new CommunityManager();
    }

    public static CommunityManager getInstance() {
        return CommunityManagerHolder.communityManager;
    }

    public void init() {
        initCommunityListener();
    }

    private void initCommunityListener() {
        if (mInternalCommunityListener == null) {
            mInternalCommunityListener = new CommunityListener() {
                @Override
                public void onCreateTopic(final String groupID, final String topicID) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onCreateTopic(groupID, topicID);
                            }

                            GroupManager.getInstance().notifyTopicCreated(groupID, topicID);
                        }
                    });
                }

                @Override
                public void onDeleteTopic(final String groupID, final List<String> topicIDList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onDeleteTopic(groupID, topicIDList);
                            }

                            GroupManager.getInstance().notifyTopicDeleted(groupID, topicIDList);
                        }
                    });
                }

                @Override
                public void onChangeTopicInfo(final String groupID, final TopicInfo topicInfo) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onChangeTopicInfo(groupID, topicInfo);
                            }

                            GroupManager.getInstance().notifyTopicInfoChanged(groupID, topicInfo);
                        }
                    });
                }

                @Override
                public void onReceiveTopicRESTCustomData(final String topicID, final byte[] customData) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onReceiveTopicRESTCustomData(topicID, customData);
                            }

                            GroupManager.getInstance().notifyReceiveRESTCustomData(topicID, customData);
                        }
                    });
                }

                @Override
                public void onCreatePermissionGroup(
                    final String groupID, final PermissionGroupInfo permissionGroupInfo) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onCreatePermissionGroup(groupID, permissionGroupInfo);
                            }
                        }
                    });
                }

                @Override
                public void onDeletePermissionGroup(final String groupID, final List<String> permissionGroupIDList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onDeletePermissionGroup(groupID, permissionGroupIDList);
                            }
                        }
                    });
                }

                @Override
                public void onChangePermissionGroupInfo(
                    final String groupID, final PermissionGroupInfo permissionGroupInfo) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onChangePermissionGroupInfo(groupID, permissionGroupInfo);
                            }
                        }
                    });
                }

                @Override
                public void onAddMembersToPermissionGroup(
                    final String groupID, final String permissionGroupID, final List<String> memberIDList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onAddMembersToPermissionGroup(
                                    groupID, permissionGroupID, memberIDList);
                            }
                        }
                    });
                }

                @Override
                public void onRemoveMembersFromPermissionGroup(
                    final String groupID, final String permissionGroupID, final List<String> memberIDList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onRemoveMembersFromPermissionGroup(
                                    groupID, permissionGroupID, memberIDList);
                            }
                        }
                    });
                }

                @Override
                public void onAddTopicPermission(final String groupID, final String permissionGroupID,
                    final HashMap<String, Long> topicPermissionMap) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onAddTopicPermission(groupID, permissionGroupID, topicPermissionMap);
                            }
                        }
                    });
                }

                @Override
                public void onDeleteTopicPermission(
                    final String groupID, final String permissionGroupID, final List<String> topicIDList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onDeleteTopicPermission(groupID, permissionGroupID, topicIDList);
                            }
                        }
                    });
                }

                @Override
                public void onModifyTopicPermission(final String groupID, final String permissionGroupID,
                    final HashMap<String, Long> topicPermissionMap) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mCommunityListener != null) {
                                mCommunityListener.onModifyTopicPermission(
                                    groupID, permissionGroupID, topicPermissionMap);
                            }
                        }
                    });
                }
            };
        }

        nativeSetCommunityListener(mInternalCommunityListener);
    }

    public void setCommunityListener(CommunityListener communityListener) {
        this.mCommunityListener = communityListener;
    }

    public void createTopic(String groupID, TopicInfo topicInfo, final IMCallback<String> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeCreateTopic(groupID, topicInfo, callback);
    }

    public void deleteTopic(String groupID, List<String> topicIDList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeleteTopic(groupID, topicIDList, callback);
    }

    public void setTopicInfo(TopicInfoModifyParam param, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetTopicInfo(param, callback);
    }

    public void getTopicList(String groupID, List<String> topicIdList, IMCallback<List<TopicInfoGetResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetTopicList(groupID, topicIdList, callback);
    }

    public void createPermissionGroupInCommunity(PermissionGroupInfo permissionGroupInfo, IMCallback<String> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeCreatePermissionGroupInCommunity(permissionGroupInfo, callback);
    }

    public void deletePermissionGroupFromCommunity(
        String groupID, List<String> permissionGroupIDList, IMCallback<List<PermissionGroupOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeletePermissionGroupFromCommunity(groupID, permissionGroupIDList, callback);
    }

    public void modifyPermissionGroupInfoOfCommunity(PermissionGroupInfoModifyParam param, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeModifyPermissionGroupInfoInCommunity(param, callback);
    }

    public void getJoinedPermissionGroupListOfCommunity(
            String groupID, IMCallback<List<PermissionGroupInfoGetResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetJoinedPermissionGroupListInCommunity(groupID, callback);
    }

    public void getPermissionGroupListOfCommunity(
        String groupID, List<String> permissionGroupIDList, IMCallback<List<PermissionGroupInfoGetResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetPermissionGroupListInCommunity(groupID, permissionGroupIDList, callback);
    }

    public void addCommunityMembersToPermissionGroup(String groupID, String permissionGroupID,
        List<String> memberIDList, IMCallback<List<PermissionGroupMemberOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeAddCommunityMembersToPermissionGroup(groupID, permissionGroupID, memberIDList, callback);
    }

    public void removeCommunityMembersFromPermissionGroup(String groupID, String permissionGroupID,
        List<String> memberIDList, IMCallback<List<PermissionGroupMemberOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeRemoveCommunityMembersFromPermissionGroup(groupID, permissionGroupID, memberIDList, callback);
    }

    public void getCommunityMemberListOfPermissionGroup(String groupID, String permissionGroupID, String nextCursor,
        IMCallback<PermissionGroupMemberInfoResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetCommunityMemberListInPermissionGroup(groupID, permissionGroupID, nextCursor, callback);
    }

    public void addTopicPermissionToPermissionGroup(String groupID, String permissionGroupID,
        HashMap<String, Long> topicPermissionMap, IMCallback<List<TopicOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeAddTopicPermissionToPermissionGroup(groupID, permissionGroupID, topicPermissionMap, callback);
    }

    public void deleteTopicPermissionToPermissionGroup(String groupID, String permissionGroupID,
        List<String> topicIDList, IMCallback<List<TopicOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeleteTopicPermissionFromPermissionGroup(groupID, permissionGroupID, topicIDList, callback);
    }

    public void modifyTopicPermissionOfPermissionGroup(String groupID, String permissionGroupID,
        HashMap<String, Long> topicPermissionMap, IMCallback<List<TopicOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeModifyTopicPermissionInPermissionGroup(groupID, permissionGroupID, topicPermissionMap, callback);
    }

    public void getTopicPermissionOfPermissionGroup(String groupID, String permissionGroupID, List<String> topicIDList,
        IMCallback<List<TopicPermissionResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetTopicPermissionInPermissionGroup(groupID, permissionGroupID, topicIDList, callback);
    }

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    protected native void nativeSetCommunityListener(CommunityListener listener);

    protected native void nativeCreateTopic(String groupID, TopicInfo topicInfo, IMCallback callback);

    protected native void nativeDeleteTopic(String groupID, List<String> topicIDList, IMCallback callback);

    protected native void nativeSetTopicInfo(TopicInfoModifyParam param, IMCallback callback);

    protected native void nativeGetTopicList(String groupID, List<String> topicIDList, IMCallback callback);

    protected native void nativeCreatePermissionGroupInCommunity(
        PermissionGroupInfo permissionGroupInfo, IMCallback callback);

    protected native void nativeDeletePermissionGroupFromCommunity(
        String groupID, List<String> permissionGroupIDList, IMCallback callback);

    protected native void nativeModifyPermissionGroupInfoInCommunity(
        PermissionGroupInfoModifyParam param, IMCallback callback);

    protected native void nativeGetJoinedPermissionGroupListInCommunity(String groupID, IMCallback callback);

    protected native void nativeGetPermissionGroupListInCommunity(
        String groupID, List<String> permissionGroupIDList, IMCallback callback);

    protected native void nativeAddCommunityMembersToPermissionGroup(
        String groupID, String permissionGroupID, List<String> memberIDList, IMCallback callback);

    protected native void nativeRemoveCommunityMembersFromPermissionGroup(
        String groupID, String permissionGroupID, List<String> memberIDList, IMCallback callback);

    protected native void nativeGetCommunityMemberListInPermissionGroup(
        String groupID, String permissionGroupID, String nextCursor, IMCallback callback);

    protected native void nativeAddTopicPermissionToPermissionGroup(
        String groupID, String permissionGroupID, HashMap<String, Long> topicPermissionMap, IMCallback callback);

    protected native void nativeDeleteTopicPermissionFromPermissionGroup(
        String groupID, String permissionGroupID, List<String> topicIDList, IMCallback callback);

    protected native void nativeModifyTopicPermissionInPermissionGroup(
        String groupID, String permissionGroupID, HashMap<String, Long> topicPermissionMap, IMCallback callback);

    protected native void nativeGetTopicPermissionInPermissionGroup(
        String groupID, String permissionGroupID, List<String> topicIDList, IMCallback callback);
}
