package com.joyy.hagorpc.internal

import com.joyy.hagorpc.IRPCNotify
import com.joyy.hagorpc.IRegisterId
import com.joyy.hagorpc.MSGTYPE
import com.joyy.hagorpc.NotifyBundle
import com.joyy.hagorpc.RPCPacketV2
import com.joyy.hagorpc.RPCPacketV3
import com.joyy.hagorpc.ResponseBundle
import com.joyy.hagorpc.SafeRunnable
import com.joyy.hagorpc.WsCode
import com.joyy.hagorpc.WsStatus
import com.joyy.hagorpc.birdge.ExecutorBridge
import com.joyy.hagorpc.birdge.GzipBridge
import com.joyy.hagorpc.birdge.LoggerBridge
import com.joyy.hagorpc.birdge.RPCPacketBridge
import com.joyy.hagorpc.birdge.RunnableBridge
import com.joyy.hagorpc.birdge.SystemBridge
import com.joyy.hagorpc.birdge.SystemClockBridge
import kotlin.jvm.Synchronized

internal class WsCallbackProxy : RPCCore.WsCallback {

    companion object {
        private const val TAG = "RPCCore.WsCallbackProxy"
    }

    var callback: RPCCore.WsCallback? = null
    private var mHeartBeatTask: HeartBeatTask? = null
    var mLastTimeoutCheck: Long = 0
    val mLastTimeoutCheckTrace = StringBuilder()

    private val mNotifyDispatcher: AbsRPCDispatcher by lazy {
        RPCDispatcherImpl()
    }

    private val mFrequencyHandler: INotifyHandler by lazy {
        NotifyFrequencyHandler(object : INotifyHandler {
            override fun notify(bundle: NotifyBundle) {
                // 也有可能是第三方分发，所以两个分发器都处理
                mNotifyDispatcher.notify(bundle)
            }

            override fun notify(bundleList: List<NotifyBundle>) {
                // 也有可能是第三方分发，所以两个分发器都处理
                mNotifyDispatcher.notify(bundleList)
            }
        })
    }

    // 用于检查超时的任务
    private val mCheckTimeoutTask = RunnableBridge {
        ExecutorBridge.getAsyncExecutor().execute(SafeRunnable {
            handleTraceForTimeoutPoll()
            handleRequestForTimeoutPoll()
        })

        triggerTimeoutTask()
    }

    /**
     * 触发定时检查超时任务
     */
    fun triggerTimeoutTask() {
        ExecutorBridge.getAsyncExecutor().let {
            it.cancel(mCheckTimeoutTask)
            it.executeDelay(mCheckTimeoutTask, RPCConst.DEFAULT_TASKS_POLLING)
        }
    }

    fun register(id: IRegisterId, notify: IRPCNotify) {
        mNotifyDispatcher.register(id, notify)

    }

    fun unregister(id: IRegisterId, notify: IRPCNotify) {
        mNotifyDispatcher.unregister(id, notify)
    }

    /**
     * 开启心跳任务
     */
    @Synchronized
    fun startHeartBeat(url: String, from: Int? = null) {
        val oldTask = mHeartBeatTask
        if (oldTask != null) {
            oldTask.stop()
            ExecutorBridge.getAsyncExecutor().cancel(oldTask)
        }

        // 开启这一次的心跳逻辑
        val task = HeartBeatTask(url, from)
        mHeartBeatTask = task

        LoggerBridge.logI(TAG, "heart beat task start, url: $url, seq: ${task.mCliSeq}")
        ExecutorBridge.getAsyncExecutor().execute(task)
    }

    override fun onConnectSucceed(wsUrl: String) {
        LoggerBridge.logI(TAG, "websocket onConnectSucceed")
        startHeartBeat(wsUrl)
        updateStatus(RPCCore.status, null, null)

        ExecutorBridge.getAsyncExecutor().execute(SafeRunnable {
            handleAfterConnected()
        })
        callback?.onConnectSucceed(wsUrl)
    }

    override fun onReConnectMax(wsUrl: String) {
        ExecutorBridge.getAsyncExecutor().execute(SafeRunnable {
            checkResendForError(
                false,
                "connect web socket failed and had try with max time", WsCode.RECONNECT_MAX_TIME
            )
        })
        callback?.onReConnectMax(wsUrl)
    }

    override fun onConnecting(wsUrl: String) {
        LoggerBridge.logD(
            TAG, "websocket onConnecting status: ${RPCCore.status}, " +
                    "url: $wsUrl"
        )

        updateStatus(RPCCore.status, null, null)
        callback?.onConnecting(wsUrl)
    }

    override fun onResponse(wsUrl: String, bytes: ByteArray) {
        val resTimestamp = SystemBridge.currentTimeMillis()
        ExecutorBridge.getAsyncExecutor().execute(SafeRunnable {
            innerHandleResponse(wsUrl, bytes, resTimestamp)
        })
        callback?.onResponse(wsUrl, bytes)
    }

    override fun onError(wsUrl: String, code: Int, reason: String) {
        // 未建连成功且code是网络异常的场景下，不回调error，而是让任务继续在等待队列等待
        val needCallbackError = RPCCore.status == WsStatus.CONNECT_SUCCESS ||
                code != WsCode.NETWORK_FAILURE_CODE

        LoggerBridge.logD(
            TAG, "websocket onError, status: ${RPCCore.status}, " +
                    "code: $code, reason: $reason"
        )

        updateStatus(RPCCore.status, code, reason)

        if (code >= WsCode.KICKOFF_CODE &&
            code < WsCode.TOKEN_PARSE_FAIL
        ) {
            LoggerBridge.logE(
                TAG,
                "WsClient onError, code:$code"
            )
        }

        if (needCallbackError) {
            ExecutorBridge.getAsyncExecutor().execute(SafeRunnable {
                checkResendForError(
                    false,
                    "web socket call onClosed or onFailure, " +
                            "for more detail to see WebSocketListener#onClosed or " +
                            "WebSocketListener#onFailure",
                    code
                )
            })
        } else {
            LoggerBridge.logD(
                TAG,
                "web socket onError but not callback, code: $code, info: $reason",
            )
        }
        callback?.onError(wsUrl, code, reason)
    }

    override fun onDisconnect(wsUrl: String, code: Int, reason: String) {
        LoggerBridge.logI(
            TAG,
            "onDisconnect status: ${RPCCore.status}, code: $code, wsUrl: $wsUrl"
        )

        updateStatus(RPCCore.status, code, reason)

        ExecutorBridge.getAsyncExecutor().execute(SafeRunnable {
            checkResendForError(
                false,
                "connect web socket was close by call WebSocket.close", code
            )
        })
        callback?.onDisconnect(wsUrl, code, reason)
    }

    private fun updateStatus(status: WsStatus, reasonCode: Int?, reasonInfo: String?) {
        LoggerBridge.logI(
            TAG,
            "update state: $status, reasonCode: $reasonCode, reasonInfo: $reasonInfo"
        )
        if (status == WsStatus.CONNECT_SUCCESS) {
            // 长连接成功时处理
            triggerTimeoutTask()
        }
    }

    /**
     * 取消定时检查超时任务
     */
    private fun cancelTimeoutTask() {
        ExecutorBridge.getAsyncExecutor().cancel(mCheckTimeoutTask)
    }

    /**
     * 处理统计跟踪（超时轮询触发的）
     */
    private fun handleTraceForTimeoutPoll() {
        if (mLastTimeoutCheck > 0) {
            val cur = SystemClockBridge.uptimeMillis()
            val consume: Long = cur - mLastTimeoutCheck
            if (consume > 10000 && consume < Int.MAX_VALUE) {
                val logString = mLastTimeoutCheckTrace.toString()
                LoggerBridge.logD(TAG, "lastTimeoutCheckTrace: $logString")
                RPCCore.mEventListener.onPollTimeout(consume, logString)

                mLastTimeoutCheckTrace.deleteRange(0, mLastTimeoutCheckTrace.length)
                mLastTimeoutCheckTrace.append(if (RPCCore.mIsBackground.value) "_back" else "_fore")
                mLastTimeoutCheck = SystemClockBridge.uptimeMillis()
            }
        } else {
            mLastTimeoutCheckTrace.deleteRange(0, mLastTimeoutCheckTrace.length)
            mLastTimeoutCheckTrace.append(if (RPCCore.mIsBackground.value) "_back" else "_fore")
            mLastTimeoutCheck = SystemClockBridge.uptimeMillis()
        }
    }

    /**
     * 处理等待队列的任务（超时轮询触发的）
     */
    private fun handleRequestForTimeoutPoll() {
        LoggerBridge.logD(TAG, "check timeout request")
        val requestMap = RPCCore.mRequestMap.value
        val requestList = requestMap.toList()
        if (requestList.isEmpty()) {
            LoggerBridge.logD(TAG, "check timeout request ignore, no waiting request")
            return
        }
        val curTime = SystemBridge.currentTimeMillis()
        for ((_, request) in requestList) {
            // 定期队列轮询需要排除采用短连接的任务，因为短连接任务的超时由grace来处理，无需这里处理
            if (request.isHttpRequesting) {
                continue
            }

            val timeCheck = curTime - request.sendTime
            request.trace += "_checktime($timeCheck)"

            val timeout = if (request.getTimeout() > 0) {
                request.getTimeout()
            } else {
                RPCConst.DEFAULT_TASK_TIMEOUT
            }
            // 如果没有超时则过
            if (timeCheck < timeout) {
                continue
            }
            onRetryWhenTimeout(request, true)
        }
    }

    /**
     * 处理建连成功后的行为
     */
    private fun handleAfterConnected() {
        if (!RPCCore.mResendIfConnected) {
            LoggerBridge.logI(TAG, "no need to resend request after ws connected")
            return
        }

        val cur = SystemBridge.currentTimeMillis()
        val requestMap = RPCCore.mRequestMap.value
        val requestList = requestMap.toList()
        if (requestList.isNotEmpty()) {
            for ((_, request) in requestList) {
                // 长连接建连成功的时候，重发等待队列的任务，但是要过滤短连接的任务（短连任务会根据结果来判断是否重试）
                if (request.isHttpRequesting || request.sendSuccess) {
                    continue
                }
                if (cur - request.sendTime > RPCConst.CONNECT_RESEND_MAX_LIMIT && request.sendTime > 0) {
                    continue
                }
                request.send()
            }
        }
    }

    /**
     * 检测是否需要重试当连接失败场景
     */
    private fun checkResendForError(canRetry: Boolean, reason: String?, code: Int) {
        val requestMap = RPCCore.mRequestMap.value
        val requestList = requestMap.toList()
        for ((_, request) in requestList) {
            // 长连接失败场景这里需要做下判断，排除那些使用短连接请求的任务
            if (!request.isHttpRequesting) {
                onRetryWhenError(request, canRetry, reason, code)
            }
        }
    }

    private fun innerHandleResponse(
        wsUrl: String,
        resData: ByteArray?,
        responseTimestamp: Long
    ) {
        if (resData == null) {
            LoggerBridge.logE(TAG, "rpc receive response data ignore, data is null")
            return
        }
        var payload: ByteArray? = resData
        val resLength: Long = resData.size.toLong()
        if (RPCCore.mUseGZip.invoke(wsUrl)) {
            payload = GzipBridge.gzipPack(resData)
        }
        if (payload == null) {
            payload = resData
        }

        val finalProtocol = tryParseForInnerProtocol(payload, responseTimestamp)
        val finalHeader = finalProtocol?.header
        if (finalHeader == null) {
            LoggerBridge.logE(
                TAG,
                "parse rpc response inner proto fail, callback will not be called, until queue task timeout"
            )
            return
        }

        LoggerBridge.logI(TAG, "rpc receive response: header: $finalHeader")

        if (finalHeader.version.isNotEmpty()) {
            payload = finalProtocol.payload
            if (finalHeader.msgtype == MSGTYPE.MSGTYPE_NOTICE) {
                innerResponseForBroadcast(
                    ResponseBundle(
                        finalProtocol,
                        payload,
                        resLength,
                        responseTimestamp
                    )
                )
            } else if (finalHeader.msgtype == MSGTYPE.MSGTYPE_RESP) {
                innerResponseForRpc(
                    ResponseBundle(
                        finalProtocol,
                        payload,
                        resLength,
                        responseTimestamp
                    )
                )
            }
        } else {
            innerResponseForNoVer(
                ResponseBundle(
                    finalProtocol,
                    payload,
                    resLength,
                    responseTimestamp
                )
            )
        }
    }

    private fun onRetryWhenTimeout(request: RPCRequest, canRetry: Boolean): Boolean {
        LoggerBridge.logI(
            TAG,
            "onRetryWhenTimeout request: $request, canRetry: $canRetry, retryCount: ${request.retryCount}"
        )

        var actualCanRetry = canRetry
        // 是否超过最大重试次数
        if (canRetry && request.retryCount >= RPCCore.mRequestMaxRetry) {
            actualCanRetry = false
        }
        val toRetry =
            request.onError(actualCanRetry, true, WsCode.UNKNOWN_CODE, "request timeout")
        if (actualCanRetry && toRetry) {
            request.resend()
            return true
        }
        return false
    }

    private fun onRetryWhenError(
        request: RPCRequest,
        canRetry: Boolean,
        reason: String?,
        code: Int
    ): Boolean {

        LoggerBridge.logI(
            TAG,
            "onRetryWhenError error: $reason, code: $code, " +
                    "request: $request"
        )

        var actualCanRetry = canRetry
        // 是否超过最大重试次数
        if (canRetry && request.retryCount >= RPCCore.mRequestMaxRetry) {
            actualCanRetry = false
        }
        val toRetry = request.onError(actualCanRetry, false, code, reason ?: "")
        if (actualCanRetry && toRetry) {
            request.resend()
            return true
        }
        return false
    }

    private fun tryParseForInnerProtocol(
        payload: ByteArray,
        responseTimestamp: Long
    ): RPCPacketV2? {
        var innerV2 = RPCPacketBridge.fromInnerV2(payload)
        val uid = RPCCore.mUid.invoke()
        LoggerBridge.logD(
            TAG,
            "rpc try parse proto innerV2"
        )
        if (innerV2?.header == null) {
            innerV2 = null
            val innerV3 = RPCPacketBridge.fromInnerV3(payload)
            LoggerBridge.logD(TAG, "rpc try parse proto innerV3: $innerV3")
            // 约定的魔法数，如果魔法数是 0x4861676eL 则认为是V3协议
            if (innerV3?.magic == RPCConst.INNERV3_MAGIC) {
                if (innerV3.msgType == RPCPacketV3.EMsgType.MSG_DOWNSTREAM_RELIABLE.value) {
                    val localSeqId = RPCLocalPersistence.getBroadcastSeq(uid)
                    if (localSeqId != 0L && localSeqId > innerV3.seqId) {
                        // duplicated broadcast, drop it
                        LoggerBridge.logD(
                            TAG,
                            "rpc parse innerV3 duplicated broadcast, drop it localSeqId: $localSeqId, " +
                                    "innerV3 seqId: ${innerV3.seqId}"
                        )
                        return null
                    }
                    // 更新seqId
                    RPCLocalPersistence.updateBroadcastSeq(uid, innerV3.seqId)
                }
                if (innerV3.version == RPCPacketV3.EMsgUnpackType.MSG_UNPACK_INNERV2.value) {
                    // 解InnerV2
                    innerV2 = RPCPacketBridge.fromInnerV2(innerV3.payload)
                    LoggerBridge.logD(
                        TAG,
                        "rpc parse innerV3 unpack innerV2"
                    )
                    if (innerV2 == null) {
                        RPCCore.mEventListener.onInnerV3UnpackFail(responseTimestamp, innerV3.seqId)
                    }
                } // drop
            } // drop
        }
        return innerV2
    }

    /**
     * 处理长连接返回的通知类消息
     */
    private fun innerResponseForBroadcast(bundle: ResponseBundle) {
        mFrequencyHandler.notify(NotifyBundle(bundle.protocol, bundle.responseBytes))
    }

    private fun getRequest(seqId: Long): RPCRequest? {
        val requestMap = RPCCore.mRequestMap.value
        return requestMap[seqId]
    }


    /**
     * 处理长连接的对于response类型的响应
     */
    private fun innerResponseForRpc(bundle: ResponseBundle) {
        val header = bundle.protocol.header
        val request = getRequest(header.seqid)
        if (header.code != 0L) {
            // 新协议 header 中code非0表示后台服务器框架出错
            LoggerBridge.logE(
                TAG, "rpc receive response code: ${header.code}, some error in service, " +
                        "tid: ${header.tid}"
            )

            if (request != null) {
                onRetryWhenError(
                    request, true,
                    "rpc receive some error in service",
                    header.code.toInt()
                )
            }
            return
        }

        if (request == null) {
            LoggerBridge.logE(
                TAG, "rpc handle response ignore, not found request"
            )
            return
        }
        val reqInner = request.packet
        if (header.msgtype == MSGTYPE.MSGTYPE_RESP && header.method == reqInner.header.method && header.sname == reqInner.header.sname
        ) {
            // 路由字段透传
            header.routingKey?.let {
                if (header.routingKey.isNotEmpty()) {
                    val routingKey = header.routingKey.decodeToString()
                    LoggerBridge.logD(
                        TAG,
                        "rpc receive response to update routing key: $routingKey"
                    )
                    RPCLocalPersistence.updateCProxyRouting(header.sname, routingKey)
                }
            }
            request.onSuccess(bundle)
        } else {
            val reason = "rpc receive response same seqId but sname or method not equals"

            request.onError(
                canRetry = true,
                fromTimeout = false,
                code = WsCode.UNKNOWN_CODE,
                reason = reason
            )

            LoggerBridge.logE(TAG, reason)
        }
    }

    /**
     * 处理长连接的对于没有版本参数类的消息
     */
    private fun innerResponseForNoVer(bundle: ResponseBundle) {
        val header = bundle.protocol.header
        val seqId = header.seqid
        val request = getRequest(seqId)
        val reqProtocol = request?.packet
        if (reqProtocol?.header != null &&
            header.sname == reqProtocol.header.sname &&
            reqProtocol.uri + 1 == bundle.protocol.uri
        ) { // 请求对应的放回
            header.routingKey?.let {
                if (header.routingKey.isNotEmpty()) {
                    val routingKey = header.routingKey.decodeToString()
                    LoggerBridge.logD(
                        TAG,
                        "rpc receive response to update routing key: $routingKey"
                    )
                    RPCLocalPersistence.updateCProxyRouting(header.sname, routingKey)
                }
            }
            request.onSuccess(bundle)
        } else {
            // 找不到对应的请求回调，可能是服务器主动通知，走广播通知
            innerResponseForBroadcast(bundle)
        }
    }
}
