package com.joyy.hagorpc.internal

/**
 * 可回收和复用的数据结构（减少不必要的重复对象创建）
 * Created by wjh on 2021/10/25
 */
abstract class RecyclableData {

    abstract class RecyclablePool {

        private var sPoolSize = 0

        private var sPoolObj: RecyclableData? = null

        protected open fun getMaxSize(): Int {
            return 10
        }

        abstract fun getLock(): Any

        fun find(): RecyclableData? {

            synchronized(getLock()) {

                if (sPoolObj != null && sPoolSize > 0) {
                    val result = sPoolObj ?: return null
                    result.handleRecycle()
                    sPoolObj = result.mNext
                    result.setNext(null)
                    result.setRecycle(false)
                    sPoolSize--
                    return result
                }
            }
            return null
        }

        fun recycle(data: RecyclableData) {
            synchronized(getLock()) {
                data.handleRecycle()
                data.setRecycle(true)
                if (sPoolSize < getMaxSize()) {
                    data.setNext(sPoolObj)
                    sPoolObj = data
                    sPoolSize++
                }
            }
        }
    }

    private var mRecycle: Boolean = false
    private var mNext: RecyclableData? = null

    protected abstract fun handleRecycle()

    protected abstract fun getPool(): RecyclablePool

    @Synchronized
    private fun setRecycle(isRecycle: Boolean) {
        mRecycle = isRecycle
    }

    @Synchronized
    private fun setNext(next: RecyclableData?) {
        mNext = next
    }

    @Synchronized
    fun hasNext(): Boolean {
        return mNext != null
    }

    @Synchronized
    fun isRecycled(): Boolean {
        return mRecycle
    }

    fun recycle() {
        getPool().recycle(this)
    }
}